/*-
 * Copyright (c) 2006 Allan Saddi <allan@saddi.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#include <Python.h>

#include <assert.h>

#include "ajp.h"
#include "wsgi-int.h"

int unquoteURL = 1;
int multiprocess = 0;

/* Populate WSGI environ dictionary */
int
wsgiPopulateEnviron(Request *self)
{
  AJPRequest *req = ((AJPContext *)self->context)->activeReq;
  PyObject *environ = self->environ;
  PyObject *val;
  int ret, i, j, k;
  const char *pathInfo, *req_uri = NULL, *envScriptName = NULL, *scriptName;
  int envScriptNameLen = 0, scriptNameLen;
  int result = -1;
  char buf[256];

  if (wsgiPutEnv(self, "REQUEST_METHOD", req->method))
    return -1;
  if (wsgiPutEnv(self, "SERVER_PROTOCOL", req->protocol))
    return -1;

  if (unquoteURL) {
    if ((req_uri = wsgiUnquote(req->uri)) == NULL) {
      PyErr_NoMemory();
      return -1;
    }
  }

  if (wsgiPutEnv(self, "REQUEST_URI", req_uri ? req_uri : req->uri))
    goto bad;
  if (wsgiPutEnv(self, "REMOTE_ADDR", req->remoteAddr))
    goto bad;
  if (wsgiPutEnv(self, "REMOTE_HOST", req->remoteHost))
    goto bad;
  if (wsgiPutEnv(self, "SERVER_NAME", req->serverName))
    goto bad;
  snprintf (buf, sizeof(buf), "%d", req->serverPort);
  if (wsgiPutEnv(self, "SERVER_PORT", buf))
    goto bad;

  /* HTTP headers */
  for (i = 0; i < req->numHeaders; i++) {
    const char *header = req->headers[2 * i];
    const char *value = req->headers[2 * i + 1];

    /* Copy/convert header name */
    strcpy(buf, "HTTP_");

    k = strlen(header);
    if (k > (sizeof(buf) - 6))
      k = sizeof(buf) - 6;
    for (j = 0; j < k; j++) {
      if (header[j] == '-')
	buf[5 + j] = '_';
      else
	buf[5 + j] = toupper(header[j]);
    }
    buf[5 + j] = '\0';

    if (!strcmp(buf, "HTTP_CONTENT_TYPE") ||
	!strcmp(buf, "HTTP_CONTENT_LENGTH")) {
      /* Strip HTTP_ */
      if (wsgiPutEnv(self, &buf[5], value))
	goto bad;
    }
    else {
      if (wsgiPutEnv(self, buf, value))
	goto bad;
    }
  }

  /* Attributes */
  for (i = 0; i < req->numAttributes; i++) {
    const char *name = req->attributes[2 * i];
    const char *value = req->attributes[2 * i + 1];

    /* Special handling of WSGI_SCRIPT_NAME */
    if (!strcmp(name, "WSGI_SCRIPT_NAME")) {
      envScriptName = value;
      envScriptNameLen = strlen(value);
    }

    if (wsgiPutEnv(self, name, value))
      goto bad;
  }

  /* SSL_KEY_SIZE attribute is special */
  if (req->sslKeySize) {
    snprintf(buf, sizeof(buf), "%d", req->sslKeySize);
    if (wsgiPutEnv(self, "SSL_KEY_SIZE", buf))
      goto bad;
  }

  /* wsgi.version */
  if ((val = Py_BuildValue("(ii)", 1, 0)) == NULL)
    goto bad;
  ret = PyDict_SetItemString(environ, "wsgi.version", val);
  Py_DECREF(val);
  if (ret)
    goto bad;

  /* wsgi.input */
  if (PyDict_SetItemString(environ, "wsgi.input", self->input))
    goto bad;

  /* wsgi.errors */
  if (PyDict_SetItemString(environ, "wsgi.errors", wsgiStderr))
    goto bad;

  /* wsgi.multithread */
  if (PyDict_SetItemString(environ, "wsgi.multithread", multiprocess ? Py_False : Py_True))
    goto bad;

  /* wsgi.multiprocess */
  if (PyDict_SetItemString(environ, "wsgi.multiprocess", multiprocess ? Py_True : Py_False))
    goto bad;

  /* wsgi.run_once */
  if (PyDict_SetItemString(environ, "wsgi.run_once", Py_False))
    goto bad;

  /* wsgi.url_scheme */
  if (wsgiPutEnv(self, "wsgi.url_scheme", req->isSSL ? "https" : "http"))
    goto bad;

  if (PyDict_SetItemString(environ, "wsgi.file_wrapper",
			   (PyObject *)&FileWrapper_Type))
    goto bad;

  /* Decide which SCRIPT_NAME to use */
  scriptName = wsgiScriptName;
  scriptNameLen = wsgiScriptNameLen;
  if (!wsgiScriptNameLen && envScriptNameLen) {
    /* wsgiInit scriptName always takes priority */
    scriptName = envScriptName;
    scriptNameLen = envScriptNameLen;
  }

  /* SCRIPT_NAME/PATH_INFO */
  pathInfo = req_uri ? req_uri : req->uri;
  if (scriptNameLen) {
    if (!strncmp(pathInfo, scriptName, scriptNameLen))
      pathInfo = &pathInfo[scriptNameLen];
    else
      fprintf (stderr, "WARNING: SCRIPT_NAME does not match REQUEST_URI\n");
  }

  if (wsgiPutEnv(self, "PATH_INFO", pathInfo))
    goto bad;

  if (wsgiPutEnv(self, "SCRIPT_NAME", scriptName))
    goto bad;

  /* Ensure QUERY_STRING is set */
  if (!PyDict_GetItemString(self->environ, "QUERY_STRING")) {
    if (wsgiPutEnv(self, "QUERY_STRING", ""))
      goto bad;
  }

  result = 0;

 bad:
  if (req_uri)
    PyMem_Free((void *)req_uri);
  return result;
}

/* Map AJP errors to Python errors. */
static int
_wsgiMapError(int error)
{
   assert(error < 0);

   switch (error) {
   case AJP_MEMORY_ERROR:
     PyErr_SetString(PyExc_MemoryError, "AJP: Out of memory");
     break;
   case AJP_SOCKET_ERROR:
     PyErr_SetString(PyExc_IOError, "AJP: Socket read/write error");
     break;
   case AJP_BUFFER_ERROR:
     PyErr_SetString(PyExc_IndexError, "AJP: Buffer overrun (protocol error)");
     break;
   case AJP_PROTOCOL_ERROR:
     PyErr_SetString(PyExc_ValueError, "AJP: Protocol error");
     break;
   case AJP_FILE_ERROR:
     PyErr_SetString(PyExc_IOError, "AJP: File error");
     break;
   default:
     PyErr_SetString(PyExc_RuntimeError, "AJP: Unknown error");
     break;
   }
   return -1;
}

/* Send HTTP response headers. */
int
wsgiSendHeaders(void *ctxt, int statusCode, const char *statusMsg,
		int headerCount, const char *headers[])
{
  int ret;

  Py_BEGIN_ALLOW_THREADS
  ret = ajpSendHeaders(ctxt, statusCode, statusMsg, headerCount, headers);
  Py_END_ALLOW_THREADS

  if (ret) return _wsgiMapError(ret);

  return 0;
}

/* Send HTTP response body. */
int
wsgiSendBody(void *ctxt, const uint8_t *data, size_t len)
{
  int pos = 0, size, ret = AJP_OK;

  Py_BEGIN_ALLOW_THREADS
  while (pos < len && ret == AJP_OK) {
    size = len - pos;
    if (size > AJP_MAX_BODY_CHUNK_SIZE)
      size = AJP_MAX_BODY_CHUNK_SIZE;
    ret = ajpSendBodyChunk(ctxt, &data[pos], size);
    pos += size;
  }
  Py_END_ALLOW_THREADS

  if (ret) return _wsgiMapError(ret);

  return 0;
}

/* Send a file as the HTTP response body. */
int
wsgiSendFile(void *ctxt, int fd)
{
  int ret;

  Py_BEGIN_ALLOW_THREADS
  ret = ajpSendFile(ctxt, fd);
  Py_END_ALLOW_THREADS

  if (ret) return _wsgiMapError(ret);

  return 0;
}

/* Receive a request body chunk from the transport layer. */
int
wsgiGetBody(void *ctxt)
{
  int ret;

  Py_BEGIN_ALLOW_THREADS
  ret = ajpGetBodyChunk(ctxt);
  Py_END_ALLOW_THREADS

  if (ret) return _wsgiMapError(ret);

  return 0;
}

/* Primes the input stream, if needed by transport layer. (AJP sends the
   first request body chunk if Content-Length is non-zero.) */
int
wsgiPrimeInput(void *ctxt)
{
  int ret;

  if (((AJPContext *)ctxt)->activeReq->contentLength > 0) {
    Py_BEGIN_ALLOW_THREADS
    ret = ajpProcessInput(ctxt);
    Py_END_ALLOW_THREADS

    if (ret) return _wsgiMapError(ret);
  }
  return 0;
}

/* Set application-specific (WSGI) request-level data. */
void
wsgiSetRequestData(void *ctxt, void *data)
{
  ((AJPContext *)ctxt)->activeReq->data = data;
}

/* Get application-specific (WSGI) request-level data. */
void *
wsgiGetRequestData(void *ctxt)
{
  return ((AJPContext *)ctxt)->activeReq->data;
}

/* Return the Content-Length associated with this request. */
int
wsgiGetContentLength(void *ctxt)
{
  return ((AJPContext *)ctxt)->activeReq->contentLength;
}
