// ===========================================================================
// File: "aidaUtils.c"
//                        Created: 2010-08-09 21:56:18
//              Last modification: 2012-03-30 17:48:55
// Author: Bernard Desgraupes
// e-mail: <bdesgraupes@users.sourceforge.net>
// (c) Copyright: Bernard Desgraupes 2010-2012
// All rights reserved.
// ===========================================================================

#include "aidaMain.h"


// ------------------------------------------------------------------------
// 
// "aida_wrongNumArgs" --
// 
// Print message on stderr and exit if wrong number of arguments.
// 
// ------------------------------------------------------------------------
void aida_wrongNumArgs(const char * inCmd)
{
	aida_print_err("wrong number of arguments\n");
	aida_print_err("hint: type '%s help%s%s' for syntax\n", gPrgName, 
				strlen(inCmd) ? " ":"" , inCmd);
	aida_exit(1);
}


// ------------------------------------------------------------------------
// 
// "aida_open" --
// 
// Open a file in the specified mode and return a standard Tcl result
// indicating if the operation succeeded (TCL_OK) or not.
// 
// ------------------------------------------------------------------------
int
aida_open(const char *inName, const char *inMode, FILE ** outFile)
{
	int			result = TCL_OK;
	FILE *		fdesc;
	
	aida_verbose(3, "opening file '%s'\n", inName);	
	fdesc = fopen(inName, inMode);
	if (fdesc == NULL) {
		aida_print_err("error: can't open file '%s'\n", inName);
		result = TCL_ERROR;
	} else {
		*outFile = fdesc;
	}
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_setIncludeBase" --
// 
// Build the basename directory for an include file. Return the full path
// of the file. The caller is responsible for disposing of this Tcl_Obj.
// 
// ------------------------------------------------------------------------
Tcl_Obj * aida_setIncludeBase(char * inFile)
{
	int				len;
	Tcl_Obj 		*baseObj = NULL, *listObj, *splitObj, *fileObj, *pathObj, *dirObj;
	
	if (gCurrInclude != NULL && gCurrInclude->caller != NULL) {
		baseObj = gCurrInclude->caller->base;
	} else {
		baseObj = Tcl_FSGetCwd(gInterp);
	}
	
	// Construct new base
	fileObj = aida_externalToTclObj(inFile);
	Tcl_IncrRefCount(fileObj);
	
	if (Tcl_FSGetPathType(fileObj) == TCL_PATH_ABSOLUTE) {
		pathObj = fileObj;
	} else {
		listObj = Tcl_NewListObj(0, NULL);
		Tcl_IncrRefCount(listObj);
		Tcl_ListObjAppendElement(gInterp, listObj, baseObj);
		Tcl_ListObjAppendElement(gInterp, listObj, fileObj);
		pathObj = Tcl_FSJoinPath(listObj, 2);
		Tcl_DecrRefCount(listObj);
	}

	Tcl_IncrRefCount(pathObj);
	splitObj = Tcl_FSSplitPath(pathObj, &len);
	Tcl_IncrRefCount(splitObj);
	
	dirObj = Tcl_FSJoinPath(splitObj, len-1);
	Tcl_IncrRefCount(dirObj);
	gCurrInclude->base = dirObj;

	Tcl_DecrRefCount(splitObj);
	Tcl_DecrRefCount(fileObj);
	
	return pathObj;
}


// ------------------------------------------------------------------------
// 
// "aida_makeDString" --
// 
// Create a new DString, initialize it with the given string and return a
// pointer.
// 
// ------------------------------------------------------------------------
Tcl_DString *
aida_makeDString(char *inStr, int inlen)
{
	Tcl_DString *	dsPtr;
	
	dsPtr = (Tcl_DString *)malloc(sizeof(Tcl_DString));
	if (dsPtr == NULL) {
		aida_abort("fatal: can't alloc dynamic string\n");
	} 
	Tcl_DStringInit(dsPtr);
	if (inStr != NULL) {
		Tcl_DStringAppend(dsPtr, inStr, inlen);
	} 
	return dsPtr; 
}


// ------------------------------------------------------------------------
// 
// "aida_appendDString" --
// 
// Append text to the DString.
// 
// ------------------------------------------------------------------------
Tcl_DString *
aida_appendDString(Tcl_DString * inDst, char *inStr, int inlen)
{
	Tcl_DStringAppend(inDst, inStr, inlen);
	return inDst; 
}


// ------------------------------------------------------------------------
// 
// "aida_makeTclList" --
// 
// Start a Tcl list to hold string objects and install the first one (the
// grammar does not allow for empty lists).
// 
// ------------------------------------------------------------------------
Tcl_Obj *
aida_makeTclList(Tcl_Obj * inObj)
{
	Tcl_Obj * listObj;
	
	listObj = Tcl_NewListObj(0, NULL);
	Tcl_IncrRefCount(listObj);
	if (inObj != NULL) {
		Tcl_ListObjAppendElement(gInterp, listObj, inObj);
	} 
	
	return listObj;
}


// ------------------------------------------------------------------------
// 
// "aida_appendTclList" --
// 
// Augment the list obj with a new string obj.
// 
// ------------------------------------------------------------------------
Tcl_Obj *
aida_appendTclList(Tcl_Obj * inList, Tcl_Obj * inObj)
{
	if (inObj != NULL) {
		Tcl_ListObjAppendElement(gInterp, inList, inObj);
	} 
	return inList;
}


// ------------------------------------------------------------------------
// 
// "aida_makeTclDict" --
// 
// Start a Tcl dict to hold key/value pairs of string objects. The 'inObj'
// argument is a two-element Tcl list.
// 
// ------------------------------------------------------------------------
Tcl_Obj *
aida_makeTclDict(Tcl_Obj * inObj)
{
	Tcl_Obj * dictObj;
	
	dictObj = Tcl_NewDictObj();
	Tcl_IncrRefCount(dictObj);
	if (inObj != NULL) {
		dictObj = aida_appendTclDict(dictObj, inObj);
	} 
	
	return dictObj;
}


// ------------------------------------------------------------------------
// 
// "aida_appendTclDict" --
// 
// Augment the dict obj with a new key/value pair. The 'inObj' argument is
// a two-element Tcl list.
// 
// ------------------------------------------------------------------------
Tcl_Obj *
aida_appendTclDict(Tcl_Obj * inDict, Tcl_Obj * inPair)
{
	int			objLen, result;
	Tcl_Obj		*keyObj, *valObj;
	
	if (inPair != NULL) {
		Tcl_ListObjLength(gInterp, inPair, &objLen);
		if (objLen != 2) {
			aida_abort("bad key/value pair in attributes\n");
		} 
		
		Tcl_ListObjIndex(gInterp, inPair, 0, &keyObj);
		Tcl_ListObjIndex(gInterp, inPair, 1, &valObj);
		
		result = Tcl_DictObjPut(gInterp, inDict, keyObj, valObj);
		if (result != TCL_OK) {
			aida_abort("failed to build attributes dictionary\n");
		} 
	} 
	
	return inDict;
}


// ------------------------------------------------------------------------
// 
// "aida_newFragment" --
// 
// Create a new fragment for output: either a file or a callback.
// 
// ------------------------------------------------------------------------
aida_frag_t *
aida_newFragment(int inKind, void * inData)
{
	aida_frag_t *	frag = NULL;
	
	aida_verbose(3, "creating new fragment of kind %d\n", inKind);
	// Allocate an aida_frag_t struct
	frag = (aida_frag_t*)malloc(sizeof(aida_frag_t));
	if (frag == NULL) {
		aida_abort("can't allocate memory for new fragment");
	} 
	
	// Fill the struct
	if (gFragmentHead == NULL) {
		gFragmentHead = frag;
	} else {
		aida_closeFragmentFile(gFragmentCurr);
		gFragmentCurr->next = frag;
	}
	frag->num = gFragmentNum++;
	gFragmentCurr = frag;
	frag->next = NULL;
	frag->kind = inKind;
	
	if (inKind == kind_proc) {
		char *	str = (char *)inData;
		frag->u.proc = Tcl_NewStringObj(str,-1);
		Tcl_IncrRefCount(frag->u.proc);
	} else if (inKind == kind_split) {
		frag->u.split.index = ++gSplitNum;
	} else {
		gCurrOutput = aida_newTempFile(&(frag->u.file), P_tmpdir);
	} 
	
	return frag;
}


// ------------------------------------------------------------------------
// 
// "aida_closeFragmentFile" --
// 
// Close the file descriptor for a fragment of type kind_file.
// 
// ------------------------------------------------------------------------
void
aida_closeFragmentFile(aida_frag_t * inFrag)
{
	if (inFrag != NULL && inFrag->kind == kind_file && inFrag->u.file.closed == false ) {
		fclose(inFrag->u.file.desc);
		inFrag->u.file.closed = true;
		aida_verbose(3, "closed temp file '%s'\n", inFrag->u.file.name);
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_newTempFile" --
// 
// Create a new temporary file for output in directory inDir. This resets
// the gCurrOutput variable to point to the new file.
// If inDir is NULL, use the default tmp dir on the system.
// 
// ------------------------------------------------------------------------
FILE *
aida_newTempFile(aida_file_t * outFile, const char * inDir)
{
	int				fd, len;
	char *			prefix = "aidatmp_frag_XXXXXX";
	char *			template;
	
	aida_verbose(3, "opening new temp file in '%s'\n", inDir);
	len = strlen(inDir)+strlen(prefix);
	template = (char*)malloc(len+2);
	sprintf(template,"%s/%s",inDir,prefix);
	template[len]=0;
	
	// Open a temporary file
	fd = mkstemp(template);
	if (fd < 0) {
		aida_abort("failed to open a new temporary file\n");
	} 	
	
	outFile->name = template;
	outFile->desc = fdopen(fd, "w+");
	if (outFile->desc == NULL) {
		aida_abort("failed to open temporary stream\n");
	} 
	aida_verbose(3, "opened new temp file '%s'\n", outFile->name);
	outFile->closed = false;
	return outFile->desc;
}


// ------------------------------------------------------------------------
// 
// "aida_forceSplit" --
// 
// ------------------------------------------------------------------------
void
aida_forceSplit(aida_parse_t * inScanParam) {
	aida_verbose(3, "force a split\n");
	if (gPrescan) {
		gSplitNum++;
	} else {
		aida_newFragment(kind_split, inScanParam);
		if (inScanParam != NULL && inScanParam->to == to_fptr) {
			inScanParam->out.fptr = aida_switchOutput();
		} 		
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_switchOutput" --
// 
// Switch to a new temporary file.
// Effects: close the current one and reset gCurrOutput.
// 
// ------------------------------------------------------------------------
FILE *
aida_switchOutput()
{
	aida_frag_t *	aft;
	
	aft = aida_newFragment(kind_file, NULL);
	return aft->u.file.desc;
}


// ------------------------------------------------------------------------
// 
// "aida_nextSplitFile" --
// 
// Switch to a new split file.
// Effects: close the previous one, rename it, open a new one and reset
// gCurrOutput.
// 
// ------------------------------------------------------------------------
FILE *
aida_nextSplitFile(int inNum, aida_file_t * ioFileData)
{
	int				result;
	FILE *			tmpout;

	if (inNum > 1) {
		aida_closeSplitFile(inNum-1, ioFileData);
	} 
	
	result = aida_executeHook("aida::splitHook", 1, aida_splitFileName(inNum));
	aida_assert_result(result);
	
	tmpout = aida_newTempFile(ioFileData, ".");
	aida_procToFile(Tcl_NewStringObj("aida::preambleProc", -1), tmpout, 0);

	result = aida_navBarProc(inNum, gSplitNum, tmpout);
	aida_assert_result(result);
	
	return tmpout;
}


// ------------------------------------------------------------------------
// 
// "aida_closeSplitFile" --
// 
// Switch to a new split file.
// Effects: close the previous one, rename it, open a new one and reset
// gCurrOutput.
// 
// ------------------------------------------------------------------------
int
aida_closeSplitFile(int inNum, aida_file_t * ioFileData)
{
	int			result;
	
	if (ioFileData->closed == false) {
		aida_procToFile(Tcl_NewStringObj("aida::postambleProc", -1), ioFileData->desc, 0);
		fclose(ioFileData->desc);
		ioFileData->closed = true;
	} 
	
	// Rename the previous split file to its final destination
	result = aida_renameWithNum(ioFileData->name, inNum);
	if (result != TCL_OK) {
		aida_abort("failed to rename split file");
	} 
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_buildSplitFormat" --
// 
// ------------------------------------------------------------------------
char *
aida_buildSplitFormat()
{
	char		*outname;
	Tcl_Obj 	*frmtObj;
	
	frmtObj = aida_buildOutNameObj();
	outname = strdup(Tcl_GetString(frmtObj));
	Tcl_DecrRefCount(frmtObj);
	
	return outname;
}


// ------------------------------------------------------------------------
// 
// "aida_buildOutFileName" --
// 
// ------------------------------------------------------------------------
char *
aida_buildOutFileName()
{
	char		*outname;
	Tcl_Obj 	*outObj;
	
	outObj = aida_buildOutNameObj();
	outname = strdup(Tcl_GetString(outObj));
	Tcl_DecrRefCount(outObj);
	
	return outname;
}


// ------------------------------------------------------------------------
// 
// "aida_checkSplitFormat" --
// 
// Check that the split filename format produces different strings. If not,
// insert a "_%d" suffix before the extension.
// 
// ------------------------------------------------------------------------
void
aida_checkSplitFormat()
{
	if (gSplitFrmt != NULL && strlen(gSplitFrmt) == 0) {
		free(gSplitFrmt);
		gSplitFrmt = NULL;
	} 
	
	if (gSplitFrmt == NULL) {
		gSplitFrmt = aida_buildSplitFormat();
	} else {
		Tcl_Obj 	*numObj[2];
		Tcl_Obj 	*outObj[2];
		char 		*newFrmt, *dot;
		char *		suffix = "_%d";

		numObj[0] = Tcl_NewIntObj(1);
		numObj[1] = Tcl_NewIntObj(2);
		outObj[0] = Tcl_Format(gInterp, gSplitFrmt, 1, &numObj[0]);
		outObj[1] = Tcl_Format(gInterp, gSplitFrmt, 1, &numObj[1]);
		
		if ( !strcmp(Tcl_GetString(outObj[0]),Tcl_GetString(outObj[1])) ) {
			newFrmt = malloc(strlen(gSplitFrmt)+strlen(suffix));
			dot = strchr(gSplitFrmt,'.');
			if (dot) {
				*dot = 0;
				sprintf(newFrmt,"%s%s.%s",gSplitFrmt,suffix,dot+1);
			} else {
				sprintf(newFrmt,"%s%s",gSplitFrmt,suffix);
			}
			
			free(gSplitFrmt);
			gSplitFrmt = newFrmt;
		} 
	}
}


// ------------------------------------------------------------------------
// 
// "aida_splitFileName" --
// 
// Build the split file name using the format string.
// 
// ------------------------------------------------------------------------
Tcl_Obj *
aida_splitFileName(int inNum)
{
	Tcl_Obj 	*fileObj, *objv;

	if (gSplitFrmt == NULL) {
		gSplitFrmt = aida_buildSplitFormat();
	} 
	objv = Tcl_NewIntObj(inNum);
	fileObj = Tcl_Format(gInterp, gSplitFrmt, 1, &objv);
	
	return fileObj;
}


// ------------------------------------------------------------------------
// 
// "aida_buildOutNameObj" --
// 
// The returned Tcl_Obj must be DecrRefCount'ed by the caller.
// 
// ------------------------------------------------------------------------
Tcl_Obj * 
aida_buildOutNameObj()
{
	int			result;
	char		*base, *dot, *slash;
	const char	*ext;
	Tcl_Obj 	*objv[2], *procObj, *outObj;
	
	result = Tcl_EvalEx(gInterp, "aida::makeOutputName", -1, TCL_EVAL_GLOBAL);
	aida_assert_result(result);

	outObj = Tcl_DuplicateObj(Tcl_GetObjResult(gInterp));
	Tcl_IncrRefCount(outObj);
	
	return outObj;
}


// ------------------------------------------------------------------------
// 
// "aida_renameFile" --
// 
// Invoke the [aida::renameOutputFile] proc which takes care of saving in
// the location defined by the DestDir parameter and to create intermediary
// dirs if necessary.
// 
// ------------------------------------------------------------------------
int
aida_renameFile(char * inOldName, char * inNewName) {
	int			i, objc, result = TCL_OK;
	Tcl_Obj *	objv[3];
	
	objv[0] = Tcl_NewStringObj("aida::renameOutputFile", -1);
	objv[1] = Tcl_NewStringObj(inOldName, -1);
	objv[2] = Tcl_NewStringObj(inNewName, -1);
	objc = 3;
	
	for (i = 0; i < objc; i++) Tcl_IncrRefCount(objv[i]);
	result = Tcl_EvalObjv(gInterp, objc, objv, TCL_EVAL_GLOBAL);
	for (i = 0; i < objc; i++) Tcl_DecrRefCount(objv[i]);
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_renameWithNum" --
// 
// Rename a temporary file to the final split file name given the split
// index number.
// 
// ------------------------------------------------------------------------
int
aida_renameWithNum(char * inOldName, int inNum) {
	int				result;
	Tcl_Obj *		fileObj;

	fileObj = aida_splitFileName(inNum);
	result = aida_renameFile(inOldName, Tcl_GetString(fileObj));

	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_addListDepth" --
// 
// ------------------------------------------------------------------------
void
aida_addListDepth(char * inType, int inDelta)
{
	if (!strcmp(inType,"dl")) {
		gDLDepth += inDelta;
	} else if (!strcmp(inType,"ol")) {
		gOLDepth += inDelta;
	} else if (!strcmp(inType,"ul")) {
		gULDepth += inDelta;
	}
	gListDepth += inDelta;
}


