// ===========================================================================
// File: "aidaMain.c"
//                        Created: 2010-08-09 21:56:18
//              Last modification: 2015-04-15 08:59:37
// Author: Bernard Desgraupes
// e-mail: <bdesgraupes@users.sourceforge.net>
// Copyright (c) 2010-2015 Bernard Desgraupes
// All rights reserved.
// ===========================================================================

#include "aidaMain.h"

// To control tracing
extern int yydebug;


// ------------------------------------------------------------------------
// 
// "main" --
// 
// Dispatch the subcommands.
// 
// ------------------------------------------------------------------------
int main(int argc, char *argv[])
{
	int		index, result = TCL_OK;
	
	// Parser traces are not enabled by default: to request a trace call
	// the aida command with the -t option
	yydebug = 0;
	
	if (argc < 2) {
		aida_usage(argv[0]);
		aida_exit(1);
	} 
	
	// Install a function to invoke at exit
	atexit(aida_terminate);
	
	gPrgName = strdup(argv[0]);
	aida_initLibDir(NULL);
	 
	if ( !strcmp(argv[1], "-h") || !strcmp(argv[1], "--help") ) {
		aida_print_help();
	} else if ( !strcmp(argv[1], "-v") || !strcmp(argv[1], "--version") ) {
		aida_print_out("%s\n", gVersionStr);
	} else {
		Tcl_Obj *		subCmd;

		// Parse the global options
		if (aida_global_options(&argc, &argv) != TCL_OK) {
			aida_exit(1);
		}
		// Check argc again
		if (argc < 2) {
			aida_wrongNumArgs("");
		} 

		// Create the Tcl interpreter
		if (aida_initTcl(argv[0]) != TCL_OK) {
			aida_exit(1);
		}

#ifdef HAVE_SIGNAL_H
		// Install a signal handler
		aida_install_sigaction();
#endif
		
		subCmd = Tcl_NewStringObj(argv[1], -1);
		
		result = Tcl_GetIndexFromObj(gInterp, subCmd, gAidaSubcommands, "subcommand", 0, &index);
		if (result != TCL_OK) {
			aida_result_to_console(result);
			aida_print_err("hint: type '%s -h' for usage\n", gPrgName);
		} else {
			switch (index) {
			  case AIDA_CMD_CONVERT:
				result = aidaCmd_convert(argc, argv);
			  break;
			  
			  case AIDA_CMD_HELP:
				result = aidaCmd_help(argc, argv);
			  break;
			  
			  case AIDA_CMD_INFO:
				result = aidaCmd_info(argc, argv);
			  break;
			  
			  case AIDA_CMD_SPLIT:
				result = aidaCmd_split(argc, argv);
			  break;
			  
			}
		}
	}
		
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_terminate" --
// 
// Cleanup before exit.
// 
// ------------------------------------------------------------------------
void aida_terminate(void)
{
	int			i;
	
	aida_verbose(2, "cleanup before exit\n");
	if (gInputFile != NULL && gInputFile != stdin) {
		fclose(gInputFile);
		aida_verbose(2, "closed input file\n");
	} 
	
	aida_verbose(3, "deallocate internal mapping array\n");
	for (i = 0; i < 256; i++) {
		if (gMap[i] != NULL) {
			Tcl_DecrRefCount(gMap[i]);
		} 
	}
	
	// Deallocate gCurrInclude
	aida_closeInclude(gCurrInclude);

	// Deallocate gEncodings
	aida_free_encodings();

	// Deallocate fragments
	aida_free_fragments();
	
	aida_verbose(2, "finished cleanup\n");	
}


// ------------------------------------------------------------------------
// 
// "aida_exit" --
// 
// Cleanup before exit. The function aida_terminate() is invoked
// automatically: it has been installed by the atexit() call in main().
// 
// ------------------------------------------------------------------------
void aida_exit(int inCode)
{
	exit(inCode);
}


// ------------------------------------------------------------------------
// 
// "aida_abort" --
// 
// Exit with a message.
// 
// ------------------------------------------------------------------------
void aida_abort(char * inMsg,...)
{
	va_list		ap;

	va_start(ap, inMsg);
	fprintf(stderr, "fatal: ");
	vfprintf(stderr, inMsg, ap);
	va_end(ap);

	aida_exit(1);
}


// ------------------------------------------------------------------------
// 
// "aida_free_fragments" --
// 
// ------------------------------------------------------------------------
void aida_free_fragments()
{
	if (gFragmentHead != NULL) {
		aida_frag_t		*curr = gFragmentHead, *next;
		do {
			next = curr->next;
			aida_verbose(3, "deallocate fragment (kind=%d)\n", curr->kind);
			switch (curr->kind) {
				case kind_file:
				aida_closeFragmentFile(curr);
				if (!gDontUnlink) {
					aida_verbose(3, "unlink '%s'\n", curr->u.file.name);
					unlink(curr->u.file.name);
				} 
				if (curr->u.file.name != NULL) {
					free(curr->u.file.name);
				} 
				break;
				
				case kind_proc:
				aida_verbose(3, "decr ref count of proc %s\n", Tcl_GetString(curr->u.proc));
				Tcl_DecrRefCount(curr->u.proc);
				break;
				
			}
			free(curr);
			curr = next;
		} while (curr != NULL);
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_global_options" --
// 
// Parse the command line arguments looking for global options. On return
// argc and argv are adjusted: the global options have been removed from
// the array.
// 
// ------------------------------------------------------------------------
int aida_global_options(int * ioArgc, char ** ioArgv[])
{
	int			i = 0, count = 0, index, result = TCL_OK;
	int			ac = *ioArgc;
	char *		av[ac];
	Tcl_Obj *	optObj;
	
	while (i < ac && result != TCL_ERROR) {
		optObj = Tcl_NewStringObj((*ioArgv)[i], -1);
		if (Tcl_GetIndexFromObj(gInterp, optObj, gAidaGlobalOptions, "global option", 0, &index) != TCL_OK) {
			av[count++] = (*ioArgv)[i];
		} else {
			switch (index) {
				case GLOBAL_OPTION_D:
				if (i < ac-1) {
					gVerbosity = atoi((*ioArgv)[++i]);
				} else {
					aida_print_err("missing value for option -d\n");
					result = TCL_ERROR;
				}
				break;
				
				case GLOBAL_OPTION_L:
				if (i < ac-1) {
					aida_initLibDir((*ioArgv)[++i]);
				} else {
					aida_print_err("missing value for option -L\n");
					result = TCL_ERROR;
				}
				break;
				
				case GLOBAL_OPTION_Q:
				gVerbosity = 0;
				break;
				
#ifdef AIDA_TRACING_OPTIONS				
				case GLOBAL_OPTION_TL:
				gTraceLexer = true;
				break;
				
				case GLOBAL_OPTION_TP:
				yydebug = 1;
				break;
#endif				
				
				case GLOBAL_OPTION_X:
				gDontUnlink = true;
				break;
			}
		} 
		i++;
	}
	
	*ioArgc = count;
	for (i = 0; i < count; i++) {
		(*ioArgv)[i] = av[i];
	}
	
	return result;
}


#ifdef HAVE_SIGNAL_H
// ------------------------------------------------------------------------
// 
// "aida_install_sigaction" --
// 
// Capture signals. Let through the SIGCONT and SIGTSTP signals, otherwise
// terminate the program.
// 
// ------------------------------------------------------------------------
void aida_install_sigaction()
{
	// Install a signal handler
	struct sigaction	act;
	int		i;
	
	aida_verbose(3, "installing a signal handler\n");
	act.sa_handler = aida_signal_handler;
	sigfillset(& (act.sa_mask));
	act.sa_flags = 0;
	
	for (i= 0; i < NSIG; i++) {
		if (sigaction(i, &act, NULL) != 0) {
			aida_verbose(3, "signal %d not captured\n", i);
		} 
	}
}


// ------------------------------------------------------------------------
// 
// "aida_signal_handler" --
// 
// Capture signals. Let through the SIGCHLD, SIGCONT and SIGTSTP signals,
// otherwise terminate the program.
// 
// ------------------------------------------------------------------------
void aida_signal_handler(int inNum)
{
	char	msg[256];
	
	sprintf(msg, "received signal %d (%s)\n", inNum, sys_siglist[inNum]);
	if (inNum != SIGCONT && inNum != SIGTSTP && inNum != SIGCHLD) {
		aida_print_err(msg);
		aida_exit(EXIT_FAILURE);
	} else {
		aida_verbose(3, msg);
	}
	
}
#endif // HAVE_SIGNAL_H


