%{
// File: aida.y
// Last modification: 2012-04-11 09:38:21
// Description: parser specification for Aida

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "aidaMain.h"

int yydebug;


// Statics
static int				aida_handleTable(Tcl_Obj * inListObj, Tcl_DString * inAttrDst, aida_parse_t * inScanParam);
static Tcl_Obj *		aida_handleRow(Tcl_DString * inDst);
static int				aida_handleList(Tcl_Obj * inListObj, Tcl_DString * inAttrDst, char * inKind, aida_parse_t * inScanParam);
static Tcl_Obj *		aida_handleListItem(Tcl_DString * inDesc, Tcl_DString * inText);
static int				aida_handleInclude(Tcl_DString * inDst, aida_parse_t * inScanParam, int inType);
static int				aida_handleEvalBlock(Tcl_DString * inDst, aida_parse_t * inScanParam);
static int				aida_handleAttrPair(Tcl_DString * inKey, Tcl_DString * inValue, aida_parse_t * inScanParam);
%}

%union {
    int				num;
    char *			str;
	Tcl_DString	*	tds;
	Tcl_Obj	*		obj;
}


// Simple tags
%token START_FULL START_TR_FRAG START_LI_FRAG START_ATTR END_INPUT

%token <str> HEAD_KEY HEAD_VAL 
%token <num> HEAD_SP 

%token <str> ATTRTEXT QUOTETEXT BLOCKTEXT LINETEXT PLAINTEXT  

%token <num> COMMENT ATTRSPACE
%token <num> TOC_TAG INDEX_TAG SPLIT_TAG
%token <num> INCLUDE_TAG INPUT_TAG
%token <num> NL_TAG HR_TAG ELSE_TAG

	// Opening tags
%token <num> VERBATIM_BTAG A_BTAG B_BTAG I_BTAG IF_BTAG IMG_BTAG LI_BTAG
%token <num> DL_BTAG LK_BTAG OL_BTAG LT_BTAG UL_BTAG RF_BTAG EVAL_BTAG 
%token <num> S1_BTAG S2_BTAG S3_BTAG S4_BTAG S5_BTAG S6_BTAG 
%token <num> TABLE_BTAG TR_BTAG U_BTAG Q_BTAG V_BTAG X_BTAG Y_BTAG

	// Closing tags
%token <num> VERBATIM_ETAG A_ETAG B_ETAG I_ETAG IF_ETAG IMG_ETAG LI_ETAG
%token <num> DL_ETAG LK_ETAG OL_ETAG LT_ETAG UL_ETAG RF_ETAG EVAL_ETAG
%token <num> S1_ETAG S2_ETAG S3_ETAG S4_ETAG S5_ETAG S6_ETAG 
%token <num> TABLE_ETAG TR_ETAG U_ETAG Q_ETAG V_ETAG X_ETAG Y_ETAG

	// Unknown tags
%token <num> UNKNOWN_BTAG UNKNOWN_ETAG

%type <tds> SectionList Section RowFragmentList RowFragment ItemFragmentList ItemFragment
%type <tds> eval_chunk eval_list 
%type <tds> attr_txt attr_val quoted_txt line_txt block_txt plain_txt 
%type <tds> comment_line header_line include_line text_section
%type <tds> eval_block if_block table_block verbatim_block div_block
%type <tds> a_pair img_pair lk_pair rf_pair v_pair x_pair
%type <num> single_tag

%type <obj> AttributeList Attribute
%type <obj> table_rows_list table_row
%type <obj> list_block dl_block ol_block ul_block
%type <obj> li_items_list lt_items_list li_item lt_item

%locations

%pure-parser
%parse-param {void *scanner}
%lex-param {yyscan_t *scanner}

%start input

%{
	// This must be included after the %union to make sure YYSTYPE and YYLTYPE are defined
	#include "aida.yy.h"
	
	void yyerror(YYLTYPE * t, yyscan_t scanner, char const * s);

    static void aida_print_tokenValue(FILE *, int, YYSTYPE);
	static void aida_dbg_locations(yyscan_t scanner);
	
    #define YYPRINT(F, N, L) aida_print_tokenValue(F, N, L)
%}

// %destructor { if ($$ != NULL) {Tcl_DStringFree($$);} } comment_line include_line
//     if_block table_block list_block 

// %destructor { if ($$ != NULL) {free($$);} } HEAD_KEY HEAD_VAL

%% // RULES SECTION 

input: /* empty */
		| START_FULL SectionList
		| START_TR_FRAG RowFragmentList
		| START_LI_FRAG ItemFragmentList
		| START_ATTR AttributeList ending
		;

SectionList: Section	 		{ $$ = NULL; if ($1 != NULL) {Tcl_DStringFree($1);} }
		| SectionList Section	{ $$ = NULL; if ($2 != NULL) {Tcl_DStringFree($2);} }
		;
	
RowFragmentList: RowFragment	 		{ $$ = NULL; if ($1 != NULL) {Tcl_DStringFree($1);} }
		| RowFragmentList RowFragment	{ $$ = NULL; if ($2 != NULL) {Tcl_DStringFree($2);} }
		;
	
ItemFragmentList: ItemFragment	 		{ $$ = NULL; if ($1 != NULL) {Tcl_DStringFree($1);} }
		| ItemFragmentList ItemFragment	{ $$ = NULL; if ($2 != NULL) {Tcl_DStringFree($2);} }
		;
	
AttributeList: Attribute	 				{ $$ = NULL; }
		| AttributeList ATTRSPACE 			{ $$ = NULL; }
		| AttributeList ATTRSPACE Attribute	{ $$ = NULL; }
		;
	
Section: ending					{ $$ = NULL; }
		| header_line			{ $$ = $1; }
		| comment_line			{ $$ = NULL; }
		| include_line			{ $$ = $1; }
		| single_tag			{ $$ = NULL; }
		| if_block				{ $$ = NULL; }
		| table_block			{ $$ = NULL; }
		| list_block			{ $$ = NULL; }
		| verbatim_block		{ $$ = NULL; }
		| div_block				{ $$ = $1; }
		| text_section ending	{ $$ = $1; }
		;

RowFragment: ending				{ $$ = NULL; }
		| verbatim_block		{ $$ = NULL; }
		| text_section ending	{ $$ = $1; }
		;

ItemFragment: ending				{ $$ = NULL; }
		| comment_line			{ $$ = NULL; }
		| include_line			{ $$ = $1; }
		| single_tag			{ $$ = NULL; }
		| if_block				{ $$ = NULL; }
		| table_block			{ $$ = NULL; }
		| list_block			{ $$ = NULL; }
		| verbatim_block		{ $$ = NULL; }
		| text_section ending	{ $$ = $1; }
		;

Attribute: attr_txt 				{ aida_handleAttrPair($1, NULL, yyget_extra(scanner)); $$ = NULL; }
		| attr_txt '=' attr_val 	{ aida_handleAttrPair($1, $3, yyget_extra(scanner)); $$ = NULL; }
		;
		
attr_txt: ATTRTEXT					{ $$ = aida_makeDString($1, -1); }
		| attr_txt ATTRTEXT			{ $$ = aida_appendDString($1, $2, -1); }
		;

quoted_txt: QUOTETEXT				{ $$ = aida_makeDString($1, -1); }
		| quoted_txt QUOTETEXT		{ $$ = aida_appendDString($1, $2, -1); }
		;

attr_val: attr_txt			{ $$ = $1; }
		| '"' quoted_txt '"' { $$ = $2; }
		| '\'' quoted_txt '\'' { $$ = $2; }
		;

ending: '\n'
		| END_INPUT 			{YYACCEPT;}
		;
		
// Free HEAD_KEY and HEAD_VAL which have been allocated with strdup()
header_line: HEAD_KEY HEAD_SP HEAD_VAL ending	{ aida_setHeaderValue($1, $3); free($1); free($3); $$ = NULL; }
		| HEAD_KEY HEAD_SP ending 				{ aida_setHeaderValue($1, ""); free($1); $$ = NULL; }
		| HEAD_KEY ending 						{ aida_setHeaderValue($1, ""); free($1); $$ = NULL; }
		;

comment_line: COMMENT ending		{ $$ = NULL; }
		| COMMENT line_txt ending	{ aida_assert_result( aida_commentProc($2, yyget_extra(scanner)) );	$$ = NULL; }
		;

// LINETEXT indicates text on a single line, stored in a DString
line_txt: LINETEXT				{ $$ = aida_makeDString($1, -1); }
		| line_txt LINETEXT		{ $$ = aida_appendDString($1, $2, -1); }
		;

// BLOCKTEXT indicates multiline text, stored in a DString
block_txt: BLOCKTEXT			{ $$ = aida_makeDString($1, -1); }
		| block_txt BLOCKTEXT	{ $$ = aida_appendDString($1, $2, -1); }
		;

// PLAINTEXT indicates multiline text which is just echoed
plain_txt: PLAINTEXT			{ $$ = NULL; }
		| plain_txt PLAINTEXT	{ $$ = NULL; }
		;

text_section: plain_txt				{ $$ = NULL; }
		| eval_list					{ $$ = NULL; }
		| plain_txt eval_list		{ $$ = NULL; }
		| tagged_env_list			{ $$ = NULL; }
		| tagged_env_list eval_list	{ $$ = NULL; }
		| plain_txt tagged_env_list	{ $$ = NULL; }
		| plain_txt tagged_env_list eval_list	{ $$ = NULL; }
		;

eval_block: EVAL_BTAG block_txt EVAL_ETAG { aida_handleEvalBlock($2, yyget_extra(scanner)); $$ = NULL; }
		;

eval_chunk: eval_block							{ $$ = NULL; }
		| eval_block plain_txt					{ $$ = NULL; }
		| eval_block tagged_env_list			{ $$ = NULL; }
		| eval_block plain_txt tagged_env_list	{ $$ = NULL; }
		;

eval_list: eval_chunk			{ $$ = NULL; }
		| eval_list eval_chunk	{ $$ = NULL; }
		;

include_line: INCLUDE_TAG line_txt ending	{ aida_handleInclude($2, yyget_extra(scanner), INCLUDE_TAG); $$ = $2; }
		| INPUT_TAG line_txt ending			{ aida_handleInclude($2, yyget_extra(scanner), INPUT_TAG); $$ = $2; }
		;

verbatim_block: VERBATIM_BTAG block_txt VERBATIM_ETAG { aida_assert_result( aida_verbatimBlockProc($2, yyget_extra(scanner)) ); $$ = NULL; }
		;

if_block: if_start SectionList IF_ETAG { $$ = NULL; }
		| if_start IF_ETAG { $$ = NULL; }
		| if_start SectionList ELSE_TAG IF_ETAG { $$ = NULL; }
		| if_start ELSE_TAG SectionList IF_ETAG { $$ = NULL; }
		;

if_start: IF_BTAG line_txt '\n' { aida_evalIfCondition($2); }
		;


// Tables
// ======
table_block: TABLE_BTAG '\n' table_rows_list TABLE_ETAG			{ aida_handleTable($3, NULL, yyget_extra(scanner)); $$ = NULL; }
		| TABLE_BTAG line_txt '\n' table_rows_list TABLE_ETAG	{ aida_handleTable($4, $2, yyget_extra(scanner)); $$ = NULL; }
		;

table_rows_list: table_row				{ $$ = aida_makeTclList($1); }
		| table_rows_list table_row		{ $$ = aida_appendTclList($1,$2); }
		;
		
table_row: TR_BTAG block_txt TR_ETAG '\n'	{ $$ = aida_handleRow($2); }
		;


// Sections
// ========
div_block: S1_BTAG block_txt S1_ETAG	{ aida_assert_result( aida_sectionProc($2, 1, yyget_extra(scanner)) ); $$ = $2; }
		| S2_BTAG block_txt S2_ETAG		{ aida_assert_result( aida_sectionProc($2, 2, yyget_extra(scanner)) ); $$ = $2; }
		| S3_BTAG block_txt S3_ETAG		{ aida_assert_result( aida_sectionProc($2, 3, yyget_extra(scanner)) ); $$ = $2; }
		| S4_BTAG block_txt S4_ETAG		{ aida_assert_result( aida_sectionProc($2, 4, yyget_extra(scanner)) ); $$ = $2; }
		| S5_BTAG block_txt S5_ETAG		{ aida_assert_result( aida_sectionProc($2, 5, yyget_extra(scanner)) ); $$ = $2; }
		| S6_BTAG block_txt S6_ETAG		{ aida_assert_result( aida_sectionProc($2, 6, yyget_extra(scanner)) ); $$ = $2; }
		;


// Lists
// =====
list_block: ol_block	{ $$ = NULL; }
		| ul_block		{ $$ = NULL; }
		| dl_block		{ $$ = NULL; }
		;

ol_block: OL_BTAG '\n' li_items_list OL_ETAG			{ aida_handleList($3, NULL, "ol", yyget_extra(scanner)); $$ = NULL; }
		| OL_BTAG block_txt '\n' li_items_list OL_ETAG	{ aida_handleList($4, $2, "ol", yyget_extra(scanner)); $$ = NULL; }
		;

ul_block: UL_BTAG '\n' li_items_list UL_ETAG			{ aida_handleList($3, NULL, "ul", yyget_extra(scanner)); $$ = NULL; }
		| UL_BTAG block_txt '\n' li_items_list UL_ETAG	{ aida_handleList($4, $2, "ul", yyget_extra(scanner)); $$ = NULL; }
		;

dl_block: DL_BTAG '\n' lt_items_list DL_ETAG			{ aida_handleList($3, NULL, "dl", yyget_extra(scanner)); $$ = NULL; }
		| DL_BTAG block_txt '\n' lt_items_list DL_ETAG	{ aida_handleList($4, $2, "dl", yyget_extra(scanner)); $$ = NULL; }
		;

li_items_list: li_item				{ $$ = aida_makeTclList($1); }
		| li_items_list li_item		{ $$ = aida_appendTclList($1,$2); }
		;

lt_items_list:  lt_item				{ $$ = aida_makeTclList($1); }
		| lt_items_list lt_item		{ $$ = aida_appendTclList($1,$2); }
		;

li_item: LI_BTAG block_txt LI_ETAG	{ $$ = aida_handleListItem(NULL, $2); }
		;

lt_item: LT_BTAG block_txt '\n' block_txt LT_ETAG	{ $$ = aida_handleListItem($2, $4); }
		;


single_tag: TOC_TAG
		| INDEX_TAG
		| NL_TAG
		| HR_TAG
		| SPLIT_TAG { aida_forceSplit(yyget_extra(scanner)); }
		;

tagged_env_list: tagged_env_chunk
		| tagged_env_list tagged_env_chunk
		;

tagged_env_chunk: tagged_env
		| tagged_env plain_txt
		;

tagged_env: i_env
		| b_env
		| u_env
		| y_env
		| simple_pair
		;

i_env: I_BTAG text_section I_ETAG
		;

b_env: B_BTAG text_section B_ETAG
		;

u_env: U_BTAG text_section U_ETAG
		;
		
y_env: Y_BTAG text_section Y_ETAG
		;
		
simple_pair: x_pair
		| v_pair
		| a_pair
		| lk_pair
		| rf_pair
		| img_pair
		;

x_pair: X_BTAG block_txt X_ETAG							{ aida_assert_result( aida_setIndexProc($2, yyget_extra(scanner)) ); $$ = NULL; }
		;

v_pair: V_BTAG block_txt V_ETAG							{ aida_assert_result( aida_verbProc($2, yyget_extra(scanner)) ); $$ = NULL; }
		| Q_BTAG block_txt Q_ETAG						{ aida_assert_result( aida_quoteProc($2, yyget_extra(scanner)) ); $$ = NULL; }
		;

img_pair: IMG_BTAG block_txt IMG_ETAG					{ aida_assert_result( aida_imageProc($2, NULL, yyget_extra(scanner)) ); $$ = NULL; }
		| IMG_BTAG '\n' block_txt IMG_ETAG				{ aida_assert_result( aida_imageProc($3, NULL, yyget_extra(scanner)) ); $$ = NULL; }
		| IMG_BTAG block_txt '\n' IMG_ETAG				{ aida_assert_result( aida_imageProc($2, NULL, yyget_extra(scanner)) ); $$ = NULL; }
		| IMG_BTAG block_txt '\n' block_txt IMG_ETAG	{ aida_assert_result( aida_imageProc($4, $2, yyget_extra(scanner)) ); $$ = NULL; }
		;

// Links
// =====

a_pair: A_BTAG block_txt A_ETAG							{ aida_assert_result( aida_anchorProc($2, yyget_extra(scanner)) ); $$ = NULL; }
		;

lk_pair: LK_BTAG '[' block_txt ']' block_txt LK_ETAG	{ aida_assert_result( aida_linkProc($3, $5, 0, yyget_extra(scanner)) ); $$ = NULL; }
		;

rf_pair: RF_BTAG '[' block_txt ']' block_txt RF_ETAG	{ aida_assert_result( aida_linkProc($3, $5, 1, yyget_extra(scanner)) ); $$ = NULL; }
		;


%%  // CODE SECTION

// ------------------------------------------------------------------------
// 
// "yyerror" --
// 
// Called by yyparse on error.
// 
// ------------------------------------------------------------------------
void
yyerror(YYLTYPE * lt, yyscan_t scanner, char const *s)
{
	aida_parse_t *		apt = yyget_extra(scanner);
	
#ifdef AIDA_DEBUG_DSTRING
	printf("scanner #%d ", apt->num);
#endif
	printf ("  error");
	if (gCurrInclude && gCurrInclude->name != NULL) {
		printf(" in file '%s'\n", gCurrInclude->name);
	} 
	
	printf("  at %spos %d.%d-%d.%d ", apt->data? "relative ":"", lt->first_line, lt->first_column, 
	                      lt->last_line, lt->last_column);
						  
	if (apt->data != NULL) {
		printf("inside string\n'%s'", Tcl_DStringValue((Tcl_DString *)apt->data));
	} 
	printf("\n");
	
	aida_abort("%s\n", s);

	return;
}


// ------------------------------------------------------------------------
// 
// "aida_parse" --
// 
// Trigger parsing.
// 
// ------------------------------------------------------------------------

int aida_parse(FILE * inStream)
{
	int				result = TCL_OK;
	yyscan_t		scanner;
	aida_parse_t	apt;

	if (inStream == NULL) {return TCL_ERROR;} 
	gParsing = true;

	yylex_init(&scanner);
	apt.scan = scanner;
	apt.to = to_fptr;
	apt.start = START_FULL;
	apt.enc = gEncodings->oenc;
	apt.out.fptr = gCurrOutput;
	apt.data = NULL;
	aida_init_contexts(&apt);
	
	yyset_extra(&apt, scanner);
	yyset_in(inStream, scanner);
	result = yyparse(scanner);
	yylex_destroy(scanner);

	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_parseDString" --
// 
// Parse the contents of a DString in memory and return the result in a
// DString. The caller is responsible for free'ing the returned DString.
// 
// ------------------------------------------------------------------------
Tcl_DString *
aida_parseDString(Tcl_DString * inDst, int inStart)
{
	int					result;
	yyscan_t			scanner;
	aida_parse_t		apt;
	YY_BUFFER_STATE		bufst;
	
	yylex_init(&scanner);
	apt.scan = scanner;
	apt.to = to_dstr;
	apt.start = inStart;
	apt.enc = NULL;
	apt.out.dstr = aida_makeDString(NULL,0);
	apt.data = inDst;
	aida_init_contexts(&apt);
	
#ifdef AIDA_DEBUG_DSTRING
	printf("-> string to parse (scanner #%d, start=%d): %s\n", apt.num, inStart, Tcl_DStringValue(inDst));
#endif
	yyset_extra(&apt, scanner);
	bufst = yy_scan_string(Tcl_DStringValue(inDst), scanner);
	result = yyparse(scanner);	
	yy_delete_buffer(bufst, scanner);
	yylex_destroy(scanner);
#ifdef AIDA_DEBUG_DSTRING
	printf("<- parsed string (scanner #%d): %s\n", apt.num, Tcl_DStringValue(apt.out.dstr));
#endif
	
	return apt.out.dstr;
}


// ------------------------------------------------------------------------
// 
// "aida_parseAttrString" --
// 
// Parse a DString representing attribute settings and return a Tcl
// dictionary of the key/value pairs.
// 
// ------------------------------------------------------------------------
Tcl_Obj *
aida_parseAttrString(Tcl_DString * inDst)
{
	int					result;
	yyscan_t			scanner;
	aida_parse_t		apt;
	YY_BUFFER_STATE		bufst;
	
	yylex_init(&scanner);
	apt.scan = scanner;
	apt.to = to_objp;
	apt.start = START_ATTR;
	apt.enc = NULL;
	apt.out.objp = Tcl_NewDictObj();
	apt.data = inDst;
	Tcl_IncrRefCount(apt.out.objp);
	aida_init_contexts(&apt);
	
#ifdef AIDA_DEBUG_DSTRING
	printf("attributes to parse (scanner #%d): %s\n", apt.num, Tcl_DStringValue(inDst));
#endif
	yyset_extra(&apt, scanner);
	bufst = yy_scan_string(Tcl_DStringValue(inDst), scanner);
	result = yyparse(scanner);	
	yy_delete_buffer(bufst, scanner);
	yylex_destroy(scanner);
#ifdef AIDA_DEBUG_DSTRING
	printf("attributes parsed (scanner #%d): %s\n", apt.num, Tcl_GetString(apt.out.objp));
#endif
	
	return apt.out.objp;
}


// ------------------------------------------------------------------------
// 
// "aida_handleInclude" --
// 
// Handle 'include' and 'input' tags. The 'inType' arg can be INCLUDE_TAG
// or INPUT_TAG.
// 
// ------------------------------------------------------------------------
int
aida_handleInclude(Tcl_DString * inDst, aida_parse_t * inScanParam, int inType)
{
	int				result = TCL_OK;
	yyscan_t		scanner;
	aida_parse_t	apt;
	aida_incl_t *	ait = NULL;

	if (gSplitting && !gPrescan && gSplitLevel == 0 && inType == INPUT_TAG) {
		aida_forceSplit(inScanParam);
	} 
	
	result = aida_openInclude(inDst, &ait);
	if (result == TCL_OK) {
		yylex_init(&scanner);
		apt.scan = scanner;
		apt.to = to_fptr;
		apt.start = START_FULL;
		apt.enc = gEncodings->oenc;
		apt.out.fptr = gCurrOutput;
		apt.data = NULL;
		aida_init_contexts(&apt);

		yyset_extra(&apt, scanner);
		yyset_in(ait->fptr, scanner);
		result = yyparse(scanner);
		yylex_destroy(scanner);
		aida_closeInclude(ait);
	} 
	
	if (result != TCL_OK) {
		aida_print_err("error parsing included file '%s'\n", Tcl_DStringValue(inDst));
		aida_exit(1);
	} 
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_handleEvalBlock" --
// 
// Handle ((e script e)) tags. The result of the evaluation is possibly a
// string containing Aida markup: it has to be parsed itself.
// 
// ------------------------------------------------------------------------
int
aida_handleEvalBlock(Tcl_DString * inDst, aida_parse_t * inScanParam)
{
	int					len, result;
	Tcl_DString			ds;
	Tcl_DString *		txtds;
	Tcl_Obj *			resObj;
	
	result = Tcl_Eval(gInterp, Tcl_DStringValue(inDst));
	
	if (result == TCL_OK) {
		resObj = Tcl_GetObjResult(gInterp);
		
		// The result is in utf-8. Convert it back to the input
		// encoding because it is going to be used again as an
		// input string.
		Tcl_DStringInit(&ds);
		Tcl_UtfToExternalDString(gEncodings->ienc, Tcl_GetStringFromObj(resObj, &len), -1, &ds);
		
		// Parse the evaluated string
		txtds = aida_parseDString(&ds, START_FULL);
		if (!gPrescan) {
			aida_writeDString(txtds, inScanParam);
		} 
		Tcl_DStringFree(txtds);
		Tcl_DStringFree(&ds);
	} else {
		aida_print_err("error in evaluation of '%s'\n", Tcl_DStringValue(inDst));
		aida_assert_result(result);
	}
	
	Tcl_DStringFree(inDst);
	
	return result; 
}


// ------------------------------------------------------------------------
// 
// "aida_handleTable" --
// 
// Handle ((table block table)) tags.
// 
// ------------------------------------------------------------------------
int
aida_handleTable(Tcl_Obj * inListObj, Tcl_DString * inAttrDst, aida_parse_t * inScanParam)
{
	int					result = TCL_OK;
	Tcl_Obj *			attrObj;
	
	if (gPrescan) {
		result = TCL_OK;
	} else {
		if (inAttrDst != NULL) {
			attrObj = aida_parseAttrString(inAttrDst);
			Tcl_DStringFree(inAttrDst);
		} else {
			attrObj = Tcl_NewDictObj();
			Tcl_IncrRefCount(attrObj);
		} 
		result = aida_executeProc("aida::tableProc", inScanParam, 2, attrObj, inListObj);
		Tcl_DecrRefCount(attrObj);
	}
		
	Tcl_DecrRefCount(inListObj);
	aida_assert_result(result);
	return result; 
}


// ------------------------------------------------------------------------
// 
// "aida_handleAttrPair" --
// 
// Handle an attribute block. If inValue is NULL, assume it is a boolean
// equivalent to true. The function returns a Tcl dict of key/value pairs.
// The return value is a pointer to the dictionary itself so that it can be
// released when the proc is executed.
// 
// ------------------------------------------------------------------------
int
aida_handleAttrPair(Tcl_DString * inKey, Tcl_DString * inValue, aida_parse_t * inScanParam)
{
	int				result = TCL_OK;
	Tcl_Obj			*keyObj, *valObj;
	
	if (gPrescan) {
		result = TCL_OK;
	} else {
		if (inScanParam->to == to_objp) {		
			// Put the key in a Tcl_Obj
			keyObj = aida_externalToTclObj(Tcl_DStringValue(inKey));

			// Put the value in a Tcl_Obj
			if (inValue != NULL) {
				valObj = aida_externalToTclObj(Tcl_DStringValue(inValue));
			} else {
				valObj = Tcl_NewBooleanObj(1);
			} 
		
			aida_verbose(3, "adding attribute pair '%s' --> '%s'\n", Tcl_DStringValue(inKey), Tcl_GetString(valObj));
			result = Tcl_DictObjPut(gInterp, inScanParam->out.objp, keyObj, valObj);
			if (result != TCL_OK) {
				aida_abort("failed to add key/value pair to attributes dictionary\n");
			} 
		} 
	}
	
	// Free the DStrings
	Tcl_DStringFree(inKey);
	if (inValue != NULL) Tcl_DStringFree(inValue); 

	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_handleRow" --
// 
// Handle ((tr block tr)) tags.
// Parse the block in memory and store the returned DString in a Tcl_Obj.
// 
// ------------------------------------------------------------------------
Tcl_Obj *
aida_handleRow(Tcl_DString * inDst)
{
	Tcl_Obj *		rowObj;
	Tcl_DString *	dsptr;
	
	if (gPrescan) {
		rowObj = NULL;
	} else {
		// Parse the row string
		dsptr = aida_parseDString(inDst, START_TR_FRAG);
		
		// Put it in a Tcl_Obj
		rowObj = aida_externalToTclObj(Tcl_DStringValue(dsptr));
		Tcl_DStringFree(dsptr);
	}
	
	// Free the DString
	Tcl_DStringFree(inDst);

	return rowObj;
}


// ------------------------------------------------------------------------
// 
// "aida_handleList" --
// 
// Handle the ((lo, ((lu, ((ld tags.
// 
// ------------------------------------------------------------------------
int
aida_handleList(Tcl_Obj * inListObj, Tcl_DString * inAttrDst, char * inKind, aida_parse_t * inScanParam)
{
	int				result = TCL_OK;
	Tcl_Obj 		*attrObj;
	
	if (gPrescan) {
		result = TCL_OK;
	} else {
		if (inAttrDst != NULL) {
			attrObj = aida_parseAttrString(inAttrDst);
			Tcl_DStringFree(inAttrDst);
		} else {
			attrObj = Tcl_NewDictObj();
			Tcl_IncrRefCount(attrObj); 
		} 

		result = aida_executeProc("aida::listProc", inScanParam, 4, 
								  Tcl_NewStringObj(inKind, -1), Tcl_NewIntObj(gListDepth), attrObj, inListObj);
		Tcl_DecrRefCount(attrObj);
	}
	
	aida_addListDepth(inKind,-1);
	Tcl_DecrRefCount(inListObj);
	aida_assert_result(result);
	return result; 
}


// ------------------------------------------------------------------------
// 
// "aida_handleListItem" --
// 
// Handle ((li ... li)) and ((lt ... lt)) tags.
// 
// Parse the TEXT block in memory and store the returned DString in a
// Tcl_Obj. The first argument 'inDesc' is the title term of a description
// list item: it is NULL in the case of an ordered or unordered list. The
// second argument 'inText' is the text of the item: it is first parsed in
// memory.
// 
// If 'inDesc' is NULL, return a Tcl_obj containing the text of the item,
// otherwise return a two-element Tcl list whose first item is the
// description title and the second item is the text.
// 
// ------------------------------------------------------------------------
Tcl_Obj *
aida_handleListItem(Tcl_DString * inDesc, Tcl_DString * inText)
{
	Tcl_Obj *		itemObj;
	Tcl_DString 	*txtPtr, *descPtr;
	
	if (gPrescan) {
		itemObj = NULL;
	} else {
		// Parse the raw string
		txtPtr = aida_parseDString(inText, START_LI_FRAG);
		
		// Put it in a Tcl_Obj
		if (inDesc != NULL) {
			descPtr = aida_parseDString(inDesc, START_TR_FRAG);
			itemObj = Tcl_NewListObj(0, NULL);
			Tcl_IncrRefCount(itemObj);
			Tcl_ListObjAppendElement(gInterp, itemObj, aida_externalToTclObj(Tcl_DStringValue(descPtr)));
			Tcl_ListObjAppendElement(gInterp, itemObj, aida_externalToTclObj(Tcl_DStringValue(txtPtr)));	
			Tcl_DStringFree(descPtr);
		} else {
			itemObj = aida_externalToTclObj(Tcl_DStringValue(txtPtr));
		} 
		
		// Free the returned string
		Tcl_DStringFree(txtPtr);
	}
	
	if (inDesc != NULL) {Tcl_DStringFree(inDesc);}
	if (inText != NULL) {Tcl_DStringFree(inText);}
	
	return itemObj;
}


// ------------------------------------------------------------------------
// 
// "aida_print_tokenValue" --
// 
// Used to define the YYPRINT macro.
// 
// ------------------------------------------------------------------------
void
aida_print_tokenValue(FILE *inFile, int inType, YYSTYPE inValue)
{
	fprintf(inFile, "aida: token ");
	
	if (inType <= 256) {
		fprintf(inFile, "%d", inType);
	} else {
		switch (inType) {
			case HEAD_KEY:
			case HEAD_VAL:
			fprintf(inFile, "'%s' (type %d)", inValue.str, inType);
			break;
			
			case BLOCKTEXT:
			case LINETEXT:
			case PLAINTEXT:
			fprintf(inFile, "'%s' (type %d)", inValue.str, inType);
			break;
			
			case HEAD_SP:
			// Ignore
			break;
			
			case COMMENT:
			fprintf(inFile, "'^!!' (type %d)", inType);
			break;
			
			default:
			fprintf(inFile, "'%d' (type %d)", inValue.num, inType);
			break;
		}
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_dbg_locations" --
// 
// Print the current value of the YYLTYPE structure.
// 
// ------------------------------------------------------------------------
void
aida_dbg_locations(yyscan_t scanner) 
{
	YYLTYPE * t;
	
	printf(">>>> aida locations: ");
	t = yyget_lloc(scanner);
	printf("first: %d.%d - last: %d.%d\n", 
			t->first_line, t->first_column, t->last_line, t->last_column);

}




