%{
// File: aida.l
// Last modification: 2015-07-20 16:28:15
// Description: lexer specification for Aida

#include "aidaMain.h"
#include "aida.tab.h"

#ifdef AIDA_DEBUG_CONTEXT
#define AIDA_PUSH_CONTEXT(Context)   BEGIN aida_push_context((Context), yyextra) 
#define AIDA_POP_CONTEXT BEGIN aida_pop_context(yyextra)

#else
// Specifying '%option stack' below enables the yy_push_state/yy_pop_state routines
#define AIDA_PUSH_CONTEXT(Context)   yy_push_state((Context), yyscanner) 
#define AIDA_POP_CONTEXT yy_pop_state(yyscanner)

#endif 


// Macros to get and set the positions
#if 0
#define PRINTLOC  printf("first: %d.%d - last: %d.%d\n", yylloc->first_line, yylloc->first_column, yylloc->last_line, yylloc->last_column);
#else
#define PRINTLOC 
#endif

#ifdef AIDA_DEBUG_LOC
#define AIDA_SHOW_LOC(Rule) \
printf("-->> (%d) seen '%s'\n", (Rule), yytext); \
printf("     at pos %d.%d-%d.%d\n", yylloc->first_line, yylloc->first_column, yylloc->last_line, yylloc->last_column)
#else
#define AIDA_SHOW_LOC(Rule) 
#endif 

#define AIDA_SET_LOC_DOT \
					yylloc->first_line = yylloc->last_line; \
					yylloc->first_column = yylloc->last_column; \
					yylloc->last_column++

#define AIDA_SET_LOC_LF \
					yylloc->first_line = yylloc->last_line; \
					yylloc->last_line++; \
					yylloc->first_column = yylloc->last_column; \
					yylloc->last_column = 0

#define AIDA_SET_LOC_BOL \
					yylloc->first_line = yylloc->last_line; \
					yylloc->first_column = 0; \
					yylloc->last_column = yyleng

#define AIDA_SET_LOC_EOF \
					yylloc->first_line = yylloc->last_line; \
					yylloc->first_column = yylloc->last_column;

#define AIDA_SET_LOC_CHUNK \
					yylloc->first_line = yylloc->last_line; \
					yylloc->first_column = yylloc->last_column; \
					yylloc->last_column += yyleng

#define AIDA_SET_LOC_BTAG \
					yylloc->first_line = yylloc->last_line; \
					yylloc->first_column = yylloc->last_column; \
					if (yytext[yyleng-1] == '\n' | yytext[yyleng-1] == '\r') { \
						yylloc->last_line++; \
						yylloc->last_column = 0; \
					} else { \
						yylloc->last_column += yyleng; \
					} 

#define AIDA_SET_LOC_ETAG \
					yylloc->first_line = yylloc->last_line; \
					yylloc->first_column = yylloc->last_column; \
					if (yytext[0] == '\n' | yytext[0] == '\r') { \
						yylloc->last_line++; \
						yylloc->last_column = yyleng-1; \
					} else { \
						yylloc->last_column += yyleng; \
					} 


// Statics

static int aida_push_context(int inContext, aida_parse_t * inScanParam);
static int aida_pop_context(aida_parse_t * inScanParam);

%}

%s s_style

%x x_head
%x x_comm
%x x_eval
%x x_incl
%x x_cond
%x x_if
%x x_table
%x x_tbl_attr
%x x_tbl_row
%x x_div
%x x_img
%x x_img_name
%x x_idx
%x x_list_attr
%x x_list_items
%x x_li
%x x_lt_dt
%x x_lt_dd
%x x_verb_v
%x x_verb_q
%x x_pass_v
%x x_pass_q
%x x_anch
%x x_link
%x x_link_targ
%x x_link_text
%x x_single_quote
%x x_double_quote
%x x_verbatim
%x x_skip_line
%x x_skip_block
%x x_attributes
%x x_attr_val


%option reentrant bison-bridge bison-locations
%option extra-type="aida_parse_t *"
%option stack


%% // RULES SECTION 

	if (yyextra->start != 0) {
		int c = yyextra->start;
		if (c == START_ATTR) {
			AIDA_PUSH_CONTEXT(x_attributes); 
			yylloc_param->last_line = 1;
			yylloc_param->last_column = 0;
		} 
    	yyextra->start = 0;
    	return(c);
    }


%{
   // Header declarations
%}


^:[[:alnum:]]+:([[:alnum:]]+|\*): |
^:[[:alnum:]]+:		{
						AIDA_PUSH_CONTEXT(x_head); 
						AIDA_SET_LOC_BOL;
						yylval->str = strdup(yytext+1);
						yylval->str[yyleng-2] = 0;
						aida_dbg_loc(1, yytext, yylloc);
						return HEAD_KEY;
					}

<x_head>[ \t]+ 		{
						AIDA_SET_LOC_CHUNK;
						aida_dbg_loc(2, yytext, yylloc);
						return HEAD_SP;
					}

<x_head>[^ \t\n].*	{
						yylval->str = strdup(yytext);
						AIDA_SET_LOC_CHUNK;
						aida_dbg_loc(3, yytext, yylloc);
						return HEAD_VAL;
					}

<x_head>[\n\r]		{
						AIDA_POP_CONTEXT; 
						AIDA_SET_LOC_LF;
						aida_dbg_loc(4, yytext, yylloc);
						return yytext[0];
					}

%{
   // Comments
%}

^!!\ ?!			{
					AIDA_PUSH_CONTEXT(x_skip_line); 
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(5, yytext, yylloc);
					return COMMENT;
				}

^!![ \t]*		{
					AIDA_PUSH_CONTEXT(x_comm); 
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(6, yytext, yylloc);
					return COMMENT;
				}

<x_comm>.		{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(7, yytext, yylloc);
					return LINETEXT;
				}

<x_comm>[\n\r]	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_LF;
					aida_writeString(yytext, yyextra);
					aida_dbg_loc(8, yytext, yylloc);
					return yytext[0];
				}
				

%{
   // Skipped text
%}

<x_skip_line>.		{
						AIDA_SET_LOC_DOT;
						aida_dbg_loc(9, yytext, yylloc);
					}

<x_skip_line>[\n\r]	{
						AIDA_POP_CONTEXT; 
						AIDA_SET_LOC_LF;
						aida_dbg_loc(10, yytext, yylloc);
						return yytext[0];
					}

<x_skip_block>.		{
						AIDA_SET_LOC_DOT;
						aida_dbg_loc(11, yytext, yylloc);
					}

<x_skip_block>[\n\r]	{
						AIDA_SET_LOC_LF;
						aida_dbg_loc(12, yytext, yylloc);
					}


%{
   // Attribute declarations
%}


<x_attributes>[ \t]+	 {
						AIDA_SET_LOC_CHUNK;
						aida_dbg_loc(13, yytext, yylloc);
						return ATTRSPACE;
					}

<x_attributes>[[:alnum:]]+	 {
						AIDA_SET_LOC_CHUNK;
						yylval->str = yytext;
						aida_dbg_loc(14, yytext, yylloc);
						return ATTRTEXT;
					}

<x_attributes>[ \t]*=[ \t]*		{
						AIDA_PUSH_CONTEXT(x_attr_val);
						AIDA_SET_LOC_CHUNK;
						aida_dbg_loc(15, yytext, yylloc);
						return '=';
					}

<x_attributes>[^[:alnum:][:space:]=|]	 {
						AIDA_SET_LOC_CHUNK;
						aida_dbg_loc(16, yytext, yylloc);
						aida_print_err("illegal char '%s' in attribute key\n", yytext);
						aida_exit(1);
					}

<x_attr_val>[[:alnum:]|]+	 {
						AIDA_POP_CONTEXT; 
						AIDA_SET_LOC_CHUNK;
						yylval->str = yytext;
						aida_dbg_loc(17, yytext, yylloc);
						return ATTRTEXT;
					}

<x_attr_val>\"		{
						AIDA_PUSH_CONTEXT(x_double_quote);
						AIDA_SET_LOC_DOT;
						aida_dbg_loc(18, yytext, yylloc);
						return yytext[0];
					}
					
<x_attr_val>'		{
						AIDA_PUSH_CONTEXT(x_single_quote);
						AIDA_SET_LOC_DOT;
						aida_dbg_loc(19, yytext, yylloc);
						return yytext[0];
					}
					
<x_single_quote,x_double_quote>\[\n\r]	{
						AIDA_SET_LOC_CHUNK;
						yytext[0] = '\n';
						yytext[1] = 0;
						yylval->str = yytext;
						aida_dbg_loc(20, yytext, yylloc);
						return QUOTETEXT;
					}

<x_single_quote,x_double_quote>\\r		{
						AIDA_SET_LOC_CHUNK;
						yytext[0] = '\r';
						yytext[1] = 0;
						yylval->str = yytext;
						aida_dbg_loc(21, yytext, yylloc);
						return QUOTETEXT;
					}

<x_single_quote,x_double_quote>\\t		{
						AIDA_SET_LOC_CHUNK;
						yytext[0] = '\t';
						yytext[1] = 0;
						yylval->str = yytext;
						aida_dbg_loc(22, yytext, yylloc);
						return QUOTETEXT;
					}

<x_single_quote,x_double_quote>\\(.|\n)	{
						// Leave escaped chars unmodified
						AIDA_SET_LOC_CHUNK;
						yylval->str = yytext;
						aida_dbg_loc(23, yytext, yylloc);
						return QUOTETEXT;
					} 
					
<x_double_quote>[^\\\n\"]+ { 
						AIDA_SET_LOC_CHUNK;
						yylval->str = yytext;
						aida_dbg_loc(24, yytext, yylloc);
						return QUOTETEXT;
					} 
					
<x_double_quote>\"		{
						AIDA_POP_CONTEXT;	// out of x_double_quote context
						AIDA_POP_CONTEXT;	// out of x_attr_val context
						AIDA_SET_LOC_DOT;
						aida_dbg_loc(25, yytext, yylloc);
						return yytext[0];
					}

<x_single_quote>[^\\\n']+ { 
						AIDA_SET_LOC_CHUNK;
						yylval->str = yytext;
						aida_dbg_loc(26, yytext, yylloc);
						return QUOTETEXT;
					} 
					
<x_single_quote>'		{
						AIDA_POP_CONTEXT;	// out of x_single_quote context
						AIDA_POP_CONTEXT;	// out of x_attr_val context
						AIDA_SET_LOC_DOT;
						aida_dbg_loc(27, yytext, yylloc);
						return yytext[0];
					}



%{
	// Section tags 
%}

^[ \t]*\(\(s1[ \t\n\r]	{
						AIDA_PUSH_CONTEXT(x_div);
						AIDA_SET_LOC_BTAG; 
						aida_dbg_loc(28, yytext, yylloc);
						return S1_BTAG;
				}
^[ \t]*\(\(s2[ \t\n\r]	{
						AIDA_PUSH_CONTEXT(x_div);
						AIDA_SET_LOC_BTAG; 
						aida_dbg_loc(29, yytext, yylloc);
						return S2_BTAG;
				}
^[ \t]*\(\(s3[ \t\n\r]	{
						AIDA_PUSH_CONTEXT(x_div);
						AIDA_SET_LOC_BTAG; 
						aida_dbg_loc(30, yytext, yylloc);
						return S3_BTAG;
				}
^[ \t]*\(\(s4[ \t\n\r]	{
						AIDA_PUSH_CONTEXT(x_div);
						AIDA_SET_LOC_BTAG; 
						aida_dbg_loc(31, yytext, yylloc);
						return S4_BTAG;
				}
^[ \t]*\(\(s5[ \t\n\r]	{
						AIDA_PUSH_CONTEXT(x_div);
						AIDA_SET_LOC_BTAG; 
						aida_dbg_loc(32, yytext, yylloc);
						return S5_BTAG;
				}
^[ \t]*\(\(s6[ \t\n\r]	{
						AIDA_PUSH_CONTEXT(x_div);
						AIDA_SET_LOC_BTAG; 
						aida_dbg_loc(33, yytext, yylloc);
						return S6_BTAG;
				}



<x_div>[ \t\n\r]s1\)\)	{
						AIDA_POP_CONTEXT; 
						AIDA_SET_LOC_ETAG; 
						aida_dbg_loc(34, yytext, yylloc);
						return S1_ETAG;
				}
<x_div>[ \t\n\r]s2\)\)	{
						AIDA_POP_CONTEXT; 
						AIDA_SET_LOC_ETAG; 
						aida_dbg_loc(35, yytext, yylloc);
						return S2_ETAG;
				}
<x_div>[ \t\n\r]s3\)\)	{
						AIDA_POP_CONTEXT; 
						AIDA_SET_LOC_ETAG; 
						aida_dbg_loc(36, yytext, yylloc);
						return S3_ETAG;
				}
<x_div>[ \t\n\r]s4\)\)	{
						AIDA_POP_CONTEXT; 
						AIDA_SET_LOC_ETAG; 
						aida_dbg_loc(37, yytext, yylloc);
						return S4_ETAG;
				}
<x_div>[ \t\n\r]s5\)\)	{
						AIDA_POP_CONTEXT; 
						AIDA_SET_LOC_ETAG; 
						aida_dbg_loc(38, yytext, yylloc);
						return S5_ETAG;
				}
<x_div>[ \t\n\r]s6\)\)	{
						AIDA_POP_CONTEXT; 
						AIDA_SET_LOC_ETAG; 
						aida_dbg_loc(39, yytext, yylloc);
						return S6_ETAG;
				}

<x_div>.		{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(40, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_div>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(41, yytext, yylloc);
					return BLOCKTEXT;
				}


%{
	// Single tags
%}

^[ \t]*\(\(toc\)\)[ \t]*	{
					AIDA_SET_LOC_BOL;
					aida_assert_result( aida_tocProc(yyextra) );
					aida_dbg_loc(42, yytext, yylloc);
					return TOC_TAG;
				}
				
^[ \t]*\(\(index\)\)[ \t]*	{
					AIDA_SET_LOC_BOL;
					aida_assert_result( aida_printIndexProc(yyextra) );
					aida_dbg_loc(43, yytext, yylloc);
					return INDEX_TAG;
				}

^\(\(nl\)\)\ *		{
					AIDA_SET_LOC_BOL;
					aida_assert_result( aida_newLineProc(yyextra) );
					aida_dbg_loc(44, yytext, yylloc);
					return NL_TAG;
				}

^\(\(hr\)\)		{
					AIDA_SET_LOC_BOL;
					aida_assert_result( aida_horizRuleProc(yyextra) );
					aida_dbg_loc(45, yytext, yylloc);
					return HR_TAG;
				}

^\(\(split\)\)	{
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(46, yytext, yylloc);
					return SPLIT_TAG;
				}


				
%{
	// Style tags
%}

\(\(i[ \t\n\r]	{
					AIDA_PUSH_CONTEXT(s_style);
					AIDA_SET_LOC_BTAG; 
					aida_assert_result( aida_styleTagProc(yytext[2], true, yyextra) );
					aida_dbg_loc(47, yytext, yylloc);
					return I_BTAG;
				}

[ \t\n\r]i\)\)	{
					AIDA_POP_CONTEXT;
					AIDA_SET_LOC_ETAG; 
					aida_assert_result( aida_styleTagProc(yytext[yyleng-3], false, yyextra) );
					aida_dbg_loc(48, yytext, yylloc);
					return I_ETAG;
				}

\(\(b[ \t\n\r]	{
					AIDA_PUSH_CONTEXT(s_style);
					AIDA_SET_LOC_BTAG; 
					aida_assert_result( aida_styleTagProc(yytext[2], true, yyextra) );
					aida_dbg_loc(49, yytext, yylloc);
					return B_BTAG;
				}

[ \t\n\r]b\)\)	{
					AIDA_POP_CONTEXT;
					AIDA_SET_LOC_ETAG; 
					aida_assert_result( aida_styleTagProc(yytext[yyleng-3], false, yyextra) );
					aida_dbg_loc(50, yytext, yylloc);
					return B_ETAG;
				}

\(\(u[ \t\n\r]	{
					AIDA_PUSH_CONTEXT(s_style);
					AIDA_SET_LOC_BTAG; 
					aida_assert_result( aida_styleTagProc(yytext[2], true, yyextra) );
					aida_dbg_loc(51, yytext, yylloc);
					return U_BTAG;
				}

[ \t\n\r]u\)\)	{
					AIDA_POP_CONTEXT;
					AIDA_SET_LOC_ETAG; 
					aida_assert_result( aida_styleTagProc(yytext[yyleng-3], false, yyextra) );
					aida_dbg_loc(52, yytext, yylloc);
					return U_ETAG;
				}

\(\(y[ \t\n\r]	{
					AIDA_PUSH_CONTEXT(s_style);
					AIDA_SET_LOC_BTAG; 
					aida_assert_result( aida_styleTagProc(yytext[2], true, yyextra) );
					aida_dbg_loc(53, yytext, yylloc);
					return Y_BTAG;
				}

[ \t\n\r]y\)\)	{
					AIDA_POP_CONTEXT;
					AIDA_SET_LOC_ETAG; 
					aida_assert_result( aida_styleTagProc(yytext[yyleng-3], false, yyextra) );
					aida_dbg_loc(54, yytext, yylloc);
					return Y_ETAG;
				}

<s_style>[\n\r]	{
					AIDA_SET_LOC_LF;
					if (gUnwrapText) {
						aida_writeString(" ", yyextra);
					} else {
						aida_writeString(yytext, yyextra);
						//yylval->str = " ";
					} 
					yylval->str = yytext;
					aida_dbg_loc(55, yytext, yylloc);
					return PLAINTEXT;
				}



%{
	// Lists 
%}

^[ \t]*\(\(dl[ \t]*	{
					AIDA_PUSH_CONTEXT(x_list_attr); 
					aida_addListDepth("dl",1);
					AIDA_SET_LOC_BOL;						
					aida_dbg_loc(56, yytext, yylloc);
					return DL_BTAG;
				}
				
<x_list_items>^[ \t]*dl\)\)	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(57, yytext, yylloc);
					return DL_ETAG;
				}
				
^[ \t]*\(\(ol[ \t]*	{
					AIDA_PUSH_CONTEXT(x_list_attr); 
					aida_addListDepth("ol",1);
					AIDA_SET_LOC_BOL;						
					aida_dbg_loc(58, yytext, yylloc);
					return OL_BTAG;
				}
				
<x_list_items>^[ \t]*ol\)\)	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(59, yytext, yylloc);
					return OL_ETAG;
				}
				
^[ \t]*\(\(ul[ \t]*	{
					AIDA_PUSH_CONTEXT(x_list_attr); 
					aida_addListDepth("ul",1);
					AIDA_SET_LOC_BOL;						
					aida_dbg_loc(60, yytext, yylloc);
					return UL_BTAG;
				}
				
<x_list_items>^[ \t]*ul\)\)	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(61, yytext, yylloc);
					return UL_ETAG;
				}
	
<x_list_attr>.	{
					// Attributes block of a list
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(62, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_list_attr>[\n\r]	{
					// End of attributes line
					AIDA_POP_CONTEXT;		// out of the <x_list_attr> state
					AIDA_PUSH_CONTEXT(x_list_items); 
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(63, yytext, yylloc);
					return yytext[0];
				}
				
<x_list_items>[ \t]	{
					// Swallow white space between list items
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(64, yytext, yylloc);
				}
				
<x_list_items>[^ \t\n]	{
					// aida_print_err("illegal char '%s' in list environment (missing list item tag?)\n", yytext);
					// aida_exit(1);
					aida_verbose(2,"ignoring char '%s' (%#0x) between list items\n", yytext, yytext[0]&0xff);
					AIDA_SET_LOC_DOT;
					aida_dbg_loc(65, yytext, yylloc);
				}
				
<x_list_items>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(66, yytext, yylloc);
				}
				
<x_list_items>^[ \t]*\(\(li[ \t]*		{
					AIDA_PUSH_CONTEXT(x_li); 
					yyextra->sublevel++;
					AIDA_SET_LOC_BOL;						
					aida_dbg_loc(67, yytext, yylloc);
					return LI_BTAG;
				}
				
<x_li>^[ \t]*\(\(li[ \t]*		{
					yyextra->sublevel++;
					AIDA_SET_LOC_BOL;						
					yylval->str = yytext;
					aida_dbg_loc(68, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_li>[ \t\n\r]li\)\)	{
					AIDA_SET_LOC_ETAG; 
					yyextra->sublevel--;
					if (yyextra->sublevel == 0) {
						AIDA_POP_CONTEXT; 
						aida_dbg_loc(69, yytext, yylloc);
						return LI_ETAG;	
					} else {
						yylval->str = yytext;
						aida_dbg_loc(70, yytext, yylloc);
						return BLOCKTEXT;
					}
				}
				
<x_li>.			{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(71, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_li>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(72, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_list_items>^[ \t]*\(\(lt[ \t]*		{
					AIDA_PUSH_CONTEXT(x_lt_dt); 
					yyextra->sublevel++;
					AIDA_SET_LOC_BOL;						
					aida_dbg_loc(73, yytext, yylloc);
					return LT_BTAG;
				}
				
<x_lt_dd>^[ \t]*\(\(lt[ \t]*		{
					yyextra->sublevel++;
					AIDA_SET_LOC_BOL;						
					yylval->str = yytext;
					aida_dbg_loc(74, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_lt_dd>[ \t\n\r]lt\)\)	{
					AIDA_SET_LOC_ETAG; 
					yyextra->sublevel--;
					if (yyextra->sublevel == 0) {
						AIDA_POP_CONTEXT; 	// out of <x_lt_dd> state
						aida_dbg_loc(75, yytext, yylloc);
						return LT_ETAG;	
					} else {
						yylval->str = yytext;
						aida_dbg_loc(76, yytext, yylloc);
						return BLOCKTEXT;
					}
				}

<x_lt_dt,x_lt_dd>.	{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(77, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_lt_dt>[\n\r]	{
					AIDA_POP_CONTEXT;		// out of <x_lt_dt> state
					AIDA_PUSH_CONTEXT(x_lt_dd); 
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(78, yytext, yylloc);
					return yytext[0];
				}

<x_lt_dd>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(79, yytext, yylloc);
					return BLOCKTEXT;
				}

%{
	// Tables
%}

^\(\(table[ \t]*	{
					AIDA_PUSH_CONTEXT(x_table); 
					AIDA_PUSH_CONTEXT(x_tbl_attr); 
					AIDA_SET_LOC_BOL;						
					aida_dbg_loc(80, yytext, yylloc);
					return TABLE_BTAG;
				}
				
<x_table>table\)\)		{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_CHUNK;
					aida_dbg_loc(81, yytext, yylloc);
					return TABLE_ETAG;
				}
				
<x_tbl_attr>.		{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(82, yytext, yylloc);
					return LINETEXT;
				}
				
<x_tbl_attr>[\n\r]	{
					AIDA_POP_CONTEXT;		// out of <x_tbl_attr> state 
					AIDA_SET_LOC_LF;
					aida_dbg_loc(83, yytext, yylloc);
					return yytext[0];
				}

<x_table>[\n\r]	{
					// Swallow eols between rows
					AIDA_SET_LOC_LF;
					aida_dbg_loc(84, yytext, yylloc);
					return yytext[0];
				}

<x_table>.		{
					AIDA_SET_LOC_DOT;
					aida_dbg_loc(85, yytext, yylloc);
					aida_print_err("illegal char '%s' between table rows\n", yytext);
					aida_exit(1);
				}

<x_table>^\(\(tr[ \t]*	{
					AIDA_PUSH_CONTEXT(x_tbl_row); 
					AIDA_SET_LOC_BOL;						
					aida_dbg_loc(86, yytext, yylloc);
					return TR_BTAG;
				}
				
<x_tbl_row>[ \t]*tr\)\)[ \t]*	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_CHUNK;
					aida_dbg_loc(87, yytext, yylloc);
					return TR_ETAG;
				}
				
<x_tbl_row>.	{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(88, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_tbl_row>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(89, yytext, yylloc);
					return BLOCKTEXT;
				}


%{
	// Links
%}
\(\(a[ \t\n\r]		{
					AIDA_PUSH_CONTEXT(x_anch); 
					AIDA_SET_LOC_BTAG; 
					aida_dbg_loc(90, yytext, yylloc);
					return A_BTAG;
				}
				
<x_anch>[ \t\n\r]a\)\)		{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_ETAG; 
					aida_dbg_loc(91, yytext, yylloc);
					return A_ETAG;
				}
				
<x_anch>.		{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(92, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_anch>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(93, yytext, yylloc);
					return BLOCKTEXT;
				}

\(\(lk[ \t\n\r]	{
					AIDA_PUSH_CONTEXT(x_link); 
					AIDA_SET_LOC_BTAG; 
					aida_dbg_loc(94, yytext, yylloc);
					return LK_BTAG;
				}

<x_link>\[		{
					AIDA_POP_CONTEXT;		// out of <x_link> state
					AIDA_PUSH_CONTEXT(x_link_targ); 
					AIDA_SET_LOC_DOT;
					aida_dbg_loc(95, yytext, yylloc);
					return yytext[0];
				}
				
<x_link>[ \t]	{
					AIDA_SET_LOC_DOT; 
					yylval->str = yytext;
					aida_dbg_loc(96, yytext, yylloc);
				}
				
<x_link>[\n\r]	{
					AIDA_SET_LOC_LF; 
					yylval->str = yytext;
					aida_dbg_loc(97, yytext, yylloc);
				}
				
<x_link>.		{
					AIDA_SET_LOC_DOT;
					aida_dbg_loc(98, yytext, yylloc);
					aida_print_err("illegal char '%s' in link - expected to find a '\['\n", yytext);
					aida_exit(1);
				}
				
<x_link_targ>[^]]*		{
					AIDA_SET_LOC_CHUNK;
					yylval->str = yytext;
					aida_dbg_loc(99, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_link_targ>\]		{
					AIDA_POP_CONTEXT;		// out of <x_link_targ> state
					AIDA_PUSH_CONTEXT(x_link_text); 
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(100, yytext, yylloc);
					return yytext[0];
				}
				
<x_link_text>.	{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(101, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_link_text>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(102, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_link_text>[ \t\n\r]lk\)\)	{
					AIDA_POP_CONTEXT;		// out of <x_link_text> state
					AIDA_SET_LOC_ETAG; 
					aida_dbg_loc(103, yytext, yylloc);
					return LK_ETAG;
				}
				
\(\(rf[ \t\n\r]	{
					AIDA_PUSH_CONTEXT(x_link); 
					AIDA_SET_LOC_BTAG; 
					aida_dbg_loc(104, yytext, yylloc);
					return RF_BTAG;
				}

<x_link_text>[ \t\n\r]rf\)\)	{
					AIDA_POP_CONTEXT;		// out of <x_link_text> state
					AIDA_SET_LOC_ETAG; 
					aida_dbg_loc(105, yytext, yylloc);
					return RF_ETAG;
				}


%{
	// Images
%}

\(\(img[ \t]+	{
					AIDA_PUSH_CONTEXT(x_img); 
					AIDA_SET_LOC_CHUNK;
					aida_dbg_loc(106, yytext, yylloc);
					return IMG_BTAG;
				}
				
<x_img>[ \t\n\r]img\)\)	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_ETAG; 
					aida_dbg_loc(107, yytext, yylloc);
					return IMG_ETAG;
				}
				
<x_img,x_img_name>. 	{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(108, yytext, yylloc);
					return BLOCKTEXT;
				}

<x_img>[\n\r]	{
					AIDA_PUSH_CONTEXT(x_img_name); 
					AIDA_SET_LOC_LF;
					aida_dbg_loc(109, yytext, yylloc);
					return yytext[0];
				}

<x_img_name>[\n\r]	{
					AIDA_SET_LOC_LF;
					aida_dbg_loc(110, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_img_name>[ \t\n\r]img\)\)	{
					AIDA_POP_CONTEXT;	// out of x_img_name context
					// Force to reparse in x_img context
					yyless(0);
					aida_dbg_loc(111, yytext, yylloc);
				}

				
%{
	// Term index
%}
				
\(\(x[ \t\n\r]	{
					AIDA_PUSH_CONTEXT(x_idx); 
					AIDA_SET_LOC_BTAG; 
					aida_dbg_loc(112, yytext, yylloc);
					return X_BTAG;
				}
				
<x_idx>[ \t\n\r]x\)\)	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_ETAG; 
					aida_dbg_loc(113, yytext, yylloc);
					return X_ETAG;
				}

<x_idx>. 		{
					AIDA_SET_LOC_DOT;
					yylval->str = AIDAMAP(yytext);
					aida_dbg_loc(114, yytext, yylloc);
					return BLOCKTEXT;
				}

<x_idx>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(115, yytext, yylloc);
					//return yytext[0];
					return BLOCKTEXT;
				}

%{
	// Inline verbatim
%}
				
\(\(v[ \t\n\r]+	{
					AIDA_PUSH_CONTEXT(x_verb_v);
					AIDA_SET_LOC_BTAG; 
					aida_dbg_loc(116, yytext, yylloc);
					return V_BTAG;
				}
				
<x_verb_v>[ \t\n\r]v\)\)	{
					AIDA_POP_CONTEXT;
					AIDA_SET_LOC_ETAG; 
					aida_dbg_loc(117, yytext, yylloc);
					return V_ETAG;
				}
				
\(\(q[ \t\n\r]	{
					AIDA_PUSH_CONTEXT(x_verb_q);
					AIDA_SET_LOC_BTAG; 
					aida_dbg_loc(118, yytext, yylloc);
					return Q_BTAG;
				}
				
<x_verb_q>[ \t\n\r]q\)\)	{
					AIDA_POP_CONTEXT;
					AIDA_SET_LOC_ETAG; 
					aida_dbg_loc(119, yytext, yylloc);
					return Q_ETAG;
				}
				
<x_verb_v,x_verb_q>. 		{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(120, yytext, yylloc);
					return BLOCKTEXT;
				}

<x_verb_v,x_verb_q>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(121, yytext, yylloc);
					return BLOCKTEXT;
				}

		/* ((v v)) or ((q q)) tags inside a table row or a list item are a pass
		   through filter */
<x_tbl_row,x_li,x_lt_dt,x_lt_dd>\(\(v[ \t\n\r]	{
					AIDA_PUSH_CONTEXT(x_pass_v);
					yyless(1);
					AIDA_SET_LOC_BTAG; 
					yylval->str = yytext;
					aida_dbg_loc(122, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_pass_v>[ \t\n\r]v\)\)	{
					AIDA_POP_CONTEXT;
					yyless(2);
					AIDA_SET_LOC_ETAG; 
					yylval->str = yytext;
					aida_dbg_loc(123, yytext, yylloc);
					return BLOCKTEXT;
				}

<x_tbl_row,x_li,x_lt_dt,x_lt_dd>\(\(q[ \t\n\r]	{
					AIDA_PUSH_CONTEXT(x_pass_q);
					yyless(1);
					AIDA_SET_LOC_BTAG; 
					yylval->str = yytext;
					aida_dbg_loc(124, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_pass_q>[ \t\n\r]q\)\)	{
					AIDA_POP_CONTEXT;
					yyless(2);
					AIDA_SET_LOC_ETAG; 
					yylval->str = yytext;
					aida_dbg_loc(125, yytext, yylloc);
					return BLOCKTEXT;
				}

<x_pass_v,x_pass_q>.		{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(126, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_pass_v,x_pass_q>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(127, yytext, yylloc);
					return BLOCKTEXT;
				}


%{
    // Eval blocks
%}
\(\(e[ \t\n\r]		{
					AIDA_PUSH_CONTEXT(x_eval); 
					AIDA_SET_LOC_BTAG; 
					aida_dbg_loc(128, yytext, yylloc);
					return EVAL_BTAG;
				}
				
<x_eval>[ \t\n\r]e\)\)	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_ETAG; 
					aida_dbg_loc(129, yytext, yylloc);
					return EVAL_ETAG;
				}
				
<x_eval>.		{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(130, yytext, yylloc);
					return BLOCKTEXT;
				}
				
<x_eval>[\n\r]	{
					AIDA_SET_LOC_LF;
					// Replace eol by space (vs 1.4)
					yytext[0] = ' ';
					yylval->str = yytext;
					aida_dbg_loc(131, yytext, yylloc);
					return BLOCKTEXT;
				}


%{
    // If blocks
%}
^[ \t]*\(\(if			{
					AIDA_PUSH_CONTEXT(x_cond); 
					gIfDepth++;
					gIfLevel = gIfDepth;
					aida_dbg_ifconds();
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(132, yytext, yylloc);
					return IF_BTAG;
				}
				
<x_skip_block>^[ \t]*\(\(if	{
					gIfDepth++;
					aida_dbg_ifconds();
					AIDA_SET_LOC_BOL;
				}
				
^[ \t]*\(\(else		{
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(133, yytext, yylloc);
					aida_dbg_ifconds();
					if (gIfDepth == gIfLevel) {
						if (gIfConds[gIfLevel] == true) {
							AIDA_PUSH_CONTEXT(x_skip_block); 
						} 
						return ELSE_TAG;
					} 
				}
				
<x_skip_block>^[ \t]*\(\(else	{
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(134, yytext, yylloc);
					aida_dbg_ifconds();
					if (gIfDepth == gIfLevel) {
						if (gIfConds[gIfLevel] == false) {
							AIDA_POP_CONTEXT;	// out of <x_skip_block> state
						} 
						return ELSE_TAG;
					} 
				}
				
^[ \t]*if\)\)		{
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(135, yytext, yylloc);
					if (gIfDepth == gIfLevel) {
						gIfLevel--;
						gIfDepth--;
						aida_dbg_ifconds();
						return IF_ETAG;
					} else {
						gIfDepth--;
						aida_dbg_ifconds();
					}
				}
				
<x_skip_block>^[ \t]*if\)\)	{
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(136, yytext, yylloc);
					if (gIfDepth == gIfLevel) {
						AIDA_POP_CONTEXT;	// out of <x_skip_block> state
						gIfLevel--;
						gIfDepth--;
						aida_dbg_ifconds();
						return IF_ETAG;
					} else {
						gIfDepth--;
						aida_dbg_ifconds();
					}
				}
				
<x_cond>.		{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(137, yytext, yylloc);
					return LINETEXT;
				}
				
<x_cond>[\n\r]	{
					AIDA_POP_CONTEXT;	// out of <x_cond> state
					// The next two lines are a trick to force the if_start
					// grammar rule action to be executed, i-e the
					// condition to be evaluated. Thus the \n will be
					// seen again by the parser, but this time in <x_if>
					// state (next lexer rule), and the result of the
					// condition can be used.
					AIDA_PUSH_CONTEXT(x_if); 
					unput('\n');
					aida_dbg_loc(138, yytext, yylloc);
					return '\n';
				}

<x_if>[\n\r]	{
					AIDA_POP_CONTEXT;	// out of <x_if> state
					aida_dbg_ifconds();
					if (gIfConds[gIfLevel] == false) {
						AIDA_PUSH_CONTEXT(x_skip_block); 
					} 
					AIDA_SET_LOC_LF;
					aida_dbg_loc(139, yytext, yylloc);
				}

%{
    // Include blocks
%}
^\(\(include[ \t]*	{
					AIDA_PUSH_CONTEXT(x_incl); 
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(140, yytext, yylloc);
					return INCLUDE_TAG;
				}
				
^\(\(input[ \t]*	{
					AIDA_PUSH_CONTEXT(x_incl); 
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(141, yytext, yylloc);
					return INPUT_TAG;
				}
				
<x_incl>.		{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(142, yytext, yylloc);
					return LINETEXT;
				}
				
<x_incl>[\n\r]	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_LF;
					aida_dbg_loc(143, yytext, yylloc);
					return yytext[0];
				}

<x_incl><<EOF>>	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_EOF;
					aida_dbg_loc(144, yytext, yylloc);
					return '\n';
				}

%{
    // Verbatim blocks
%}
^\(\(\|			{
					AIDA_PUSH_CONTEXT(x_verbatim); 
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(145, yytext, yylloc);
					return VERBATIM_BTAG;
				}
				
<x_verbatim>^\|\)\)	{
					AIDA_POP_CONTEXT; 
					AIDA_SET_LOC_BOL;
					aida_dbg_loc(146, yytext, yylloc);
					return VERBATIM_ETAG;
				}

<x_verbatim>.		{
					AIDA_SET_LOC_DOT;
					yylval->str = yytext;
					aida_dbg_loc(147, yytext, yylloc);
					return BLOCKTEXT;
				}

<x_verbatim>[\n\r]	{
					AIDA_SET_LOC_LF;
					yylval->str = yytext;
					aida_dbg_loc(148, yytext, yylloc);
					return BLOCKTEXT;
				}

			
%{
    // Last resort
%}

\n\n+			{
					AIDA_SET_LOC_LF;
					if (gCollapse > 0 && yyleng > gCollapse) {
						yyleng = gCollapse;
						yytext[yyleng]=0;
					} 
					yylloc->last_line += yyleng-1; 
					aida_writeString(yytext, yyextra);
					aida_dbg_loc(149, yytext, yylloc);
					return yytext[0];
				}

[\n\r]			{
					AIDA_SET_LOC_LF;
					if (gUnwrapText) {
						aida_writeString(" ", yyextra);
					} else {
						aida_writeString(yytext, yyextra);
					} 
					aida_dbg_loc(150, yytext, yylloc);
					return yytext[0];
				}

. 				{
					unsigned char ch = yytext[0];
					
					AIDA_SET_LOC_DOT;
					if (gMapChars && gMap[ch] != NULL && gInLen == 0) {					
						aida_writeTclObj(gMap[ch], yyextra);
					} else {
						aida_writeByte(yytext[0], yyextra);
					}
					yylval->str = AIDAMAP(yytext);
					aida_dbg_loc(151, yytext, yylloc);
					return PLAINTEXT;
				}


<<EOF>> 		{
					aida_verbose(3,"scanner %d reached <<EOF>>\n", yyextra->num);
					AIDA_SET_LOC_EOF;
					aida_dbg_loc(152, yytext, yylloc);
					return END_INPUT;
				}

%%  // CODE SECTION


void aida_init_contexts(aida_parse_t * inScanParam)
{
	inScanParam->contexts[0] = 0;
	inScanParam->ctxCount = 0;
	inScanParam->sublevel = 0;
	inScanParam->num = gScanNum++;
#ifdef AIDA_DEBUG_CONTEXT
	printf("Initialized scanner #%d\n", inScanParam->num);
#endif
}


// ------------------------------------------------------------------------
// 
// "aida_push_context" --
// 
// ------------------------------------------------------------------------
static int aida_push_context(int inContext, aida_parse_t * inScanParam) 
{
	inScanParam->ctxCount++;
	if (inScanParam->ctxCount == AIDA_CONTEXTS_MAX) {
		aida_abort("too many nested contexts\n");
	} 
	inScanParam->contexts[inScanParam->ctxCount] = inContext;
#ifdef AIDA_DEBUG_CONTEXT
	printf("\n>>> push(%d) -> ", inScanParam->num); aida_dbg_context(inScanParam);
#endif
	return inContext;
}


// ------------------------------------------------------------------------
// 
// "aida_pop_context" --
// 
// ------------------------------------------------------------------------
static int aida_pop_context(aida_parse_t * inScanParam)
{
	if (inScanParam->ctxCount > 0) {
		inScanParam->ctxCount--;
	} else {
		printf("\n>>> scanner(%d) context stack underflow\n", inScanParam->num);
		aida_exit(1);
	}
#ifdef AIDA_DEBUG_CONTEXT
	printf("\n>>> pop(%d) -> ", inScanParam->num); aida_dbg_context(inScanParam);
#endif
	return inScanParam->contexts[inScanParam->ctxCount];
}


// ------------------------------------------------------------------------
// 
// "yywrap" --
// 
// Provide a default definition for yywrap so that we don't have to link
// with -ll or -lfl. The prototype is automatically inserted by flex in
// aida.yy.c and aida.yy.h.
// 
// ------------------------------------------------------------------------
int yywrap(yyscan_t yyscanner)
{
	return 1;
}




