# ===========================================================================
# File: "aidaUtils.tcl"
#                        Created: 2010-09-25 12:05:28
#              Last modification: 2012-03-28 14:41:45
# Author: Bernard Desgraupes
# e-mail: <bdesgraupes@users.sourceforge.net>
# Description: this file is part of the Aida package.
# It contains various utility procs for the Aida package.
# 
# (c) Copyright: Bernard Desgraupes, 2010-2012
#         All rights reserved.
# This software is free software. 
# See licensing terms in the Aida Help file.
# ===========================================================================

namespace eval Aida {}


proc Aida::quickReference {} {
	set quickref [file join [Aida::pathToSelf] aidaQuickRef.stx]
	if {![file exists $quickref]} {
		alertnote "Couldn't find file '$quickref'"
		return
	} 
	new -n "Aida Syntax Quick Ref" -m Setx -info [file::readAll $quickref] -shell 1
	goto [minPos]
	markFile
}


proc Aida::pathToSelf {} {
	global auto_index
	if {[info exists auto_index(aidaMenu)]} {
		return [file dir [lindex $auto_index(aidaMenu) 1]]
	} else {
		return ""
	}
}


# The 'action' argument can be "insert" or "remove".
proc Aida::strongComment {action} {
	if {![win::checkIfWinToEdit]} {return}
	
	set prefix "!!! "
	set reselect [isSelection]	
	set pos0 [pos::lineStart]
	set lng0 [pos::diff $pos0 [getPos]]
	set offset 0
	set pos1 [selEnd]
	set txt1 [getText $pos0 $pos1]
	set lng1 [string length $txt1]
	set txt2 [doPrefixText $action $prefix $txt1]
	set lng2 [string length $txt2]
	if {$lng1 == $lng2} {return}
	replaceText $pos0 $pos1 $txt2
	set pos0 [pos::math $pos0 + $offset]
	set pos2 [pos::math $pos0 + $lng2]
	if {$reselect} {
		set lng3 [string length $prefix]
		if {$lng0} {set pos0 [pos::math $pos0 + $lng0 + $lng3]}
		selectText $pos0 $pos2
	} else {
		goto [pos::math $pos2 + $lng0 - $lng1]
	}
}


proc Aida::displayBindings {} {
	global mac_p
	binding::displayInfoForType "Aida Menu" "tag" Aida
}


proc Aida::aidaHelp {} {
	set indexurl "http://aidadoc.sourceforge.net/AidaHelp.html"
	url::execute $indexurl
}


proc Aida::buildConvertDialog {} {
	global screenWidth aida_params
	variable hitkn
	variable currTrgtIndex
	
	set wd 300
	set left [expr ($screenWidth - $wd)/2]
	set wb [list 120 $left 250 [expr {$left + $wd}]]
	hi::defaultRect vb
	set dial [hiview root dialog -bounds $wb -title "Aida Conversion" -resizable 0]
	
	# Create the targets popup
	hi::setRect 0 [hi::margin] 15 240
	hi::lf
	set aida_params(targets) [exec aida info targets]
	set hitkn(trgtPopup) [hiview create popupbutton -bounds [hi::rect] \
	  -title "Target: " -items $aida_params(targets)]
	hiview configure $hitkn(trgtPopup) -menuValue $currTrgtIndex
	hiview add $dial $hitkn(trgtPopup)
	
	# Create the check box
	hi::lf
	hi::vskip 10
	hi::sizeRect 250 15
	set hitkn(splitBox) [hiview create checkbox -bounds [hi::rect] \
	  -title "Split the file"]
	hiview add $dial $hitkn(splitBox)
	
	# Row of buttons at the bottom of the dialog
	set label "Process"
	eval hi::setRect [hi::okButtonRect $wb $label]
	set syncBtn [hiview create pushbutton -bounds [hi::rect] -title $label \
	  -command Aida::convertButtonProc -constraints {0 0 1 1} \
	  -help "Perform the synchronization."]
	hiview configure $syncBtn -default 1
	
	set label "Cancel"
	set dims [hi::buttonDims $label]
	hi::moveRect [expr {-[hi::margin] - [lindex $dims 0]}] 0
	hi::sizeRect {*}$dims
	set cnclBtn [hiview create pushbutton -bounds [hi::rect] -title $label \
	  -command hi::exit -constraints {0 0 1 1}]
	hiview configure $cnclBtn -cancel 1
	
	# Embed the buttons
	hiview add $dial $syncBtn $cnclBtn 
	# Declare the default and cancel buttons
	hiview configure $dial -default $syncBtn -cancel $cnclBtn

	# Run the dialog
	hiview show $dial 
}


proc Aida::convertButtonProc {token} {
	global aida_params
	variable hitkn
	variable currTrgtIndex
	
	set splitting [hiview configure $hitkn(splitBox) -value]
	set idx [hiview configure $hitkn(trgtPopup) -menuValue]
	hi::exit $token
	
	set frmt [lindex $aida_params(targets) $idx]
	set currTrgtIndex $idx
	set currFile [win::Current]
	# Look for an :Output: parameter
	set title ""
	set ext ""
	set encOptions [list -from macRoman -to macRoman]
	if {![catch {search -r 1 "^:Output:${frmt}:.*" 0} res]} {
		set line [eval getText $res]
		regsub ":Output:${frmt}:\\s*" $line "" title
	} elseif {![catch {search -r 1 "^:Output:\s.*" 0} res]} {
		set line [eval getText $res]
		regsub ":Output:\\s*" $line "" title
	} else {
		set title [file root [file tail [win::Current]]]
		if {$frmt ne "text" && ![catch {exec aida info ext ${frmt}} res]} {
			set ext $res
		} 
	}
	set title [string trim $title]
	cd [file dir $currFile]
	if {$splitting} {
		set caught [catch {exec aida split -target ${frmt} {*}$encOptions [file tail $currFile]} res]
	} else {
		set caught [catch {exec aida convert -out $title$ext -target ${frmt} {*}$encOptions [file tail $currFile]} res]
	} 
	if {$caught} {
		alertnote $res
	} 	
}