(* See LICENSE for licensing details. *)

(* BEGIN VERSION *)
let major_version  : int           = 2
let minor_version  : int           = 0
let patch_version  : int           = 7
let flavor_version : string option = None
(* END VERSION *)

let version_string =
  Printf.sprintf "%d.%d.%d%s" major_version minor_version patch_version
    (match flavor_version with
     | Some v -> "-" ^ v
     | None -> "")

let version_file_contents =
  let buf = Buffer.create 255 in
  let printf fmt = Printf.bprintf buf (fmt ^^ "\n") in
  printf "(* DO NOT EDIT *)" ;
  printf "(* this file is automatically generated by myocamlbuild.ml *)" ;
  printf "let major : int = %d;;" major_version ;
  printf "let minor : int = %d;;" minor_version ;
  printf "let patch : int = %d;;" patch_version ;
  printf "let flavor : string option = %s;;" begin
    match flavor_version with
    | None -> "None"
    | Some v -> "Some \"" ^ String.escaped v ^ "\""
  end ;
  printf "let version = %S;;" version_string ;
  printf "let self_digest = Digest.file Sys.executable_name;;" ;
  Buffer.contents buf

let version_file = "src/version.ml"

let maybe_make_version_file () =
  if not begin
      Sys.file_exists version_file
      && Digest.file version_file = Digest.string version_file_contents
    end then begin
    Printf.printf "Recreating %S\n" version_file ;
    let oc = open_out version_file in
    output_string oc version_file_contents ;
    close_out oc
  end

open Ocamlbuild_plugin ;;

let () =
  maybe_make_version_file () ;
  Options.use_ocamlfind := true ;
  dispatch begin function
  | After_rules ->
      (* flag ["ocaml" ; "compile"] (A "-g") ; *)
      (* flag ["ocaml" ; "link"] (A "-g") ; *)
      Scanf.sscanf Sys.ocaml_version "%d.%d."
        (fun major minor ->
           if major >= 4 then begin
             flag ["ocaml" ; "compile"] (A "-bin-annot") ;
             if minor >= 2 && minor < 6 then
               flag ["ocaml" ; "compile"] (A "-safe-string") ;
             if minor >= 6 then
               flag ["ocaml" ; "compile"] (A "-strict-formats") ;
           end) ;
      if Sys.os_type = "Unix" then
        flag ["ocaml" ; "compile"] (S [A "-w" ; A "@a-4-29-40-41-42-44-45-48-58-59-60"]) ;
      flag ["ocaml" ; "native" ; "compile"] (A "-nodynlink") ;
  | _ -> ()
  end
