/*
 *      Copyright (C) 2005-2008 Team XBMC
 *      http://www.xbmc.org
 *
 *  This Program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This Program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with XBMC; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *  http://www.gnu.org/copyleft/gpl.html
 *
 */

#include "include.h"
#include "GUIWindow.h"
#include "GUIWindowManager.h"
#include "LocalizeStrings.h"
#include "TextureManager.h"
#include "Settings.h"
#include "GuiControlFactory.h"
#include "GUIControlGroup.h"
#ifdef PRE_SKIN_VERSION_2_1_COMPATIBILITY
#include "GUIListContainer.h"
#include "GUIPanelContainer.h"
#include "GUIEditControl.h"
#endif

#include "SkinInfo.h"
#include "utils/GUIInfoManager.h"
#include "utils/SingleLock.h"
#include "ButtonTranslator.h"
#include "XMLUtils.h"

#ifdef HAS_PERFORMANCE_SAMPLE
#include "utils/PerformanceSample.h"
#endif

using namespace std;

CStdString CGUIWindow::CacheFilename = "";

CGUIWindow::CGUIWindow(DWORD dwID, const CStdString &xmlFile)
{
  m_dwWindowId = dwID;
  m_xmlFile = xmlFile;
  m_dwIDRange = 1;
  m_saveLastControl = false;
  m_dwDefaultFocusControlID = 0;
  m_lastControlID = 0;
  m_focusedControl = 0;
  m_bRelativeCoords = false;
  m_posX = m_posY = m_width = m_height = 0;
  m_overlayState = OVERLAY_STATE_PARENT_WINDOW;   // Use parent or previous window's state
  m_WindowAllocated = false;
  m_coordsRes = g_guiSettings.m_LookAndFeelResolution;
  m_isDialog = false;
  m_needsScaling = true;
  m_visibleCondition = 0;
  m_windowLoaded = false;
  m_loadOnDemand = true;
  m_renderOrder = 0;
  m_dynamicResourceAlloc = true;
  m_hasRendered = false;
  m_hasCamera = false;
  m_previousWindow = WINDOW_INVALID;
  m_animationsEnabled = true;
}

CGUIWindow::~CGUIWindow(void)
{}

void CGUIWindow::FlushReferenceCache()
{
  CacheFilename.clear();
}

bool CGUIWindow::LoadReferences()
{
  // load references.xml
  TiXmlDocument xmlDoc;
  RESOLUTION res;
  CStdString strReferenceFile = g_SkinInfo.GetSkinPath("references.xml", &res);
  // check if we've already loaded it previously
  if (CacheFilename == strReferenceFile)
    return true;

  // nope - time to load it in
  if ( !xmlDoc.LoadFile(strReferenceFile.c_str()) )
  {
//    CLog::Log(LOGERROR, "unable to load:%s, Line %d\n%s", strReferenceFile.c_str(), xmlDoc.ErrorRow(), xmlDoc.ErrorDesc());
    return false;
  }

  CLog::Log(LOGINFO, "Loading references file: %s", strReferenceFile.c_str());
  TiXmlElement* pRootElement = xmlDoc.RootElement();
  CStdString strValue = pRootElement->Value();
  if (strValue != CStdString("controls"))
  {
    CLog::Log(LOGERROR, "references.xml doesn't contain <controls>");
    return false;
  }
  RESOLUTION includeRes;
  g_SkinInfo.GetSkinPath("includes.xml", &includeRes);
  g_SkinInfo.ResolveIncludes(pRootElement);
  CGUIControlFactory factory;
  CStdString strType;
  TiXmlElement *pControl = pRootElement->FirstChildElement();
  TiXmlElement includes("includes");
  while (pControl)
  {
    // ok, this is a <control> block, find the type
    strType = factory.GetType(pControl);
    if (!strType.IsEmpty())
    { // we construct a new <default type="type"> block in our includes document
      TiXmlElement include("default");
      include.SetAttribute("type", strType.c_str());
      // and add the rest of the items under this controlblock to it
      TiXmlElement *child = pControl->FirstChildElement();
      while (child)
      {
        TiXmlElement element(*child);
        // scale element if necessary
        factory.ScaleElement(&element, res, includeRes);
        include.InsertEndChild(element);
        child = child->NextSiblingElement();
      }
      includes.InsertEndChild(include);
    }
    pControl = pControl->NextSiblingElement();
  }
  CacheFilename = strReferenceFile;
  // now load our includes
  g_SkinInfo.LoadIncludes(&includes);
  return true;
}

bool CGUIWindow::Load(const CStdString& strFileName, bool bContainsPath)
{
#ifdef HAS_PERFORMANCE_SAMPLE
  CPerformanceSample aSample("WindowLoad-" + strFileName, true);
#endif

  if (m_windowLoaded)
    return true;      // no point loading if it's already there
    
  LARGE_INTEGER start;
  QueryPerformanceCounter(&start);

  RESOLUTION resToUse = INVALID;
  CLog::Log(LOGINFO, "Loading skin file: %s", strFileName.c_str());
  TiXmlDocument xmlDoc;
  // Find appropriate skin folder + resolution to load from
  CStdString strPath;
  CStdString strLowerPath;
  if (bContainsPath)
    strPath = strFileName;
  else 
  {
    // FIXME: strLowerPath needs to eventually go since resToUse can get incorrectly overridden
    strLowerPath =  g_SkinInfo.GetSkinPath(CStdString(strFileName).ToLower(), &resToUse);
    strPath = g_SkinInfo.GetSkinPath(strFileName, &resToUse);
  }

  if (!bContainsPath)
    m_coordsRes = resToUse;

  bool ret = LoadXML(strPath.c_str(), strLowerPath.c_str());

  LARGE_INTEGER end, freq;
  QueryPerformanceCounter(&end);
  QueryPerformanceFrequency(&freq);
  CLog::Log(LOGDEBUG,"Load %s: %.2fms", m_xmlFile.c_str(), 1000.f * (end.QuadPart - start.QuadPart) / freq.QuadPart);

  return ret;
}

bool CGUIWindow::LoadXML(const CStdString &strPath, const CStdString &strLowerPath)
{
  TiXmlDocument xmlDoc;
  if ( !xmlDoc.LoadFile(strPath.c_str()) && !xmlDoc.LoadFile(CStdString(strPath).ToLower().c_str()) && !xmlDoc.LoadFile(strLowerPath.c_str()))
  {
    CLog::Log(LOGERROR, "unable to load:%s, Line %d\n%s", strPath.c_str(), xmlDoc.ErrorRow(), xmlDoc.ErrorDesc());
#ifdef PRE_SKIN_VERSION_2_1_COMPATIBILITY
    if (g_SkinInfo.GetVersion() < 2.1 && GetID() == WINDOW_VIDEO_NAV && m_xmlFile != "myvideotitle.xml")
    {
      m_xmlFile = "myvideotitle.xml";
      return Load(m_xmlFile);
    }
#endif
    m_dwWindowId = WINDOW_INVALID;
    return false;
  }
  
  return Load(xmlDoc);
}

bool CGUIWindow::Load(TiXmlDocument &xmlDoc)
{
  TiXmlElement* pRootElement = xmlDoc.RootElement();
  if (strcmpi(pRootElement->Value(), "window"))
  {
    CLog::Log(LOGERROR, "file : XML file doesnt contain <window>");
    return false;
  }

  // set the scaling resolution so that any control creation or initialisation can
  // be done with respect to the correct aspect ratio
  g_graphicsContext.SetScalingResolution(m_coordsRes, 0, 0, m_needsScaling);

  // Resolve any includes that may be present
  g_SkinInfo.ResolveIncludes(pRootElement);
  // now load in the skin file
  SetDefaults();

  LoadReferences();
  TiXmlElement *pChild = pRootElement->FirstChildElement();
  while (pChild)
  {
    CStdString strValue = pChild->Value();
    if (strValue == "type" && pChild->FirstChild())
    {
      // if we have are a window type (ie not a dialog), and we have <type>dialog</type>
      // then make this window act like a dialog
      if (!IsDialog() && strcmpi(pChild->FirstChild()->Value(), "dialog") == 0)
        m_isDialog = true;
    }
    else if (strValue == "previouswindow" && pChild->FirstChild())
    {
      m_previousWindow = g_buttonTranslator.TranslateWindowString(pChild->FirstChild()->Value());
    }
    else if (strValue == "defaultcontrol" && pChild->FirstChild())
    {
      const char *always = pChild->Attribute("always");
      if (always && strcmpi(always, "true") == 0)
        m_saveLastControl = false;
      m_dwDefaultFocusControlID = atoi(pChild->FirstChild()->Value());
    }
    else if (strValue == "visible" && pChild->FirstChild())
    {
      CGUIControlFactory::GetConditionalVisibility(pRootElement, m_visibleCondition);
    }
    else if (strValue == "animation" && pChild->FirstChild())
    {
      FRECT rect = { 0, 0, (float)g_settings.m_ResInfo[m_coordsRes].iWidth, (float)g_settings.m_ResInfo[m_coordsRes].iHeight };
      CAnimation anim;
      anim.Create(pChild, rect);
      m_animations.push_back(anim);
    }
    else if (strValue == "zorder" && pChild->FirstChild())
    {
      m_renderOrder = atoi(pChild->FirstChild()->Value());
    }
    else if (strValue == "coordinates")
    {
      // resolve any includes within coordinates tag (such as multiple origin includes)
      g_SkinInfo.ResolveIncludes(pChild);
      TiXmlNode* pSystem = pChild->FirstChild("system");
      if (pSystem)
      {
        int iCoordinateSystem = atoi(pSystem->FirstChild()->Value());
        m_bRelativeCoords = (iCoordinateSystem == 1);
      }

      CGUIControlFactory::GetFloat(pChild, "posx", m_posX);
      CGUIControlFactory::GetFloat(pChild, "posy", m_posY);

      TiXmlElement *originElement = pChild->FirstChildElement("origin");
      while (originElement)
      {
        COrigin origin;
        g_SkinInfo.ResolveConstant(originElement->Attribute("x"), origin.x);
        g_SkinInfo.ResolveConstant(originElement->Attribute("y"), origin.y);
        if (originElement->FirstChild())
          origin.condition = g_infoManager.TranslateString(originElement->FirstChild()->Value());
        m_origins.push_back(origin);
        originElement = originElement->NextSiblingElement("origin");
      }
    }
    else if (strValue == "camera")
    { // z is fixed
      g_SkinInfo.ResolveConstant(pChild->Attribute("x"), m_camera.x);
      g_SkinInfo.ResolveConstant(pChild->Attribute("y"), m_camera.y);
      m_hasCamera = true;
    }
    else if (strValue == "controls")
    {
      // resolve any includes within controls tag (such as whole <control> includes)
      g_SkinInfo.ResolveIncludes(pChild);

      TiXmlElement *pControl = pChild->FirstChildElement();
      while (pControl)
      {
        if (strcmpi(pControl->Value(), "control") == 0)
        {
          LoadControl(pControl, NULL);
        }
        else if (strcmpi(pControl->Value(), "controlgroup") == 0)
        {
          // backward compatibility...
          LoadControl(pControl, NULL);
        }
        pControl = pControl->NextSiblingElement();
      }
    }
    else if (strValue == "allowoverlay")
    {
      bool overlay = false;
      if (XMLUtils::GetBoolean(pRootElement, "allowoverlay", overlay))
        m_overlayState = overlay ? OVERLAY_STATE_SHOWN : OVERLAY_STATE_HIDDEN;
    }

    pChild = pChild->NextSiblingElement();
  }

  m_windowLoaded = true;
  OnWindowLoaded();
  return true;
}

void CGUIWindow::LoadControl(TiXmlElement* pControl, CGUIControlGroup *pGroup)
{
  // get control type
  CGUIControlFactory factory;

  FRECT rect = { 0, 0, (float)g_settings.m_ResInfo[m_coordsRes].iWidth, (float)g_settings.m_ResInfo[m_coordsRes].iHeight };
  if (pGroup)
  {
    rect.left = pGroup->GetXPosition();
    rect.top = pGroup->GetYPosition();
    rect.right = rect.left + pGroup->GetWidth();
    rect.bottom = rect.top + pGroup->GetHeight();
  }
  CGUIControl* pGUIControl = factory.Create(m_dwWindowId, rect, pControl);
  if (pGUIControl)
  {
    float maxX = pGUIControl->GetXPosition() + pGUIControl->GetWidth();
    if (maxX > m_width)
    {
      m_width = maxX;
    }

    float maxY = pGUIControl->GetYPosition() + pGUIControl->GetHeight();
    if (maxY > m_height)
    {
      m_height = maxY;
    }
    // if we are in a group, add to the group, else add to our window
    pGUIControl->SetParentControl(pGroup);
    if (pGroup)
      pGroup->AddControl(pGUIControl);
    else
      Add(pGUIControl);
#ifdef PRE_SKIN_VERSION_2_1_COMPATIBILITY
    if (pGUIControl->GetControlType() == CGUIControl::GUICONTAINER_LIST)
    {
      CGUIListContainer *list = (CGUIListContainer *)pGUIControl;
      if (list->m_spinControl)
      {
        list->m_spinControl->SetParentControl(pGroup);
        if (pGroup)
          pGroup->AddControl(list->m_spinControl);
        else
          Add(list->m_spinControl);
        list->m_spinControl = NULL;
      }
    }
    if (pGUIControl->GetControlType() == CGUIControl::GUICONTAINER_PANEL)
    {
      CGUIPanelContainer *panel = (CGUIPanelContainer *)pGUIControl;
      if (panel->m_spinControl)
      {
        panel->m_spinControl->SetParentControl(pGroup);
        if (pGroup)
          pGroup->AddControl(panel->m_spinControl);
        else
          Add(panel->m_spinControl);
        panel->m_spinControl = NULL;
      }
      if (panel->m_largePanel)
      {
        panel->m_largePanel->SetParentControl(pGroup);
        if (pGroup)
          pGroup->AddControl(panel->m_largePanel);
        else
          Add(panel->m_largePanel);
        panel->m_largePanel = NULL;
      }
    }
#endif
    // if the new control is a group, then add it's controls
    if (pGUIControl->IsGroup())
    {
      TiXmlElement *pSubControl = pControl->FirstChildElement("control");
      while (pSubControl)
      {
        LoadControl(pSubControl, (CGUIControlGroup *)pGUIControl);
        pSubControl = pSubControl->NextSiblingElement("control");
      }
    }
  }
}

void CGUIWindow::OnWindowLoaded() 
{
  DynamicResourceAlloc(true);
#ifdef PRE_SKIN_VERSION_2_1_COMPATIBILITY
  // we hook up the controlgroup navigation as desired
  if (g_SkinInfo.GetVersion() < 2.1)
  { // run through controls, and check navigation into a controlgroup
    for (ivecControls i = m_vecControls.begin(); i != m_vecControls.end(); ++i)
    {
      CGUIControl *group = *i;
      if (group->IsGroup())
      {
        // first thing first: We have to have a unique id
        if (!group->GetID())
        {
          DWORD id = 9000;
          while (GetControl(id++) && id < 9100)
            ;
          group->SetID(id);
        }
      }
    }
  }
#endif
}

void CGUIWindow::SetPosition(float posX, float posY)
{
  m_posX = posX;
  m_posY = posY;
}

void CGUIWindow::CenterWindow()
{
  if (m_bRelativeCoords)
  {
    m_posX = (g_settings.m_ResInfo[m_coordsRes].iWidth - GetWidth()) / 2;
    m_posY = (g_settings.m_ResInfo[m_coordsRes].iHeight - GetHeight()) / 2;
  }
}

void CGUIWindow::Render()
{
  // If we're rendering from a different thread, then we should wait for the main
  // app thread to finish AllocResources(), as dynamic resources (images in particular)
  // will try and be allocated from 2 different threads, which causes nasty things
  // to occur.
  if (!m_WindowAllocated) return;

  // find our origin point
  float posX = m_posX;
  float posY = m_posY;
  for (unsigned int i = 0; i < m_origins.size(); i++)
  {
    // no condition implies true
    if (!m_origins[i].condition || g_infoManager.GetBool(m_origins[i].condition, GetID()))
    { // found origin
      posX = m_origins[i].x;
      posY = m_origins[i].y;
      break;
    }
  }
  g_graphicsContext.SetRenderingResolution(m_coordsRes, posX, posY, m_needsScaling);
  if (m_hasCamera)
    g_graphicsContext.SetCameraPosition(m_camera);

  DWORD currentTime = timeGetTime();
  // render our window animation - returns false if it needs to stop rendering
  if (!RenderAnimation(currentTime))
    return;

  for (int i = 0; i < (int)m_vecControls.size(); i++)
  {
    CGUIControl *pControl = m_vecControls[i];
    if (pControl)
    {
      pControl->UpdateVisibility();
      pControl->DoRender(currentTime);
    }
  }
  m_hasRendered = true;
}

void CGUIWindow::Close(bool forceClose)
{
  CLog::Log(LOGERROR,"%s - should never be called on the base class!", __FUNCTION__);
}

bool CGUIWindow::OnAction(const CAction &action)
{
  if (action.wID == ACTION_MOUSE)
    return OnMouseAction();

  CGUIControl *focusedControl = GetFocusedControl();
  if (focusedControl)
    return focusedControl->OnAction(action);

  // no control has focus?
  // set focus to the default control then
  CGUIMessage msg(GUI_MSG_SETFOCUS, GetID(), m_dwDefaultFocusControlID);
  OnMessage(msg);
  return false;
}

// OnMouseAction - called by OnAction()
bool CGUIWindow::OnMouseAction()
{
  // we need to convert the mouse coordinates to window coordinates
  float posX = m_posX;
  float posY = m_posY;
  for (unsigned int i = 0; i < m_origins.size(); i++)
  {
    // no condition implies true
    if (!m_origins[i].condition || g_infoManager.GetBool(m_origins[i].condition, GetID()))
    { // found origin
      posX = m_origins[i].x;
      posY = m_origins[i].y;
      break;
    }
  }
  g_graphicsContext.SetScalingResolution(m_coordsRes, posX, posY, m_needsScaling);
  CPoint mousePoint(g_Mouse.GetLocation());
  g_graphicsContext.InvertFinalCoords(mousePoint.x, mousePoint.y);
  m_transform.InverseTransformPosition(mousePoint.x, mousePoint.y);

  bool bHandled = false;
  // check if we have exclusive access
  if (g_Mouse.GetExclusiveWindowID() == GetID())
  { // we have exclusive access to the mouse...
    CGUIControl *pControl = (CGUIControl *)GetControl(g_Mouse.GetExclusiveControlID());
    if (pControl)
    { // this control has exclusive access to the mouse
      HandleMouse(pControl, mousePoint + g_Mouse.GetExclusiveOffset());
      return true;
    }
  }

  // run through the controls, and unfocus all those that aren't under the pointer,
  for (ivecControls i = m_vecControls.begin(); i != m_vecControls.end(); ++i)
  {
    CGUIControl *pControl = *i;
    pControl->UnfocusFromPoint(mousePoint);
  }
  // and find which one is under the pointer
  // go through in reverse order to make sure we start with the ones on top
  bool controlUnderPointer(false);
  for (vector<CGUIControl *>::reverse_iterator i = m_vecControls.rbegin(); i != m_vecControls.rend(); ++i)
  {
    CGUIControl *pControl = *i;
    CGUIControl *focusableControl = NULL;
    CPoint controlPoint;
    if (pControl->CanFocusFromPoint(mousePoint, &focusableControl, controlPoint))
    {
      controlUnderPointer = focusableControl->OnMouseOver(controlPoint);
      bHandled = HandleMouse(focusableControl, controlPoint);
      if (bHandled || controlUnderPointer)
        break;
    }
  }
  if (!bHandled)
  { // haven't handled this action - call the window message handlers
    bHandled = OnMouse(mousePoint);
  }
  // and unfocus everything otherwise
  if (!controlUnderPointer)
    m_focusedControl = 0;
  return bHandled;
}

// Handles any mouse actions that are not handled by a control
// default is to go back a window on a right click.
// This function should be overridden for other windows
bool CGUIWindow::OnMouse(const CPoint &point)
{
  if (g_Mouse.bClick[MOUSE_RIGHT_BUTTON])
  { // no control found to absorb this click - go to previous menu
    CAction action;
    action.wID = ACTION_PREVIOUS_MENU;
    return OnAction(action);
  }
  return false;
}

bool CGUIWindow::HandleMouse(CGUIControl *pControl, const CPoint &point)
{
  if (g_Mouse.bClick[MOUSE_LEFT_BUTTON])
  { // Left click
    return pControl->OnMouseClick(MOUSE_LEFT_BUTTON, point);
  }
  else if (g_Mouse.bClick[MOUSE_RIGHT_BUTTON])
  { // Right click
    return pControl->OnMouseClick(MOUSE_RIGHT_BUTTON, point);
  }
  else if (g_Mouse.bClick[MOUSE_MIDDLE_BUTTON])
  { // Middle click
    return pControl->OnMouseClick(MOUSE_MIDDLE_BUTTON, point);
  }
  else if (g_Mouse.bDoubleClick[MOUSE_LEFT_BUTTON])
  { // Left double click
    return pControl->OnMouseDoubleClick(MOUSE_LEFT_BUTTON, point);
  }
  else if (g_Mouse.bHold[MOUSE_LEFT_BUTTON] && g_Mouse.HasMoved())
  { // Mouse Drag
    return pControl->OnMouseDrag(g_Mouse.GetLastMove(), point);
  }
  else if (g_Mouse.GetWheel())
  { // Mouse wheel
    return pControl->OnMouseWheel(g_Mouse.GetWheel(), point);
  }
  // no mouse stuff done other than movement
  return false;
}

DWORD CGUIWindow::GetID(void) const
{
  return m_dwWindowId;
}

void CGUIWindow::SetID(DWORD dwID)
{
  m_dwWindowId = dwID;
}

/// \brief Called on window open.
///  * Restores the control state(s)
///  * Sets initial visibility of controls
///  * Queue WindowOpen animation
///  * Set overlay state
/// Override this function and do any window-specific initialisation such
/// as filling control contents and setting control focus before
/// calling the base method.
void CGUIWindow::OnInitWindow()
{
  // set our rendered state
  m_hasRendered = false;
  ResetAnimations();  // we need to reset our animations as those windows that don't dynamically allocate
                      // need their anims reset. An alternative solution is turning off all non-dynamic
                      // allocation (which in some respects may be nicer, but it kills hdd spindown and the like)

  // set our initial control visibility before restoring control state and
  // focusing the default control, and again afterward to make sure that
  // any controls that depend on the state of the focused control (and or on
  // control states) are active.
  SetControlVisibility();
  RestoreControlStates();
  SetControlVisibility();
  QueueAnimation(ANIM_TYPE_WINDOW_OPEN);
  m_gWindowManager.ShowOverlay(m_overlayState);
}

// Called on window close.
//  * Executes the window close animation(s)
//  * Saves control state(s)
// Override this function and call the base class before doing any dynamic memory freeing
void CGUIWindow::OnDeinitWindow(int nextWindowID)
{
  if (nextWindowID != WINDOW_FULLSCREEN_VIDEO)
  {
    // Dialog animations are handled in Close() rather than here
    if (HasAnimation(ANIM_TYPE_WINDOW_CLOSE) && !IsDialog() && IsActive())
    {
      // Perform the window out effect
      QueueAnimation(ANIM_TYPE_WINDOW_CLOSE);
      while (IsAnimating(ANIM_TYPE_WINDOW_CLOSE))
      {
        m_gWindowManager.Process(true);
      }
    }
  }
  SaveControlStates();
}

bool CGUIWindow::OnMessage(CGUIMessage& message)
{
  switch ( message.GetMessage() )
  {
  case GUI_MSG_WINDOW_INIT:
    {
      OutputDebugString("------------------- GUI_MSG_WINDOW_INIT ");
      OutputDebugString(g_localizeStrings.Get(GetID()).c_str());
      OutputDebugString("------------------- \n");
      if (m_dynamicResourceAlloc || !m_WindowAllocated) AllocResources();
      OnInitWindow();
      return true;
    }
    break;

  case GUI_MSG_WINDOW_DEINIT:
    {
      OutputDebugString("------------------- GUI_MSG_WINDOW_DEINIT ");
      OutputDebugString(g_localizeStrings.Get(GetID()).c_str());
      OutputDebugString("------------------- \n");
      OnDeinitWindow(message.GetParam1());
      // now free the window
      if (m_dynamicResourceAlloc) FreeResources();
      return true;
    }
    break;

  case GUI_MSG_CLICKED:
    {
      // a specific control was clicked
      CLICK_EVENT clickEvent = m_mapClickEvents[ message.GetSenderId() ];

      // determine if there are any handlers for this event
      if (clickEvent.HasAHandler())
      {
        // fire the message to all handlers
        clickEvent.Fire(message);
      }
      break;
    }

  case GUI_MSG_SELCHANGED:
    {
      // a selection within a specific control has changed
      SELECTED_EVENT selectedEvent = m_mapSelectedEvents[ message.GetSenderId() ];

      // determine if there are any handlers for this event
      if (selectedEvent.HasAHandler())
      {
        // fire the message to all handlers
        selectedEvent.Fire(message);
      }
      break;
    }
  case GUI_MSG_FOCUSED:
    { // a control has been focused
      if (HasID(message.GetSenderId()))
      {
        m_focusedControl = message.GetControlId();
        return true;
      }
      break;
    }
  case GUI_MSG_MOVE:
    {
      if (HasID(message.GetSenderId()))
        return OnMove(message.GetControlId(), message.GetParam1());
      break;
    }
  case GUI_MSG_SETFOCUS:
    {
//      CLog::Log(LOGDEBUG,"set focus to control:%i window:%i (%i)\n", message.GetControlId(),message.GetSenderId(), GetID());
      if ( message.GetControlId() )
      {
#ifdef PRE_SKIN_VERSION_2_1_COMPATIBILITY
        if (g_SkinInfo.GetVersion() < 2.1)
        { // to support the best backwards compatibility, we reproduce the old method here
          const CGUIControl *oldGroup = NULL;
          // first unfocus the current control
          CGUIControl *control = GetFocusedControl();
          if (control)
          {
            CGUIMessage msgLostFocus(GUI_MSG_LOSTFOCUS, GetID(), control->GetID(), message.GetControlId());
            control->OnMessage(msgLostFocus);
            oldGroup = control->GetParentControl();
          }
          // get the control to focus
          CGUIControl* pFocusedControl = GetFirstFocusableControl(message.GetControlId());
          if (!pFocusedControl) pFocusedControl = (CGUIControl *)GetControl(message.GetControlId());

          // and focus it
          if (pFocusedControl)
          {
            // check for group changes
            if (pFocusedControl->GetParentControl() && pFocusedControl->GetParentControl() != oldGroup)
            { // going to a different group, focus the group instead
              CGUIControlGroup *group = (CGUIControlGroup *)pFocusedControl->GetParentControl();
              if (group->GetFocusedControlID())
                pFocusedControl = group;
            }
            return pFocusedControl->OnMessage(message);
          }
        }
        else
          {
#endif
        // first unfocus the current control
        CGUIControl *control = GetFocusedControl();
        if (control)
        {
          CGUIMessage msgLostFocus(GUI_MSG_LOSTFOCUS, GetID(), control->GetID(), message.GetControlId());
          control->OnMessage(msgLostFocus);
        }

        // get the control to focus
        CGUIControl* pFocusedControl = GetFirstFocusableControl(message.GetControlId());
        if (!pFocusedControl) pFocusedControl = (CGUIControl *)GetControl(message.GetControlId());

        // and focus it
        if (pFocusedControl)
          return pFocusedControl->OnMessage(message);
#ifdef PRE_SKIN_VERSION_2_1_COMPATIBILITY
        }
#endif
      }
      return true;
    }
    break;
  case GUI_MSG_NOTIFY_ALL:
    {
      // only process those notifications that come from this window, or those intended for every window
      if (HasID(message.GetSenderId()) || !message.GetSenderId())
      {
        if (message.GetParam1() == GUI_MSG_PAGE_CHANGE ||
            message.GetParam1() == GUI_MSG_REFRESH_THUMBS ||
            message.GetParam1() == GUI_MSG_REFRESH_LIST)
        { // alter the message accordingly, and send to all controls
          for (ivecControls it = m_vecControls.begin(); it != m_vecControls.end(); ++it)
          {
            CGUIControl *control = *it;
            CGUIMessage msg(message.GetParam1(), message.GetControlId(), control->GetID(), message.GetParam2());
            control->OnMessage(msg);
          }
        }
      }
    }
    break;
  }

  ivecControls i;
  // Send to the visible matching control first
  for (i = m_vecControls.begin();i != m_vecControls.end(); ++i)
  {
    CGUIControl* pControl = *i;
    if (pControl->HasVisibleID(message.GetControlId()))
    {
      if (pControl->OnMessage(message))
        return true;
    }
  }
  // Unhandled - send to all matching invisible controls as well
  bool handled(false);
  for (i = m_vecControls.begin();i != m_vecControls.end(); ++i)
  {
    CGUIControl* pControl = *i;
    if (pControl->HasID(message.GetControlId()))
    {
      if (pControl->OnMessage(message))
        handled = true;
    }
  }
  return handled;
}

void CGUIWindow::AllocResources(bool forceLoad /*= FALSE */)
{
  CSingleLock lock(g_graphicsContext);

  LARGE_INTEGER start;
  QueryPerformanceCounter(&start);

  // load skin xml file
  bool bHasPath=false; 
  if (m_xmlFile.Find("\\") > -1 || m_xmlFile.Find("/") > -1 ) 
    bHasPath = true; 
  if (m_xmlFile.size() && (forceLoad || m_loadOnDemand || !m_windowLoaded))
    Load(m_xmlFile,bHasPath);

  LARGE_INTEGER slend;
  QueryPerformanceCounter(&slend);

  // and now allocate resources
  g_TextureManager.StartPreLoad();
  ivecControls i;
  for (i = m_vecControls.begin();i != m_vecControls.end(); ++i)
  {
    CGUIControl* pControl = *i;
    if (!pControl->IsDynamicallyAllocated()) 
      pControl->PreAllocResources();
  }
  g_TextureManager.EndPreLoad();

  LARGE_INTEGER plend;
  QueryPerformanceCounter(&plend);

  for (i = m_vecControls.begin();i != m_vecControls.end(); ++i)
  {
    CGUIControl* pControl = *i;
    if (!pControl->IsDynamicallyAllocated()) 
      pControl->AllocResources();
  }
  g_TextureManager.FlushPreLoad();

  LARGE_INTEGER end, freq;
  QueryPerformanceCounter(&end);
  QueryPerformanceFrequency(&freq);
  CLog::Log(LOGDEBUG,"Alloc resources: %.2fms (%.2f ms skin load, %.2f ms preload)", 1000.f * (end.QuadPart - start.QuadPart) / freq.QuadPart, 1000.f * (slend.QuadPart - start.QuadPart) / freq.QuadPart, 1000.f * (plend.QuadPart - slend.QuadPart) / freq.QuadPart);

  m_WindowAllocated = true;
}

void CGUIWindow::FreeResources(bool forceUnload /*= FALSE */)
{
  m_WindowAllocated = false;
  ivecControls i;
  for (i = m_vecControls.begin();i != m_vecControls.end(); ++i)
  {
    CGUIControl* pControl = *i;
    pControl->FreeResources();
  }
  //g_TextureManager.Dump();
  // unload the skin
  if (m_loadOnDemand || forceUnload) ClearAll();
}

void CGUIWindow::DynamicResourceAlloc(bool bOnOff)
{
  m_dynamicResourceAlloc = bOnOff;
  for (ivecControls i = m_vecControls.begin();i != m_vecControls.end(); ++i)
  {
    CGUIControl* pControl = *i;
    pControl->DynamicResourceAlloc(bOnOff);
  }
}

void CGUIWindow::Add(CGUIControl* pControl)
{
  m_vecControls.push_back(pControl);
}

void CGUIWindow::Insert(CGUIControl *control, const CGUIControl *insertPoint)
{
  // get the insertion point
  for (unsigned int i = 0; i < m_vecControls.size(); i++)
  {
    CGUIControl *child = m_vecControls[i];
    if (child->IsGroup() && ((CGUIControlGroup *)child)->InsertControl(control, insertPoint))
      return;
    else if (child == insertPoint)
    {
      m_vecControls.insert(m_vecControls.begin() + i, control);
      return;
    }
  }
}

// Note: This routine doesn't delete the control.  It just removes it from the control list
bool CGUIWindow::Remove(const CGUIControl *control)
{
  for (ivecControls i = m_vecControls.begin(); i != m_vecControls.end(); i++)
  {
    CGUIControl* pControl = *i;
    if (pControl->IsGroup() && ((CGUIControlGroup *)pControl)->RemoveControl(control))
      return true;
    if (pControl == control)
    {
      m_vecControls.erase(i);
      return true;
    }
  }
  return false;
}

void CGUIWindow::ClearAll()
{
  OnWindowUnload();

  for (int i = 0; i < (int)m_vecControls.size(); ++i)
  {
    CGUIControl* pControl = m_vecControls[i];
    delete pControl;
  }
  m_vecControls.erase(m_vecControls.begin(), m_vecControls.end());
  m_windowLoaded = false;
  m_dynamicResourceAlloc = true;
}

const CGUIControl* CGUIWindow::GetControl(int iControl) const
{
  const CGUIControl* pPotential=NULL;
  for (int i = 0;i < (int)m_vecControls.size(); ++i)
  {
    const CGUIControl* pControl = m_vecControls[i];
    if (pControl->IsGroup())
    {
      CGUIControlGroup *group = (CGUIControlGroup *)pControl;
      const CGUIControl *control = group->GetControl(iControl);
      if (control) pControl = control;
    }
    if ((int) pControl->GetID() == iControl) 
    {
      if (pControl->IsVisible())
        return pControl;
      else if (!pPotential)
        pPotential = pControl;
    }
  }
  return pPotential;
}

int CGUIWindow::GetFocusedControlID() const
{
  if (m_focusedControl) return m_focusedControl;
  CGUIControl *control = GetFocusedControl();
  if (control) return control->GetID();
  return -1;
}

CGUIControl *CGUIWindow::GetFocusedControl() const
{
  for (vector<CGUIControl *>::const_iterator it = m_vecControls.begin(); it != m_vecControls.end(); ++it)
  {
    const CGUIControl* pControl = *it;
    if (pControl->HasFocus())
    {
      if (pControl->IsGroup())
      {
        CGUIControlGroup *group = (CGUIControlGroup *)pControl;
        return group->GetFocusedControl();
      }
      return (CGUIControl *)pControl;
    }
  }
  return NULL;
}

bool CGUIWindow::Initialize()
{
  return Load(m_xmlFile);
}

void CGUIWindow::SetControlVisibility()
{
  // reset our info manager caches
  g_infoManager.ResetCache();
  for (unsigned int i=0; i < m_vecControls.size(); i++)
  {
    CGUIControl *pControl = m_vecControls[i];
    pControl->SetInitialVisibility();
  }
}

bool CGUIWindow::IsActive() const
{
  return m_gWindowManager.IsWindowActive(GetID());
}

void CGUIWindow::QueueAnimation(ANIMATION_TYPE animType)
{
  // special cases first
  if (animType == ANIM_TYPE_WINDOW_CLOSE)
  {
    if (!m_WindowAllocated || !m_hasRendered) // can't render an animation if we aren't allocated or haven't rendered
      return;
    // make sure we update our visibility prior to queuing the window close anim
    for (unsigned int i = 0; i < m_vecControls.size(); i++)
      m_vecControls[i]->UpdateVisibility();
  }
  // we first check whether the reverse animation is in progress (and reverse it)
  // then we check for the normal animation, and queue it
  CAnimation *reverse = GetAnimation((ANIMATION_TYPE)-animType, false);
  CAnimation *forward = GetAnimation(animType);
  if (reverse && reverse->IsReversible() && (reverse->GetState() == ANIM_STATE_IN_PROCESS || reverse->GetState() == ANIM_STATE_DELAYED))
  {
    reverse->QueueAnimation(ANIM_PROCESS_REVERSE);
    if (forward) forward->ResetAnimation();
  }
  else if (forward)
  {
    forward->QueueAnimation(ANIM_PROCESS_NORMAL);
    if (reverse) reverse->ResetAnimation();
  }
  else if (reverse)
    reverse->ResetAnimation();

  // and queue any anims for the controls as well
  for (unsigned int i = 0; i < m_vecControls.size(); i++)
  {
    CGUIControl *pControl = m_vecControls[i];
    pControl->QueueAnimation(animType);
  }
}

CAnimation *CGUIWindow::GetAnimation(ANIMATION_TYPE animType, bool checkConditions)
{
  for (unsigned int i = 0; i < m_animations.size(); i++)
  {
    CAnimation &anim = m_animations[i];
    if (anim.GetType() == animType)
    {
      if (!checkConditions || (!anim.GetCondition() || g_infoManager.GetBool(anim.GetCondition())))
        return &anim;
    }
  }
  return NULL;
}

bool CGUIWindow::IsAnimating(ANIMATION_TYPE animType)
{
  if (!m_animationsEnabled) return false;
  
  for (unsigned int i = 0; i < m_animations.size(); i++)
  {
    CAnimation &anim = m_animations[i];
    if (anim.GetType() == animType)
    {
      if (anim.GetQueuedProcess() == ANIM_PROCESS_NORMAL)
        return true;
      if (anim.GetProcess() == ANIM_PROCESS_NORMAL)
        return true;
    }
    else if (anim.GetType() == -animType)
    {
      if (anim.GetQueuedProcess() == ANIM_PROCESS_REVERSE)
        return true;
      if (anim.GetProcess() == ANIM_PROCESS_REVERSE)
        return true;
    }
  }
  for (unsigned int i = 0; i < m_vecControls.size(); i++)
  {
    CGUIControl *pControl = m_vecControls[i];
    if (pControl->IsAnimating(animType)) return true;
  }
  return false;
}

bool CGUIWindow::RenderAnimation(DWORD time)
{
  m_transform.Reset();
  if (m_animationsEnabled)
  {
    CPoint center(m_posX + m_width * 0.5f, m_posY + m_height * 0.5f);
    // show animation
    for (unsigned int i = 0; i < m_animations.size(); i++)
    {
      CAnimation &anim = m_animations[i];
      anim.Animate(time, true);
      UpdateStates(anim.GetType(), anim.GetProcess(), anim.GetState());
      anim.RenderAnimation(m_transform, center);
    }
  }
  g_graphicsContext.SetWindowTransform(m_transform);
  return true;
}

void CGUIWindow::UpdateStates(ANIMATION_TYPE type, ANIMATION_PROCESS currentProcess, ANIMATION_STATE currentState)
{
}

bool CGUIWindow::HasAnimation(ANIMATION_TYPE animType)
{
  for (unsigned int i = 0; i < m_animations.size(); i++)
  {
    CAnimation &anim = m_animations[i];
    if (anim.GetType() == animType && (!anim.GetCondition() || g_infoManager.GetBool(anim.GetCondition())))
      return true;
  }

  // Now check the controls to see if we have this animation
  for (unsigned int i = 0; i < m_vecControls.size(); i++)
    if (m_vecControls[i]->HasAnimation(animType)) return true;
  return false;
}

void CGUIWindow::DisableAnimations()
{
  m_animationsEnabled = false;
}

void CGUIWindow::ResetAnimations()
{
  for (unsigned int i = 0; i < m_animations.size(); i++)
    m_animations[i].ResetAnimation();

  for (ivecControls it = m_vecControls.begin(); it != m_vecControls.end(); ++it)
    (*it)->ResetAnimations();
}

// returns true if the control group with id groupID has controlID as
// its focused control
bool CGUIWindow::ControlGroupHasFocus(int groupID, int controlID)
{
  // 1.  Run through and get control with groupID (assume unique)
  // 2.  Get it's selected item.
#ifdef PRE_SKIN_VERSION_2_1_COMPATIBILITY
  if (g_SkinInfo.GetVersion() < 2.1)
    groupID += 9000;
#endif
  CGUIControl *group = GetFirstFocusableControl(groupID);
  if (!group) group = (CGUIControl *)GetControl(groupID);

  if (group && group->IsGroup())
  {
    if (controlID == 0)
    { // just want to know if the group is focused
      return group->HasFocus();
    }
    else
    {
      CGUIMessage message(GUI_MSG_ITEM_SELECTED, GetID(), group->GetID());
      group->OnMessage(message);
      return (controlID == (int) message.GetParam1());
    }
  }
  return false;
}

void CGUIWindow::SaveControlStates()
{
  ResetControlStates();
  if (m_saveLastControl)
    m_lastControlID = GetFocusedControlID();
  for (ivecControls it = m_vecControls.begin(); it != m_vecControls.end(); ++it)
    (*it)->SaveStates(m_controlStates);
}

void CGUIWindow::RestoreControlStates()
{
  for (vector<CControlState>::iterator it = m_controlStates.begin(); it != m_controlStates.end(); ++it)
  {
    CGUIMessage message(GUI_MSG_ITEM_SELECT, GetID(), (*it).m_id, (*it).m_data);
    OnMessage(message);
  }
  int focusControl = (m_saveLastControl && m_lastControlID) ? m_lastControlID : m_dwDefaultFocusControlID;
#ifdef PRE_SKIN_VERSION_2_1_COMPATIBILITY
  if (g_SkinInfo.GetVersion() < 2.1)
  { // skins such as mc360 focus a control in a group by default.
    // In 2.1 they should set the focus to the group, rather than the control in the group
    CGUIControl *control = GetFirstFocusableControl(focusControl);
    if (!control) control = (CGUIControl *)GetControl(focusControl);
    if (control && control->GetParentControl())
    {
      CGUIControlGroup *group = (CGUIControlGroup *)control->GetParentControl();
      if (group->GetFocusedControlID())
        focusControl = group->GetID();
    }
  }
#endif
  SET_CONTROL_FOCUS(focusControl, 0);
}

void CGUIWindow::ResetControlStates()
{
  m_lastControlID = 0;
  m_focusedControl = 0;
  m_controlStates.clear();
}

// find the first focusable control with this id.
// if no focusable control exists with this id, return NULL
CGUIControl *CGUIWindow::GetFirstFocusableControl(int id)
{
  for (ivecControls i = m_vecControls.begin();i != m_vecControls.end(); ++i)
  {
    CGUIControl* pControl = *i;
    if (pControl->IsGroup())
    {
      CGUIControlGroup *group = (CGUIControlGroup *)pControl;
      CGUIControl *control = group->GetFirstFocusableControl(id);
      if (control) return control;
    }
    if (pControl->GetID() == (DWORD) id && pControl->CanFocus())
      return pControl;
  }
  return NULL;
}

bool CGUIWindow::OnMove(int fromControl, int moveAction)
{
  const CGUIControl *control = GetFirstFocusableControl(fromControl);
  if (!control) control = GetControl(fromControl);
  if (!control)
  { // no current control??
    CLog::Log(LOGERROR, "Unable to find control %i in window %u",
              fromControl, GetID());
    return false;
  }
  vector<int> moveHistory;
  int nextControl = fromControl;
  while (control)
  { // grab the next control direction
    moveHistory.push_back(nextControl);
    nextControl = control->GetNextControl(moveAction);
    // check our history - if the nextControl is in it, we can't focus it
    for (unsigned int i = 0; i < moveHistory.size(); i++)
    {
      if (nextControl == moveHistory[i])
        return false; // no control to focus so do nothing
    }
    control = GetFirstFocusableControl(nextControl);
    if (control)
      break;  // found a focusable control
    control = GetControl(nextControl); // grab the next control and try again
  }
  if (!control)
    return false;   // no control to focus
  // if we get here we have our new control so focus it (and unfocus the current control)
  SET_CONTROL_FOCUS(nextControl, 0);
  return true;
}

void CGUIWindow::SetDefaults()
{
  m_renderOrder = 0;
  m_saveLastControl = true;
  m_dwDefaultFocusControlID = 0;
  m_bRelativeCoords = false;
  m_posX = m_posY = m_width = m_height = 0;
  m_overlayState = OVERLAY_STATE_PARENT_WINDOW;   // Use parent or previous window's state
  m_visibleCondition = 0;
  m_previousWindow = WINDOW_INVALID;
  m_animations.clear();
  m_origins.clear();
  m_hasCamera = false;
  m_animationsEnabled = true;
}

FRECT CGUIWindow::GetScaledBounds() const
{
  CSingleLock lock(g_graphicsContext);
  g_graphicsContext.SetScalingResolution(m_coordsRes, m_posX, m_posY, m_needsScaling);
  FRECT rect = {0, 0, m_width, m_height};
  float z = 0;
  g_graphicsContext.ScaleFinalCoords(rect.left, rect.top, z);
  g_graphicsContext.ScaleFinalCoords(rect.right, rect.bottom, z);
  return rect;
}

void CGUIWindow::GetContainers(vector<CGUIControl *> &containers) const
{
  for (ciControls it = m_vecControls.begin();it != m_vecControls.end(); ++it)
  {
    if ((*it)->IsContainer())
      containers.push_back(*it);
    else if ((*it)->IsGroup())
      ((CGUIControlGroup *)(*it))->GetContainers(containers);
  }
}

void CGUIWindow::OnEditChanged(int id, CStdString &text)
{
  CGUIMessage msg(GUI_MSG_ITEM_SELECTED, GetID(), id);
  OnMessage(msg);
  text = msg.GetLabel();
}

bool CGUIWindow::SendMessage(DWORD message, DWORD id, DWORD param1 /* = 0*/, DWORD param2 /* = 0*/)
{
  CGUIMessage msg(message, GetID(), id, param1, param2);
  return OnMessage(msg);
}

#ifdef _DEBUG
void CGUIWindow::DumpTextureUse()
{
  CLog::Log(LOGDEBUG, "%s for window %u", __FUNCTION__, GetID());
  for (ivecControls it = m_vecControls.begin();it != m_vecControls.end(); ++it)
  {
    (*it)->DumpTextureUse();
  }
}
#endif

void CGUIWindow::ChangeButtonToEdit(int id, bool singleLabel /* = false*/)
{
#ifdef PRE_SKIN_VERSION_2_1_COMPATIBILITY
  CGUIControl *name = (CGUIControl *)GetControl(id);
  if (name && name->GetControlType() == CGUIControl::GUICONTROL_BUTTON)
  { // change it to an edit control
    CGUIEditControl *edit = new CGUIEditControl(*(const CGUIButtonControl *)name);
    if (edit)
    {
      if (singleLabel)
        edit->SetLabel("");
      Insert(edit, name);
      Remove(name);
      name->FreeResources();
      delete name;
    }
  }
#endif
}

