/********************************************************************************************************
 * QRNA - Comparative analysis of biological sequences 
 *         with pair hidden Markov models, pair stochastic context-free
 *        grammars, and probabilistic evolutionary  models.
 *       
 * Version 2.0.0 (JUN 2003)
 *
 * Copyright (C) 2000-2003 Howard Hughes Medical Institute/Washington University School of Medicine
 * All Rights Reserved
 * 
 *     This source code is distributed under the terms of the
 *     GNU General Public License. See the files COPYING and LICENSE
 *     for details.
 ***********************************************************************************************************/

/* funcs.h
 * ANSI prototypes for all external functions.
 * 
 */
#ifndef FUNCSH_INCLUDED
#define FUNCSH_INCLUDED

#include <stdio.h>
#include "cfg.h"
#include "globals.h"
#include "squid.h"
#include "structs.h"

/* from cfgio.c:
 */
extern void PairsSCFG (double **cfg, double **ret_pair);
extern void Pairs5SCFG (double **cfg, double **ret_pair);
extern int  ReadSCFG(FILE *ofp, double ***ret_cfg);
extern int  SaveSCFG(FILE *ofp, double **cfg);
extern void WriteRdbSCFG(FILE *ofp, double **cfg);
extern void WriteRdbLogSCFG(FILE *ofp, double **cfg);
extern void WriteRdbSummary(FILE *ofp, double **cfg);

/* from cfgmodel.c:
 */
extern double  **AllocSCFG(void);
extern double ***AllocSCFGNodes(void);
extern int     **AllocIntSCFG(void);
extern void      CheckSCFG(double **cfg);
extern void      CheckSCFGEmit(double **cfg_emit);
extern void      CheckSCFGNode(double ***cfg_node);
extern int       Connects(int dp, int node);
extern char     *dpcNAME(int node);
extern double  **DupSCFG(double **cfg);
extern void      FreeSCFG(double **cfg);
extern void      FreeSCFGNode(double ***cfg_node);
extern void      Log2ProbSCFG(double **cfg);
extern void      MdfySCFGEmit(double ***ret_cfg_emit, double *pmodel);
extern void      MdfySCFGNode(double ****ret_cfg_node);
extern int       Nemit(int node);
extern int       Ntype(int dp, int tr);
extern void      PrintPerDp(int dp, int idx, double **cfg, double **cfg_emit, double ***cfg_node);
extern void      ReassembleSCFG(double ***cfg_node, double **cfg_emit, double ***ret_cfg);
extern double    Size(int dp, int tr);
extern int       Symi(int dp, int tr);
extern int       Symj(int dp, int tr);
extern int       Symk(int dp, int tr);
extern int       Syml(int dp, int tr);
extern void      TieProbs(double **cfg, double ****ret_cfg_node, double ***ret_cfg_emit);
extern int       Ttype(int node, int symi, int symj, int symk, int syml, int size);

/* from cfgparse.c:
 */
extern void GraphicTrace(FILE *fp, struct tracekn_s *tr, char *seq, int len);
extern int  KHS2Trace(FILE *outf, char *seq, char *ss, int len, struct tracekn_s **ret_tr,
		      int allow_pseudoknots, int debug);
extern void TraceCount(char *seq, int len, double wgt, struct tracekn_s *tr, double **cfg);

/* from codingcreatealign.c:
 */
extern struct emitcodon_s *AllocEmitCodon (void);
extern void                FillEmitCodon (struct codmodel_s *cod,struct nullmodel_s *null,  struct emitcodon_s *emitcodon);
extern void                FreeEmitCodon (struct emitcodon_s *emitcodon);
extern void                SimulateCodingSequences(int L, double pammodel[64][64], char **ret_s1, char **ret_s2, 
						   double *ret_first_pos, double *ret_second_pos, double *ret_third_pos);
extern void                SimulateCODAlign(FILE *ofp, SQINFO *sqinfoX, int **seqX, SQINFO *sqinfoY, int **seqY, 
					    int L, int start, int *ret_L, struct codmodel_s *cod, 
					    struct emitcodon_s *emitcodon, struct nullmodel_s *null, 
					    int traceback, int alignment, char *string_name, 
					    double *ret_first_pos, double *ret_second_pos, double *ret_third_pos);
extern void                SimulateCODSequence(FILE *ofp, int *seq, SQINFO *sqinfoX, int *seqX, SQINFO *sqinfoY, 
					       int *seqY, int start, int *L, struct codmodel_s *cod, 
					       struct nullmodel_s *null, struct ali_s *ali, int traceback, 
					       int alignment, char *string_name, 
					       double *ret_first_pos, double *ret_second_pos, double *ret_third_pos);
/* from coddp.c:
 */
extern struct coddp_s *AllocDpDiagCOD(int L);
extern struct coddp_s *AllocDpFullCOD(int Lx, int Ly);
extern void            ForwardCODDiag(FILE *ofp, int *seqX, int *seqY, int start, int L, 
				      struct codmodel_s *cod, struct dpd_s *dp, 
				      struct nullmodel_s *null, struct sc2_s *codsc, int logodds, struct dpsc3_s *sc, int ones);
extern void            ViterbiCODFull(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY,  
				      int startX, int startY, int Lx, int Ly, int *ret_leg, int Lw,
				      struct codmodel_s *cod, struct dpf_s *dp, struct nullmodel_s *null, struct sc2_s *codsc, 
				      struct ali_s *ali, int alignment, int logodds, int ones, int traceback, 
				      int doends, struct windowends_s *windowends);
extern void            FreeDpCOD(struct coddp_s *dp);
extern void            PatternDpDiagCOD(int L, struct coddp_s *coddp);
extern void            ViterbiCODDiag(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, 
				      int start, int L, struct codmodel_s *cod, struct dpd_s *dp, 
				      struct nullmodel_s *null, struct sc2_s *codsc, struct ali_s *ali,
				      int alignment,  int logodds, int ones, int traceback, 
				      int doends, struct windowends_s *windowends);

/* from coddpscanfast.c:
 */
extern struct coddpscanfast_s  *AllocDpDiagScanFastCOD(int L);
extern struct coddpscanfast2_s *AllocDpDiagScanFast2COD(int L);
extern void                     FreeDpXFastCOD(struct coddpxfast_s *dp);
extern void                     FreeDpScanFastCOD(struct coddpscanfast_s *coddpscan);
extern void                     FreeDpScanFast2COD(struct coddpscanfast2_s *coddpscan2);
extern double                   ViterbiCODDiagScanFast(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, 
						       int leg, int win, int st, int stmod, int l, int lmax, 
						       struct codmodel_s *cod, struct coddpscanfast_s *dp,
						       struct nullmodel_s *null, struct ali_s *ali, int alignment, int logodds, 
						       int revstrand, int traceback, int doends, struct end3scan_s *ends);

/* from codmodel.c:
 */
extern struct codmodel_s *AllocCODModel(void);
extern struct pamcond_s  *AllocPamCond(void);
extern void     CheckCODProbs(struct codmodel_s *cod);
extern void     CheckPAMModel(double **pammodel);
extern void     CODToLog2(struct codmodel_s *cod);
extern void     CODLog2ToOdds(struct codmodel_s *cod, struct nullmodel_s *null);
extern void     CODmodelTransitionProbabilities(int len, double tfactor, struct codmodel_s *cod, int pedantic, int verbose);
extern void     ConstructCODModel(double **pammodel_star, int **pam, double scale, double *codon_joint, int add_codon, double **hexa, 
				  int win, double tfactor, double *targetfreq, int changefreq, 
				  struct codmodel_s *cod, int add_hexamer, int pedantic, int verbose);
extern void     ConstructCODEmitProbs(double **pammodel, double *pmut, double **pcodon);
extern void     ConstructESTModel(double **estmodel);
extern void     ConstructHexamerProbs(double **hexa, double **phexa, int add_hexamer);
extern void     ConstructPAMModel(int **pam, double scale, double *codon_joint, double ***ret_pammodel,
				  int add_codon, double *targetfreq, int changefreq, int pedantic, int verbose);
extern void     ConstructTimeDependentPAMModel(int **pam, double scale, double time, double *codon_joint, double ***ret_pammodel, 
					       int add_codon, int pedantic, int verbose);
extern void     ConstructPAMCondi(int **pam, double scale, double tfactor, double *codon_joint, struct pamcond_s **ret_pamcond, 
				  int add_codon, int pedantic, int verbose);
extern void     ConstructPsingle (double **pammodel, double **ret_psingle);
extern void     ConstructPAMJoint(struct pamcond_s *pamcond, double ***ret_pammodel, double *freqX, double *freqY, 
				  int pedantic, int verbose);
extern void     ConstructPAM5Model(int **pam, double scale, double ***ret_pammodel);
extern void     DefaultCodonBias(double codebias[64]);
extern void     DefaultGeneticCode(int aacode[64]);
extern void     FreeCODModel(struct codmodel_s *cod);
extern void     FreePAMModel(double **pammodel);
extern void     FreePamCond(struct pamcond_s *pamcond);
extern void     NormalizePAMModel(double **pammodel);
extern void     PatternCODModel(struct codmodel_s *cod);
extern void     PrintCodonString(int x);
extern void     PrintCODModel(struct codmodel_s *cod);
extern void     PrintPAMModel(double **pammodel, double *mutpxy);
extern void     PrintCODTrProbs(struct codmodel_s *cod);

/* from codon.c: Parse disk file with codon-codon frequencies calculated from real data
 */
extern int ParseCodonFile(FILE *fp, double **ret_codon);
extern int ParseTargetFreqFile(FILE *fp, double **ret_targetfreq);

/* from codscorealign.c:
 */
extern double        average(int pos, int x1, int x2, int codon, struct codmodel_s *cod, 
			     struct nullmodel_s *null, int logodds);
extern double        average_codon(int codon, struct codmodel_s *cod, struct nullmodel_s *null, int logodds);
extern double        fourcodonX(int xleft, int gappos, int cod5X, int cod5Y, struct codmodel_s *cod, 
				struct nullmodel_s *null, int logodds, int *ret_tr1, int *ret_tr2);
extern double        fourcodonY(int yleft, int gappos, int cod5X, int cod5Y, struct codmodel_s *cod,  
				struct nullmodel_s *null, int logodds, int *ret_tr1, int *ret_tr2);
extern int           isitagap(int x);
extern int           iscomplete(int cod5);
extern int           isgap1(int cod5);
extern int           isgap2(int cod5);
extern int           isgap3(int cod5);
extern int           isnocod(int cod5);
extern int           miss1(int cod5);
extern int           miss2(int cod5);
extern int           miss3(int cod5);
extern double        ScoreWithCOD_ungapped(int *isX, int *isY, int L, struct codmodel_s *cod, 
					   int traceback);
extern void          ScoreWithCOD(FILE *ofp, int *isX, int *isY, int start, int L, struct codmodel_s *cod, 
				  struct nullmodel_s *null, struct coddp_s  *coddp, struct sc2_s *codsc, 
				  int logodds, int ones, int traceback);
extern double        ScoreWithPAM(int *isX, int *isY, int L, struct codmodel_s *cod);


/* from er_math.c:
 */
extern void   CheckDoubleProb(double **pdouble, int dimx, int dimy);
extern void   CheckDoubleLog2Prob(double **pdouble, int dimx, int dimy);
extern void   CheckSingleProb(double *psingle, int size);
extern void   CheckSingleLog2Prob(double *psingle, int size);
extern void   CheckSYMLog2Prob(double *mat, int edge_size);
extern void   DistributionMeanVar(double *p, int dim, double *ret_mean, double *ret_var);
extern void   DistributionMean(double *p, int dim, double *ret_mean);
extern void   DistributionLogMean(double *p, int dim, double *ret_mean);
extern void   DistributionLogMeanVar(double *p, int dim, double *ret_mean, double *ret_var);
extern int    DENChoose(double *p, int N); 
extern void   DExp2(double *vec, int n);
extern void   DLog2(double *vec, int n);
extern int    DLog2Choose(double *p, int N);
extern void   DLog2Norm(double *log2p, int n);
extern void   DLog2SYMNorm(double *log2p, int n);
extern double DLog2Sum(double *log2p, int n);
extern double DLog2SYMSum(double *log2p, int n);
extern int    ILogSum(int *logp, int n);
extern int    ILog2Sum(int *log2p, int n);
extern void   FLog(float *vec, int n);
extern void   FLog2(float *vec, int n);
extern void   FExp(float *vec, int n);
extern void   FExp2(float *vec, int n);
extern void   FAddConst(float *vec, float plus, int n);
extern void   IAddConst(int *vec, int plus, int n);


/* from evd.c
 */
extern void   AllocHistogram (int N, double k, int **ret_histo);
extern int    FillHistogram (FILE *ofp, double score, int N, double k, int shift, int *histo);
extern void   FillScoreHistograms (FILE *ofp, struct scores_s *sc, int N, double k, int shift, int *histo, int ones);
extern double FitEVD (double fp, int L, double *sc, double *scnull);
extern void   FitHistoEVD (FILE *ofp, double censor_cutoff, int N, double k, int shift, int *sc_histo);
extern void   PrintHistogram (FILE *ofp, double censor_cutoff, double censor, int n_censor, double lambda, 
			      double mu, int N, double k, int shift, int *sc_histo);

/* from evolve.c
 */
extern double                Cal_lambda(FILE *ofp, double *S, double *pml, double *pmr, int L, int verbose);
extern void                  Cal_YuAltschul_ModifProbs(FILE *ofp, double *P, double *targetfreq_l, double *targetfreq_r, int L, int verbose);
extern void                  Cal_YuAltschul_Modif3DProbs(FILE *ofp, double *P, double *targetfreq_1, double *targetfreq_2, double *targetfreq_3, 
							 int L1, int L2, int L3, int verbose);
extern void                  CalculateConditionalsAndMarginals(FILE *ofp, double *P, double **ret_QL, double **ret_QR, 
							       double **ret_ml, double **ret_mr, int L, int hasindel, int verbose);
extern void                  ComputeConditionalsAndMarginals(FILE *ofp, double *P, double *Q, double *pm, int L, int hasindel, int verbose);
extern void                  ComputeLRConditionalsAndMarginals(FILE *ofp, double *P, double *Ql, double *Qr, double *ml, double *mr, 
							       int L, int hasindel, int verbose);
extern void                  ChangeFrequencies(FILE *ofp, double *Q_0, double *R, double *pm, int L, double *targetfreq, int hasindel, int verbose);
extern void                  ChangePairProbs(int L5, double *pair5prob, double *psingle_target, int hasindel, int verbose);
extern void                  ChangeStationaryProbs(int L, double *jointprob, double *targetfreq, int hasindel, int verbose);
extern void                  ChangeStationaryProbsGW(int L, double *jointprob, double *targetfreq, int hasindel, int verbose);
extern void                  ChangeStationaryPAMProbsGW(double **pammodel, double *targetfreq, int verbose);
extern void                  ChangePairProbsIterate(FILE *ofp, double *pairprobs, int L, double *targetfreq, int hasindel, int verbose);
extern void                  ChangeConditionalMatrix(FILE *ofp, double *Qstar, double *ml, double *mr, int L, double *targetfreq, 
						     int hasindel, int isleft, int verbose);
extern void                  ChangeQ_0Matrix(FILE *ofp, double *Q_0, double *ml, double *mr, int L, double *targetfreq, 
					     int hasindel, int isleft, int verbose);
extern void                  ChangeQ_0MatrixNaive(FILE *ofp, double *Q_0, double *QR_0, int L, double *targetfreq, 
						  int hasindel, int isleft, int verbose);
extern void                  ChangeQ_0MatrixIterate(FILE *ofp, double *Q_0, double *Q_0_inv, double *pml, double *pmr, int L, double *targetfreq, 
						    int hasindel, int isleft, int verbose);
extern void                  ChangeRateMatrix(FILE *ofp, double *Q_0R, double *Q_0_inv, double *R, 
					      double *ml, double *mr, int L, double *targetfreq, int hasindel, int isleft, int verbose);
extern int                   Check_Accuracy(double *vec, int L);
extern int                   CompareFreqs(double *pml, double *pmr, double *targetfreq, int L);
extern double               *Condi_From_Joint(FILE *ofp, double *P, int L, int verbose);
extern double               *CondiR_From_Joint(FILE *ofp, double *P, int L, int verbose);
extern void                  Condi_From_Rate(double **ret_Q, double *R, double *Q_0, double tfactor, int L, int pedantic, int verbose);
extern void                  ConditionalsFromRate(double *Q, double *R, double *Q_0, double tfactor, int L, int pedantic, int verbose);
extern void                  ZCondi_From_Rate(double **ret_Q, struct zmatrix_s *R, struct zmatrix_s *Q_0, double tfactor, 
					      int L, int pedantic, int verbose);
extern void                  ConditionalsEvolved(FILE *ofp, double *Ql, double *Qr, double *Ql_0, double *Qr_0, double *ml, double *mr, 
						 int L, double tfactor, double *targetfreq, 
						 int changefreq, int hasindel, int pedantic, int verbose);
extern void                  RIBOConditionalsEvolved(FILE *ofp, fullcondmat_t *ribocond, condmatrix_t *ribocond_nought, 
						     double tfactor, double *targetpair, int changepair, int pedantic, int verbose);
extern void                  ConditionalsEvolved_2(FILE *ofp, double *Q, double *Q_0, double *R, int L, double tfactor, double *targetfreq, 
						   int changefreq, int pedantic, int verbose);
extern void                  EvolveIndelMarginals(FILE *ofp, double *R, double *Q_0, double *pm, double tfactor, 
						  int L, int pedantic, int verbose);
extern int                   IsQConsistent(FILE *ofp, double *Q, int L, int pedantic, int verbose);
extern void                  IslogQConsistent(FILE *ofp, double *Q_0, double *K, int Lx, int Ly, int Lz, int verbose);
extern void                  IslogQConsistent_2(FILE *ofp, double *Q_0, double *A, double *K, int Lx, int Ly, int Lz, int verbose);
extern void                  Joint_From_Condi(FILE *ofp, double *P, double *Q, double *pm, int L, int verbose);
extern void                  RIBOJoint_From_Condi(FILE *ofp, double *P, double *Q, double *pm, int L, int verbose);
extern void                  Joint_From_CondiR(FILE *ofp, double *P, double *Q, double *pm, int L, int verbose);
extern void                  Joint_From_Condi_Symmetrize(FILE *ofp, double *P, double *Q, double *p, int L, int verbose);
extern void                  Joint2Joint(double *p_joint, double *ql_nought, double *qr_nought, int L, double tfactor, double *targetfreq, 
					 int changefreq, int hasindel, int ispairprob, int pedantic, int verbose);
extern void                  Joint2JointGap(double *p_joint, double *q_nought, double *pv, int L, double tfactor, double *targetfreq, 
					    int changefreq,  int pedantic, int verbose);
extern void                  MarginalizeJointProbs(double *jp, double *sp, int dim, int which);
extern void                  OrderVector(int **ret_perm, double *b, int L, int isdescendent, int verbose);
extern void                  QOMRegularizationAlgorithm(double *row, int size, int verbose);
extern void                  QOGRegularizationAlgorithm(double *row, int size, int whichrow, int verbose);
extern void                  RateFromConditionals(FILE *ofp, double *R, double *Q, double *Q_0, int L, double tfactor, 
						  int hasindel, int pedantic, int verbose);
extern double               *SaturationProbs(double *R, double *q_nought, int L, int pedantic, int verbose);
extern void                  ComputeSaturationProbs(double *R, double *Q_0, double *pm, int L, int pedantic, int verbose);
extern double                TimeIdCorrelation (struct three_divergence_s div, double id);
extern struct three_times_s  TimeIdCorrelation3Models(struct three_divergence_s div_o, struct three_divergence_s div_c, 
						      struct three_divergence_s div_r, double id);
extern void                  TransitionsEvolvedLinear(FILE *ofp, double *q, double *q_0, double *r, double *r_diag, double *r_fix, 
						      int L, double tfactor, 
						      int pedantic, int verbose);
extern void                  TransitionsEvolved(FILE *ofp, double *q, double *q_0, double *q_infty, double *r_diag, int L, double tfactor, 
						int pedantic, int verbose);
extern void                  TransitionsDirectoryCosines(double *q, double *q_zero, double *q_infty, double *R_diag, int dim);
extern double               *TransitionsExpVector(double *T, double *T_zero, double *T_infty, int dim);
extern double               *TransitionsDiagVector(double *T, double *T_zero, double *T_rate, double *A_fix,  int dim);
extern double               *TransitionsFixVector (double *T, double *T_zero, double *T_rate, double *A_diag, int dim);
extern double               *TransitionsRateVector(double *T, double *T_zero, double *A_fix,  double *A_diag, int dim);

/* from hexamer.c: Parse disk file with hexamer frequencies for diferent organisms
 */
extern int ParseHexamerFile(FILE *fp, double ***ret_hexap);

/* from matrix.c: functions for matrix manipulations
 */
extern struct eigenvalue_s *AllocEigenvalues(int L);
extern struct zmatrix_s    *AllocZmatrix(int L);
extern double              *Cal_Id(int L);
extern double              *Cal_LUP_Decomp(FILE *ofp, double *M, int *perm, int L, int verbose);
extern struct eigenvalue_s *Cal_M_Eigenvalues(FILE *ofp, double *M, int L, int verbose);
extern double              *Cal_M_Exp(FILE *ofp, double *M, int L, double power, int verbose);
extern double              *Cal_M_Inv(FILE *ofp, double *M, int L, int verbose);
extern struct zmatrix_s    *Cal_M_Vandermonde_Log(FILE *ofp, double *M, int L, int verbose);
extern double              *Cal_M_Taylor_Log(FILE *ofp, double *M, int L, int verbose);
extern double              *Cal_M_MinId(double *M, int L);
extern double              *Cal_M_num_Prod(FILE *ofp, double *M, double num, int Li, int Lj, int verbose);
extern double              *Cal_V_M_Prod(FILE *ofp, double *vec, double *M, int L, int verbose);
extern double              *Cal_M_N_Prod(FILE *ofp, double *M, double *N, int Li, int Lj, int Lk, int verbose);
extern double              *Cal_M_N_Sum(FILE *ofp, double *M, double *N, int Li, int Lj, int add, int verbose);
extern double              *Col_From_M(double *A, int Lrow, int Lcol, int col_i);
extern void                 Comp_Id(double *Id, int L);
extern void                 Comp_M_Exp(FILE *ofp, double *M, int L, double power, int verbose);
extern void                 Comp_M_Inv(FILE *ofp, double *M, double *Minv, int L, int verbose);
extern void                 Comp_M_Taylor_Log(FILE *ofp, double *M, int L, int verbose);
extern void                 Comp_M_MinId(double *M, int L);
extern void                 Comp_M_num_Prod(FILE *ofp, double *M, double num, int Li, int Lj, int verbose);
extern void                 Comp_M_N_Prod(FILE *ofp, double *M, double *N, int L, int verbose);
extern void                 Comp_M_N_Sum(FILE *ofp, double *M, double *N, int Li, int Lj, int add, int verbose);
extern void                 CopyMatrix (double *copyQ, double *Q, int N, int M);
extern void                 CopyVector(double *copyVec, double *Vec, int N);
extern int                  IsDiagonal (double *A, int dim);
extern void                 FreeEigenvalues(struct eigenvalue_s *eigen);
extern void                 FreeZmatrix(struct zmatrix_s *zmatrix);
extern double              *HessenbergForm(FILE *ofp, double *A, int L, int verbose);
extern void                 Hessenberg2Eigenvalues(FILE *ofp, double *H, int L, struct eigenvalue_s **ret_eigen, int pedantic, int verbose);
extern void                 MultiplyMatrix (double tfactor, int L, double *K);
extern void                 PrintMatrix(FILE *ofp, double *m, int L);
extern void                 PrintVector(FILE *ofp, double *v, int L) ;
extern void                 PrintZmatrix(FILE *ofp, struct zmatrix_s *m, int L);
extern void                 PrintZvector(FILE *ofp, struct zmatrix_s *v, int L) ;
extern void                 QR_Decomposition (FILE *ofp, double *X, double *Q, double *R, int n, int pedantic, int verbose);
extern double              *Row_From_M(double *A, int Lrow, int Lcol, int row_i);
extern void                 SecondDegree_Solutions(double a, double b, double c, struct eigenvalue_s *eigen, int idx);
extern void                 Solve_LU_Eq(FILE *ofp, double *LU, double *b, int L, int verbose);
extern struct zmatrix_s    *ZCal_Id(int L);
extern struct zmatrix_s    *ZCal_LUP_Decomp(FILE *ofp, struct zmatrix_s *M, int *perm, int L, int verbose);
extern struct zmatrix_s    *ZCal_M_Exp(FILE *ofp, struct zmatrix_s *M, int L, double r, int verbose);
extern struct zmatrix_s    *ZCal_M_Inv(FILE *ofp, struct zmatrix_s  *M, int L, int verbose);
extern struct zmatrix_s    *ZCal_M_num_Prod(FILE *ofp, struct zmatrix_s *M, double real, double imag, int Li, int Lj, int verbose);
extern struct zmatrix_s    *ZCal_M_N_Prod(FILE *ofp, struct zmatrix_s *M, struct zmatrix_s *N, int Li, int Lk, int Lj, int verbose);
extern struct zmatrix_s    *ZCal_M_N_Sum(FILE *ofp, struct zmatrix_s *M, struct zmatrix_s *N, int Li, int Lj, int add, int verbose);
extern void                 ZComp_M_N_Prod(FILE *ofp, struct zmatrix_s *M, struct zmatrix_s *N, int L, int verbose);
extern void                 ZComp_M_N_Sum(FILE *ofp, struct zmatrix_s *M, struct zmatrix_s *N, int Li, int Lj, int add, int verbose);
extern void                 CopyZmatrix (struct zmatrix_s *copyQ, struct zmatrix_s *Q, int N, int M);
extern void                 MultiplyZmatrix (double real, double imag, int L, struct zmatrix_s *K);
extern void                 ZSolve_LU_Eq(FILE *ofp, struct zmatrix_s *LU, struct zmatrix_s *b, int L, int verbose);

/* from misc.c: miscellaneous functions with no home
 */
extern void                  AdjustAliSS(FILE *ofp, SQINFO sqinfo, char *aliss, int *ctX, int *ctY);
extern void                  AllocAli(int L, struct ali_s **ret_ali);
extern struct dpd_s         *AllocDpDiagMtx(int L);
extern struct dpdscanfast_s *AllocDpDiagScanFastMtx(int L);
extern struct dpf_s         *AllocDpSemiMtx(int L);
extern struct dpf_s         *AllocDpFullMtx(int Lx, int Ly);
extern struct dpsc3_s       *AllocDpDiagScores(int L);
extern struct dpsc3_s       *AllocDpScores(int L);
extern struct scanends_s    *AllocScanEnds(int L);
extern struct scansc_s      *AllocScanSc(int L);
extern struct scanfast_s    *AllocScanFast(int L, int win, int off);
extern void                  AllocModels(struct model_s **ret_model);
extern void     ConstructModels(FILE *ofp, char *codonfile, char *hexapfile, char *pamfile, char *cfgfile, char *ribofile,
				struct model_s **ret_model, int win, struct three_times_s time, double *targetfreq,
				int changefreq, int logodds, int pedantic, int verbose);
extern void     ConstructModels_phase1(FILE *ofp, char *codonfile, char *hexapfile, char *pamfile, char *cfgfile, char *ribofile,
				       struct model_s **ret_model, 
				       double ****ret_cfg_node, double ***ret_hexa, double **ret_pair5prob, double **ret_codon_joint, 
				       int ***ret_pam, double *ret_scale, fullmat_t **ret_riboprob,
				       int add_codon, int add_hexamer, int logodds, int pedantic, int verbose);
extern void     ConstructModels_phase2(FILE *ofp, int win, fullmat_t *riboprob, double ***cfg_node, int **pam, double scale, double **hexa, 
				       double *pair5prob, double *codon_joint, double *freqX, double *freqY,
				       struct model_s *model, struct three_times_s tfactor, 
				       int add_codon, int add_hexamer, int changefreq, int logodds, int pedantic, int verbose);
extern void     ConstructTiedProbs(double **pammodel, double tfactor, double **ret_mutpxy, 
				   double **ret_mut5pxy, double *targetfreq, int changefreq, int pedantic, int verbose);
extern void     FormatSeqs(FILE *ofp, int Lmax, int format, char *seqX, char *seqY, 
			   SQINFO *sqinfoX, SQINFO *sqinfoY, int **ret_ctX, int **ret_ctY, 
			   int *isegX, int *isegY, int allow_pseudoknots, int verbose);
extern void     FreeAli(struct ali_s *ali);
extern void     FreeScanEnds(struct scanends_s *scanends, int L);
extern void     FreeScanSc(struct scansc_s *scansc);
extern void     FreeScanFast(struct scanfast_s *fast, int L, int win, int off);
extern void     FreeDpDiagMtx(struct dpd_s *dp);
extern void     FreeDpDiagScanMtx(struct dpdscan_s *dp);
extern void     FreeDpDiagScanFastMtx(struct dpdscanfast_s *dp);
extern void     FreeDpSemiMtx(struct dpf_s *dp);
extern void     FreeDpFullMtx(struct dpf_s *dp);
extern void     FreeDpScores(struct dpsc3_s *sc);
extern void     FreeModels(struct model_s *model);
extern void     ModelLog2ToOdds(struct model_s *model);
extern void     PatternAli(int L, struct ali_s *ali);
extern void     PatternDpDiag(int L, struct dpd_s *dp);
extern void     PatternDpFull(int Lx, int Ly, struct dpf_s *dp);
extern void     PatternModels(struct model_s *model);
extern void     PrintTrProbs(struct model_s *model);
extern void     PrintProbs(FILE *ofp, double *prob, int L);
extern void     PrintProbs2D(FILE *ofp, double **prob, int Lx, int Ly);
extern void     PrintVectorProbs(FILE *ofp, double *prob, int L);

/* from nrnscfg.c: Dynamic programming for the Non Redundant Nussionv grammar (R Dowell) 
 */
extern struct nrn_s      *AllocNRNModel(int L);
extern struct nrnscfg_s  *AllocScfgNRN(int L);
extern void     Best_nrnS(int *sX, int *sY, struct nrn_s *nrn, int start, int L, int j, int d, 
			  struct nrnscfg_s *mtx, struct end_s *rnaends);
extern void     Best_nrnL(int *sX, int *sY, struct nrn_s *nrn, int start, int L, int j, int d, 
			  struct nrnscfg_s *mtx, struct end_s *rnaends);
extern void     Best_nrnR(int *sX, int *sY, struct nrn_s *nrn, int start, int L, int j, int d, 
			  struct nrnscfg_s *mtx, struct end_s *rnaends);
extern void     ConstructNRNModel(int L, struct nrn_s **ret_nrn, double **pp, double *ps);
extern void     ConstructNRNModel2(int L, struct nrn_s **ret_nrn, double **pp, double *ps);
extern void     Cyk_NRN(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, int start, int L,  
			struct nrn_s *nrn, struct nrnscfg_s *mtx, struct end_s *rnaends);
extern void     FreeNRNModel(struct nrn_s *nrn);
extern void     FreeScfgNRN(struct nrnscfg_s *mx);
extern void     PrintNRNModel (FILE *ofp, struct nrn_s *nrn, int L);
extern void     Tr_nrnS(FILE *ofp, int *sX, int *sY, struct nrn_s *nrn, int start, int L, int j, int d, 
			struct nrnscfg_s *mtx, 
			int *flag, struct tracekn_s *curr_tr, struct traceknstack_s *dolist, int traceback, struct end_s *rnaends);
extern void     Tr_nrnL(FILE *ofp, int *sX, int *sY, struct nrn_s *nrn, int start, int L, int j, int d, 
			struct nrnscfg_s *mtx, 
			int *flag, struct tracekn_s *curr_tr, struct traceknstack_s *dolist, int traceback, struct end_s *rnaends);
extern void     Tr_nrnR(FILE *ofp, int *sX, int *sY, struct nrn_s *nrn, int start, int L, int j, int d, 
			struct nrnscfg_s *mtx, 
			int *flag, struct tracekn_s *curr_tr, struct traceknstack_s *dolist, int traceback, struct end_s *rnaends);
extern void     Trk_NRN(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, int start, int L, int j, int d,
			struct nrn_s *nrn, struct nrnscfg_s *mtx, struct tracekn_s **ret_trace, int traceback, struct end_s *rnaends);

/* from nullmodel.c:
 */
extern struct nullmodel_s  *AllocNullModel(void);
extern void    CheckNullProbs(struct nullmodel_s *null);
extern void    ConstructNullModel(double **pammodel_star, struct nullmodel_s *null, struct nullparam_s param, struct nullparam_s param_zero, 
				  struct nullparam_s param_infty, double tfactor, double *targetfreq, int changefreq, int verbose);
extern void    ConstructXemNullProb(double *nullpxy, double *xem);
extern void    ConstructYemNullProb(double *nullpxy, double *yem);
extern void    FreeNullModel(struct nullmodel_s *null);
extern double  NullAddOnePair(int cur_x, int cur_y, struct nullmodel_s *null);
extern double  NullAddOneX(int cur_x, struct nullmodel_s *null);
extern double  NullAddOneY(int cur_y, struct nullmodel_s *null);
extern void    NullToLog2(struct nullmodel_s *null);
extern void    NullLog2ToOdds(struct nullmodel_s *null1, struct nullmodel_s *null2);
extern void    PatternNullModel(struct nullmodel_s *null);
extern void    PrintNullModel(struct nullmodel_s *null);
extern double  ScoreWithNull(int *seqX, int *seqY, int startX, int Lx, int startY, int Ly, 
			     struct nullmodel_s *null);
extern double  ScoreWithNullDiag(int *sX, int *sY, int start, int L, struct nullmodel_s *null);
extern void    ScoreWithNullFJScan(int *seqX, int *seqY, int win, int j, int jmod, int d, struct nullmodel_s *null, double **fjmx);
extern double  ScoreWithNullX(int *sX, int startX, int Lx, struct nullmodel_s *null);
extern double  ScoreWithNullY(int *sY, int startY, int Ly, struct nullmodel_s *null);
extern double  ScoreWithNullRev(int *seqX, int *seqY, int startX, int Lx, int startY, int Ly, 
				struct nullmodel_s *null);
extern double  ScoreWithNullRevDiag(int *seqX, int *seqY, int start, int L, struct nullmodel_s *null);
extern void    SimulateNullAlign(int *s1, int *s2, int L1, int L2, struct nullmodel_s *null);
extern char   *SimulateNullSequence(char *s1, int L, double nullmodel[4][4]);


/* from othercreatealign.c:
 */
extern void  SimulateOTHAlign(FILE *ofp, SQINFO *sqinfoX, int **seqX, SQINFO *sqinfoY, int **seqY, 
			      int L, int start, int *ret_L, struct othmodel_s *oth,
			      int traceback, int alignment, char *string_name);
extern void  SimulateOTHSequence(FILE *ofp, int *seq, SQINFO *sqinfoX, int *seqX, SQINFO *sqinfoY, int *seqY, 
				 int start, int *L, struct othmodel_s *oth, struct ali_s *ali, 
				 int traceback, int alignment, char *string_name);
extern void SimulateMutSequence(FILE *ofp, int *seq, SQINFO *sqinfoX, int *seqX, SQINFO *sqinfoY, int *seqY, 
				int start, int L, double *pxy, struct ali_s *ali, int alignment, 
				char *string_name);
extern void SimulateMut5Sequence(FILE *ofp, int *seq, SQINFO *sqinfoX, int *seqX, SQINFO *sqinfoY, int *seqY, 
				 int start, int *L, double *pxy, struct ali_s *ali, int alignment, 
				 char *string_name);

/* from othdp.c:
 */
extern struct othdpd_s *AllocDpDiagOTH(int L);
extern struct othdpf_s *AllocDpFullOTH(int Lx, int Ly);
extern double      **AllocMatrixDiag(int L);
extern double      **AllocMatrixFull(int Lx,  int Ly);
extern void          CompareOTH(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, 
				int start, int L, struct othmodel_s *oth, struct othdpd_s *othdp, 
				double *sc, struct ali_s *ali, int traceback, int alignment);
extern void          CompareOTHFull(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, 
				    int startX, int Lx, int startY, int Ly,  int *ret_leg, int Lw,
				    struct othmodel_s *oth, struct othdpf_s *othdp, struct ali_s *ali, struct windowends_s *windoends);
extern void          ForwardOTHFull(FILE *ofp, int *seqX, int *seqY, int startX, int startY, int Lx, int Ly, int Lw,
				    struct othmodel_s *oth, struct othdpf_s *othdp, struct sc2_s *othsc, double *sc, int ones);
extern double        ForwardOTH_L2(FILE *ofp, int *seqX, int *seqY, int startX, int startY, int Lx, int Ly, 
				   int Lw, struct othmodel_s *oth, struct othdpf_s *othdp, double *sc);
extern void          ForwardOTHDiag(FILE *ofp, int *seqX, int *seqY, int start, int L, 
				    struct othmodel_s *oth, struct othdpd_s *othdp, struct sc2_s *othsc, double *sc, int ones);
extern double        ForwardOTHDiag_L(FILE *ofp, int *seqX, int *seqY, int start, int L, 
				      struct othmodel_s *oth, struct othdpd_s *othdp, double *sc);
extern void          ForwardOTHDiagMatrix(FILE *ofp, double **mt, int *seqX, int *seqY, int start, int L, 
					  struct othmodel_s *oth, struct othdpd_s *othdp, double *sc);
extern void          ForwardOTHDiagVector(FILE *ofp, double *v, int *seqX, int *seqY, int start, int L, 
					  struct othmodel_s *oth, struct othdpd_s *othdp, double *sc);
extern void          FreeDpDiagOTH(struct othdpd_s *othdp);
extern void          FreeDpFullOTH(struct othdpf_s *othdp);
extern double        NullOJ(int iabs, int jabs, int dx, int dy, int *seqX, int *seqY, struct othmodel_s *oth);
extern void          PatternDpDiagOTH(int L, struct othdpd_s *dp);
extern void          PatternDpFullOTH(int Lx, int Ly, struct othdpf_s *dp);
extern void          PatternOTHModel(struct othmodel_s *oth);
extern void          TestForwardOTH(FILE *ofp, int *seqX, int *seqY, int start, int L, struct othmodel_s *oth, 
				    struct othdpd_s *othdp, double *sc);
extern void          TestViterbiOTH(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, 
				    int start, int L, 
				    struct othmodel_s *oth, struct othdpd_s *othdp, struct ali_s *ali, 
				    int alignment, int traceback, struct end_s *othends);
extern void          ViterbiOTHFull(FILE *ofp, 
				    SQINFO sqinfoX, int *seqX, 
				    SQINFO sqinfoY, int *seqY, 
				    int startX, int startY,
				    int Lx, int Ly, 
				    int *ret_len_ali,  int Lw,
				    struct othmodel_s *oth, struct othdpf_s *othdp, 
				    struct sc2_s *othsc, 
				    struct ali_s *ali, 
				    int alignment, int sweep, int ones, int traceback, 
				    int doends, struct windowends_s *windowends);
extern double        ViterbiOTH_L2(FILE *ofp, 
				   SQINFO sqinfoX, int *seqX, 
				   SQINFO sqinfoY, int *seqY, 
				   int startX, int startY,
				   int Lx, int Ly, 
				   int *ret_len_ali,  int Lw,
				   struct othmodel_s *oth, struct othdpf_s *othdp, 
				   struct ali_s *ali, 
				   int alignment, int sweep, int traceback, 
				   int doends, struct end_s *othends);
extern void          ViterbiOTHDiag(FILE *ofp, 
				    SQINFO sqinfoX, int *seqX, 
				    SQINFO sqinfoY, int *seqY, 
				    int start, int L, 
				    struct othmodel_s *oth, struct othdpd_s *othdp, 
				    struct sc2_s *othsc, 
				    struct ali_s *ali, 
				    int alignment, int ones, int sweep, int traceback, 
				    int doends, struct windowends_s *windowends);
extern double        ViterbiOTHDiag_L(FILE *ofp, 
				      SQINFO sqinfoX, int *seqX, 
				      SQINFO sqinfoY, int *seqY, 
				      int start, int L, 
				      struct othmodel_s *oth, struct othdpd_s *othdp, struct ali_s *ali, 
				      int alignment, int sweep, int traceback, 
				      int doends, struct end_s *othends);
extern void          ViterbiOTHDiagMatrix(FILE *ofp, double **mt, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, 
					  int *seqY, int start, int L, struct othmodel_s *oth, 
					  struct othdpd_s *othdp);
extern void          ViterbiOTHFullMatrix(FILE *ofp, double **mt, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, 
					  int *seqY, int startX, int startY, int Lx, int Ly, int Lw,
					  struct othmodel_s *oth, struct othdpf_s *othdp);
extern void          ViterbiOTHFullVector(FILE *ofp, double *v, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, 
					  int *seqY, int startX, int startY, int Lx, int Ly, 
					  struct othmodel_s *oth, struct othdpf_s *othdp);
extern void          ViterbiOTHDiagVector(FILE *ofp, double *v, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, 
					  int *seqY, int start, int L, struct othmodel_s *oth, 
					  struct othdpd_s *othdp);

/* from othdpscanfast.c:
 */
extern struct othdpscanfast_s  *AllocDpDiagScanFastOTH(int L);
extern struct othdpscanfast2_s *AllocDpDiagScanFast2OTH(int L);
extern void                     FillOTHMtxScanFast(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, 
						   int leg, int win, int j, int jmod, int d, 
						   struct othmodel_s *othmodel, struct nullmodel_s *null, struct othdpscanfast_s *othdp, 
						   double  **othj, int exact, int logodds, struct endscan_s *othends);
extern void                     FreeDpDiagScanFastOTH(struct othdpscanfast_s *dp);
extern void                     FreeDpDiagScanFast2OTH(struct othdpscanfast2_s *othdpscan2);
extern void                     PatternDpDiagScanFastOTH(int L, struct othdpscanfast_s *dp);
extern double                   ViterbiOTHDiagScanFast(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, 
						       int leg, int win, int st, int stmod, int l, int lmax, 
						       struct othmodel_s *oth, struct othdpscanfast_s *othdp, 
						       int revstrand, int traceback, int do_ends, struct endscan_s *othends);
extern double                   ViterbiBackwardsOTHDiagScanFast(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, 
								int win, int st, int stmod, int l, int lmax, 
								struct othmodel_s *oth, struct othdpscanfast_s *othdp);

/* from othmodel.c:
 */
extern struct othmodel_s *AllocOTHModel(void);
extern void    CheckOTHParam(struct othparam_s othparam);
extern void    CheckOTHProbs(struct othmodel_s *oth);
extern void    ConstructMemOTHProbs(double *mutpxy, double *mem);
extern void    ConstructOTHModel(double **pammodel_star, struct othparam_s param,  struct othparam_s othparam_zero, struct othparam_s othparam_infty, 
				 double win, double win_zero, double win_infty, double tfactor, double *targetfreq, int changefreq, 
				 struct othmodel_s *oth, int pedantic, int verbose);
extern void    ConstructXemOTHProbs(double *mem, double *xem);
extern void    ConstructYemOTHProbs(double *mem, double *yem);
extern void    FreeOTHModel(struct othmodel_s *oth);
extern void    OTHToLog2(struct othmodel_s *other);
extern void    OTHLog2ToOdds(struct othmodel_s *oth, struct nullmodel_s *null);
extern void    OTHmodelTransitionProbabilities(struct othparam_s othparam, struct othparam_s othparam_zero, struct othparam_s othparam_infty, 
					       double tfactor, struct othmodel_s *oth, int pedantic, int verbose);
extern void    PrintMutProbs(double *mutpxy, double *pnull);
extern void    PrintMut5Probs(double *mut5pxy, double *pnull);
extern void    PrintOTHModel(struct othmodel_s *oth);
extern void    PrintOTHTrProbs(struct othmodel_s *oth);

/* from otherscorealing.c:
 */
extern double        ScoreWithOTH_ungapped(int *sX, int *sY, int L, struct othmodel_s *oth);
extern void          ScoreWithOTH2(FILE *ofp, int *seqX, int *seqY, int start, int L,
				   struct othmodel_s *oth, struct sc2_s *othsc, int ones, int traceback);
extern double        ScoreWithOTH(FILE *ofp, int *seqX, int *seqY, int start, int L,
				  struct othmodel_s *oth, int traceback);
extern double        ScoreWithOTHMatrix(int *sX, int *sY, int j, struct othmodel_s *oth);

/* from posterior.c:
 */
extern double Posterior(double sc1, double sc2, double sc3);
extern double PosteriorLog2(double sc1, double sc2, double sc3);
extern double SigmoidalLog2(double sc1, double sc2, double sc3);

/* from qrnaversion.c:
 */
extern int  IndexForwardWindow (int L, int win, int off, int j);
extern int  IndexBackwardWindow(int L, int win, int off, int j);
extern int  IndexWindow(int j, int L, int win, int off, int revstrand);
extern int  NumberScanningWindows(int L, int win, int off);
extern void PrintScanBanner (FILE *ofp, int start, int end, 
			     int lenX, int lenY, int fstX, int fstY, double *freqX, double *freqY, 
			     double id, double gap, double mut, int use_win_id, struct three_times_s time);
extern void RNAbanner (FILE *ofp, int cyk);
extern void ScoreScanFast(FILE *ofp, 
			  FILE *regressfp, char *regressionfile, 
			  int format, 
			  SQINFO sqinfoX, int *isegX, int *iseqX, double *freqX,
			  SQINFO sqinfoY, int *isegY, int *iseqY, double *freqY,
			  char *aliss,
			  int Lw, int leg, int win, int slide,	
			  double id_ali, double id_win_mean, double id_win_sd,
			  fullmat_t           *riboprob, 
			  double            ***cfg_node,
			  double             **hexa,
			  double              *pair5prob,
			  double              *codon_joint,
			  int                **pam,
			  double               scale,
			  struct dos_s             dos, 
			  struct model_s           *model, 
			  struct dpdscanfast_s     *dpdscan, 
			  struct dpd_s             *dpd, 
			  struct dpf_s             *dpf, 
			  struct rnascfgscanfast_s *mxscan,   
			  struct rnascfg_s         *mx, 
			  struct scores_s          *sc, 
			  struct ali_s             *ali,
			  struct scanfast_s        *scanfast,
			  int add_codon, int add_hexamer, int alignment, int cyk, int changefreqwin, int evd,
			  int fastintloop, int logodds, int pedantic, int shuffle, int sre_shuffle, struct three_times_s  tfactor, 
			  int traceback, int verbose, int ones, int parse, int rnass, int doends, int N, double k, int shift, int *histo_sc);
extern void ScoreWindow(FILE *ofp, 
			FILE *regressfp, char *regressionfile, 
			int format, 
			SQINFO sqinfoX, int *isegX, int *iseqX, double *freqX,
			SQINFO sqinfoY, int *isegY, int *iseqY, double *freqY,
			char *aliss,
			int Lw, int leg, int win, int slide,	      
			double id_ali, double id_win_mean, double id_win_sd,
			fullmat_t           *riboprob, 
			double            ***cfg_node,
			double             **hexa,
			double              *pair5prob,
			double              *codon_joint,
			int                **pam,
			double               scale,
			struct dos_s         d, 
			struct model_s      *model, 
			struct dpd_s        *dpd, 
			struct dpf_s        *dpf, 
			struct rnascfg_s    *mx, 
			struct scores_s     *sc, 
			struct ali_s        *ali,
			int add_codon, int add_hexamer, int alignment, int cyk, int changefreq, int changefreqwin, int evd,
			int fastintloop, int logodds, int pedantic, int shuffle, int sre_shuffle, int sweep, struct three_times_s  tfactor, 
			int traceback, int verbose, int ones, int parse, int rnass, int doends, int shtoo, int twindow, int N, double k, int shift, int *histo_sc);


/* from riboprob.c:
 */
extern fullmat_t     *AllocFullMatrix (int L);
extern fullcondmat_t *AllocFullCondMatrix (int L);
extern matrix_t      *AllocMatrix (int L);
extern condmatrix_t  *AllocCondMatrix (int L);
extern void           CalculatePPConditionalsAndMarginals(FILE *ofp, double *pp, double **ret_Qx, double **ret_Qy, double **ret_Ql, double **ret_Qr, 
							  double **ret_pairx, double **ret_pairy, double **ret_pmutl, double **ret_pmutr, 
							  int L, int verbose);
extern void           CalculatePPMarginals(FILE *ofp, double *pp, double **ret_pairx, double **ret_pairy, double **ret_pmutl, double **ret_pmutr, 
					   int L, int verbose);
extern void           ComputePPMarginals(FILE *ofp, double *pp, double *pairx, double *pairy, double *pmutl, double *pmutr, int L, int verbose);
extern void           CalculateRIBOConditionals (FILE *fp, fullmat_t *fullmat, fullcondmat_t **ret_cond_mut, fullcondmat_t **ret_cond_pair, int verbose);
extern void           CalculateRIBO5Conditionals (FILE *fp, fullmat_t *riboprob, fullcondmat_t **ret_ribocond5_mut, fullcondmat_t **ret_ribocond5_pair, 
						  double *mut5pxy, double *pair5prob, int verbose);
extern void           CalculateRIBO5ConditionalsUncorr_201(FILE *fp, fullmat_t *riboprob, fullcondmat_t **ret_ribocond5_mut, 
							   fullcondmat_t **ret_ribocond5_pair, double *mut5pxy, double *pair5prob, 
							   double *target, int verbose);
extern void           CalculateRIBO5ConditionalsUncorr (FILE *fp, fullmat_t *riboprob, fullcondmat_t **ret_ribocond5_mut, 
							fullcondmat_t **ret_ribocond5_pair, double *mut5pxy, double *pair5prob, double *target, 
							int verbose);
extern void           CalculateRIBOConditionalsUncorr (FILE *fp, fullmat_t *riboprob, fullcondmat_t **ret_ribocond_mut, 
						       fullcondmat_t **ret_ribocond_pair, 
						       double *targetfreq, int verbose);
extern void           CalculateRIBO5MutConditionals_201(FILE *fp, fullmat_t *riboprob, fullcondmat_t **ret_ribocond5_mut, 
							double *mut5pxy, double *pair5prob, int verbose);
extern void           CalculateRIBO5MutConditionals (FILE *fp, fullmat_t *riboprob, fullcondmat_t **ret_ribocond5_mut, 
						     double *mut5pxy, double *pair5prob, int verbose);
extern void           CalculateRIBOMutConditionals (FILE *fp, fullmat_t *riboprob, fullcondmat_t **ret_ribocond_mut, int verbose);
extern void           CalculateRIBOMutConditionalsApprox(FILE *ofp, int L, double *riboprobs, fullcondmat_t **ret_ribocond_mut, int verbose);
extern void           CalculateRIBO5PairConditionals_201(FILE *fp, fullmat_t *riboprob, fullcondmat_t **ret_ribocond5_pair, 
							 double *pair5prob, double *mut5pxy, int verbose);
extern void           CalculateRIBO5PairConditionals (FILE *fp, fullmat_t *riboprob, fullcondmat_t **ret_ribocond5_pair, 
						      double *pair5prob, double *mut5pxy, int verbose);
extern void           CalculateRIBOPairConditionals (FILE *fp, fullmat_t *riboprob, fullcondmat_t **ret_ribocond_pair, int verbose);
extern void           CalculateRIBOPairConditionalsApprox(FILE *ofp, int L, double *riboprobs, fullcondmat_t **ret_ribocond_pair, int verbose);
extern void           CalculateRIBOMarginals(FILE *ofp, fullmat_t *riboprobs, double **ret_pmxl, double **ret_pmxr, double **ret_pmyl,
					     double **ret_pmyr, int verbose);
extern void           ComputeRIBOMarginals(FILE *ofp, fullmat_t *riboprobs, double *pmxl, double *pmxr, double *pmyl, double *pmyr, int verbose);
extern void           CalculateRIBOProbsApprox(FILE *ofp, int L, double *mutpxy, double *pair, double **ret_pairprobs, int verbose);
extern void           ChangeRIBOProbsIterate(FILE *ofp, fullmat_t *riboprobs, double *targetfreq, int verbose);
extern void           ChangePPMutProbsIterate(FILE *ofp, double *pp, double *mut5pxy, int L, int verbose);
extern void           ChangePPPairProbsIterate(FILE *ofp, double *pp, double *pair5prob, int L, int verbose);
extern fullcondmat_t *CopyFullCondMatrix (fullcondmat_t *fullcondmat);
extern fullmat_t     *CopyFullMatrix (fullmat_t *fullmat);
extern void           CopyRIBOMatrix (matrix_t *matcopy, matrix_t *mat);
extern void           CopyRIBOCondMatrix (condmatrix_t *condmatcopy, condmatrix_t *condmat);
extern void           EvolveRIBO5PairConditionals(fullcondmat_t *ribo5cond, condmatrix_t *ribo5cond_nought, 
						 double *pair5prob, double tfactor, int ispair, int pedantic, int verbose);
extern void           EvolveRIBOPairConditionals(fullcondmat_t *ribocond, condmatrix_t *ribocond_nought, 
						 double tfactor, double *targetpair, int changepair, int pedantic, int verbose);
extern void           FreeFullMatrix (fullmat_t *fullmat);
extern void           FreeFullCondMatrix (fullcondmat_t *fullmat);
extern void           FreeMatrix (matrix_t *mat);
extern void           FreeCondMatrix (condmatrix_t *mat);
extern int            numbered_basepair (char c, char d);
extern int            numbered_nucleotide (char c);
extern void           PrintFullMatrix (FILE *fp, fullmat_t *fullmat);
extern void           PrintFullRIBOCondMatrix (FILE *fp, fullcondmat_t *fullcondmat, int ispaircond, int islog2);
extern void           PrintRIBOCondMatrix (FILE *fp, condmatrix_t *condmat, int ispaircond, int islog2, char *title); 
extern void           PrintRIBOMatrix (FILE *fp, double *mat, int L, int ispaircond, int change, int takelog2, char *title);
extern void           RIBOMutCondFromRIBOPairCond (fullcondmat_t **ret_ribo5cond_mut, fullcondmat_t *ribo5cond_pair, double *mut5pxy, 
						   double tfactor, int pedantic, int verbose);
extern fullmat_t     *ReadRIBOPROBMatrix(FILE *matfp);

/* from rnacreatealing.c:
 */
extern void SimulateNUSSequences(struct rnamodel_s *rna, int *ct, int L, 
				 char **ret_sX, char **ret_sY);
extern void SimulateRNAAlign(FILE *ofp, SQINFO *sqinfoX, int **seqX, SQINFO *sqinfoY, int **seqY, int L, int start, int *ret_len, 	
			     struct rnamodel_s *rna, int traceback, int alignment, char *string_name, 
			     int *ret_pairs, int *ret_comp_pairs, int *ret_noncomp_pairs);
extern void SimulateRNAAlignFromPosterior(FILE *ofp, SQINFO *sqinfoX, int *seqX, SQINFO *sqinfoY, int *seqY, int start, 
					  int L, struct rnamodel_s *rna, struct rnamtx_s *in, struct ali_s *ali, 
					  int *ct, int traceback, int alignment, char *string_name, 
					  int *ret_pairs, int *ret_comp_pairs, int *ret_noncomp_pairs);
extern void SimulateRNASequence(FILE *ofp, int *seq, SQINFO *sqinfoX, int *seqX, SQINFO *sqinfoY, 
				int *seqY, int start, int L, struct rnamodel_s *rna, 
				struct ali_s *ali, int *ct, int traceback, int alignment, char *string_name, 
				int *ret_pairs, int *ret_comp_pairs, int *ret_noncomp_pairs);
/* from rnadp.c:
 */
extern struct rnadp_s *AllocDpDiagRNA(int L);
extern struct rnadp_s *AllocDpFullRNA(int L);
extern void            ForwardRNADiag(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, char *gss, int start, int L, 
				      struct model_s *model, struct dpd_s *dp, struct rnascfg_s *mx, struct sc2_s *rnasc,
				      struct dpsc3_s *sc, int *ret_scfg_status, int *ret_scfgrv_status, 
				      int cyk, int fastintloop, int logodds, int do_nus, int ones, int parse);
extern void            FreeDpRNA(struct rnadp_s *dp);
extern void            PatternDpRNA(int L, struct rnadp_s *rnadp);
extern double          ViterbiRNAFull(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, 
				      int *seqY, char *gss, int startX, int Lx, int startY, int Ly, 
				      struct rnamodel_s *rna, struct dpf_s *dp, struct rnascfg_s *mx, 
				      struct ali_s *ali, int alignment, int ones, int parse, int traceback, 
				      int doends, struct end3_s *ends);
extern void            ViterbiRNADiag(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, char *gss, 
				      int start, int L, struct model_s *model, struct dpd_s *dp, 
				      struct rnascfg_s *mx, struct sc2_s *rnasc, struct ali_s *ali, int *ret_scfg_status, int *ret_scfgrv_status, 
				      int alignment, int cyk, int fastintloop, int logoods, int do_nus, int ones, int parse, int traceback, 
				      int doends, struct windowends_s *windowends);

/* from rnadpscanfast.c:
 */
extern struct rnadpscanfast_s  *AllocDpDiagScanFastRNA(int L);
extern struct rnadpscanfast2_s *AllocDpDiagScanFast2RNA(int L);
extern void                     FreeDpScanFastRNA(struct rnadpscanfast_s *rnadpscan);
extern void                     FreeDpScanFast2RNA(struct rnadpscanfast2_s *rnadpscan2);
extern void                     FreeDpXFastRNA(struct rnadpxfast_s *dp);
extern void                     PatternDpXFastRNA(int L, struct rnadpxfast_s *rnadp);
extern double                   ViterbiRNADiagScanFast(FILE *ofp, SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, char *gss, 
						       int leg, int win, int st, int stmod, int l, int lmax, 
						       struct model_s *model, struct rnadpscanfast_s *dp, struct rnamtxscanfast_s *mtx,
						       struct ali_s *ali, int alignment, 
						       int cyk, int logodds, int revstrand, int parse, int traceback, 
						       int doends, struct end3scan_s *ends);

/* from rnaio.c:
 */
extern int   IsCompensatory (int xl, int xr, int yl, int yr);
extern int   IsNonCompensatory (int xl, int xr, int yl, int yr);
extern int   IsWcPair (int xl, int xr, int yl, int yr);
extern int   KHS2ct(char *ss, int len, int allow_pseudoknots, int **ret_ct);
extern int   KHS2ctInfernal(char *ss, int len, int allow_pseudoknots, int **ret_ct);
extern char *RevSS(char *ss, int L);
extern int   VerifyKHS(char *name, char *ss, int L, int wordy);

/* from rnamodel.c:
 */
extern struct nusmodel_s *AllocNUSModel(void);
extern struct pi2model_s *AllocPI2Model(void);
extern struct rnamodel_s *AllocRNAModel(void);
extern void       CheckRNAProbs(struct rnamodel_s *rna);
extern void       ConstructRNAModel_201(double **pammodel_star, double ***cfg_node, double *pair5, fullmat_t *riboprob,
					int win, double tfactor, struct rnamodel_s *rna, struct nullmodel_s *null,
					double *targetfreq, int changefreq, int pedantic, int verbose);
extern void       ConstructRNAModel(double **pammodel_star, double ***cfg_node, double *pair5, fullmat_t *riboprob, 
				    int win, double tfactor, struct rnamodel_s *rna, struct nullmodel_s *null, 
				    double *targetfreq, int changefreq, int pedantic, int verbose);
extern struct  pi2model_s *CopyPI2Model(struct pi2model_s *pi2);
extern void       FreeNUSModel(struct nusmodel_s *nus);
extern void       FreePI2Model(struct pi2model_s *pi2);
extern void       FreeRNAModel(struct rnamodel_s *rna);
extern void       NUSToLog2(struct nusmodel_s *nus);
extern void       NUSLog2ToOdds(struct nusmodel_s *nus, struct nullmodel_s *null);
extern void       PI2ToLog2(struct pi2model_s *pi2, struct nullmodel_s *null);
extern void       PI2Log2ToOdds(struct pi2model_s *pi2, struct nullmodel_s *null);
extern void       PatternRNAModel(struct rnamodel_s *rna);
extern void       RNAToLog2(struct rnamodel_s *rna, struct nullmodel_s *null);
extern void       RNALog2ToOdds(struct rnamodel_s *rna, struct nullmodel_s *null);
extern void       PrintNUSModel(struct nusmodel_s *nus);
extern void       PrintPI2Model(struct pi2model_s *pi2);
extern void       PrintRNAModel(struct rnamodel_s *rna);
extern void       PrintRNATrProbs(struct rnamodel_s *rna);
extern void       RNAmodelTransitionProbabilities(double tfactor, struct rnamodel_s *rna, int pedantic, int verbose);
extern void       SCFGTransitionProbabilities(int len, double tfactor, struct pi2model_s *pi2, int pedantic, int verbose);

/* from rnaoutput.c: conversion of internal data structures into
 *                   more user-friendly forms
 */
extern void CompareRNAStructures(FILE *ofp, int start, int L, char *ss_true, char *ss);
extern void CompareRNAParses(FILE *ofp, int start, int L, char *ss_true, char *ss);
extern void Ct2KHS_ER(int start, int L, int *ct, char **ret_cc);
extern int  IsRNAComplement(char sym1, char sym2, int allow_gu);
extern void RNAPairs(SQINFO sqinfo, int *ss, int format, float *ret_pairs, float *ret_cykpairs);
extern void TraceRNA(struct tracekn_s *tr, int rlen, int lend, int watsoncrick, char **ret_ss, char **ret_cc) ;
extern void Tracenrn(struct tracekn_s *tr, int rlen, int lend, int watsoncrick, char **ret_ss, char **ret_cc);
                     

/* from rnascorealing.c:
 */
extern double ScoreWithNUS_ungapped(int *sX, int *sY, int L, struct nusmodel_s *nus, int traceback);
extern void   ScoreWithRNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, char *gss, int start, int L, 
			   struct model_s *model, struct othdpd_s *othdp, struct rnascfg_s *mx, struct sc2_s *rnasc,
			   int *ret_scfg_status, int *ret_scfgrv_status, int cyk, int fastintloop, int logodds, 
			   int do_nus, int ones, int parse, int traceback);

/* from rnascfg.c:
 */
extern struct rnamtx_s     *AllocRNAMtx(int L);
extern struct rnamtx_ou_s  *AllocRNAMtxOutside(int L);
extern struct rnascfg_s    *AllocScfgRNA(int L, int fastintloop, int rnass);
extern void     BestV (int *sX, int *sY, struct pi2model_s *pi2, int start, int L, int j, int d, 
		       struct rnamtx_s *mtx, double *vp, int fastversion);
extern void     BestParseV (int *sX, int *sY, int *ct, struct pi2model_s *pi2, int start, int L, int j, int d, 
			    struct rnamtx_s *mtx, double *vp);
extern void     BestW (int *sX, int *sY, struct pi2model_s *pi2, int start, int L, int j, int d, 
		       struct rnamtx_s *mtx);
extern void     BestParseW (int *sX, int *sY, int *ct, struct pi2model_s *pi2, int start, int L, int j, int d, 
			    struct rnamtx_s *mtx);
extern void     BestWB(int *sX, int *sY, struct pi2model_s *pi2, int start, int L, int j, int d, 
		       struct rnamtx_s *mtx);
extern void     BestParseWB(int *sX, int *sY, int *ct, struct pi2model_s *pi2, int start, int L, int j, int d, 
			    struct rnamtx_s *mtx);
extern void     CalPairs (int *sX, int *sY, int start, int L, struct pi2model_s *pi2, 
			  struct rnamtx_s *in, struct rnamtx_ou_s *ou, double ***pp, double **ps, struct end_s *rnaends);
extern int      CYKParseRNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, int *ct, int start, int L,
			    struct rnamodel_s *rna, struct nullmodel_s *null, struct othdpd_s *othdp, 
			    struct rnamtx_s *mtx, double *vp, int logodds, int do_nus, int traceback);
extern int      CYKRNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, int start, int L, 
		       struct rnamodel_s *rna, struct nullmodel_s *null, struct othdpd_s *othdp, 
		       struct rnamtx_s *mtx, double *vp, 
		       int logoods, int do_nus, int traceback);
extern void     CYKParseTracebackRNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, char *gss, int start, int L,
				     struct pi2model_s *pi2, struct rnascfg_s *mx, 
				     int do_nus, int logodds, int traceback, struct end_s *rnaends, int isrnarev);
extern void     CYKTracebackRNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, char *gss, int start, int L,
				struct pi2model_s *pi2, struct rnascfg_s *mx, 
				int do_nus, int logodds, int traceback, struct end_s *rnaends, int isrnarev);
extern void     FillV (int *sX, int *sY, struct pi2model_s *pi2, int start, int L, int j, int d, 
		       struct rnamtx_s *mtx, double *vp, double *sc, int fastversion);
extern void     FillVSS (struct pi2model_s *pi2, int j, int d, struct rnamtx_s *mtx, double *vp, double *sc);
extern void     FillparseV (int *sX, int *sY, int *ct, struct pi2model_s *pi2, int start, int L, int j, int d, 
			    struct rnamtx_s *mtx, double *vp, double *sc);
extern void     FillVO(int *sX, int *sY, struct pi2model_s *pi2, double score, int start, int L, int j, int d, 
		       double *vp, double **wox, double **wbox, double **vox, double **vx, double *sc, struct end_s *rnaends);
extern void     FillW (int *sX, int *sY, struct pi2model_s *pi2, int start, int L, int j, int d, 
		       struct rnamtx_s *mtx, double *sc);
extern void     FillWSS(struct pi2model_s *pi2, int j, int d, struct rnamtx_s *mtx, double *sc);
extern void     FillParseW (int *sX, int *sY, int *ct, struct pi2model_s *pi2, int start, int L, int j, int d, 
			    struct rnamtx_s *mtx, double *sc);
extern void     FillWO(int *sX, int *sY, struct pi2model_s *pi2, double score, int start, int L, int j, int d, 
		       double **wox, double **wx, double *sc, struct end_s *rnaends);
extern void     FillWB(int *sX, int *sY, struct pi2model_s *pi2, int start, int L, int j, int d, 
		       struct rnamtx_s *mtx, double *sc);
extern void     FillWBSS(struct pi2model_s *pi2, int j, int d, struct rnamtx_s *mtx, double *sc);
extern void     FillParseWB(int *sX, int *sY,int *ct,  struct pi2model_s *pi2, int start, int L, int j, int d, 
			    struct rnamtx_s *mtx, double *sc);
extern void     FillWBO(int *sX, int *sY, struct pi2model_s *pi2, double score, int start, int L, int j, int d, double **wbox,
			double **wbx, double **vox, double *sc, struct end_s *rnaends);
extern void     FreeRNAMtx(struct rnamtx_s *mtx);
extern void     FreeRNAMtxOutside(struct rnamtx_ou_s *mtx);
extern void     FreeScfgRNA(struct rnascfg_s *mx, int rnass);
extern int      InsideRNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, 
			  int start, int L, struct rnamodel_s *rna,  struct nullmodel_s *null, 
			  struct othdpd_s *othdp, struct rnamtx_s *mtx, double *sc, double *vp, 
			  int do_nus, int fastintloop, int logodds);
extern void     InsideRNASS(FILE *ofp, int L, struct pi2model_s *pi2, struct rnascfg_s *mx);
extern int      InsideParseRNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, int *ct,
			       int start, int L, struct rnamodel_s *rna,  struct nullmodel_s *null, 
			       struct othdpd_s *othdp, struct rnamtx_s *mtx, double *sc, double *vp, 
			       int do_nus, int logodds);
extern int      OutsideRNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, int start, int L,
			   struct rnamodel_s *rna, struct rnamtx_s *in, struct rnamtx_ou_s *ou, double *vp, double *sc,
			   int do_nus, int logodds, struct end_s *rnaends);
extern void     SCFGTransFromPosteriors(FILE *ofp, int j, int d, struct pi2model_s *pi2, struct rnamtx_s *mtx, struct pi2model_s *pi2post);
extern void     Tr_V(FILE *ofp, int *sX, int *sY, struct pi2model_s *pi2, int start, int L, int j, int d, 
		     struct rnamtx_s *mtx, double *vp,
		     int *flag, struct tracekn_s *curr_tr, struct traceknstack_s *dolist, int traceback, struct end_s *rnaends);
extern void     Tr_Parse_V(FILE *ofp, int *sX, int *sY, int *ct, struct pi2model_s *pi2, int start, int L, int j, int d, 
			   struct rnamtx_s *mtx, double *vp,
			   int *flag, struct tracekn_s *curr_tr, struct traceknstack_s *dolist, int traceback, struct end_s *rnaends);
extern void     Tr_W(FILE *ofp, int *sX, int *sY, struct pi2model_s *pi2, int start, int L, int j, int d, struct rnamtx_s *mtx, 
		     int *flag, struct tracekn_s *curr_tr, struct traceknstack_s *dolist, int traceback, struct end_s *rnaends);
extern void     Tr_Parse_W(FILE *ofp, int *sX, int *sY, int *ct, struct pi2model_s *pi2, int start, int L, int j, int d, struct rnamtx_s *mtx, 
			   int *flag, struct tracekn_s *curr_tr, struct traceknstack_s *dolist, int traceback, struct end_s *rnaends);
extern void     Tr_WB(FILE *ofp, int *sX, int *sY, struct pi2model_s *pi2, int start, int L, int j, int d, struct rnamtx_s *mtx, 
		      int *flag, struct tracekn_s *curr_tr, struct traceknstack_s *dolist, int traceback, struct end_s *rnaends);
extern void     Tr_Parse_WB(FILE *ofp, int *sX, int *sY, int *ct, struct pi2model_s *pi2, int start, int L, int j, int d, struct rnamtx_s *mtx, 
			    int *flag, struct tracekn_s *curr_tr, struct traceknstack_s *dolist, int traceback, struct end_s *rnaends);
extern void     Trk_RNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, int start, int L,  int j, int d,
			struct pi2model_s *pi2, struct rnamtx_s *mtx, double *vp,
			struct tracekn_s **ret_trace, int traceback, struct end_s *rnaends);
extern void     Trk_RNA_FromPosteriors(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, int start, int L,  int j, int d,
				       struct pi2model_s *pi2, struct rnascfg_s *mx_ss, struct rnamtx_s *mtx, double *vp,
				       struct tracekn_s **ret_trace, int traceback, struct end_s *rnaends);
extern void     Trk_Parse_RNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, int *ct, int start, int L, int j, int d,
			      struct pi2model_s *pi2, struct rnamtx_s *mtx, double *vp,
			      struct tracekn_s **ret_trace, int traceback, struct end_s *rnaends);
extern void     PatternVec(int L, double *vec);
extern void     PatternMtx(int L, double **m);
extern void     PatternRNAMtx(int L, struct rnamtx_s *rnamtx);
extern void     PatternRNAMtxOutside(int L, struct rnamtx_ou_s *rnamtx);
extern void     PatternScfgRNA(int L, struct rnascfg_s *mx, int rnass, int rvrna);
extern void     PosteriorRNA(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, char *gss, int start, int L,
			     struct rnamodel_s *rna, struct rnascfg_s *mx, 
			     int do_nus, int logodds, int traceback, struct end_s *rnaends, int isrnarev);


/* from rnascfgscanfast.c:
 */
extern struct rnamtxscanfast_s  *AllocRNAMtxScanFast(int L, int fastintloop);
extern struct rnascfgscanfast_s *AllocScfgScanFastRNA(int L, int fastintloop);
extern void                      CYKRNAScanFast(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, 
						int win, int j, int jmod, int d, 
						struct rnamodel_s *rna, struct nullmodel_s *null, 
						struct rnamtxscanfast_s *mtx, int logodds, int traceback);
extern void                      InsideRNAScanFast(FILE *ofp, SQINFO sqinfoX, int *sX, SQINFO sqinfoY, int *sY, 
						   int win, int j, int jmod, int l,
						   struct rnamodel_s *rna, struct nullmodel_s *null, 
						   struct rnamtxscanfast_s *mtx, double *sc, double *vp, int fastintloop, int logodds, int traceback);
extern void                      FreeRNAMtxScanFast(struct rnamtxscanfast_s *mtx);
extern void                      FreeScfgScanFastRNA(struct rnascfgscanfast_s *mx);
extern void                      PatternRNAMtxScanFast(int L, struct rnamtxscanfast_s *rnamtx);
extern void                      PatternScfgScanFastRNA(int L, struct rnascfgscanfast_s *mx);

/* from scorewithmodels.c:
 */
extern struct windowends_s *AllocWindowEnds();
extern struct end_s        *AllocEnd();
extern struct end3_s       *AllocEnds3();
extern struct sc3_s        *AllocSc3();
extern struct scores_s     *AllocScores();
extern void             FreeEnd(struct end_s *ends);
extern void             FreeEnds3(struct end3_s *ends);
extern void             FreeWindowEnds(struct windowends_s *windowends);
extern void             FreeSc3(struct sc3_s *sc3);
extern void             FreeScores(struct scores_s *sc);
extern void             PatternEnd(struct end_s *ends);
extern void             PrintScores(FILE *ofp, struct sc3_s *sc, int doends, struct windowends_s *windowends, int start, int win, int ones);
extern void             ScoreWithModels(FILE *ofp, 
					FILE *regressfp, char *regressionfile, 
					SQINFO sqinfoX, int *isegX, int *iseqX, 
					SQINFO sqinfoY, int *isegY, int *iseqY, 
					char *gss, int start, int leg, int Lmax, 
					struct dos_s dos, 
					struct model_s *model, 
					struct dpd_s *dpd, struct dpf_s *dpf, 
					struct rnascfg_s *mx, 
					struct scores_s *sc, 
					struct ali_s *ali,  
					int alignment, int cyk, int doends, int fastintloop, int logodds, int ones, int parse, int rnass, 
					int shuffle, int sre_shuffle, int sweep, int traceback, int verbose, int use_win_id, struct three_times_s time);
extern char            *WinnerOfThree(double othsc, double codsc, double rnacs);
extern char            *WinnerScore(struct sc3_s *sc, int ones);

/* from scorewithmodelscanfast.c:
 */
extern struct endscan_s  *AllocEndScanFast();
extern struct end3scan_s *AllocEnd3ScanFast();
extern void               CopyEnd3ScanFast(struct end3scan_s *copy_ends, struct end3scan_s *ends);
extern void               FreeEndScanFast(struct endscan_s *end);
extern void               FreeEnd3ScanFast(struct end3scan_s *ends);
extern int                IsFullWindow(int revstrand, int L, int win, int off, int j);
extern void               PatternEndScanFast(struct endscan_s *end);
extern void               PatternEnd3ScanFast(struct end3scan_s *ends);
extern void               PrintEndScanFast(struct endscan_s *ends);
extern void               PosteriorScoresScanFast(FILE *ofp, struct scanfast_s *scanfast, int leg, int j, int win, int off, int doends, int ones);
extern int                ScoreWithModelsScanFast(FILE *ofp, 
						  SQINFO sqinfoX, int *isegX,
						  SQINFO sqinfoY, int *isegY,
						  char *gss,
						  int L, int win, int slide, int j, int jmod, int l, int lmax, 
						  struct model_s           *model, 
						  struct dpdscanfast_s     *dpdscan, 		   
						  struct rnascfgscanfast_s *mx, 
						  struct scores_s          *sc, 
						  struct ali_s             *ali, 
						  struct scanfast_s        *scanfast,
						  int alignment, int cyk,  int doends, int fastintloop, int logodds, int ones, int parse, int rnass, 
						  int revstrand, int shuffle, int traceback, int verbose);

/* from seqio.c:
 */
extern void          AdjustCT(FILE *ofp, SQINFO *sqinfo, char *seq, int *ct);
extern void          AliIdStats(int *iseqX, int *iseqY, int j, int d, int win, int slide, double *ret_id_win_mean, double *ret_id_win_sd);
extern void          AlignStat(FILE *ofp, SQINFO *sqinfoX, int *seqX, SQINFO *sqinfoY, int *seqY, int len, 
			       double *id, double *gap, double *mut);
extern double       *AllocBaseFreq(void);
extern void          AllocCharSeq(int Lmax, char **ret_seq);
extern void          AllocCharSeqs(int Lmax, char **ret_seqX, char **ret_seqY);
extern void          AllocIntSeq(int Lmax, int **ret_iseq);
extern void          AllocIntSeqs(int Lmax, int **ret_iseqX, int **ret_iseqY);
extern void          AllocCt(int Lmax, int **ret_ct);
extern void          ArrangeSequence(FILE *ofp, SQINFO *sqinfo, int format, char *seq, int **ret_ct, 
				     int allow_pseudoknots, int verbose);
extern void          ArrangeGapSequence(FILE *ofp, SQINFO *sqinfo, int format, char *seq, int **ret_ct, 
					int allow_pseudoknots, int verbose);
extern void          BaseComp(FILE *ofp, int *s, int j, int d, double *basefreq);
extern void          BaseCount(FILE *ofp, int *s, int j, int d, double *basecount);
extern void          BaseCompChar(FILE *ofp, char *s, int j, int d, double *basefreq);
extern void          BaseCompNorm(int nbases, double *basefreq);
extern void          CalculatePairs(FILE *outf, char *seq, SQINFO sqinfo, int *ss);
extern void          CheckMaxLength(SQFILE *sqfp, int format, long int *ret_Lmax, int *ret_nseqs, long int maxhitlen);
extern int           CleanUpAliCt(struct ali_s *ali, int *ct, int j, int d);
extern int           CleanUpCharSeqsCt(char *charX, char *charY, int *ct, int j, int d);
extern void          CleanUpSeqs(int *iseqX, int *iseqY, int j, int d, int *ret_len);
extern void          Count2Freq(FILE *ofp, int dim, double *basecount);
extern char         *DedigitizeSequence(int *dsq, int start, int L);
extern int          *DigitizeSequence(char *seq, int L);
extern void          DupIntAseq(int *seq, int *copyseq, int len);
extern void          DupIntSeq(int *iseq, int *copyseq, int j, int d);
extern char         *DupSeq(char *seq, int j, int d);
extern void          FillAliStruct(int *iseqX, int *iseqY, int j, int d, struct ali_s *ali);
extern void          IntizeGapAsequence(char *seq, int start, int len, int *iseq, int verbose);
extern void          IntizeGapSequence(char *seq, int len, int **ret_iseq);
extern void          IntizeSequence(char *seq, int len, int **ret_iseq);
extern int           LenNoGaps(int *seq, int j, int d);
extern void          PatternIntSeqs(int Lmax, int *iseqX, int *iseqY);
extern void          PatternCharSeqs(int Lmax, char *seqX, char *seqY);
extern void          PatternCt(int Lmax, int *ct);
extern void          PercIdSeqs(int *iseqX, int *iseqY, int j, int d, double *ret_id, double *ret_gap, double *ret_mut);
extern int           PosNoGaps(int *seq, int pos);
extern void          PrintAlign(FILE *ofp, SQINFO sqinfoX, SQINFO sqinfoY, int start, int L, 
				struct ali_s *ali);
extern void          PrintAlignIntSeqs(FILE *ofp, SQINFO *sqinfoX, SQINFO *sqinfoY, char *string_name, double tfactor,
				       int start, int len, int *seqX, int *seqY);
extern void          PrintNewAlign(FILE *ofp, SQINFO *sqinfoX, SQINFO *sqinfoY, int start, int L, 
				   struct ali_s *ali, char *string_name);
extern void          PrintNewCharSeqs(FILE *ofp, SQINFO *sqinfoX, SQINFO *sqinfoY, int start, int L, char *charX, char *charY,
				      char *string_name);
extern void          PrintCtAlign(FILE *ofp, SQINFO *sqinfoX, SQINFO *sqinfoY, int start, int L, 
				  struct ali_s *ali, int *ct, char *string_name);
extern void          PrintCtCharSeqs(FILE *ofp, SQINFO *sqinfoX, SQINFO *sqinfoY, int start, int L, char *charX, char *charY, 
				     int *ct, char *string_name);
extern void          PrintCtSeqs(FILE *ofp, SQINFO *sqinfoX, int *sX, SQINFO *sqinfoY, int *sY, int start, int L, char *ss);
extern void          PrintCtSeq(FILE *ofp, SQINFO *sqinfo, int *iseq, int start, int L, char *ss);
extern void          PrintQfile (FILE *ofp, char *qfile, SQINFO sqinfoX, int *isegX, SQINFO sqinfoY, int *isegY, int start, int L, int abs_shift);
extern void          PrintSSseqs(FILE *ofp, SQINFO *sqinfoX, int *seqX, SQINFO *sqinfoY, int *seqY, 
				 int start, int L, char *ss, char *cc);
extern void          Print2Seqs(FILE *ofp, SQINFO *sqinfoX, int *sX, SQINFO *sqinfoY, int *sY, int start, int L);
extern void          PrintBanner(FILE *ofp, char *pamfile, char *rnafile, char *ribofile, char *seqfile, 
				 char *seqfile_b, int allow_pseudoknots, int nseqs, long int Lmax, long int maxhitlen, 
				 int minhitlen, int use_ribo_approx, int sweep, int window, int slide, int scan);
extern void          PrintSeq(FILE *ofp, SQINFO *sqinfo, int leg, char *seg);
extern void          PrintIntSeq(FILE *ofp, SQINFO *sqinfo, int leg, int *iseg);
extern void          PrintAlignSequences(FILE *ofp, SQINFO *sqinfoX, SQINFO *sqinfoY, char *string_name, 
					 double tfactor, int start, int len, int *seqX, int *seqY, struct ali_s *ali);
extern void          PrintIntSequence(FILE *ofp, SQINFO *sqinfo, char *name, int start, int len, int *seq);
extern void          PrintHitBanner(FILE *ofp, FILE *regressfp, char *regressionfile, 
				    SQINFO sqinfoX, SQINFO sqinfoY, int nseqs, int ones, 
				    int shuffle, int sre_shuffle, int con_shuffle, int strand);
extern void          PrintNull(FILE *ofp, struct nullmodel_s *null, double nullsc, double nullscrev);
extern void          RemoveGaps(int *s, int Lmax, int *len, int verbose);
extern void          RemoveJointGaps(SQINFO sqinfoX, int *seqX, SQINFO sqinfoY, int *seqY, int *ret_leg, int sweep, char *ss);
extern void          RevComp (int *rev, int *seq, int L);
extern void          CompNoRev (int *rev, int *seq, int L);
extern void          RevCompChar (char *rev, char *seq, int L);
extern void          ShuffleSequence(char *seq, int len, int endpos, int verbose);
extern void          ShuffleIntSequence(int *iseq, int len, int endpos, int verbose);
extern void          Shuffle2IntSequences(int *iseqX, int *iseqY, int len, int endposX, int endposY, int verbose);
extern int           QRNAShuffle(char *xs, char *ys, char *x, char *y);
extern int           QRNAIntShuffle(int *xs, int *ys, int L);
extern int           QRNAIntConservedShuffle(int *xs, int *ys, int L);
extern void          StripDegeneracy(char *seq, int *len);
extern void          StripSomeDegeneracy(char *seq, int len);
extern void          ToFold(SQINFO sqinfo, int pos_init, int pos_end, int *ret_begin, 
			    int *ret_end, int *ret_len);
extern void          WriteSeqinfo(FILE *outf, SQINFO *sqinfo, int nfile, int nseq, int nbases);
extern int           WriteSeqkn(FILE *ofp, char *seq, SQINFO *sqinfo, int *ret_ss);


/* from trace.c
 */
extern struct tracer_s      *InitTracer(void);
extern struct tracerstack_s *InitTracerstack(void);
extern struct tracer_s      *AttachTracer(struct tracer_s *parent, int emit, int type);
extern struct tracerstack_s *InitTracerstack(void);
extern struct tracer_s      *PopTracerstack(struct tracerstack_s *stack);
extern void                  FreeTracer(struct tracer_s *tr);
extern void                  DeleteTracernode(struct tracer_s *oldtr);
extern void                  PushTracerstack(struct tracerstack_s *stack, struct tracer_s *node);
extern void                  FreeTracerstack(struct tracerstack_s *stack);

/* from traceSCFG.c
 */
extern struct trace_s   *InitTrace(void);
extern struct tracekn_s *InitTracekn(void);
extern struct trace_s   *AttachTrace(struct trace_s *parent, int emitr, int emitl, int type);
extern struct tracekn_s *AttachTracekn(struct tracekn_s *parent, 
				       int emiti, int emitj, int emitk, int emitl, 
                                       int type, int node);
extern struct tracestack_s   *InitTracestack(void);
extern struct traceknstack_s *InitTraceknstack(void);
extern struct trace_s        *PopTracestack(struct tracestack_s *stack);
extern struct tracekn_s      *PopTraceknstack(struct traceknstack_s *stack);
extern void   FreeTrace(struct trace_s *tr);
extern void   FreeTracekn(struct tracekn_s *tr);
extern void   DeleteTracenode(struct trace_s *oldtr);
extern void   DeleteTraceknnode(struct tracekn_s *oldtr);
extern void   PushTracestack(struct tracestack_s *stack, struct trace_s *node);
extern void   PushTraceknstack(struct traceknstack_s *stack, struct tracekn_s *node);
extern void   FreeTracestack(struct tracestack_s *stack);
extern void   FreeTraceknstack(struct traceknstack_s *stack);

/* from tying.c:
 */
extern void    BootstrapConfidence(double **cfg, double *counts, int **enslave, int nboot, double conf,
				   double ***ret_high, double ***ret_low);
extern void    CountFreeParameters(int **enslave, int *ret_free, int *ret_nonzero, int allow_pseudoknots);
extern double *CountsPerState(double **cfg);
extern int   **EnslaveTrans(void);
extern void    TieCounts(double **cfg, int **enslave);

#endif /*FUNCSH_INCLUDED*/



