///////////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2015-2018 Edouard Griffiths, F4EXB.                             //
// Copyright (C) 2021 Jon Beniston, M7RCE                                        //
//                                                                               //
// This program is free software; you can redistribute it and/or modify          //
// it under the terms of the GNU General Public License as published by          //
// the Free Software Foundation as version 3 of the License, or                  //
// (at your option) any later version.                                           //
//                                                                               //
// This program is distributed in the hope that it will be useful,               //
// but WITHOUT ANY WARRANTY; without even the implied warranty of                //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the                  //
// GNU General Public License V3 for more details.                               //
//                                                                               //
// You should have received a copy of the GNU General Public License             //
// along with this program. If not, see <http://www.gnu.org/licenses/>.          //
///////////////////////////////////////////////////////////////////////////////////

#ifndef INCLUDE_RADIOCLOCK_H
#define INCLUDE_RADIOCLOCK_H

#include <vector>

#include <QNetworkRequest>
#include <QThread>
#include <QDateTime>

#include "dsp/basebandsamplesink.h"
#include "channel/channelapi.h"
#include "util/message.h"

#include "radioclockbaseband.h"
#include "radioclocksettings.h"

class QNetworkAccessManager;
class QNetworkReply;
class QThread;
class DeviceAPI;
class ScopeVis;

class RadioClock : public BasebandSampleSink, public ChannelAPI {
    Q_OBJECT
public:
    class MsgConfigureRadioClock : public Message {
        MESSAGE_CLASS_DECLARATION

    public:
        const RadioClockSettings& getSettings() const { return m_settings; }
        bool getForce() const { return m_force; }

        static MsgConfigureRadioClock* create(const RadioClockSettings& settings, bool force)
        {
            return new MsgConfigureRadioClock(settings, force);
        }

    private:
        RadioClockSettings m_settings;
        bool m_force;

        MsgConfigureRadioClock(const RadioClockSettings& settings, bool force) :
            Message(),
            m_settings(settings),
            m_force(force)
        { }
    };

    class MsgDateTime : public Message {
        MESSAGE_CLASS_DECLARATION

    public:

        QDateTime getDateTime() const { return m_dateTime; }
        RadioClockSettings::DST getDST() const { return m_dst; }

        static MsgDateTime* create(QDateTime dateTime, RadioClockSettings::DST dst = RadioClockSettings::DST::UNKNOWN)
        {
            return new MsgDateTime(dateTime, dst);
        }

    private:
        QDateTime m_dateTime;
        RadioClockSettings::DST m_dst;

        MsgDateTime(QDateTime dateTime, RadioClockSettings::DST dst) :
            Message(),
            m_dateTime(dateTime),
            m_dst(dst)
        {
        }
    };

    class MsgStatus : public Message {
        MESSAGE_CLASS_DECLARATION

    public:
        QString getStatus() const { return m_status; }

        static MsgStatus* create(QString status)
        {
            return new MsgStatus(status);
        }

    private:
        QString m_status;

        MsgStatus(QString status) :
            Message(),
            m_status(status)
        {
        }
    };

    RadioClock(DeviceAPI *deviceAPI);
    virtual ~RadioClock();
    virtual void destroy() { delete this; }

    using BasebandSampleSink::feed;
    virtual void feed(const SampleVector::const_iterator& begin, const SampleVector::const_iterator& end, bool po);
    virtual void start();
    virtual void stop();
    virtual bool handleMessage(const Message& cmd);

    virtual void getIdentifier(QString& id) { id = objectName(); }
    virtual const QString& getURI() const { return getName(); }
    virtual void getTitle(QString& title) { title = m_settings.m_title; }
    virtual qint64 getCenterFrequency() const { return 0; }

    virtual QByteArray serialize() const;
    virtual bool deserialize(const QByteArray& data);

    virtual int getNbSinkStreams() const { return 1; }
    virtual int getNbSourceStreams() const { return 0; }

    virtual qint64 getStreamCenterFrequency(int streamIndex, bool sinkElseSource) const
    {
        (void) streamIndex;
        (void) sinkElseSource;
        return 0;
    }

    virtual int webapiSettingsGet(
            SWGSDRangel::SWGChannelSettings& response,
            QString& errorMessage);

    virtual int webapiSettingsPutPatch(
            bool force,
            const QStringList& channelSettingsKeys,
            SWGSDRangel::SWGChannelSettings& response,
            QString& errorMessage);

    virtual int webapiReportGet(
            SWGSDRangel::SWGChannelReport& response,
            QString& errorMessage);

    static void webapiFormatChannelSettings(
            SWGSDRangel::SWGChannelSettings& response,
            const RadioClockSettings& settings);

    static void webapiUpdateChannelSettings(
            RadioClockSettings& settings,
            const QStringList& channelSettingsKeys,
            SWGSDRangel::SWGChannelSettings& response);

    ScopeVis *getScopeSink();
    double getMagSq() const { return m_basebandSink->getMagSq(); }

    void getMagSqLevels(double& avg, double& peak, int& nbSamples) {
        m_basebandSink->getMagSqLevels(avg, peak, nbSamples);
    }

    uint32_t getNumberOfDeviceStreams() const;

    static const char * const m_channelIdURI;
    static const char * const m_channelId;

private:
    DeviceAPI *m_deviceAPI;
    QThread m_thread;
    RadioClockBaseband* m_basebandSink;
    RadioClockSettings m_settings;
    int m_basebandSampleRate; //!< stored from device message used when starting baseband sink
    qint64 m_centerFrequency;
    QDateTime m_dateTime; //!< Received data and time

    QNetworkAccessManager *m_networkManager;
    QNetworkRequest m_networkRequest;

    void applySettings(const RadioClockSettings& settings, bool force = false);
    void webapiFormatChannelReport(SWGSDRangel::SWGChannelReport& response);
    void webapiReverseSendSettings(QList<QString>& channelSettingsKeys, const RadioClockSettings& settings, bool force);
    void webapiFormatChannelSettings(
        QList<QString>& channelSettingsKeys,
        SWGSDRangel::SWGChannelSettings *swgChannelSettings,
        const RadioClockSettings& settings,
        bool force
    );

private slots:
    void networkManagerFinished(QNetworkReply *reply);

};

#endif // INCLUDE_RADIOCLOCK_H
