% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter-Spat.R
\name{filter.Spat}
\alias{filter.Spat}
\alias{filter.SpatRaster}
\alias{filter.SpatVector}
\title{Subset cells/geometries of Spat* objects}
\usage{
\method{filter}{SpatRaster}(.data, ..., .preserve = FALSE, .keep_extent = TRUE)

\method{filter}{SpatVector}(.data, ..., .preserve = FALSE)
}
\arguments{
\item{.data}{A SpatRaster created with \code{\link[terra:rast]{terra::rast()}} or a SpatVector
created with \code{\link[terra:vect]{terra::vect()}}.}

\item{...}{\code{\link[dplyr:filter]{data-masking}} Expressions that return a logical
value, and are defined in terms of the layers/attributes in \code{.data}. If
multiple expressions are included, they are combined with the \code{&} operator.
Only cells/geometries for which all conditions evaluate to \code{TRUE} are kept.
See \strong{Methods}.}

\item{.preserve}{Ignored for Spat* objects.}

\item{.keep_extent}{Should the extent of the resulting SpatRaster be kept? On
\code{FALSE}, \code{\link[terra:trim]{terra::trim()}} is called so the extent of the result may be
different of the extent of the output. See also \code{\link[=drop_na.SpatRaster]{drop_na.SpatRaster()}}.}
}
\value{
A Spat* object  of the same class than \code{.data}. See \strong{Methods}.
}
\description{
The \code{filter()} function is used to subset Spat* objects, retaining all
cells/geometries that satisfy your conditions. To be retained, the
cell/geometry must produce a value of \code{TRUE} for all conditions.

\strong{It is possible to filter a SpatRaster by its geographic coordinates}.
You need to use \code{filter(.data, x > 42)}. Note that \code{x} and \code{y} are reserved
names on terra, since they refer to the geographic coordinates of the layer.

See \strong{Examples} and section \verb{About layer names} on \code{\link[=as_tibble]{as_tibble()}}.
}
\section{Methods}{


Implementation of the \strong{generic} \code{\link[dplyr:filter]{dplyr::filter()}} function.
\subsection{SpatRaster}{

Cells that do not fulfill the conditions on \code{...} are returned with value
\code{NA}. On a multi-layer SpatRaster the \code{NA} is propagated across all the
layers.

If \code{.keep_extent = TRUE} the returning SpatRaster has the same crs, extent,
resolution and hence the same number of cells than \code{.data}. If
\code{.keep_extent = FALSE} the outer \code{NA} cells are trimmed with \code{\link[terra:trim]{terra::trim()}},
so the extent and number of cells may differ. The output would present in
any case the same crs and resolution than \code{.data}.

\code{x} and \code{y} variables (i.e. the longitude and latitude of the SpatRaster)
are also available internally for filtering. See \strong{Examples}.
}

\subsection{SpatVector}{

The result is a SpatVector with all the geometries that produce a value of
\code{TRUE} for all conditions.
}
}

\examples{

library(terra)
f <- system.file("extdata/cyl_temp.tif", package = "tidyterra")

r <- rast(f) \%>\% select(tavg_04)

plot(r)


# Filter temps
r_f <- r \%>\% filter(tavg_04 > 11.5)

# Extent is kept
plot(r_f)


# Filter temps and extent
r_f2 <- r \%>\% filter(tavg_04 > 11.5, .keep_extent = FALSE)

# Extent has changed
plot(r_f2)


# Filter by geographic coordinates
r2 <- project(r, "epsg:4326")

r2 \%>\% plot()

r2 \%>\%
  filter(
    x > -4,
    x < -2,
    y > 42
  ) \%>\%
  plot()
}
\seealso{
\code{\link[dplyr:filter]{dplyr::filter()}}

Other single table verbs: 
\code{\link{arrange.SpatVector}()},
\code{\link{mutate.Spat}},
\code{\link{rename.Spat}},
\code{\link{select.Spat}},
\code{\link{slice.Spat}},
\code{\link{summarise.SpatVector}()}

Other dplyr verbs that operate on rows:
\code{\link{arrange.SpatVector}()},
\code{\link{distinct.SpatVector}()},
\code{\link{slice.Spat}}

Other dplyr methods:
\code{\link{arrange.SpatVector}()},
\code{\link{bind_cols.SpatVector}},
\code{\link{bind_rows.SpatVector}},
\code{\link{count.SpatVector}()},
\code{\link{distinct.SpatVector}()},
\code{\link{filter-joins.SpatVector}},
\code{\link{glimpse.Spat}},
\code{\link{group-by.SpatVector}},
\code{\link{mutate-joins.SpatVector}},
\code{\link{mutate.Spat}},
\code{\link{pull.Spat}},
\code{\link{relocate.Spat}},
\code{\link{rename.Spat}},
\code{\link{rowwise.SpatVector}()},
\code{\link{select.Spat}},
\code{\link{slice.Spat}},
\code{\link{summarise.SpatVector}()}
}
\concept{dplyr.methods}
\concept{dplyr.rows}
\concept{single table verbs}
