#' Get tweets data for given statuses (status IDs).
#'
#' @inheritParams lookup_users
#' @param statuses User id or screen name of target user.
#' @references <https://developer.twitter.com/en/docs/twitter-api/v1/tweets/post-and-engage/api-reference/get-statuses-lookup>
#' @examples
#'
#' if (auth_has_default()) {
#'   statuses <- c(
#'     "567053242429734913",
#'     "266031293945503744",
#'     "440322224407314432"
#'   )
#'
#'   ## lookup tweets data for given statuses
#'   tw <- lookup_tweets(statuses)
#'   tw
#' }
#' @return A tibble of tweets data.
#' @family tweets
#' @export
lookup_tweets <- function(statuses, parse = TRUE, token = NULL,
                          retryonratelimit = NULL, verbose = TRUE) {
  # Prevents problems if someone passes a full table.
  if (is.data.frame(statuses)) {
    statuses <- ids(statuses)
  }
  chunks <- unname(split(statuses, (seq_along(statuses) - 1) %/% 100))
  params_list <- lapply(chunks, function(id) {
    list(
      id = paste(id, collapse = ","),
      tweet_mode = "extended",
      include_ext_alt_text = "true"
    )
  })

  results <- TWIT_paginate_chunked(token, "/1.1/statuses/lookup", params_list,
    retryonratelimit = retryonratelimit,
    verbose = verbose
  )

  if (parse) {
    results <- tweets_with_users(results)
    results$created_at <- format_date(results$created_at)
    # Reorder to match input, skip NA if the status is not found.
    m <- match(statuses, results$id_str)
    m <- m[!is.na(m)]
    results <- results[m, ]
    attr(results, "users") <- users_data(results)[m, ]
  }
  results
}

#' @export
#' @rdname lookup_tweets
#' @usage NULL
lookup_statuses <- function(statuses, parse = TRUE, token = NULL) {
  lifecycle::deprecate_warn("1.0.0", "lookup_statuses()", "lookup_tweets()")

  lookup_tweets(statuses = statuses, parse = parse, token = token)
}

#' Get tweet information
#'
#' Look up tweets up to 100 at the same time.
#' @inheritParams tweet_retweeted_by
#' @inheritParams stream
#' @param token This endpoint accepts a OAuth2.0 authentication (can be
#' created via [rtweet_oauth2()]) or a bearer token (can be created via [rtweet_app()]).
#' @param id At least a tweet id.
#' @seealso [lookup_tweets()]
#' @references
#' One tweet: <https://developer.twitter.com/en/docs/twitter-api/tweets/lookup/api-reference/get-tweets-id>
#'
#' Multiple tweets: <https://developer.twitter.com/en/docs/twitter-api/tweets/lookup/api-reference/get-tweets>
#' @export
#' @examples
#' if (FALSE){
#'  tweet_get("567053242429734913", parse = FALSE)
#'  tweet_get(c("567053242429734913", "567053242429734913"), parse = FALSE)
#'  tweet_get(c("567053242429734913", "567053242429734913"), parse = TRUE)
#' }
tweet_get <- function(id, expansions = NULL, fields = NULL, ..., token = NULL,
                      parse = TRUE, verbose = FALSE) {
  expansions <- check_expansions(
    arg_def(expansions,
            set_expansions(user = NULL, list = NULL)),
    set_expansions(user = NULL, list = NULL))
  fields <- check_fields(arg_def(fields, set_fields(list = NULL)),
                         metrics = NULL, list = NULL)
  expansions_for_fields(expansions, fields)
  parsing(parse, expansions, fields)
  data <- c(list(expansions = expansions), fields, ...)
  data <- unlist(prepare_params(data), recursive = FALSE)
  stopifnot("Requires valid ids." = is_id(id))
  if (length(id) == 1) {
    url <- paste0("tweets/", id)
  } else if (length(id) <= 100 ) {
    data <- c(ids = paste(id, collapse = ","), data)
    url <- "tweets"
  } else {
    abort("Only 100 tweets can be processed at once.")
  }

  # Rates from the website app and user limits
  token <- check_token_v2(token, c("bearer", "pkce"))
  check_scopes_token(token, c("tweet.read", "users.read"))
  rate <- check_rate(token, 300/(60*15), 900/(60*15))
  req_archive <- endpoint_v2(token, url, rate)
  req_final <- httr2::req_url_query(req_archive, !!!data)
  p <- pagination(req_final, 1, length(ids), verbose = verbose)
  if (!parse) {
    return(p)
  }
  parse(p, expansions, fields)
}
