% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lazytensor_preprocess.R
\name{LazyTensor}
\alias{LazyTensor}
\title{Build and return a LazyTensor object.}
\usage{
LazyTensor(x, index = NA, is_complex = FALSE)
}
\arguments{
\item{x}{A matrix or a vector of numeric values, or a scalar value}

\item{index}{A text string that should be either \strong{i} or \strong{j}, or an \strong{NA}
value (the default), to specify whether if the \strong{x} variable is indexed
by \strong{i} (rows), by \strong{j} (columns), or is a fixed parameter across indices.
If \strong{x} is a matrix, \strong{index} must be \strong{i} or \strong{j}.}

\item{is_complex}{A boolean (default is FALSE). Whether if we want to create a
\code{ComplexLazyTensor} (is_complex = TRUE) or a \code{LazyTensor} (is_complex = FALSE).}
}
\value{
An object of class "LazyTensor" or "ComplexLazyTensor".
}
\description{
\code{LazyTensor}s objects are wrappers around R matrices or vectors that are used
to create symbolic formulas for the \code{KeOps} reduction operations.
}
\details{
The \code{LazyTensor()} function builds a \code{LazyTensor}, which is a
list containing the following elements:
\itemize{
\item{\strong{formula}:}{ A string defining the mathematical operation to
be computed by the \code{KeOps} routine - each variable is encoded with the
pointer address of its argument, suffixed by 'i', 'j', or 'NA', to
give it a unique identifier;}
\item{\strong{args}:}{ A vector of arguments containing a unique identifier
associated to the type of the argument:
\itemize{
\item{\strong{Vi(n)}:}{ vector indexed by \strong{i} of dim \strong{n}}
\item{\strong{Vj(n)}:}{ vector indexed by \strong{j} of dim \strong{n}}
\item{\strong{Pm(n)}:}{ fixed parameter of dim \strong{n}}
}}
\item{\strong{data}:}{ A list of R matrices which will be the inputs of the
\code{KeOps} routine;}
\item{\strong{dimres}:}{ An integer corresponding to the inner dimension of
the \code{LazyTensor}. \strong{dimres} is used when creating new
\code{LazyTensor}s that result from operations,
to keep track of the dimension.}
}

\strong{Note}

Setting the argument \code{is_complex} to \code{TRUE} will build a \code{ComplexLazyTensor},
which is also a \code{LazyTensor}. Run \code{browseVignettes("rkeops")} and see
"RKeOps LazyTensor" vignette for further details on how \code{ComplexLazyTensor}s
are build.

\strong{Note}

If \code{x} is an integer, \code{LazyTensor(x)} builds a \code{LazyTensor} whose
formula is simply \code{IntCst(x)} and contains all the necessary information;
\code{args} and \code{data} remains empty, to avoid useless storage.

\strong{Alternatives}

\itemize{
\item LazyTensor(x, "i") is equivalent to Vi(x) (see \strong{Vi()} function)
\item LazyTensor(x, "j") is equivalent to Vi(x) (see \strong{Vj()} function)
\item LazyTensor(x) is equivalent to Pm(x) (see \strong{Pm()} function)
}

Run \code{browseVignettes("rkeops")} to access the vignettes and see how to use
\code{LazyTensor}s.
}
\examples{
\dontrun{
# Data
nx <- 100
ny <- 150
x <- matrix(runif(nx * 3), nrow = nx, ncol = 3) # arbitrary R matrix representing 
                                                # 100 data points in R^3
y <- matrix(runif(ny * 3), nrow = ny, ncol = 3) # arbitrary R matrix representing 
                                                # 150 data points in R^3
s <- 0.1                                        # scale parameter

# Turn our Tensors into KeOps symbolic variables:
x_i <- LazyTensor(x, "i")   # symbolic object representing an arbitrary row of x, 
                            # indexed by the letter "i"
y_j <- LazyTensor(y, "j")   # symbolic object representing an arbitrary row of y, 
                            # indexed by the letter "j"

# Perform large-scale computations, without memory overflows:
D_ij <- sum((x_i - y_j)^2)    # symbolic matrix of pairwise squared distances, 
                              # with 100 rows and 150 columns
K_ij <- exp(- D_ij / s^2)     # symbolic matrix, 100 rows and 150 columns
res <- sum(K_ij, index = "i") # actual R matrix (in fact a row vector of 
                              # length 150 here)
                              # containing the column sums of K_ij
                              # (i.e. the sums over the "i" index, for each 
                              # "j" index)


# Example : create ComplexLazyTensor:
z <- matrix(1i^ (-6:5), nrow = 4)                     # create a complex 4x3 matrix
z_i <- LazyTensor(z, index = 'i', is_complex = TRUE)  # create a ComplexLazyTensor, 
                                                      # indexed by 'i'

}
}
\author{
Joan Glaunes, Chloe Serre-Combe, Amelie Vernay
}
