% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/variable_timesteps.R
\name{add_timesteps_columns}
\alias{add_timesteps_columns}
\alias{add_remind_timesteps_columns}
\alias{ggplot_bar_vts}
\alias{ggplot_bar_remind_vts}
\title{Bar plots with variable time steps}
\usage{
add_timesteps_columns(
  data,
  timesteps,
  periods = "period",
  gaps = 0,
  interval_shift = c(-0.5, 0.5),
  timesteps_period = "period",
  timesteps_interval = "year"
)

add_remind_timesteps_columns(data, periods = "period", gaps = 0)

ggplot_bar_vts(
  data,
  timesteps,
  mapping = aes(x = !!sym("period"), y = !!sym("value"), fill = !!sym("variable")),
  gaps = 0.1,
  position_fill = FALSE,
  interval_shift = c(-0.5, 0.5),
  timesteps_period = "period",
  timesteps_interval = "year"
)

ggplot_bar_remind_vts(
  data,
  mapping = aes(x = !!sym("period"), y = !!sym("value"), fill = !!sym("variable")),
  gaps = 0.1,
  position_fill = FALSE
)
}
\arguments{
\item{data}{A data frame.}

\item{timesteps}{A data frame like \code{\link{remind_timesteps}} with columns
\code{period}, \code{year}, and \code{weight}, where \code{weight} determines which share of
\code{year} belongs to \code{period}.}

\item{periods}{The column holding the period information in \code{data} (either a
string or an object).  Defaults to \code{'period'}.}

\item{gaps}{Gaps between bars as a fraction of the smallest bar width.
Defaults to 0.1 (e.g. 0.1 * 5 years = 0.5 years).}

\item{interval_shift}{numeric of length 2.  Shifts added to the interval fix
point to obtain the beginning and end of time interval.
If the interval for period 1 should be \verb{[0.5, 1.5]}, \code{interval_shift}
should be set to \code{c(-0.5, 0.5)} (default).  If the interval for period 1
should be \verb{[0, 1]}, \code{interval_shift} should be set to \code{c(-1, 0)}.}

\item{timesteps_period}{character string giving the column name of the
\code{period} in the \code{timesteps} data frame.  Defaults to \code{'period'}.}

\item{timesteps_interval}{character string giving the column name of the time
interval in the \code{timesteps} data frame.  Defaults to \code{'year'}.}

\item{mapping}{\code{\link[ggplot2:aes]{aes()}} mapping with aesthetics \code{x}, \code{y}, and
optionally \code{fill}.}

\item{position_fill}{If \code{TRUE}, stacks bars and standardises each stack to
have constant height.}
}
\value{
\code{add_timesteps_columns()} and \code{add_remind_timesteps_columns()} return
a data frame.
\code{ggplot_bar_vts()} and \code{ggplot_bar_remind_vts()} return a
\code{\link[ggplot2:ggplot]{ggplot()}}-like object.
}
\description{
Utility functions for plotting stacked bars with variable widths for
displaying time-series data with variable time steps (like REMIND data).
}
\details{
\code{add_timesteps_columns()} calculates the x-axis position and width of bars
based on the information in \code{timesteps} and joins it to \code{data}.
\code{add_remind_timesteps_columns()} uses the \code{\link{remind_timesteps}} data frame.
\code{ggplot_bar_vts()} produces a bar plot with bars positioned according to
\code{timesteps}.  \code{ggplot_bar_remind_vts()} uses the \code{\link{remind_timesteps}} data
frame.
}
\examples{
require(tidyverse)

# some example data
(data <- quitte_example_data \%>\%
    filter(first(scenario) == scenario,
           last(region) == region,
           first(variable) == variable))

# adding individual timesteps
add_timesteps_columns(data, remind_timesteps)

# adding remind timesteps with gaps
add_remind_timesteps_columns(data, gaps = 0.1)

# plotting individual timesteps without gaps
ggplot_bar_vts(data, remind_timesteps, gaps = 0)

# plotting remind timegaps, using further ggplot2 functions
ggplot_bar_remind_vts(
    data = quitte_example_data \%>\%
        filter(scenario \%in\% levels(quitte_example_data$scenario)[1:3],
               last(region) == region,
               grepl('PE\\\\|', variable),
               2100 >= period)) +
    scale_fill_manual(
        values = mip::plotstyle(grep('^PE\\\\|',
                                     levels(quitte_example_data$variable),
                                     value = TRUE))) +
    facet_wrap(~ scenario)

# another data set with a different time column
data2 <- data.frame(variable = c('Wind', 'Solar', 'Wind', 'Solar'),
    tau = c(1,1,2,2),
    value = 1:4)

# some timesteps dataframe with hourly data
timesteps <- data.frame(tau = c(rep(1,2),rep(2,4)),
                            hour = 1:6,
                            weight = 1)

# plotting with different timesteps than periods and years
ggplot_bar_vts(data2, timesteps,
              mapping = aes(tau, value, group = variable, fill = variable),
              timesteps_period = 'tau',
              timesteps_interval = 'hour',
              interval_shift = c(-1,0))
}
\author{
Michaja Pehl
}
