% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/interpolate_missing_periods.R
\name{interpolate_missing_periods}
\alias{interpolate_missing_periods}
\alias{interpolate_missing_periods_}
\title{Interpolate missing periods}
\usage{
interpolate_missing_periods(
  data,
  ...,
  value = "value",
  expand.values = FALSE,
  method = "linear",
  combinations = "nesting"
)

interpolate_missing_periods_(
  data,
  periods,
  value = "value",
  expand.values = FALSE,
  method = "linear",
  combinations = "nesting"
)
}
\arguments{
\item{data}{A data frame or a quitte object.}

\item{...}{A name-value pair of periods to fill. If unnamed, defaults to
\code{'period'}. If empty (but possibly named) uses only periods present in
\code{data}.}

\item{value}{Name of the column to fill, defaults to \code{'value'}.}

\item{expand.values}{If \code{FALSE} (the default), values are not expanded
beyond the range of available data.  If \code{TRUE} values at the closest
extreme is used for linear interpolation.  Results for spline
interpolation are possibly nonsensical.}

\item{method}{Specifies the interpolation method. Either \code{'linear'} for
linear interpolation or \code{'spline'}, \code{'spline_fmm'}, or \code{'spline_natural'}
for spline interpolation.  \code{'spline'} is an alias for \code{'spline_fmm'}.
See \code{\link[=spline]{spline()}} for details.}

\item{combinations}{Specifies the method with which other columns are
treated.  They are either preserved as-is (\code{'nesting'}, the default), or
are expanded to all unique combinations (\code{'crossing'}).
See \code{\link[tidyr:expand]{tidyr::expand()}} for details.}

\item{periods}{A named list of periods to fill.}
}
\value{
A data frame or a quitte object, the same as \code{data}.
}
\description{
Adds missing periods to data frame and interpolates missing values linearly
or using splines from adjacent existing ones. Values for periods
smaller/bigger than the existing ones can be filled with the values for the
first/last available period in the case of linear interpolation.
}
\examples{
require(dplyr)

# generate some test data with explicit (A-y-2025) and implicit (B-x-2030)
# missing values
(data <- tibble(
    group  = rep(c('A', 'B'), c(8, 4)),
    item   = c(rep('x', 4), rep('y', 4), rep('x', 4)),
    period = rep(c(2015, 2025, 2030, 2035), 3),
    value  = c(2, 4, 5, 6, 20, NA, 50, 60, NA, 400, 500, NA)))

# fill values for already existing periods
interpolate_missing_periods(data)

# fill values for existing periods, with full combinations of other columns
interpolate_missing_periods(data, combinations = 'crossing')

# add additional periods and fill values
interpolate_missing_periods(data, period = seq(2010, 2035, 5))

# also fill values outside the original data range
interpolate_missing_periods(data, seq(2010, 2035, 5), expand.values = TRUE)

# works on data frames with different column names
(data <- data \%>\%
        rename(year = period, coeff = value))

interpolate_missing_periods(data, year, value = 'coeff')

# works on quitte objects too
(quitte <- data \%>\%
        rename(model = group, scenario = item, period = year, value = coeff) \%>\%
        mutate(variable = 'Var 1', unit = 'u1') \%>\%
        as.quitte())

interpolate_missing_periods(quitte, expand.values = TRUE)

# and works with POSIXct periods
(quitte <- quitte \%>\%
        mutate(period = ISOyear(period)))

interpolate_missing_periods(quitte, period = ISOyear(seq(2010, 2035, 5)))

# standard evaluation example
interpolate_missing_periods_(data, periods = list(year = seq(2010, 2035, 5)),
                             value = 'coeff', expand.values = TRUE)
}
\author{
Michaja Pehl
}
