% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scan1blup.R
\name{scan1blup}
\alias{scan1blup}
\title{Calculate BLUPs of QTL effects in scan along one chromosome}
\usage{
scan1blup(
  genoprobs,
  pheno,
  kinship = NULL,
  addcovar = NULL,
  nullcovar = NULL,
  contrasts = NULL,
  se = FALSE,
  reml = TRUE,
  tol = 0.000000000001,
  cores = 1,
  quiet = TRUE
)
}
\arguments{
\item{genoprobs}{Genotype probabilities as calculated by
\code{\link[=calc_genoprob]{calc_genoprob()}}.}

\item{pheno}{A numeric vector of phenotype values (just one phenotype, not a matrix of them)}

\item{kinship}{Optional kinship matrix, or a list of kinship matrices (one
per chromosome), in order to use the LOCO (leave one chromosome
out) method.}

\item{addcovar}{An optional numeric matrix of additive covariates.}

\item{nullcovar}{An optional numeric matrix of additional additive
covariates that are used under the null hypothesis (of no QTL) but
not under the alternative (with a QTL). This is needed for the X
chromosome, where we might need sex as a additive covariate under
the null hypothesis, but we wouldn't want to include it under the
alternative as it would be collinear with the QTL effects. Only
used if \code{kinship} is provided but \code{hsq} is not, to get
estimate of residual heritability.}

\item{contrasts}{An optional numeric matrix of genotype contrasts, size
genotypes x genotypes. For an intercross, you might use
\code{cbind(mu=c(1,0,0), a=c(-1, 0, 1), d=c(0, 1, 0))} to get
mean, additive effect, and dominance effect. The default is the
identity matrix.}

\item{se}{If TRUE, also calculate the standard errors.}

\item{reml}{If \code{reml=TRUE}, use
REML to estimate variance components; otherwise maximum likelihood.}

\item{tol}{Tolerance value for convergence of linear mixed model fit.}

\item{cores}{Number of CPU cores to use, for parallel calculations.
(If \code{0}, use \code{\link[parallel:detectCores]{parallel::detectCores()}}.)
Alternatively, this can be links to a set of cluster sockets, as
produced by \code{\link[parallel:makeCluster]{parallel::makeCluster()}}.}

\item{quiet}{If FALSE, print message about number of cores used when multi-core.}
}
\value{
An object of class \code{"scan1coef"}: a matrix of estimated regression coefficients, of dimension
positions x number of effects. The number of effects is
\code{n_genotypes + n_addcovar + (n_genotypes-1)*n_intcovar}.
May also contain the following attributes:
\itemize{
\item \code{SE} - Present if \code{se=TRUE}: a matrix of estimated
standard errors, of same dimension as \code{coef}.
\item \code{sample_size} - Vector of sample sizes used for each
phenotype
}
}
\description{
Calculate BLUPs of QTL effects in scan along one chromosome, with a
single-QTL model treating the QTL effects as random, with possible
allowance for covariates and for a residual polygenic effect.
}
\details{
For each of the inputs, the row names are used as
individual identifiers, to align individuals.

If \code{kinship} is provided, the linear mixed model accounts for
a residual polygenic effect, with a the polygenic variance
estimated under the null hypothesis of no (major) QTL, and then
taken as fixed as known in the scan to estimate QTL effects.

If \code{contrasts} is provided, the genotype probability matrix,
\eqn{P}, is post-multiplied by the contrasts matrix, \eqn{A}, prior
to fitting the model. So we use \eqn{P \cdot A}{P A} as the \eqn{X}
matrix in the model. One might view the rows of
\ifelse{html}{\out{<em>A</em><sup>-1</sup>}}{\eqn{A^{-1}}}
as the set of contrasts, as the estimated effects are the estimated
genotype effects pre-multiplied by
\ifelse{html}{\out{<em>A</em><sup>-1</sup>}}{\eqn{A^{-1}}}.
}
\examples{
# read data
iron <- read_cross2(system.file("extdata", "iron.zip", package="qtl2"))
\dontshow{iron <- iron[,c(7,19)] # reduce to chr 7 and 19}

# insert pseudomarkers into map
map <- insert_pseudomarkers(iron$gmap, step=1)

# calculate genotype probabilities
probs <- calc_genoprob(iron, map, error_prob=0.002)
\dontshow{probs[["7"]] <- probs[["7"]][,,1:5] # reduce to very small number}

# convert to allele probabilities
aprobs <- genoprob_to_alleleprob(probs)

# grab phenotypes and covariates; ensure that covariates have names attribute
pheno <- iron$pheno[,1]
covar <- match(iron$covar$sex, c("f", "m")) # make numeric
names(covar) <- rownames(iron$covar)

# calculate BLUPs of coefficients for chromosome 7
blup <- scan1blup(aprobs[,"7"], pheno, addcovar=covar)

# leave-one-chromosome-out kinship matrix for chr 7
kinship7 <- calc_kinship(probs, "loco")[["7"]]

# calculate BLUPs of coefficients for chromosome 7, adjusting for residual polygenic effect
blup_pg <- scan1blup(aprobs[,"7"], pheno, kinship7, addcovar=covar)

}
\references{
Haley CS, Knott SA (1992) A simple
regression method for mapping quantitative trait loci in line
crosses using flanking markers.  Heredity 69:315--324.

Kang HM, Zaitlen NA, Wade CM, Kirby A, Heckerman D, Daly MJ, Eskin
E (2008) Efficient control of population structure in model
organism association mapping. Genetics 178:1709--1723.

Robinson GK (1991) That BLUP is a good thing: The estimation of
random effects. Statist Sci 6:15--32.
}
