##' Contacts model
##'
##' A panel model for dynamic variation in sexual contacts, with
##' data from Vittinghof et al (1999). The model was developed by
##' Romero-Severson et al (2015) and discussed by \Breto et al (2020).
##'
##' @param params parameter vector.
##' @return
##' A \code{panelPomp} object.
##'
##' @author Edward L. Ionides
##'
##' @examples
##' \donttest{
##' contacts()
##' }
##' @export
##'
##' @references \breto2020
##'
##' \romeroseverson2015
##'
##' \vittinghoff1999
##' @family panelPomp examples
contacts <- function(params=c(mu_X=1.75,sigma_X=2.67,mu_D=3.81,
  sigma_D=4.42,mu_R=0.04,sigma_R=0,alpha=0.90)){

  contact_data <- read.table(
  text = '"y1"	"y2"	"y3"	"y4"	"individual"
  5	0	0	0	1
  2	2	31	4	2
  9	2	0	0	3
  12	8	10	6	4
  0	10	4	1	5
  25	23	15	2	6
  12	6	0	0	7
  10	2	0	4	8
  31	58	24	48	9
  0	0	0	0	10
  32	42	9	1	11
  5	4	12	2	12
  0	1	0	1	13
  2	0	3	0	14
  1	3	0	0	15
  92	40	7	5	16
  3	0	1	10	17
  2	4	3	3	18
  5	1	3	2	19
  18	1	22	2	20
  0	12	12	11	21
  5	9	15	19	22
  20	10	0	0	23
  1	0	0	6	24
  9	8	5	6	25
  17	7	10	10	26
  0	1	0	3	27
  0	3	13	5	28
  15	16	12	8	29
  9	0	1	1	30
  0	1	0	0	31
  15	5	9	0	32
  0	25	0	0	33
  75	59	45	30	34
  160	34	40	145	35
  2	0	0	4	36
  2	0	26	3	37
  4	0	0	0	38
  4	11	2	0	39
  2	1	4	1	40
  15	2	0	0	41
  150	30	49	22	42
  0	0	0	0	43
  2	10	3	2	44
  2	0	0	0	45
  58	23	6	4	46
  0	0	1	0	47
  12	15	14	8	48
  0	0	0	0	49
  3	5	6	15	50
  15	7	0	4	51
  1	3	0	3	52
  0	2	3	6	53
  0	0	0	0	54
  13	7	13	10	55
  2	2	2	0	56
  0	0	0	0	57
  33	17	2	0	58
  0	2	15	8	59
  0	0	0	0	60
  4	7	4	4	61
  5	1	2	0	62
  1	2	14	13	63
  14	13	1	8	64
  12	3	0	0	65
  0	6	2	0	66
  4	0	0	0	67
  0	0	0	0	68
  6	6	3	3	69
  60	75	10	13	70
  2	4	2	2	71
  113	23	24	12	72
  8	127	99	45	73
  3	1	4	13	74
  60	31	10	35	75
  1	1	20	7	76
  3	1	0	0	77
  0	0	0	0	78
  4	3	0	3	79
  0	0	0	0	80
  1	0	0	2	81
  23	10	1	2	82
  0	0	2	0	83
  3	0	5	5	84
  5	0	0	0	85
  0	0	0	0	86
  15	2	6	7	87
  14	1	0	0	88
  0	78	52	4	89
  4	7	1	1	90
  4	4	0	2	91
  3	4	1	2	92
  5	40	1	52	93
  15	0	6	0	94
  0	2	6	0	95
  23	10	4	3	96
  7	25	51	3	97
  16	30	5	1	98
  10	2	0	0	99
  0	1	0	4	100
  10	22	11	3	101
  2	0	0	0	102
  0	0	0	0	103
  55	13	0	11	104
  0	3	8	3	105
  27	2	1	13	106
  10	4	5	14	107
  16	3	1	1	108
  3	3	1	11	109
  4	2	25	0	110
  21	15	16	32	111
  11	0	0	0	112
  10	9	22	11	113
  6	10	2	72	114
  6	3	6	4	115
  3	1	5	2	116
  8	8	7	12	117
  1	1	0	0	118
  0	0	0	0	119
  236	101	26	32	120
  0	2	1	18	121
  0	0	0	0	122
  91	0	0	0	123
  10	13	4	2	124
  0	1	0	0	125
  2	7	3	4	126
  0	0	2	0	127
  27	35	15	16	128
  0	0	0	0	129
  7	10	3	0	130
  0	0	0	0	131
  0	0	0	0	132
  3	2	2	3	133
  1	0	0	8	134
  1	0	0	0	135
  5	103	8	2	136
  1	6	2	2	137
  17	30	98	40	138
  9	0	0	0	139
  0	0	0	0	140
  10	6	4	19	141
  103	44	45	80	142
  50	3	0	0	143
  0	1	7	0	144
  4	1	5	4	145
  10	15	6	6	146
  33	46	105	96	147
  0	0	0	0	148
  0	0	0	0	149
  4	0	0	0	150
  0	0	0	0	151
  0	0	0	0	152
  0	0	0	0	153
  6	6	8	12	154
  20	3	21	5	155
  0	0	0	0	156
  0	10	0	2	157
  1	0	0	1	158
  1	12	6	0	159
  6	1	1	1	160
  0	0	0	0	161
  31	31	15	5	162
  72	43	56	61	163
  0	0	0	0	164
  0	3	0	0	165
  10	37	25	2	166
  20	17	19	2	167
  5	6	6	11	168
  12	0	0	0	169
  22	58	5	4	170
  1	23	14	0	171
  25	2	9	4	172
  0	0	0	1	173
  0	0	0	0	174
  4	4	5	2	175
  2	4	0	8	176
  4	4	0	0	177
  15	26	30	26	178
  2	12	2	0	179
  22	3	9	58	180
  0	0	0	0	181
  4	5	3	1	182
  14	50	29	0	183
  6	3	1	1	184
  2	2	1	1	185
  4	0	5	1	186
  37	57	20	9	187
  31	1	0	0	188
  2	6	11	3	189
  54	12	3	4	190
  32	17	6	2	191
  3	0	1	2	192
  13	0	3	9	193
  3	30	2	1	194
  0	0	0	0	195
  0	0	0	0	196
  69	0	0	0	197
  1	1	1	0	198
  36	30	22	80	199
  0	0	0	0	200
  2	2	60	84	201
  44	0	0	2	202
  0	0	0	1	203
  0	11	31	4	204
  2	1	0	4	205
  5	11	18	17	206
  33	45	6	15	207
  34	18	60	54	208
  0	0	0	0	209
  0	0	0	0	210
  17	20	45	6	211
  0	0	11	6	212
  6	0	0	0	213
  0	11	2	5	214
  4	6	2	3	215
  9	11	10	14	216
  9	0	0	0	217
  1	4	1	1	218
  5	5	1	2	219
  1	5	3	2	220
  4	0	70	6	221
  0	0	0	0	222
  9	0	0	9	223
  0	0	0	0	224
  5	2	3	2	225
  1	2	22	14	226
  0	2	3	3	227
  5	4	3	2	228
  3	1	0	0	229
  1	0	0	0	230
  0	1	4	2	231
  12	8	18	3	232
  0	6	18	10	233
  4	0	12	4	234
  46	80	71	33	235
  5	0	1	1	236
  2	10	0	3	237
  3	7	6	9	238
  6	0	0	0	239
  7	1	14	2	240
  30	9	4	8	241
  16	5	54	76	242
  12	5	0	0	243
  6	3	2	2	244
  3	2	3	1	245
  7	14	13	3	246
  50	134	4	51	247
  0	0	0	0	248
  19	0	0	0	249
  0	0	0	0	250
  15	10	3	1	251
  2	0	0	1	252
  5	0	0	1	253
  13	38	17	23	254
  0	7	1	0	255
  8	1	3	1	256
  33	5	29	41	257
  10	1	2	4	258
  18	54	30	20	259
  0	0	0	0	260
  43	43	3	0	261
  2	0	0	0	262
  0	0	0	0	263
  7	2	0	7	264
  0	2	0	0	265
  1	0	0	0	266
  3	4	5	8	267
  3	4	3	1	268
  33	26	16	17	269
  6	2	16	4	270
  0	3	4	5	271
  10	10	3	9	272
  8	10	8	7	273
  0	0	0	1	274
  4	9	0	0	275
  27	26	21	5	276
  12	2	2	0	277
  1	12	4	3	278
  16	2	0	0	279
  27	0	0	0	280
  15	14	10	9	281
  2	2	5	5	282
  0	0	7	0	283
  15	8	19	1	284
  9	2	2	0	285
  10	30	12	15	286
  0	0	0	0	287
  0	3	3	12	288
  47	47	17	2	289
  0	8	0	0	290
  2	0	0	0	291
  2	0	0	26	292
  64	21	10	5	293
  12	100	30	1	294
  0	0	0	0	295
  35	18	13	10	296
  3	9	6	10	297
  0	0	0	0	298
  0	4	0	0	299
  24	30	12	10	300
  0	0	0	0	301
  3	0	0	0	302
  4	9	4	27	303
  7	11	1	14	304
  1	1	0	0	305
  1	1	0	2	306
  41	6	17	27	307
  47	0	0	0	308
  0	1	4	7	309
  23	31	11	4	310
  0	1	0	1	311
  0	0	0	0	312
  3	1	2	2	313
  5	11	3	16	314
  3	1	0	0	315
  3	2	5	0	316
  0	0	0	0	317
  17	0	0	0	318
  9	3	1	0	319
  5	5	8	9	320
  1	0	3	0	321
  1	0	0	0	322
  14	8	20	18	323
  0	7	3	10	324
  5	3	8	25	325
  2	2	0	3	326
  2	0	1	2	327
  2	4	0	3	328
  4	3	5	5	329
  47	0	4	7	330
  3	2	4	4	331
  1	0	0	0	332
  29	1	46	4	333
  55	48	53	146	334
  0	18	15	4	335
  3	1	0	0	336
  6	0	1	0	337
  6	1	10	8	338
  0	1	3	53	339
  3	7	9	2	340
  0	0	0	0	341
  4	4	83	51	342
  0	0	1	4	343
  35	33	31	0	344
  11	3	3	8	345
  94	0	15	19	346
  19	17	20	8	347
  2	4	0	0	348
  12	27	15	13	349
  24	3	8	25	350
  161	0	40	7	351
  0	0	1	0	352
  3	6	17	0	353
  28	19	0	16	354
  1	1	10	0	355
  15	2	0	0	356
  34	76	138	45	357
  0	0	0	0	358
  0	0	2	11	359
  0	0	0	0	360
  0	0	0	0	361
  15	10	3	10	362
  1	4	1	3	363
  0	4	0	0	364
  10	46	3	4	365
  1	52	1	0	366
  7	1	0	4	367
  3	11	0	0	368
  6	10	0	0	369
  17	27	34	32	370
  2	9	7	2	371
  18	2	0	4	372
  0	0	0	1	373
  72	104	0	0	374
  0	0	3	0	375
  39	22	15	33	376
  37	29	0	0	377
  0	0	0	0	378
  15	3	8	17	379
  0	0	0	1	380
  2	6	10	7	381
  27	16	15	19	382
  2	7	1	19	383
  0	10	1	17	384
  6	6	6	13	385
  1	0	1	0	386
  11	0	0	0	387
  8	12	10	18	388
  0	0	0	0	389
  0	0	0	0	390
  15	16	1	4	391
  0	1	1	0	392
  2	0	0	0	393
  1	4	1	1	394
  5	2	4	4	395
  1	0	1	3	396
  0	0	0	0	397
  7	13	8	13	398
  0	0	0	60	399
  8	17	28	44	400
  2	11	1	12	401
  2	0	0	0	402
  15	14	75	25	403
  14	94	2	9	404
  2	5	2	3	405
  20	0	0	0	406
  4	0	5	3	407
  1	6	2	1	408
  0	0	0	0	409
  0	0	0	0	410
  0	0	0	0	411
  38	18	7	0	412
  27	1	0	0	413
  2	7	27	6	414
  61	0	0	0	415
  6	1	7	2	416
  0	0	0	0	417
  5	11	17	9	418
  0	0	0	0	419
  6	1	0	2	420
  0	0	0	4	421
  50	21	1	16	422
  0	0	0	0	423
  0	0	4	0	424
  0	5	2	0	425
  5	8	13	16	426
  0	0	0	0	427
  8	16	19	5	428
  25	0	0	0	429
  0	2	5	3	430
  4	4	10	1	431
  2	20	21	17	432
  0	2	0	2	433
  24	25	52	14	434
  35	20	21	37	435
  3	0	0	0	436
  0	0	0	9	437
  3	18	5	8	438
  13	7	0	0	439
  0	12	12	0	440
  9	5	3	3	441
  105	105	53	50	442
  5	13	8	6	443
  0	0	0	0	444
  0	0	225	0	445
  0	5	2	3	446
  11	12	11	8	447
  12	1	0	0	448
  1	0	0	0	449
  0	1	51	15	450
  5	1	9	8	451
  14	10	8	16	452
  8	23	70	0	453
  4	13	18	0	454
  2	0	1	0	455
  0	0	4	0	456
  10	3	0	0	457
  3	5	3	1	458
  30	65	28	75	459
  0	0	0	0	460
  2	0	2	0	461
  1	4	16	13	462
  0	1	0	1	463
  2	0	0	0	464
  16	16	7	0	465
  2	0	6	0	466
  0	0	2	0	467
  15	3	25	27	468
  0	0	0	0	469
  125	41	35	67	470
  8	1	3	1	471
  26	15	123	32	472
  0	0	0	0	473
  4	0	0	0	474
  12	6	0	0	475
  4	3	2	0	476
  12	1	3	2	477
  8	14	18	9	478
  3	5	4	35	479
  15	20	26	11	480
  0	0	0	1	481
  1	0	3	6	482
  0	0	0	0	483
  4	4	4	1	484
  1	0	0	0	485
  39	76	33	2	486
  0	0	30	4	487
  40	0	1	19	488
  5	29	4	10	489
  0	0	0	2	490
  10	5	7	8	491
  4	2	2	0	492
  1	6	2	1	493
  18	4	12	22	494
  2	1	0	0	495
  0	0	0	28	496
  5	0	0	15	497
  4	1	8	15	498
  0	0	0	0	499
  0	0	0	0	500
  3	0	0	0	501
  0	0	0	0	502
  2	3	1	1	503
  15	1	11	0	504
  2	2	4	3	505
  20	10	24	0	506
  11	35	5	2	507
  0	0	0	0	508
  7	0	0	9	509
  10	13	5	14	510
  0	0	0	0	511
  1	2	2	4	512
  2	0	0	3	513
  6	1	13	0	514
  1	0	2	1	515
  9	7	3	14	516
  0	0	0	0	517
  3	4	3	5	518
  0	0	0	0	519
  4	15	14	42	520
  0	0	0	0	521
  4	0	0	0	522
  0	1	0	0	523
  41	0	3	0	524
  0	4	0	0	525
  13	3	1	2	526
  19	1	8	6	527
  10	21	14	15	528
  0	10	57	0	529
  0	3	1	0	530
  6	1	1	9	531
  0	6	1	1	532
  9	0	4	7	533
  0	0	1	0	534
  1	4	0	0	535
  4	6	3	6	536
  4	0	6	20	537
  3	0	1	9	538
  8	4	16	18	539
  2	5	4	2	540
  21	31	24	37	541
  3	2	1	1	542
  0	0	0	0	543
  26	23	20	12	544
  3	5	9	7	545
  0	25	20	3	546
  2	5	5	7	547
  1	52	52	22	548
  15	4	8	5	549
  1	2	4	2	550
  0	0	0	0	551
  7	20	21	16	552
  0	0	0	0	553
  4	0	0	0	554
  34	6	7	2	555
  7	5	0	37	556
  106	20	4	0	557
  0	1	2	0	558
  2	0	0	2	559
  1	0	1	13	560
  37	0	10	9	561
  3	2	0	0	562
  27	2	0	2	563
  2	0	0	0	564
  3	9	3	1	565
  3	1	0	0	566
  40	10	7	45	567
  0	0	0	0	568
  24	25	17	25	569
  4	0	5	144	570
  17	14	6	1	571
  6	52	1	2	572
  14	5	26	4	573
  3	1	1	1	574
  0	0	1	2	575
  5	0	0	0	576
  2	0	0	0	577
  3	5	4	4	578
  2	1	4	9	579
  3	0	0	0	580
  4	1	3	0	581
  7	3	3	3	582
  0	1	0	0	583
  10	20	14	6	584
  1	1	3	2	585
  4	5	1	2	586
  1	1	2	2	587
  52	25	0	1	588
  0	0	0	0	589
  17	23	1	3	590
  4	2	8	1	591
  5	4	7	14	592
  0	0	0	0	593
  1	7	10	6	594
  0	0	0	0	595
  0	0	18	30	596
  1	0	0	0	597
  16	58	3	15	598
  2	1	3	1	599
  0	4	1	1	600
  3	12	9	20	601
  38	0	0	75	602
  5	1	6	7	603
  0	0	0	0	604
  13	4	4	1	605
  4	0	0	0	606
  0	22	3	41	607
  2	0	87	23	608
  1	0	5	1	609
  1	4	0	0	610
  3	1	0	0	611
  24	0	0	1	612
  27	40	25	26	613
  0	1	5	1	614
  4	7	3	20	615
  14	2	6	6	616
  2	5	0	1	617
  1	0	5	8	618
  1	16	15	33	619
  0	0	0	0	620
  41	3	0	0	621
  1	0	1	1	622
  16	5	0	0	623
  0	1	1	8	624
  6	1	5	2	625
  3	3	3	3	626
  0	0	0	62	627
  1	7	4	8	628
  15	2	2	2	629
  46	73	2	1	630
  6	24	0	0	631
  3	0	5	4	632
  3	5	1	1	633
  15	1	15	112	634
  21	6	14	4	635
  86	0	0	0	636
  0	7	7	7	637
  3	1	1	0	638
  0	0	0	0	639
  2	2	18	0	640
  60	39	13	6	641
  3	14	11	28	642
  1	2	0	1	643
  8	3	6	2	644
  2	9	10	0	645
  3	53	75	20	646
  14	1	0	0	647
  13	12	9	15	648
  50	20	42	50	649
  3	11	18	6	650
  7	4	2	1	651
  5	11	5	3	652
  25	6	19	30	653
  2	9	80	0	654
  0	0	0	0	655
  1	0	0	0	656
  32	21	6	5	657
  0	115	379	6	658
  0	0	0	0	659
  0	0	0	0	660
  0	0	0	0	661
  1	3	3	2	662
  1	2	0	4	663
  2	0	0	0	664
  1	3	0	0	665
  10	0	8	8	666
  0	0	0	0	667
  10	8	9	2	668
  20	4	0	0	669
  0	1	1	0	670
  2	65	5	1	671
  12	17	21	27	672
  40	0	0	0	673
  0	23	3	2	674
  0	19	0	4	675
  0	1	3	0	676
  4	0	0	0	677
  1	1	1	0	678
  0	0	0	0	679
  9	10	7	1	680
  3	0	6	4	681
  4	6	14	10	682
  3	1	2	35	683
  5	3	13	47	684
  0	0	0	0	685
  0	8	2	0	686
  6	0	0	0	687
  2	0	0	0	688
  18	3	6	0	689
  0	1	0	0	690
  8	10	17	5	691
  4	7	7	2	692
  24	75	0	0	693
  0	0	0	0	694
  1	0	3	42	695
  64	60	0	0	696
  38	0	1	0	697
  0	1	7	2	698
  2	2	0	0	699
  0	0	0	1	700
  40	52	52	3	701
  1	0	0	0	702
  0	0	0	2	703
  7	4	2	4	704
  30	10	13	11	705
  0	1	0	0	706
  1	0	0	0	707
  45	37	0	13	708
  0	0	0	0	709
  25	76	60	40	710
  0	6	53	2	711
  4	2	2	1	712
  0	0	0	0	713
  54	23	23	7	714
  1	1	1	2	715
  34	0	0	0	716
  2	11	0	2	717
  1	10	7	1	718
  2	0	0	1	719
  15	8	5	4	720
  0	0	0	0	721
  37	13	0	28	722
  12	0	36	1	723
  6	3	3	2	724
  2	3	0	2	725
  0	0	0	0	726
  0	0	0	0	727
  2	0	2	0	728
  1	0	0	0	729
  0	0	0	0	730
  0	0	0	0	731
  10	0	2	4	732
  62	0	0	0	733
  3	0	0	0	734
  0	0	0	0	735
  31	27	19	8	736
  6	50	0	0	737
  5	5	3	0	738
  26	12	16	6	739
  0	0	0	0	740
  8	3	3	5	741
  10	2	10	17	742
  11	9	9	16	743
  0	0	0	0	744
  0	0	0	0	745
  4	4	1	4	746
  4	6	3	12	747
  8	8	11	11	748
  13	17	7	6	749
  5	2	0	0	750
  5	0	0	0	751
  4	0	0	0	752
  3	0	0	0	753
  0	0	0	7	754
  3	1	2	3	755
  7	3	14	22	756
  6	6	13	2	757
  2	1	2	2	758
  1	0	0	1	759
  54	40	13	0	760
  2	0	0	1	761
  76	119	84	12	762
  20	24	4	24	763
  0	0	0	0	764
  0	0	0	0	765
  0	0	0	0	766
  0	8	0	12	767
  0	0	0	4	768
  7	0	0	0	769
  0	4	0	0	770
  4	1	0	0	771
  2	2	0	0	772
  61	4	21	13	773
  24	16	25	18	774
  1	7	3	4	775
  268	3	15	5	776
  48	0	0	0	777
  0	0	0	0	778
  19	13	56	40	779
  12	10	18	3	780
  0	0	0	0	781
  0	2	0	0	782
  17	4	27	25	783
  1	0	0	4	784
  12	11	6	11	785
  0	11	0	7	786
  33	29	28	24	787
  15	0	19	13	788
  3	2	0	0	789
  0	12	39	24	790
  3	8	0	1	791
  6	3	14	7	792
  0	0	1	1	793
  53	24	34	105	794
  1	5	2	0	795
  0	5	4	8	796
  10	6	6	3	797
  51	10	5	3	798
  6	10	3	22	799
  0	0	0	0	800
  4	31	13	1	801
  4	6	5	8	802
  0	0	0	0	803
  13	18	12	18	804
  0	4	10	7	805
  25	16	6	1	806
  0	0	12	0	807
  3	0	1	4	808
  0	0	0	0	809
  10	0	0	0	810
  1	0	0	0	811
  277	16	10	48	812
  43	12	4	5	813
  0	0	0	0	814
  4	2	4	0	815
  13	5	7	19	816
  1	2	0	10	817
  0	11	0	12	818
  2	7	3	6	819
  3	3	0	0	820
  17	1	1	1	821
  0	0	0	0	822
  6	7	5	59	823
  67	16	28	2	824
  0	0	0	0	825
  0	2	0	2	826
  0	0	25	12	827
  10	6	3	6	828
  4	0	4	0	829
  214	0	2	0	830
  3	2	0	1	831
  3	2	4	1	832
  1	0	0	1	833
  7	4	3	7	834
  4	4	42	18	835
  8	5	10	37	836
  2	0	25	0	837
  3	3	1	8	838
  0	0	0	3	839
  120	221	5	98	840
  0	2	20	2	841
  10	43	80	32	842
  8	5	0	0	843
  2	1	6	6	844
  8	1	3	3	845
  1	0	0	0	846
  208	0	0	0	847
  13	8	5	23	848
  1	15	3	0	849
  19	6	7	4	850
  3	1	2	2	851
  0	1	0	0	852
  1	2	0	0	853
  3	0	4	5	854
  9	38	17	12	855
  0	0	2	2	856
  8	13	33	22	857
  0	0	4	6	858
  1	0	1	0	859
  0	0	1	3	860
  10	5	1	0	861
  13	0	0	0	862
  7	2	0	2	863
  15	121	4	0	864
  5	14	14	2	865
  6	0	0	0	866
  3	3	1	1	867
  1	2	1	0	868
  10	1	12	5	869
  20	27	5	20	870
  4	10	4	2	871
  0	0	1	6	872
  82	20	0	0	873
  0	0	0	7	874
  26	2	0	6	875
  2	0	0	0	876
  16	40	20	12	877
  0	0	3	3	878
  0	2	0	0	879
  7	76	86	103	880
  6	3	3	1	881
  0	0	0	0	882',
  header = TRUE
  )


  dmeas <- Csnippet("lik = dnbinom(y, D, D/(D+C), give_log);")
  rmeas <- Csnippet("y = rnbinom(D, D/(D+C));")

  rinit <- Csnippet("
    double tol=0.000001;
    D = (sigma_D < tol || mu_D < tol) ? mu_D :
      rgamma(pow(mu_D/sigma_D,2), pow(sigma_D,2)/mu_D);
    if(D < tol) {D = tol;}
    R = (sigma_R < tol || mu_R < tol) ? mu_R :
      rgamma(pow(mu_R/sigma_R, 2), pow(sigma_R, 2)/mu_R);
    X = (sigma_X < tol || mu_X < tol) ? mu_X :
      rgamma(pow(mu_X/sigma_X, 2), pow(sigma_X, 2)/mu_X);
    Z = (R < tol) ? 1/tol : rexp(1/R);
    C = 0;
  ")

  rproc <- Csnippet("
    double Zcum, tol=0.000001;
    C = 0;
    Zcum = Z;
    while(Zcum < 6){ // time in months within a 6 month observation interval
      C  += Z * X;
      Z = (R < tol) ? 1/tol : rexp(1/R);
      X = (sigma_X < tol || mu_X < tol) ? mu_X :
        rgamma(pow(mu_X/sigma_X, 2), pow(sigma_X, 2)/mu_X);
      Zcum += Z;
    }
    C += (6 - (Zcum - Z)) * X;
    C *= pow(alpha, (int)t % 4);
    Z = Zcum - 6;
  ")

  # check for existing 'cdir' (to make 'testthat' package work)
  cdir <- if (exists("cdir",inherits=FALSE)) cdir else NULL

  template <- pomp(data=data.frame(t=1:4,y=NA),
    times="t",
    t0=0,
    rprocess=discrete_time(step.fun=rproc,delta.t=1),
    rmeasure=rmeas,
    dmeasure=dmeas,
    params=params,
    paramnames=c("mu_X","sigma_X","mu_D","sigma_D","mu_R","sigma_R","alpha"),
    statenames=c("X","D","R","C","Z"),
    obsnames="y",
    partrans=parameter_trans(
      log=c("mu_X","sigma_X","mu_D","sigma_D"),
      logit="alpha"
    ),
    rinit=rinit,
    cdir=cdir
  )

  ## build list of pomps
  U <- nrow(contact_data)
  poList <- setNames(vector(mode="list",length=U),
    nm=paste0("unit",1:U))
  for (u in seq_len(U)) {
    poList[[u]] <- template
    data_u <- pomp(
      data.frame(t=time(template),y=as.numeric(contact_data[u,time(template)])),
      times="t",
      t0=timezero(template)
      )
    poList[[u]]@data <- data_u@data
  }

  ## Construct panelPomp
  panelPomp(object=poList,shared=coef(template))
}
