#    nfer - a system for inferring abstractions of event streams
#    Copyright (C) 2021  Sean Kauffman
#
#    This file is part of nfer.
#    nfer is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.

#' Apply a loaded nfer specification to a dataframe of events.
#'
#' This function obtains the interval abstractions generated by applying the 
#' nfer specification to the events.
#'
#' Event traces are passed as dataframes with at least two columns.
#' The first two columns contain event names and timestamps, respectively.
#' Names should be strings.  Timestamps should be integers or strings, ideally, but may be numeric.
#' Subsequent columns contain data, where the column name is the data key.
#' The column value will be the data value for each event, where NA means no value is present.
#' 
#' Example dataframe events: \preformatted{
#' | name | timestamp | x |    y |
#' -------------------------------
#' | foo  | 123       | 2 | NA   |
#' | bar  | 987       | 3 | TRUE |
#' }
#'
#' The result of the function is also a dataframe, but this contains intervals.
#' The difference is that it has an extra timestamp column.
#' Column two is now the start time of the interval, and column 3 is the end time.
#' Name is still column one and columns 4+ are still data.
#' 
#' Dataframe interval output: \preformatted{
#' | name | start | end | z  |
#' ---------------------------
#' | far  | 123   | 987 | NA |
#' | baf  | 654   | 987 | 3  |
#' }
#'
#' @param handle The loaded nfer specification 
#'               (using \code{nfer::load}, or \code{nfer::learn})
#' @param events The dataframe of events to abstract using nfer.
#' @return A dataframe containing intervals.
#' @examples
#' ops <- nfer::load(system.file("extdata", "ops.nfer", package = "nfer"))
#' events <- nfer::read(system.file("extdata", "ops.events", package = "nfer"))
#' intervals <- nfer::apply(ops, events)
#' @export
apply <- function(handle, events) {
  # transform to char from factor
  f <- sapply(events, is.factor)
  events[f] <- lapply(events[f], as.character)

  # input validation
  stopifnot(length(handle) == 2)
  stopifnot(sapply(handle, class)[1] == "integer")
  stopifnot(length(events) >= 2)
  stopifnot(sapply(events, class)[1] == "character")
  # this is weird, but we don't want to lose precision by converting to floating point numbers
  stopifnot(sapply(events, class)[2] == "integer" || sapply(events, class)[2] == "character" || sapply(events, class)[2] == "numeric")

  .Call(apply_, handle, events)
}
