% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{stdt}
\alias{stdt}
\alias{dt0}
\alias{pt0}
\alias{rt0}
\alias{qt0}
\title{Standardized Student t Distribution}
\usage{
dt0(x, df = 10, log = FALSE, grad_x = FALSE, grad_df = FALSE)

pt0(x, df = 10, log = FALSE, grad_x = FALSE, grad_df = FALSE, n = 10L)

rt0(n = 1L, df = 10)

qt0(x = 1L, df = 10)
}
\arguments{
\item{x}{numeric vector of quantiles.}

\item{df}{positive real value representing the number of degrees of freedom.
Since this function deals with standardized Student distribution, argument
\code{df} should be greater than \code{2} because otherwise variance is
undefined.}

\item{log}{logical; if \code{TRUE} then probabilities (or densities) p 
are given as log(p) and derivatives will be given respect to log(p).}

\item{grad_x}{logical; if \code{TRUE} then function returns a derivative
respect to \code{x}.}

\item{grad_df}{logical; if \code{TRUE} then function returns a derivative
respect to \code{df}.}

\item{n}{positive integer. If \code{rt0} function is used then this 
argument represents the number of random draws. Otherwise \code{n} states 
for the number of iterations used to calculate the derivatives associated 
with \code{pt0} function via \code{\link[mnorm]{pbetaDiff}} function.}
}
\value{
Function \code{rt0} returns a numeric vector of random numbers. 
Function \code{qt0} returns a numeric vector of quantiles. 
Functions \code{pt0} and \code{dt0} return a list which may contain the 
following elements:
\itemize{
\item \code{prob} - numeric vector of probabilities calculated for each 
element of \code{x}. Exclusively for \code{pt0} function.
\item \code{den} - numeric vector of densities calculated for each 
each element of \code{x}. Exclusively for \code{dt0} function.
\item \code{grad_x} - numeric vector of derivatives respect to \code{p} for
each element of \code{x}. 
This element appears only if input argument \code{grad_x} is \code{TRUE}.
\item \code{grad_df} - numeric vector of derivatives respect to \code{q} for
each element of \code{x}.
This element appears only if input argument \code{grad_df} is \code{TRUE}.
}
}
\description{
These functions calculate and differentiate a cumulative 
distribution function and density function of the standardized 
(to zero mean and unit variance) Student distribution. Quantile function 
and random numbers generator are also provided.
}
\details{
Standardized (to zero mean and unit variance) Student distribution 
has the following density and cumulative distribution functions:
\deqn{
f(x) = \frac{\Gamma\left(\frac{v + 1}{2}\right)}{
\sqrt{(v - 2)\pi}\Gamma\left(\frac{v}{2}\right)}
\left(1 + \frac{x^2}{v - 2}\right)^{-\frac{v+1}{2}},
}
\deqn{
F(x) = 
\begin{cases}
1 - \frac{1}{2}I(\frac{v - 2}{x^2 + v - 2}, 
\frac{v}{2}, \frac{1}{2})\text{, if }x \geq 0\\
\frac{1}{2}I(\frac{v - 2}{x^2 + v - 2}, 
\frac{v}{2}, \frac{1}{2})\text{, if }x < 0
\end{cases},
}
where \eqn{v > 2} is the number of degrees of freedom \code{df} and 
\eqn{I(.)} is a cumulative distribution function of beta distribution 
which is calculated by \code{\link[stats]{pbeta}} function.
}
\examples{
# Simple examples
pt0(x = 0.3, df = 10, log = FALSE, grad_x = TRUE, grad_df = TRUE)
dt0(x = 0.3, df = 10, log = FALSE, grad_x = TRUE, grad_df = TRUE)
qt0(x = 0.3, df = 10)

# Compare analytical and numeric derivatives
delta <- 1e-6
x <- c(-2, -1, 0, 1, 2)
df <- 5

# For probabilities
out <- pt0(x, df = df, grad_x = TRUE, grad_df = TRUE)
p0 <- out$prob
  # grad_x
p1 <- pt0(x + delta, df = df)$prob
data.frame(numeric = (p1 - p0) / delta, analytical = out$grad_x)
  # grad_df
p1 <- pt0(x, df = df + delta)$prob
data.frame(numeric = (p1 - p0) / delta, analytical = out$grad_df)

# For densities
out <- dt0(x, df = df, grad_x = TRUE, grad_df = TRUE)
p0 <- out$den
  # grad_x
p1 <- dt0(x + delta, df = df)$den
data.frame(numeric = (p1 - p0) / delta, analytical = out$grad_x)
  # grad_df
p1 <- dt0(x, df = df + delta)$den
data.frame(numeric = (p1 - p0) / delta, analytical = out$grad_df)

}
