% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/block.plsda.R
\name{block.plsda}
\alias{block.plsda}
\title{N-integration with Projection to Latent Structures models (PLS) with
Discriminant Analysis}
\usage{
block.plsda(
  X,
  Y,
  indY,
  ncomp = 2,
  design,
  scheme,
  scale = TRUE,
  init = "svd",
  tol = 1e-06,
  max.iter = 100,
  near.zero.var = FALSE,
  all.outputs = TRUE,
  verbose.call = FALSE
)
}
\arguments{
\item{X}{A named list of data sets (called 'blocks') measured on the same 
samples. Data in the list should be arranged in matrices, samples x variables, 
with samples order matching in all data sets.}

\item{Y}{a factor or a class vector for the discrete outcome.}

\item{indY}{To supply if \code{Y} is missing, indicates the position of 
the matrix response in the list \code{X}.}

\item{ncomp}{the number of components to include in the model. Default to 2.
Applies to all blocks.}

\item{design}{numeric matrix of size (number of blocks in X) x (number of
blocks in X) with values between 0 and 1. Each value indicates the strenght
of the relationship to be modelled between two blocks; a value of 0
indicates no relationship, 1 is the maximum value. Alternatively, one of
c('null', 'full') indicating a disconnected or fully connected design,
respecively, or a numeric between 0 and 1 which will designate all
off-diagonal elements of a fully connected design (see examples in
\code{block.splsda}). If \code{Y} is provided instead of \code{indY}, the
\code{design} matrix is changed to include relationships to \code{Y}.}

\item{scheme}{Character, one of 'horst', 'factorial' or 'centroid'. Default =
\code{'horst'}, see reference.}

\item{scale}{Logical. If scale = TRUE, each block is standardized to zero
means and unit variances (default: TRUE)}

\item{init}{Mode of initialization use in the algorithm, either by Singular
Value Decomposition of the product of each block of X with Y ('svd') or each
block independently ('svd.single'). Default = \code{svd.single}}

\item{tol}{Positive numeric used as convergence criteria/tolerance during the
iterative process. Default to \code{1e-06}.}

\item{max.iter}{Integer, the maximum number of iterations. Default to  100.}

\item{near.zero.var}{Logical, see the internal \code{\link{nearZeroVar}}
function (should be set to TRUE in particular for data with many zero
values). Setting this argument to FALSE (when appropriate) will speed up the
computations. Default value is FALSE.}

\item{all.outputs}{Logical. Computation can be faster when some specific
(and non-essential) outputs are not calculated. Default = \code{TRUE}.}

\item{verbose.call}{Logical (Default=FALSE), if set to TRUE then the \code{$call}
component of the returned object will contain the variable values for all 
parameters. Note that this may cause large memory usage.}
}
\value{
\code{block.plsda} returns an object of class
\code{"block.plsda","block.pls"}, a list that contains the following
components:

\item{X}{the centered and standardized original predictor matrix.}
\item{indY}{the position of the outcome Y in the output list X.}
\item{ncomp}{the number of components included in the model for each block.}
\item{mode}{the algorithm used to fit the model.} \item{variates}{list
containing the variates of each block of X.} \item{loadings}{list containing
the estimated loadings for the variates.} \item{names}{list containing the
names to be used for individuals and variables.} \item{nzv}{list containing
the zero- or near-zero predictors information.} \item{iter}{Number of
iterations of the algorithm for each component}
\item{prop_expl_var}{Percentage of explained variance for each
component and each block}
\item{call}{if \code{verbose.call = FALSE}, then just the function call is returned.
If \code{verbose.call = TRUE} then all the inputted values are accessable via
this component}
}
\description{
Integration of multiple data sets measured on the same samples or
observations to classify a discrete outcome, ie. N-integration with
Discriminant Analysis. The method is partly based on Generalised Canonical
Correlation Analysis.
}
\details{
\code{block.plsda} function fits a horizontal integration PLS-DA model with
a specified number of components per block). A factor indicating the
discrete outcome needs to be provided, either by \code{Y} or by its position
\code{indY} in the list of blocks \code{X}.

\code{X} can contain missing values. Missing values are handled by being
disregarded during the cross product computations in the algorithm
\code{block.pls} without having to delete rows with missing data.
Alternatively, missing data can be imputed prior using the
\code{\link{impute.nipals}} function.

The type of algorithm to use is specified with the \code{mode} argument.
Four PLS algorithms are available: PLS regression \code{("regression")}, PLS
canonical analysis \code{("canonical")}, redundancy analysis
\code{("invariant")} and the classical PLS algorithm \code{("classic")} (see
References and \code{?pls} for more details).

Note that our method is partly based on Generalised Canonical Correlation
Analysis and differs from the MB-PLS approaches proposed by Kowalski et al.,
1989, J Chemom 3(1) and Westerhuis et al., 1998, J Chemom, 12(5).
}
\examples{
data(nutrimouse)
data = list(gene = nutrimouse$gene, lipid = nutrimouse$lipid, Y = nutrimouse$diet)
# with this design, all blocks are connected
design = matrix(c(0,1,1,1,0,1,1,1,0), ncol = 3, nrow = 3,
byrow = TRUE, dimnames = list(names(data), names(data)))

res = block.plsda(X = data, indY = 3) # indY indicates where the outcome Y is in the list X
plotIndiv(res, ind.names = FALSE, legend = TRUE)
plotVar(res)

\dontrun{
# when Y is provided
res2 = block.plsda(list(gene = nutrimouse$gene, lipid = nutrimouse$lipid),
Y = nutrimouse$diet, ncomp = 2)
plotIndiv(res2)
plotVar(res2)
}
}
\references{
On PLSDA:

Barker M and Rayens W (2003). Partial least squares for discrimination.
\emph{Journal of Chemometrics} \bold{17}(3), 166-173. Perez-Enciso, M. and
Tenenhaus, M. (2003). Prediction of clinical outcome with microarray data: a
partial least squares discriminant analysis (PLS-DA) approach. \emph{Human
Genetics} \bold{112}, 581-592. Nguyen, D. V. and Rocke, D. M. (2002). Tumor
classification by partial least squares using microarray gene expression
data. \emph{Bioinformatics} \bold{18}, 39-50.

On multiple integration with PLS-DA: Gunther O., Shin H., Ng R. T. ,
McMaster W. R., McManus B. M. , Keown P. A. , Tebbutt S.J. , Lê Cao K-A. ,
(2014) Novel multivariate methods for integration of genomics and proteomics
data: Applications in a kidney transplant rejection study, OMICS: A journal
of integrative biology, 18(11), 682-95.

On multiple integration with sPLS-DA and 4 data blocks:

Singh A., Gautier B., Shannon C., Vacher M., Rohart F., Tebbutt S. and Lê
Cao K.A. (2016). DIABLO: multi omics integration for biomarker discovery.
BioRxiv available here:
\url{http://biorxiv.org/content/early/2016/08/03/067611}

mixOmics article:

Rohart F, Gautier B, Singh A, Lê Cao K-A. mixOmics: an R package for 'omics
feature selection and multiple data integration. PLoS Comput Biol 13(11):
e1005752
}
\seealso{
\code{\link{plotIndiv}}, \code{\link{plotArrow}},
\code{\link{plotLoadings}}, \code{\link{plotVar}}, \code{\link{predict}},
\code{\link{perf}}, \code{\link{selectVar}}, \code{\link{block.pls}},
\code{\link{block.splsda}} and http://www.mixOmics.org for more details.
}
\author{
Florian Rohart, Benoit Gautier, Kim-Anh Lê Cao, Al J Abadi
}
\keyword{multivariate}
\keyword{regression}
