% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_marss2.R
\name{estimate_marss2}
\alias{estimate_marss2}
\title{Internal function: MARSS parameter estimation using TMB}
\usage{
estimate_marss2(
  MLEobj,
  method = c("TMB", "nlminb_TMB", "BFGS_TMB"),
  opt.control = NULL,
  ...
)
}
\arguments{
\item{MLEobj}{A properly formatted MARSS model as output by \code{\link[=MARSS]{MARSS()}}}

\item{method}{Normally passed in as MLEobj$method, but allows user to pass in a new method if they want to use MLEobj with another method. Allowed values are "TMB", "nlminb.TMB", "BFGS.TMB".}

\item{opt.control}{Normally this is passed in as MLEobj$control, but if the MLEobj was set up using a different method, then you will need to set the opt.control options. See details.}

\item{...}{not used}
}
\value{
A list with the objective and optimization objects.
\itemize{
\item \code{obj} is the raw output from the \code{\link[TMB:MakeADFun]{TMB::MakeADFun()}} call.
\item \code{op} is the raw output from the optimization call (optim or nlminb). Note that the function is minimizing the negative log-likelihood so the sign will be opposite of the log-likelihood reported by MARSS()
\item \code{opt.control} is the controls sent to the optimization function.
\item \code{method} method used for optimization
}
}
\description{
This model is in the general "marss" vectorized form. In \code{\link[=estimate_marss2]{estimate_marss2()}}, I use the approach in \code{\link[MARSS:MARSSoptim]{MARSS::MARSSoptim()}} where I just use the \code{\link[=chol]{chol()}} of these matrices. Technically there are 2 equal solutions since the diagonals appear as the square so -a and a are the same. But I have not observed that this affects the LL of optim() but it definitely seems to slow things down. In \code{\link[=estimate_marss]{estimate_marss()}}, the diagonals and offdiagonals of R and Q are split apart like Tim Cline did. I had some problems with the splitting approach for some models with Q unconstrained, though now it seems fixed.
}
\details{
Minimal error checking is done in this function.
Normal calling function is \code{\link[MARSS:MARSS]{MARSS::MARSS()}} with \code{method="TMB"}.

The main thing this does is
\itemize{
\item collapse the 3D fixed and free matrices into 2D
\item separate out the diag and offdiag parameter elements of R and Q
}

Restrictions
\itemize{
\item V0 fixed (not estimated)
\item Q and R cannot be time-varying (at the moment)
}

\code{opt.control} is what is passed to the control argument in \code{\link[=nlminb]{nlminb()}} or \code{\link[=optim]{optim()}}. If you use \code{MARSS(x, method="TMB")}, this will be set to appropriate defaults which you can see via \code{MLEobj$control}. But if you call \code{estimate_marss()} with a MLEobj from a call such as \code{MARSS(x, method="kem")} (so not a TMB method), you will need to set \code{opt.control} if you want values different from the base defaults for those functions. Note as a shortcut for \code{nlminb()}, you can set both \code{eval.max}, \code{iter.max} to the same value with \code{opt.control=list(maxit=1000)}. Note, if you pass in \code{opt.control}, this will replace all values currently in \code{MLEobj$control} that are associated with the optimizer function.

The defaults set in \code{\link[MARSS:MARSS]{MARSS::MARSS()}} are
\itemize{
\item \code{nlminb}: \code{eval.max = 5000}, \code{iter.max = 5000} and \code{trace = 0}.
\item \code{optim}: \code{maxit = 5000} and \code{trace = 0}
}

All other controls for the optimization function are left at NULL.
}
\examples{
library(MARSS)
data(lakeWAplankton, package = "MARSS")
phytoplankton <- c("Cryptomonas", "Diatoms", "Greens", "Unicells", "Other.algae")
dat <- as.data.frame(lakeWAplanktonTrans) |>
  subset(Year >= 1980 & Year <= 1989) |>
  subset(select=phytoplankton) |>
  t() |>
  MARSS::zscore()

# set-up the model
mod <- MARSS(dat, model=list(m=3, tinitx=1), form="dfa", fit=FALSE, silent=TRUE)
# fit
fit <- estimate_marss(mod)

}
\seealso{
\code{\link[MARSS:MARSSoptim]{MARSS::MARSSoptim()}}, \code{\link[MARSS:MARSSkem]{MARSS::MARSSkem()}}
}
\author{
Eli Holmes.
}
