% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_lpjml.R
\name{run_lpjml}
\alias{run_lpjml}
\title{Run LPJmL model}
\usage{
run_lpjml(
  x,
  model_path = ".",
  sim_path = NULL,
  parallel_cores = 1,
  write_stdout = FALSE,
  raise_error = TRUE,
  output_path = NULL
)
}
\arguments{
\item{x}{A \link[tibble]{tibble} with at least one column named \code{"sim_name"}.
Each simulation gets a separate row. Optional run parameters are \code{"order"}
and \code{"dependency"} which are used for subsequent simulations (see details).
\code{\link[=write_config]{write_config()}} returns a tibble in the required format.
OR provide a character string (vector) with the file name of one or
multiple generated configuration file(s).}

\item{model_path}{Character string providing the path to LPJmL
(equal to \code{LPJROOT} environment variable). Defaults to "."}

\item{sim_path}{Character string defining path where all simulation data are
written, including output, restart and configuration files. If \code{NULL},
\code{model_path} is used. See also \link{write_config}}

\item{parallel_cores}{Integer defining the number of available CPU
cores/nodes for parallelization. Defaults to \code{1} (no parallelization).
Please note that parallelization is only supported for SLURM jobs and not
for interactive runs.}

\item{write_stdout}{Logical. If \code{TRUE}, \code{stdout} as well as \code{stderr} files
are written. If \code{FALSE} (default), these are printed instead. Within a
SLURM job \code{write_stdout} is automatically set to \code{TRUE}.}

\item{raise_error}{Logical. Whether to raise an error if sub-process has
non-zero exit status. Defaults to \code{TRUE}.}

\item{output_path}{Argument is deprecated as of version 1.0; use sim_path
instead.}
}
\value{
See \code{x}, extended by columns \code{"type"}, \code{"job_id"} and \code{"status"}.
}
\description{
Runs LPJmL using \code{"config_*.json"} files written by
\code{\link[=write_config]{write_config()}}. \code{write_config()} returns a tibble
that can be used as an input (see \code{x}). It contains the details to run single
or multiple (dependent/subsequent) model runs.
}
\details{
A \link[tibble]{tibble} for \code{x} that has been generated by
\code{\link[=write_config]{write_config()}} and can look like the following examples can
supplied:\tabular{l}{
   \strong{sim_name} \cr
   scen1_spinup \cr
   scen2_transient \cr
}


To perform subsequent or rather dependent runs the optional run parameter
\code{"dependency"} needs to be provided within the initial
\link[tibble]{tibble} supplied as \code{param} to \code{\link[=write_config]{write_config()}}.\tabular{lrl}{
   \strong{sim_name} \tab \strong{order} \tab \strong{dependency} \cr
   scen1_spinup \tab 1 \tab NA \cr
   scen2_transient \tab 2 \tab scen1 _spinup \cr
}


As a shortcut it is also possible to provide the config file
\code{"config_*.json"} as a character string or multiple config files as a
character string vector directly as the \code{x} argument to \code{run_lpjml}. \cr
Also be aware that the order of the supplied config files is important
(e.g. make sure the spin-up run is run before the transient one).
}
\examples{

\dontrun{
library(tibble)

model_path <- "./LPJmL_internal"
sim_path <-"./my_runs"

# Basic usage
my_params1 <- tibble(
  sim_name = c("scen1", "scen2"),
  startgrid = c(27410, 27410),
  river_routing = c(FALSE, FALSE),
  random_seed = c(42, 404),
  `pftpar[[1]]$name` = c("first_tree", NA),
  `param$k_temp` = c(NA, 0.03),
  new_phenology = c(TRUE, FALSE)
)

config_details1 <- write_config(my_params1, model_path, sim_path)

 run_details1 <- run_lpjml(
  x = config_details1,
  model_path = model_path,
  sim_path = sim_path
)

run_details1
#   sim_name      job_id   status
#   <chr>           <int>  <chr>
# 1 scen1              NA  run
# 2 scen2              NA  run


# With run parameters dependency and order being set (also less other
#   parameters than in previous example)
my_params2 <- tibble(
  sim_name = c("scen1", "scen2"),
  startgrid = c(27410, 27410),
  river_routing = c(FALSE, FALSE),
  random_seed = c(42, 404),
  dependency = c(NA, "scen1_spinup")
)

config_details2 <- write_config(my_params2, model_path, sim_path)

run_details2 <- run_lpjml(config_details2, model_path, sim_path)

run_details2
#   sim_name        order dependency   type       job_id   status
#   <chr>           <dbl> <chr>        <chr>      <chr>    <chr>
# 1 scen1_spinup        1 NA           simulation NA       run
# 2 scen1_transient     2 scen1_spinup simulation NA       run


# Same but by using the pipe operator
library(magrittr)

run_details2 <- tibble(
  sim_name = c("scen1_spinup", "scen1_transient"),
  random_seed = as.integer(c(1, 42)),
  dependency = c(NA, "scen1_spinup")
) \%>\%
  write_config(model_path, sim_path) \%>\%
  run_lpjml(model_path, sim_path)


# Shortcut approaches
run_details3 <- run_lpjml(
  x = "./config_scen1_transient.json",
  model_path = model_path,
  sim_path = sim_path
)

run_details4 <- run_lpjml(
  c("./config_scen1_spinup.json", "./config_scen1_transient.json"),
  model_path,
  sim_path
)

}

}
