
# jagstargets <img src='man/figures/logo.png' align="right" height="139"/>

[![JOSS](https://joss.theoj.org/papers/759f48d9ae7bc57e318e2d0ecc00569e/status.svg)](https://joss.theoj.org/papers/10.21105/joss.03877)
[![ropensci](https://badges.ropensci.org/425_status.svg)](https://github.com/ropensci/software-review/issues/425)
[![DOI](https://zenodo.org/badge/321076424.svg)](https://zenodo.org/badge/latestdoi/321076424)
[![R
Targetopia](https://img.shields.io/badge/R_Targetopia-member-blue?style=flat&labelColor=gray)](https://wlandau.github.io/targetopia/)
[![cran](https://www.r-pkg.org/badges/version/jagstargets)](https://cran.r-project.org/package=jagstargets)
[![status](https://www.repostatus.org/badges/latest/active.svg)](https://www.repostatus.org/#active)
[![check](https://github.com/ropensci/jagstargets/workflows/check/badge.svg)](https://github.com/ropensci/jagstargets/actions?query=workflow%3Acheck)
[![codecov](https://codecov.io/gh/ropensci/jagstargets/branch/main/graph/badge.svg?token=3T5DlLwUVl)](https://app.codecov.io/gh/ropensci/gittargets)
[![lint](https://github.com/ropensci/jagstargets/workflows/lint/badge.svg)](https://github.com/ropensci/jagstargets/actions?query=workflow%3Alint)

Bayesian data analysis usually incurs long runtimes and cumbersome
custom code, and the process of prototyping and deploying custom
[JAGS](https://mcmc-jags.sourceforge.io) models can become a daunting
software engineering challenge. To ease this burden, the `jagstargets` R
package creates [JAGS](https://mcmc-jags.sourceforge.io) pipelines that
are concise, efficient, scalable, and tailored to the needs of Bayesian
statisticians. Leveraging
[`targets`](https://docs.ropensci.org/targets/), `jagstargets` pipelines
automatically parallelize the computation and skip expensive steps when
the results are already up to date. Minimal custom user-side code is
required, and there is no need to manually configure branching, so
`jagstargets` is easier to use than
[`targets`](https://docs.ropensci.org/targets/) and
[`R2jags`](https://CRAN.R-project.org/package=R2jags) directly.

## Prerequisites

1.  The [prerequisites of the `targets` R
    package](https://docs.ropensci.org/targets/#prerequisites).
2.  Basic familiarity with
    [`targets`](https://docs.ropensci.org/targets/): watch minutes 7
    through 40 of [this video](https://youtu.be/Gqn7Xn4d5NI?t=439), then
    read [this
    chapter](https://books.ropensci.org/targets/walkthrough.html) of the
    [user manual](https://books.ropensci.org/targets/).
3.  Familiarity with Bayesian Statistics and
    [JAGS](https://mcmc-jags.sourceforge.io/). Prior knowledge of
    [`rjags`](https://cran.r-project.org/package=rjags) or
    [`R2jags`](https://cran.r-project.org/package=R2jags) helps.

## How to get started

Read the `jagstargets` [introductory
vignette](https://docs.ropensci.org/jagstargets/articles/introduction.html),
and then use <https://docs.ropensci.org/jagstargets/> as a reference
while constructing your own workflows. If you need to analyze large
collections of simulated datasets, please consult the [simulation
vignette](https://docs.ropensci.org/jagstargets/articles/simulation.html).

## Installation

`jagstargets` requires the user to install
[JAGS](https://mcmc-jags.sourceforge.io/),
[`rjags`](https://CRAN.R-project.org/package=rjags), and
[`R2jags`](https://CRAN.R-project.org/package=R2jags) beforehand. You
can install JAGS from <https://mcmc-jags.sourceforge.io/>, and you can
install the rest from CRAN.

``` r
install.packages(c("rjags", "R2jags"))
```

Then, install the latest release from CRAN.

``` r
install.packages("jagstargets")
```

Alternatively, install the GitHub development version to access the
latest features and patches.

``` r
install.packages("remotes")
remotes::install_github("ropensci/jagstargets")
```

## Usage

Begin with one or more models: for example, the simple regression model
below with response variable $y$ and covariate $x$.

<center>
<img src="./man/figures/model.gif">
</center>

Next, write a JAGS model file for each model like the `model.jags` file
below.

``` jags
model {
  for (i in 1:n) {
    y[i] ~ dnorm(x[i] * beta, 1)
  }
  beta ~ dnorm(0, 1)
}
```

To begin a reproducible analysis pipeline with this model, write a
[`_targets.R` file](https://books.ropensci.org/targets/walkthrough.html)
that loads your packages, defines a function to generate JAGS data, and
lists a pipeline of targets. The target list can call target factories
like
[`tar_jags()`](https://docs.ropensci.org/jagstargets/reference/tar_jags.html)
as well as ordinary targets with
[`tar_target()`](https://docs.ropensci.org/targets/reference/tar_target.html).
The following minimal example is simple enough to contain entirely
within the `_targets.R` file, but for larger projects, you may wish to
store functions in separate files as in the
[`targets-stan`](https://github.com/wlandau/targets-stan) example.

``` r
# _targets.R
library(targets)
library(jagstargets)

generate_data <- function() {
  true_beta <- stats::rnorm(n = 1, mean = 0, sd = 1)
  x <- seq(from = -1, to = 1, length.out = n)
  y <- stats::rnorm(n, x * true_beta, 1)
  out <- list(n = n, x = x, y = y, true_beta = true_beta)
}

list(
  tar_jags(
    example,
    jags_files = "model.jags", # You provide this file.
    parameters.to.save = "beta",
    data = generate_data()
  )
)
```

Run
[`tar_visnetwork()`](https://docs.ropensci.org/targets/reference/tar_visnetwork.html)
to check `_targets.R` for correctness, then call
[`tar_make()`](https://docs.ropensci.org/targets/reference/tar_make.html)
to run the pipeline. Access the results using
[`tar_read()`](https://docs.ropensci.org/targets/reference/tar_read.html),
e.g. `tar_read(tar_read(example_summary_x)`. Visit the [introductory
vignette](https://docs.ropensci.org/jagstargets/articles/introduction.html)
to read more about this example.

## How the package works

`jagstargets` supports specialized [target
factories](https://ropensci.org/blog/2021/02/03/targets/#target-factories)
that create ensembles of [target
objects](https://docs.ropensci.org/targets/reference/tar_target.html)
for [`R2jags`](https://CRAN.R-project.org/package=R2jags) workflows.
These [target
factories](https://ropensci.org/blog/2021/02/03/targets/#target-factories)
abstract away the details of
[`targets`](https://docs.ropensci.org/targets/) and
[`R2jags`](https://CRAN.R-project.org/package=R2jags) and make both
packages easier to use. For details, please read the [introductory
vignette](https://docs.ropensci.org/jagstargets/articles/introduction.html).

## Help

Please read the `targets` help guide at
<https://books.ropensci.org/targets/help.html> to learn how to ask for
help.

If you have trouble using `jagstargets`, you can ask for help in the
[GitHub discussions
forum](https://github.com/ropensci/jagstargets/discussions/categories/help).
Because the purpose of `jagstargets` is to combine
[`targets`](https://docs.ropensci.org/targets/) and
[`R2jags`](https://CRAN.R-project.org/package=R2jags), your issue may
have something to do with one of the latter two packages, a [dependency
of
`targets`](https://github.com/ropensci/targets/blob/4e3ef2a6c986f558a25e544416f480fc01236b6b/DESCRIPTION#L49-L88),
or [`R2jags`](https://CRAN.R-project.org/package=R2jags) itself. When
you troubleshoot, peel back as many layers as possible to isolate the
problem. For example, if the issue comes from
[`R2jags`](https://CRAN.R-project.org/package=R2jags), create a
[reproducible example](https://reprex.tidyverse.org) that directly
invokes [`R2jags`](https://CRAN.R-project.org/package=R2jags) without
invoking `jagstargets`. The GitHub discussion and issue forums of those
packages are great resources.

## Participation

Development is a community effort, and we welcome discussion and
contribution. By participating in this project, you agree to abide by
the [code of conduct](https://ropensci.org/code-of-conduct/) and the
[contributing
guide](https://github.com/ropensci/jagstargets/blob/main/CONTRIBUTING.md).

## Citation

``` r
citation("jagstargets")
#> 
#> To cite jagstargets in publications use:
#> 
#>   Landau, W. M., (2021). The jagstargets R package: a reproducible
#>   workflow framework for Bayesian data analysis with JAGS. Journal of
#>   Open Source Software, 6(68), 3877,
#>   https://doi.org/10.21105/joss.03877
#> 
#> A BibTeX entry for LaTeX users is
#> 
#>   @Article{,
#>     title = {The jagstargets R package: a reproducible workflow framework for Bayesian data analysis with JAGS},
#>     author = {William Michael Landau},
#>     journal = {Journal of Open Source Software},
#>     year = {2021},
#>     volume = {6},
#>     number = {68},
#>     pages = {3877},
#>     url = {https://doi.org/10.21105/joss.03877},
#>   }
```
