#ifndef COVARIANCE_HPP
#define COVARIANCE_HPP

#define _USE_MATH_DEFINES

#include "openmpheader.h"
#include "general.h"
#include "interpreter.h"
#include "formula.hpp"
#include "sparse.h"

using namespace Eigen;

namespace glmmr {

class Covariance {
public:
  glmmr::Formula form_;
  const ArrayXXd data_;
  const strvec colnames_;
  dblvec parameters_;
  dblvec other_pars_;
  
  Covariance(const str& formula,
             const ArrayXXd &data,
             const strvec& colnames) :
    form_(formula), data_(data), colnames_(colnames), Q_(0),
    size_B_array((parse(),B_)), dmat_matrix(max_block_dim(),max_block_dim()),
    zquad(max_block_dim()) { 
      Z_constructor();
    };

  Covariance(const glmmr::Formula& form,
             const ArrayXXd &data,
             const strvec& colnames) :
    form_(form), data_(data), colnames_(colnames), Q_(0),
    size_B_array((parse(),B_)), dmat_matrix(max_block_dim(),max_block_dim()),
    zquad(max_block_dim()) {
      Z_constructor();
    };

  Covariance(const str& formula,
             const ArrayXXd &data,
             const strvec& colnames,
             const dblvec& parameters) :
    form_(formula), data_(data), colnames_(colnames), parameters_(parameters),
    Q_(0),size_B_array((parse(),B_)), dmat_matrix(max_block_dim(),max_block_dim()),
    zquad(max_block_dim()), spchol((make_sparse(),mat)) {
      L_constructor();
      Z_constructor();
    };

  Covariance(const glmmr::Formula& form,
             const ArrayXXd &data,
             const strvec& colnames,
             const dblvec& parameters) :
    form_(form), data_(data), colnames_(colnames), parameters_(parameters),
    Q_(0),size_B_array((parse(),B_)), dmat_matrix(max_block_dim(),max_block_dim()),
    zquad(max_block_dim()), spchol((make_sparse(),mat)) {
      L_constructor();
      Z_constructor();
    };

  Covariance(const str& formula,
             const ArrayXXd &data,
             const strvec& colnames,
             const ArrayXd& parameters) :
    form_(formula), data_(data), colnames_(colnames),
    parameters_(parameters.data(),parameters.data()+parameters.size()),Q_(0), 
    size_B_array((parse(),B_)), dmat_matrix(max_block_dim(),max_block_dim()),
    zquad(max_block_dim()), spchol((make_sparse(),mat)) {
      L_constructor();
      Z_constructor();
    };

  Covariance(const glmmr::Formula& form,
             const ArrayXXd &data,
              const strvec& colnames,
              const ArrayXd& parameters) :
    form_(form), data_(data), colnames_(colnames),
    parameters_(parameters.data(),parameters.data()+parameters.size()),Q_(0), 
    size_B_array((parse(),B_)), dmat_matrix(max_block_dim(),max_block_dim()),
    zquad(max_block_dim()), spchol((make_sparse(),mat)) {
    L_constructor();
    Z_constructor();
  };

  void update_parameters(const dblvec& parameters);
  
  void update_parameters_extern(const dblvec& parameters);

  void update_parameters(const ArrayXd& parameters);

  void parse();

  double get_val(int b, int i, int j);

  MatrixXd Z();

  MatrixXd D(bool chol = false,
             bool upper = false){
    MatrixXd D(Q_,Q_);
    if(isSparse){
      D = D_sparse_builder(chol,upper);
    } else {
      D = D_builder(0,chol,upper);
    }
    return D;
  };
  

  int npar(){
    return npars_;
  };

  
  int B(){
    return B_;
  }

  VectorXd sim_re();

  int Q(){
    if(Q_==0)Rcpp::stop("Random effects not initialised");
    return Q_;
  }
  
  int max_block_dim(){
    int max = 0;
    for(int i = 0; i<B_; i++){
      if(block_dim(i) > max)max = block_dim(i);
    }
    return max;
  }
  
  double log_likelihood(const VectorXd &u);

  double log_determinant();

  int block_dim(int b){
    return re_data_[b].rows();
  };

  void make_sparse();
  
  MatrixXd ZL();
  
  MatrixXd LZWZL(const VectorXd& w);
  
  MatrixXd ZLu(const MatrixXd& u);
  
  MatrixXd Lu(const MatrixXd& u);
  
  void set_sparse(bool sparse);
  
  bool any_group_re();
  
  intvec parameter_fn_index(){
    return re_fn_par_link_;
  }
  
  intvec re_count(){
    return re_count_;
  }
  
  sparse ZL_sparse();
  
  sparse Z_sparse();
  
  strvec parameter_names();
  
  void derivatives(std::vector<MatrixXd>& derivs,
                   int order = 1);

private:
  std::vector<glmmr::calculator> calc_;
  intvec z_;
  intvec3d re_pars_;
  intvec3d re_cols_data_;
  strvec2d fn_;
  dblvec2d par_for_calcs_;
  std::vector<MatrixXd> re_data_;
  intvec re_fn_par_link_;
  intvec re_count_;
  intvec re_order_;
  int Q_;
  int n_;
  int B_;
  int npars_;
  ArrayXd size_B_array;
  MatrixXd dmat_matrix;
  VectorXd zquad;
  bool isSparse = true;
  sparse mat;
  sparse matZ;
  sparse matL;
  SparseChol spchol;
  
  void update_parameters_in_calculators();
  
  MatrixXd get_block(int b);
  
  MatrixXd get_chol_block(int b,bool upper = false);

  MatrixXd D_builder(int b,
                            bool chol = false,
                            bool upper = false);

  void update_ax();
  
  void L_constructor();
  
  void Z_constructor();
  
  MatrixXd D_sparse_builder(bool chol = false,
                                   bool upper = false);
};

}

inline void glmmr::Covariance::derivatives(std::vector<MatrixXd>& derivs,
                                           int order){
  // get unique parameters
  strvec pars = parameter_names();
  int R = pars.size();
  int matrix_n = order==2 ? R + R*(R+1)/2 + 1 : R+1;
  //std::vector<MatrixXd> derivs;
  // initialise all the matrices to zero
  for(int i = 0; i < matrix_n; i++)derivs.push_back(MatrixXd::Zero(Q_,Q_));
  int block_count = 0;
  
  // iterate over the blocks and insert if the parameter is in the list.
  for(int b = 0; b < B_; b++){
    int block_size = block_dim(b);
    int R_block = calc_[b].parameter_names.size();
    intvec par_index;
    for(int k = 0; k < R_block; k++){
      auto par_pos = std::find(pars.begin(),pars.end(),calc_[b].parameter_names[k]);
      int par_pos_int = par_pos - pars.begin();
      par_index.push_back(par_pos_int);
    }
    //check speed of adding parallelisation here with collapse...
    for(int i = 0; i < block_size; i++){
      for(int j = i; j < block_size; j++){
        dblvec out = calc_[b].calculate(i,par_for_calcs_[b],re_data_[b],j,order);
        derivs[0](block_count+i,block_count+j) = out[0];
        if(i!=j)derivs[0](block_count+j,block_count+i) = out[0];
        int index_count = R_block + 1;
        for(int k = 0; k < R_block; k++){
          derivs[par_index[k]+1](block_count+i,block_count+j) = out[k+1];
          if(i!=j)derivs[par_index[k]+1](block_count+j,block_count+i) = out[k+1];
          //second order derivatives
          if(order >= 2){
            for(int l=k; l < R_block; l++){
              int second_pos = par_index[l]*(R-1) - par_index[l]*(par_index[l]-1)/2 + par_index[k];
              derivs[R+1+second_pos](block_count+i,block_count+j) = out[index_count];
              if(i!=j)derivs[R+1+second_pos](block_count+j,block_count+i) = out[index_count];
              index_count++;
            }
          }
        }
      }
    }
    block_count += block_size;
  }
}

#include "covariance.ipp"

#endif