% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/d.spls.ridge.R
\name{d.spls.ridge}
\alias{d.spls.ridge}
\title{Dual Sparse Partial Least Squares (Dual-SPLS) regression for the ridge norm}
\usage{
d.spls.ridge(X,y,ncp,ppnu,nu2,verbose=TRUE)
}
\arguments{
\item{X}{a numeric matrix of predictors values of dimension \code{(n,p)}. Each row represents one observation and each column one predictor variable.}

\item{y}{a numeric vector or a one column matrix of responses. It represents the response variable for each observation.}

\item{ncp}{a positive integer. \code{ncp} is the number of Dual-SPLS components.}

\item{ppnu}{a positive real value, in \eqn{[0,1]}. \code{ppnu} is the desired
proportion of variables to shrink to zero for each component (see Dual-SPLS methodology).}

\item{nu2}{a positive real value. \code{nu2} is a regularization parameter on \eqn{X^TX}.}

\item{verbose}{a Boolean value indicating whether or not to display the iterations steps. Default value is \code{TRUE}.}
}
\value{
A \code{list} of the following attributes
\item{Xmean}{the mean vector of the predictors matrix \code{X}.}
\item{scores}{the matrix of dimension \code{(n,ncp)} where \code{n} is the number of observations. The \code{scores} represents
the observations in the new component basis computed by the compression step
of the Dual-SPLS.}
\item{loadings}{the matrix of dimension \code{(p,ncp)} that represents the Dual-SPLS components.}
\item{Bhat}{the matrix of dimension \code{(p,ncp)} that regroups the regression coefficients for each component.}
\item{intercept}{the vector of intercept values for each component.}
\item{fitted.values}{the matrix of dimension \code{(n,ncp)} that represents the predicted values of \code{y}}
\item{residuals}{the matrix of dimension \code{(n,ncp)} that represents the residuals corresponding
to the difference between the responses and the fitted values.}
\item{lambda1}{the vector of length \code{ncp} collecting the parameters of sparsity used to fit the model at each iteration.}
\item{zerovar}{the vector of length \code{ncp} representing the number of variables shrank to zero per component.}
\item{ind_diff0}{the list of \code{ncp} elements representing the index of the none null regression coefficients elements.}
\item{type}{a character specifying the Dual-SPLS norm used. In this case it is \code{ridge}. }
}
\description{
The function \code{d.spls.lasso} performs dimensional reduction as in PLS methodology combined to variable selection via the
Dual-SPLS algorithm with the norm \deqn{\Omega(w)=\lambda_1 \|w\|_1 +\lambda_2 \|Xw\|_2 + \|w\|_2.}
In the algorithm, the parameters \eqn{\lambda}, \eqn{\lambda_1} and \eqn{\lambda_2}are transformed into more meaningful values, \code{ppnu} and \eqn{\nu_2}.
}
\details{
The resulting solution for \eqn{w} and hence for the coefficients vector, in the case of \code{d.spls.ridge}, has
a simple closed form expression (ref) deriving from the fact that \eqn{w} is collinear to a vector \eqn{z_{\nu_1}} of coordinates
\deqn{z_{\nu_1,j}=\textrm{sign}(z_{X,\nu_2,j})(|z_{X,\nu_2,j}|-\nu_1)_+.}
Here \eqn{\nu_1} is the threshold for which \code{ppnu} of
the absolute values of the coordinates of \eqn{z_{X,\nu_2}} are greater than \eqn{\nu_1} and \eqn{z_{X,\nu_2}=(\nu_2 X^TX + I_p)^{-1}X^Ty}.
Therefore, the ridge norm is beneficial to the situation where \eqn{X^TX} is singular. If \eqn{X^TX} is invertible, one
can choose to use the Dual-SPLS for the least squares norm instead.
}
\examples{
### load dual.spls library
library(dual.spls)
### parameters
oldpar <- par(no.readonly = TRUE)
n <- 200
p <- 100
nondes <- 150
sigmaondes <- 0.01
data=d.spls.simulate(n=n,p=p,nondes=nondes,sigmaondes=sigmaondes)

X <- data$X
y <- data$y


#fitting the model
mod.dspls <- d.spls.ridge(X=X,y=y,ncp=10,ppnu=0.9,nu2=100,verbose=TRUE)

str(mod.dspls)

### plotting the observed values VS predicted values
plot(y,mod.dspls$fitted.values[,6], xlab="Observed values", ylab="Predicted values",
main="Observed VS Predicted for 6 components")
points(-1000:1000,-1000:1000,type='l')

### plotting the regression coefficients
par(mfrow=c(3,1))
i=6
nz=mod.dspls$zerovar[i]
plot(1:dim(X)[2],mod.dspls$Bhat[,i],type='l',
    main=paste(" Dual-SPLS (ridge), ncp =", i, " #0coef =", nz, "/", dim(X)[2]),
    ylab='',xlab='' )
inonz=which(mod.dspls$Bhat[,i]!=0)
points(inonz,mod.dspls$Bhat[inonz,i],col='red',pch=19,cex=0.5)
legend("topright", legend ="non null values", bty = "n", cex = 0.8, col = "red",pch=19)
par(oldpar)
}
\seealso{
\link{d.spls.LS}
}
\author{
Louna Alsouki François Wahl
}
