% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/description.R
\docType{class}
\name{description}
\alias{description}
\title{Read, write, update, validate DESCRIPTION files}
\format{
An R6 class.
}
\description{
Read, write, update, validate DESCRIPTION files
}
\section{Constructors}{


There are two ways of creating a description object. The first
is reading an already existing \code{DESCRIPTION} file; simply give
the name of the file as an argument. The default is \code{DESCRIPTION}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x <- description$new()
x2 <- description$new("path/to/DESCRIPTION")
}\if{html}{\out{</div>}}

The second way is creating a description object from scratch,
supply \code{"!new"} as an argument to do this.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x3 <- description$new("!new")
}\if{html}{\out{</div>}}

The complete API reference:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$new(cmd = NULL, file = NULL, text = NULL,
    package = NULL)
}\if{html}{\out{</div>}}
\itemize{
\item \code{cmd}: a command to create a description from scratch.
Currently only \code{"!new"} is implemented. If it does not start
with an exclamation mark, it will be interpreted as a \code{file}
argument.
\item \code{file}: name of the \code{DESCRIPTION} file to load. If it is
a directory, then we assume that it is inside an R package and
conduct a search for the package root directory, i.e. the first
directory up the tree that contains a \code{DESCRIPTION} file.
If \code{cmd}, \code{file}, \code{text} and \code{package} are all \code{NULL} (the default),
then the search is started from the working directory. The file can
also be an R package (source, or binary), in which case the
\code{DESCRIPTION} file is extracted from it, but note that in this case
\verb{$write()} cannot write the file back in the package archive.
\item \code{text}: a character scalar containing the full DESCRIPTION.
Character vectors are collapsed into a character scalar, with
newline as the separator.
\item \code{package}: if not NULL, then the name of an installed package
and the DESCRIPTION file of this package will be loaded.
}
}

\section{Setting and Querying fields}{

Set a field with \verb{$set} and query it with \verb{$get}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x <- description$new("!new")
x$get("Package")
x$set("Package", "foobar")
x$set(Title = "Example Package for 'description'")
x$get("Package")
}\if{html}{\out{</div>}}

Note that \verb{$set} has two forms. You can either give the field name
and new value as two arguments; or you can use a single named argument,
the argument name is the field name, the argument value is the field
value.

The \verb{$fields} method simply lists the fields in the object:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x$fields()
}\if{html}{\out{</div>}}

The \verb{$has_fields} method checks if one or multiple fields are
present in a description object:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x$has_fields("Package")
x$has_fields(c("Title", "foobar"))
}\if{html}{\out{</div>}}

The \verb{$del} method removes the specified fields:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x$set(foo = "bar")
x$del("foo")
}\if{html}{\out{</div>}}

\verb{$get_field} is similar to \verb{$get}, but it queries a single
field, it returns an unnamed vector if found, and returns the
specified \code{default} value if not. By default it throws an error
if the field is not found.

The complete API reference:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$get(keys)
description$get_field(key, default, trim_ws = TRUE, squish_ws = trim_ws)
description$set(..., check = TRUE)
description$fields()
description$has_fields(keys)
description$del(keys)
}\if{html}{\out{</div>}}
\itemize{
\item \code{key}: a character string (length one), the key to query.
\item \code{default}: If specified and \code{key} is missing, this value
is returned. If not specified, an error is thrown.
\item \code{trim_ws}: whether to trim leading and trailing whitespace
from the returned value.
\item \code{squish_ws}: whether to reduce repeated whitespace in the
returned value.
\item \code{keys}: a character vector of keys to query, check or delete.
\item \code{...}: this must be either two unnamed arguments, the key and
and the value to set; or an arbitrary number of named arguments,
names are used as keys, values as values to set.
\item \code{check}: A logical scalar. Whether to check the validity of the
new values.
}
}

\section{Normalizing}{

Format DESCRIPTION in a standard way. \verb{$str} formats each
field in a standard way and returns them (it does not change the
object itself), \verb{$print} is used to print it to the
screen. The \verb{$normalize} function normalizes each field (i.e.
it changes the object). Normalization means reformatting the fields,
via \verb{\{$reformat_fields()} and also reordering them via
\verb{$reorder_fields()}. The format of the various fields is
opinionated and you might like it or not. Note that \code{desc} only
re-formats fields that it updates, and only on demand, so if your
formatting preferences differ, you can still manually edit
\code{DESCRIPTION} and \code{desc} will respect your edits.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$str(by_field = FALSE, normalize = TRUE,
    mode = c("file", "screen"))
description$normalize()
description$reformat_fields()
description$reorder_fields()
description$print()
}\if{html}{\out{</div>}}
\itemize{
\item \code{by_field}: whether to return the normalized format
by field, or collapsed into a character scalar.
\item \code{normalize}: whether to reorder and reformat the fields.
\item \code{mode}: \code{file} mode formats the fields as they are
written to a file with the \code{write} method. \code{screen}
mode adds extra markup to some fields, e.g. formats the
\code{Authors@R} field in a readable way.
}
}

\section{Writing it to file}{

The \verb{$write} method writes the description to a file.
By default it writes it to the file it was created from, if it was
created from a file. Otherwise giving a file name is compulsory:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x$write(file = "DESCRIPTION")
}\if{html}{\out{</div>}}

The API:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$write(file = NULL)
}\if{html}{\out{</div>}}
\itemize{
\item \code{file}: path to write the description to. If it was created
from a file in the first place, then it is written to the same
file. Otherwise this argument must be specified.
}
}

\section{Version numbers}{


\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$get_version()
description$set_version(version)
description$bump_version(which = c("patch", "minor", "major", "dev"))
}\if{html}{\out{</div>}}
\itemize{
\item \code{version}: a string or a \link[base:numeric_version]{base::package_version} object.
\item \code{which}: which component of the version number to increase.
See details just below.
}

These functions are simple helpers to make it easier to query, set and
increase the version number of a package.

\verb{$get_version()} returns the version number as a
\link[base:numeric_version]{base::package_version} object. It throws an error if the
package does not have a \code{Version} field.

\verb{$set_version()} takes a string or a \link[base:numeric_version]{base::package_version} object and
sets the \code{Version} field to it.

\verb{$bump_version()} increases the version number. The \code{which}
parameter specifies which component to increase.
It can be a string referring to a component: \code{major},
\code{minor}, \code{patch} or \code{dev}, or an integer
scalar, for the latter components are counted from one, and the
beginning. I.e. component one is equivalent to \code{major}.

If a component is bumped, then the ones after it are zeroed out.
Trailing zero components are omitted from the new version number,
but if the old version number had at least two or three components, then
the one will also have two or three.

The bumping of the \code{dev} version (the fourth component) is
special: if the original version number had less than four components,
and the \code{dev} version is bumped, then it is set to \code{9000}
instead of \code{1}. This is a convention often used by R developers,
it was originally invented by Winston Chang.

Both \verb{$set_version()} and \verb{$bump_version()} use dots to
separate the version number components.
}

\section{Dependencies}{

These functions handle the fields that define how the R package
uses another R packages. See \link{dep_types} for the
list of fields in this group.

The \verb{$get_deps} method returns all declared dependencies, in a
data frame with columns: \code{type}, \code{package} and \code{version}.
\code{type} is the name of the dependency field, \code{package} is the
name of the R package, and \code{version} is the required version. If
no specific versions are required, then this is a \code{"*"}.

The \verb{$set_deps} method is the opposite of \verb{$get_deps} and
it sets all dependencies. The input is a data frame, with the same
structure as the return value of \verb{$get_deps}.

The \verb{$has_dep} method checks if a package is included in the
dependencies. It returns a logical scalar. If \code{type} is not
\code{any}, then it has to match as well.

The \verb{$del_deps} method removes all declared dependencies.

The \verb{$set_dep} method adds or updates a single dependency. By
default it adds the package to the \code{Imports} field.

The API:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$set_dep(package, type = dep_types, version = "*")
description$set_deps(deps)
description$get_deps()
description$has_dep(package, type = c("any", dep_types))
description$del_dep(package, type = c("all", dep_types))
description$del_deps()
}\if{html}{\out{</div>}}
\itemize{
\item \code{package}: name of the package to add to or remove from the
dependencies.
\item \code{type}: dependency type, see \link{dep_types}. For
\verb{$del_dep} it may also be \code{"all"}, and then the package
will be deleted from all dependency types.
\item \code{version}: required version. Defaults to \code{"*"}, which means
no explicit version requirements.
\item \code{deps}: a data frame with columns \code{type}, \code{package} and
\code{version}. \verb{$get_deps} returns the same format.
}
}

\section{Collate fields}{

Collate fields contain lists of file names with R source code,
and the package has a separate API for them. In brief, you can
use \verb{$add_to_collate} to add one or more files to the main or
other collate field. You can use \verb{$del_from_collate} to remove
it from there.

The API:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$set_collate(files, which = c("main", "windows", "unix"))
description$get_collate(which = c("main", "windows", "unix"))
description$del_collate(which = c("all", "main", "windows", "unix"))
description$add_to_collate(files, which = c("default", "all", "main",
  "windows", "unix"))
description$del_from_collate(files, which = c("all", "main",
  "windows", "unix"))
}\if{html}{\out{</div>}}
\itemize{
\item \code{iles}: the files to add or remove, in a character vector.
\item \verb{which: which collate field to manipulate. }"default"\code{for}$add_to_collate` means all existing collate fields, or the
main one if none exist.
}
}

\section{Authors}{

There is a specialized API for the \code{Authors@R} field,
to add and remove authors, update their roles, change the maintainer,
etc.

The API:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$get_authors()
description$set_authors(authors)
description$get_author(role)
description$get_maintainer()
description$coerce_authors_at_r()
}\if{html}{\out{</div>}}
\itemize{
\item \code{authors}: a \code{person} object, a list of authors.
\item \code{role}: The role to query. See \code{person} for details.
}

\verb{$get_authors} returns a \code{person} object, the parsed
authors. See \code{\link[utils:person]{utils::person()}} for details.

\verb{$get_author} returns a \code{person} object, all authors with
the specified role.

\verb{$get_maintainer} returns the maintainer of the package. It works
with \code{Authors@R} fields and with traditional \code{Maintainer}
fields as well.

\verb{$coerce_authors_at_r} converts an \code{Author} field to one with
a \code{person} object. This coercion may be necessary for other
functions such as \verb{$get_authors}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$add_author(given = NULL, family = NULL, email = NULL,
    role = NULL, comment = NULL, orcid = NULL)
description$add_me(role = "ctb", comment = NULL, orcid = NULL)
description$add_author_gh(username, role = "ctb", comment = NULL, orcid = NULL)
}\if{html}{\out{</div>}}

Add a new author. The arguments correspond to the arguments of the
\code{\link[utils:person]{utils::person()}} function. \code{add_me} is a convenience
function, it adds the current user as an author, and it needs the
\code{whoami} package to be installed. It'll add your ORCID ID
if you provide it as argument or save it as \code{ORCID_ID} environment
variable in .Renviron.
The full name is parsed by \code{add_me} and \code{add_author_gh} using
\code{as.person} and collapsing the given name and the family name
in order to e.g. have the first and middle names together as given
name. This approach might be limited to some full name structures.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$del_author(given = NULL, family = NULL, email = NULL,
    role = NULL, comment = NULL, orcid = NULL)
}\if{html}{\out{</div>}}

Remove an author, or multiple authors. The author(s) to be removed
can be specified via any field(s). All authors matching all
specifications will be removed. E.g. if only \code{given = "Joe"}
is supplied, then all authors whole given name matches \code{Joe} will
be removed. The specifications can be (PCRE) regular expressions.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$add_role(role, given = NULL, family = NULL, email = NULL,
    comment = NULL, orcid = NULL)
description$add_orcid(orcid, given = NULL, family = NULL, email = NULL,
    comment = NULL, role = NULL)
description$del_role(role, given = NULL, family = NULL, email = NULL,
    comment = NULL, orcid = NULL)
description$change_maintainer(given = NULL, family = NULL,
    email = NULL, comment = NULL, orcid = NULL)
}\if{html}{\out{</div>}}

\code{role} is the role to add or delete. The other arguments
are used to select a subset of the authors, on which the operation
is performed, similarly to \verb{$del_author}.
}

\section{URLs}{


We provide helper functions for manipulating URLs in the \code{URL}
field:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$get_urls()
description$set_urls(urls)
description$add_urls(urls)
description$del_urls(pattern)
description$clear_urls()
}\if{html}{\out{</div>}}
\itemize{
\item \code{urls}: character vector of URLs to set or add.
\item \code{pattern}: Perl compatible regular expression to specify the
URLs to be removed.
}

\verb{$get_urls()} returns all urls in a character vector. If no URL
fields are present, a zero length vector is returned.

\verb{$set_urls()} sets the URL field to the URLs specified in the
character vector argument.

\verb{$add_urls()} appends the specified URLs to the URL field. It
creates the field if it does not exists. Duplicate URLs are removed.

\verb{$del_urls()} deletes the URLs that match the specified pattern.

\verb{$clear_urls()} deletes all URLs.
}

\section{Remotes}{


\code{devtools}, \code{remotes} and some other packages support the
non-standard \code{Remotes} field in \code{DESCRIPTION}. This field
can be used to specify locations of dependent packages: GitHub or
BitBucket repositories, generic git repositories, etc. Please see the
\verb{Package remotes} vignette in the \code{devtools} package.

\code{desc} has helper functions for manipulating the \code{Remotes}
field:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{description$get_remotes()
description$get_remotes()
description$set_remotes(remotes)
description$add_remotes(remotes)
description$del_remotes(pattern)
description$clear_remotes()
}\if{html}{\out{</div>}}
\itemize{
\item \code{remotes}: character vector of remote dependency locations to
set or add.
\item \code{pattern}: Perl compatible regular expression to specify the
remote dependency locations to remove.
}

\verb{$get_remotes()} returns all remotes in a character vector.
If no URL fields are present, a zero length vector is returned.

\verb{$set_remotes()} sets the URL field to the Remotes specified in the
character vector argument.

\verb{$add_remotes()} appends the specified remotes to the
\code{Remotes} field. It creates the field if it does not exists.
Duplicate remotes are removed.

\verb{$del_remotes()} deletes the remotes that match the specified
pattern.

\verb{$clear_remotes()} deletes all remotes.
}

\section{Built}{


The \code{Built} field is used in binary packages to store information
about when and how a binary package was built.

\verb{$get_built()} returns the built information as a list with fields
\code{R}, \code{Platform}, \code{Date}, \code{OStype}. It throws an
error if the package does not have a \code{Built} field.
}

\section{Encodings}{

When creating a \code{description} object, \code{desc} observes the \code{Encoding}
field, if present, and uses the specified encoding to parse the file.
Internally, it converts all fields to UTF-8.

When writing a \code{description} object to a file, \code{desc} uses the
\code{Encoding} field (if present), and converts all fields to the specified
encoding.

We suggest that whenever you need to use non-ASCII characters in your
package, you use the UTF-8 encoding, for maximum portability.
}

\examples{
## Create a template
desc <- description$new("!new")
desc

## Read a file
desc2 <- description$new(file = system.file("DESCRIPTION",
                           package = "desc"))
desc2

## Remove a field
desc2$del("LazyData")

## Add another one
desc2$set(VignetteBuilder = "knitr")
desc2$get("VignetteBuilder")
desc2
}
