% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/normFactors.R
\name{normFactors}
\alias{normFactors}
\title{Scaling normalization across libraries}
\usage{
normFactors(
  object,
  method = NULL,
  weighted = FALSE,
  ...,
  assay.id = "counts",
  se.out = TRUE
)
}
\arguments{
\item{object}{A \linkS4class{SummarizedExperiment} object containing a count matrix and library sizes in the \code{totals} field of the \code{\link{colData}}.

Alternatively, a \link{DGEList} object containing a count matrix in \code{object$counts} and library sizes in \code{object$samples$lib.size}.

Alternatively, an ordinary matrix containing counts.}

\item{method}{Deprecated argument, ignored.}

\item{weighted}{A logical scalar indicating whether precision weights should be used for TMM normalization.}

\item{...}{Other arguments to be passed to \code{\link{calcNormFactors}}.}

\item{assay.id}{An integer scalar or string specifying the assay values to use for normalization.}

\item{se.out}{A logical scalar indicating whether or not a SummarizedExperiment object should be returned.

Alternatively, a SummarizedExperiment or DGEList object in which normalization factors are to be stored.}
}
\value{
If \code{se.out=FALSE}, a numeric vector containing the relative normalization factors for each library.

If \code{se.out=TRUE}, the same vector is stored in the \code{norm.factors} field of \code{mcols(object)} (if \code{object} is a SummarizedExperiment)
or \code{object$samples} (if \code{object} is a DGEList) and the modified \code{object} is returned.

If \code{se.out} is a separate SummarizedExperiment or DGEList object, 
the normalization factors are stored inside \code{se.out} and the modified object is returned.
}
\description{
Calculate normalization factors using count data from multiple libraries.
}
\details{
This function uses the trimmed mean of M-values (TMM) method to remove composition biases, typically in background regions of the genome.
The key difference from standard TMM is that precision weighting is turned off by default so as to avoid upweighting high-abundance regions. 
These are more likely to be bound and thus more likely to be differentially bound. 
Assigning excessive weight to such regions will defeat the purpose of trimming when normalizing the coverage of background regions.

% Large changes in the normalization factor estimates with changes in the prior suggest that the counts are too low i.e. not
% enough new information in the dataset. This can be overcome by (obviously) increasing the counts. For example, binning
% can be performed with a larger bin size in \code{windowCounts} to obtain proportionally larger counts.

The normalization factors are always computed from \code{object}.
However, if \code{se.out} is a (different) SummarizedExperiment object, these factors are stored in \code{se.out} and the modified object.
This is useful when \code{se.out} contains counts for windows, but the normalization factors are computed using larger bins in \code{object}.
The same logic applies when \code{se.out} is a (different) DGEList object.

Note that an error is raised if the library sizes in \code{se.out} are not identical to \code{object$totals}.
This is because the normalization factors are only comparable when the library sizes are the same.
Consistent library sizes can be achieved by using the same \code{\link{readParam}} object in \code{\link{windowCounts}} and related functions.
}
\examples{
counts <- matrix(rnbinom(400, mu=10, size=20), ncol=4)
data <- SummarizedExperiment(list(counts=counts))
data$totals <- colSums(counts)

# TMM normalization.
normFactors(data)

}
\references{
Robinson MD, Oshlack A (2010). 
A scaling normalization method for differential expression analysis of RNA-seq data. 
\emph{Genome Biology} 11, R25.
}
\seealso{
\code{\link{calcNormFactors}}, for the base method.

\code{\link{normOffsets}}, for the trended normalization strategy.
}
\author{
Aaron Lun
}
\keyword{normalization}
