% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orsf_vi.R
\name{orsf_vi}
\alias{orsf_vi}
\alias{orsf_vi_negate}
\alias{orsf_vi_permute}
\alias{orsf_vi_anova}
\title{ORSF variable importance}
\usage{
orsf_vi(
  object,
  group_factors = TRUE,
  importance = NULL,
  oobag_fun = NULL,
  n_thread = 1,
  verbose_progress = FALSE,
  ...
)

orsf_vi_negate(
  object,
  group_factors = TRUE,
  oobag_fun = NULL,
  n_thread = 1,
  verbose_progress = FALSE,
  ...
)

orsf_vi_permute(
  object,
  group_factors = TRUE,
  oobag_fun = NULL,
  n_thread = 1,
  verbose_progress = FALSE,
  ...
)

orsf_vi_anova(object, group_factors = TRUE, ...)
}
\arguments{
\item{object}{(\emph{orsf_fit}) a trained oblique random survival forest
(see \link{orsf}).}

\item{group_factors}{(\emph{logical}) if \code{TRUE}, the importance of factor variables will be reported overall by aggregating the importance of individual levels of the factor. If \code{FALSE}, the importance of individual factor levels will be returned.}

\item{importance}{(\emph{character}) Indicate method for variable importance:
\itemize{
\item 'anova': compute analysis of variance (ANOVA) importance
\item 'negate': compute negation importance
\item 'permute': compute permutation importance
}}

\item{oobag_fun}{(\emph{function}) to be used for evaluating out-of-bag prediction accuracy after negating coefficients
(if importance = 'negate') or permuting the values of a predictor
(if importance = 'permute')
\itemize{
\item When \code{oobag_fun = NULL} (the default), Harrell's C-statistic (1982) is used to evaluate accuracy.
\item if you use your own \code{oobag_fun} note the following:
\itemize{
\item \code{oobag_fun} should have two inputs: \code{y_mat} and \code{s_vec}
\item \code{y_mat} is a two column matrix with first column named 'time', second named 'status'
\item \code{s_vec} is a numeric vector containing predicted survival probabilities.
\item \code{oobag_fun} should return a numeric output of length 1
\item the same \code{oobag_fun} should have been used when you created \code{object}
so that the initial value of out-of-bag prediction accuracy is
consistent with the values that will be computed while variable
importance is estimated.
}
}

For more details, see the out-of-bag
\href{https://docs.ropensci.org/aorsf/articles/oobag.html}{vignette}.}

\item{n_thread}{(\emph{integer}) number of threads to use while computing predictions. Default is one thread. To use the maximum number of threads that your system provides for concurrent execution, set \code{n_thread = 0}.}

\item{verbose_progress}{(\emph{logical}) if \code{TRUE}, progress messages are
printed in the console. If \code{FALSE} (the default), nothing is printed.}

\item{...}{Further arguments passed to or from other methods (not currently used).}
}
\value{
\code{orsf_vi} functions return a named numeric vector.
\itemize{
\item Names of the vector are the predictor variables used by \code{object}
\item Values of the vector are the estimated importance of the given predictor.
}

The returned vector is sorted from highest to lowest value, with higher
values indicating higher importance.
}
\description{
Estimate the importance of individual variables using oblique random
survival forests.
}
\details{
When an \code{orsf_fit} object is fitted with importance = 'anova', 'negate', or
'permute', the output will have a vector of importance values based on
the requested type of importance. However, you may still want to call
\code{orsf_vi()} on this output if you want to group factor levels into one
overall importance value.

\code{orsf_vi()} is a general purpose function to extract or compute variable
importance estimates from an \code{'orsf_fit'} object (see \link{orsf}).
\code{orsf_vi_negate()}, \code{orsf_vi_permute()}, and \code{orsf_vi_anova()} are wrappers
for \code{orsf_vi()}. The way these functions work depends on whether the
\code{object} they are given already has variable importance estimates in it
or not (see examples).
}
\section{Variable importance methods}{


\strong{negation importance}: Each variable is assessed separately by multiplying the variable's coefficients by -1 and then determining how much the model's performance changes. The worse the model's performance after negating coefficients for a given variable, the more important the variable. This technique is promising b/c it does not require permutation and it emphasizes variables with larger coefficients in linear combinations, but it is also relatively new and hasn't been studied as much as permutation importance. See Jaeger, (2023) for more details on this technique.

\strong{permutation importance}: Each variable is assessed separately by randomly permuting the variable's values and then determining how much the model's performance changes. The worse the model's performance after permuting the values of a given variable, the more important the variable. This technique is flexible, intuitive, and frequently used. It also has several \href{https://christophm.github.io/interpretable-ml-book/feature-importance.html#disadvantages-9}{known limitations}

\strong{analysis of variance (ANOVA) importance}: A p-value is computed for each coefficient in each linear combination of variables in each decision tree. Importance for an individual predictor variable is the proportion of times a p-value for its coefficient is < 0.01. This technique is very efficient computationally, but may not be as effective as permutation or negation in terms of selecting signal over noise variables. See \href{https://link.springer.com/chapter/10.1007/978-3-642-23783-6_29}{Menze, 2011} for more details on this technique.
}

\section{Examples}{
\subsection{ANOVA importance}{

The default variable importance technique, ANOVA, is calculated while
you fit an ORSF ensemble.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit <- orsf(pbc_orsf, Surv(time, status) ~ . - id)

fit
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## ---------- Oblique random survival forest
## 
##      Linear combinations: Accelerated
##           N observations: 276
##                 N events: 111
##                  N trees: 500
##       N predictors total: 17
##    N predictors per node: 5
##  Average leaves per tree: 21
## Min observations in leaf: 5
##       Min events in leaf: 1
##           OOB stat value: 0.84
##            OOB stat type: Harrell's C-statistic
##      Variable importance: anova
## 
## -----------------------------------------
}\if{html}{\out{</div>}}

ANOVA is the default because it is fast, but it may not be as decisive
as the permutation and negation techniques for variable selection.
}

\subsection{Raw VI values}{

the ‘raw’ variable importance values can be accessed from the fit object

\if{html}{\out{<div class="sourceCode r">}}\preformatted{attr(fit, 'importance_values')
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##     edema_1   ascites_1        bili      copper     albumin         age 
##  0.53189300  0.49950642  0.39598881  0.30443254  0.26028060  0.24758399 
##     protime       stage        chol   edema_0.5   spiders_1         ast 
##  0.22874192  0.20974576  0.20353982  0.18401760  0.18090452  0.17457962 
##    hepato_1       sex_f        trig    alk.phos    platelet trt_placebo 
##  0.16402406  0.14803440  0.13009809  0.11627907  0.07853659  0.06939410
}\if{html}{\out{</div>}}

these are ‘raw’ because values for factors have not been aggregated into
a single value. Currently there is one value for k-1 levels of a k level
factor. For example, you can see edema_1 and edema_0.5 in the importance
values above because edema is a factor variable with levels of 0, 0.5,
and 1.
}

\subsection{Collapse VI across factor levels}{

To get aggregated values across all levels of each factor,
\itemize{
\item access the \code{importance} element from the \code{orsf} fit:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit$importance
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##    ascites       bili      edema     copper    albumin        age    protime 
## 0.49950642 0.39598881 0.32482431 0.30443254 0.26028060 0.24758399 0.22874192 
##      stage       chol    spiders        ast     hepato        sex       trig 
## 0.20974576 0.20353982 0.18090452 0.17457962 0.16402406 0.14803440 0.13009809 
##   alk.phos   platelet        trt 
## 0.11627907 0.07853659 0.06939410
}\if{html}{\out{</div>}}
\item use \code{orsf_vi()} with group_factors set to \code{TRUE} (the default)

\if{html}{\out{<div class="sourceCode r">}}\preformatted{orsf_vi(fit)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##    ascites       bili      edema     copper    albumin        age    protime 
## 0.49950642 0.39598881 0.32482431 0.30443254 0.26028060 0.24758399 0.22874192 
##      stage       chol    spiders        ast     hepato        sex       trig 
## 0.20974576 0.20353982 0.18090452 0.17457962 0.16402406 0.14803440 0.13009809 
##   alk.phos   platelet        trt 
## 0.11627907 0.07853659 0.06939410
}\if{html}{\out{</div>}}
}

Note that you can make the default returned importance values ungrouped
by setting \code{group_factors} to \code{FALSE} in the \code{orsf_vi} functions or the
\code{orsf} function.
}

\subsection{Add VI to an ORSF}{

You can fit an ORSF without VI, then add VI later

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_no_vi <- orsf(pbc_orsf,
                  Surv(time, status) ~ . - id,
                  importance = 'none')

# Note: you can't call orsf_vi_anova() on fit_no_vi because anova
# VI can only be computed while the forest is being grown.

orsf_vi_negate(fit_no_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##        bili      copper         sex     protime       stage     albumin 
## 0.118355612 0.048917049 0.037068840 0.027044335 0.023867241 0.021214168 
##         age     ascites        chol         ast      hepato       edema 
## 0.020517824 0.014993236 0.014726515 0.011441749 0.007711157 0.007218808 
##     spiders        trig    alk.phos         trt    platelet 
## 0.006372905 0.003230269 0.002823511 0.002469395 0.001550349
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{orsf_vi_permute(fit_no_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##          bili        copper       protime       albumin       ascites 
##  0.0546201463  0.0248826626  0.0154623867  0.0135573758  0.0134222183 
##           age         stage          chol         edema           ast 
##  0.0119054385  0.0113940805  0.0074991392  0.0052943907  0.0051219919 
##        hepato       spiders           sex          trig      alk.phos 
##  0.0050381864  0.0046277553  0.0039401072  0.0024125340  0.0009602607 
##      platelet           trt 
##  0.0004343594 -0.0018248238
}\if{html}{\out{</div>}}
}

\subsection{ORSF and VI all at once}{

fit an ORSF and compute vi at the same time

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_permute_vi <- orsf(pbc_orsf,
                        Surv(time, status) ~ . - id,
                        importance = 'permute')

# get the vi instantly (i.e., it doesn't need to be computed again)
orsf_vi_permute(fit_permute_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##          bili        copper           age       ascites       albumin 
##  5.352210e-02  2.610549e-02  1.286639e-02  1.251888e-02  1.205836e-02 
##       protime         stage           ast          chol         edema 
##  1.084665e-02  1.057182e-02  8.228770e-03  6.002428e-03  5.834663e-03 
##       spiders        hepato           sex          trig      alk.phos 
##  4.760070e-03  3.437825e-03  3.388559e-03  2.274705e-03  2.226034e-03 
##      platelet           trt 
##  1.424997e-03 -4.992912e-06
}\if{html}{\out{</div>}}

You can still get negation VI from this fit, but it needs to be computed

\if{html}{\out{<div class="sourceCode r">}}\preformatted{orsf_vi_negate(fit_permute_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##        bili      copper         sex       stage     protime     albumin 
## 0.124726411 0.052319578 0.038681018 0.027479231 0.022737472 0.022214271 
##         age         ast     ascites        chol     spiders       edema 
## 0.020375826 0.013404081 0.013208974 0.011856865 0.008990815 0.007340934 
##      hepato         trt        trig    alk.phos    platelet 
## 0.006432032 0.004392664 0.004237792 0.003229450 0.002576929
}\if{html}{\out{</div>}}
}
}

\references{
Harrell FE, Califf RM, Pryor DB, Lee KL, Rosati RA. Evaluating the Yield of Medical Tests. \emph{JAMA} 1982; 247(18):2543-2546. DOI: 10.1001/jama.1982.03320430047030

Breiman L. Random forests. \emph{Machine learning} 2001 Oct; 45(1):5-32. DOI: 10.1023/A:1010933404324

Menze BH, Kelm BM, Splitthoff DN, Koethe U, Hamprecht FA. On oblique random forests. \emph{Joint European Conference on Machine Learning and Knowledge Discovery in Databases} 2011 Sep 4; pp. 453-469. DOI: 10.1007/978-3-642-23783-6_29

Jaeger BC, Welden S, Lenoir K, Speiser JL, Segar MW, Pandey A, Pajewski NM. Accelerated and interpretable oblique random survival forests. \emph{Journal of Computational and Graphical Statistics} Published online 08 Aug 2023. DOI: 10.1080/10618600.2023.2231048
}
