% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orsf_pd.R
\name{orsf_ice_oob}
\alias{orsf_ice_oob}
\alias{orsf_ice_inb}
\alias{orsf_ice_new}
\title{ORSF Individual Conditional Expectations}
\usage{
orsf_ice_oob(
  object,
  pred_spec,
  pred_horizon = NULL,
  pred_type = "risk",
  expand_grid = TRUE,
  boundary_checks = TRUE,
  n_thread = 1,
  ...
)

orsf_ice_inb(
  object,
  pred_spec,
  pred_horizon = NULL,
  pred_type = "risk",
  expand_grid = TRUE,
  boundary_checks = TRUE,
  n_thread = 1,
  ...
)

orsf_ice_new(
  object,
  pred_spec,
  new_data,
  pred_horizon = NULL,
  pred_type = "risk",
  na_action = "fail",
  expand_grid = TRUE,
  boundary_checks = TRUE,
  n_thread = 1,
  ...
)
}
\arguments{
\item{object}{(\emph{orsf_fit}) a trained oblique random survival forest
(see \link{orsf}).}

\item{pred_spec}{(\emph{named list} or \emph{data.frame}).
\itemize{
\item If \code{pred_spec} is a named list,
Each item in the list should be a vector of values that will be used as
points in the partial dependence function. The name of each item in the
list should indicate which variable will be modified to take the
corresponding values.
\item If \code{pred_spec} is a \code{data.frame}, columns will
indicate variable names, values will indicate variable values, and
partial dependence will be computed using the inputs on each row.
}}

\item{pred_horizon}{(\emph{double}) a value or vector indicating the time(s)
that predictions will be calibrated to. E.g., if you were predicting
risk of incident heart failure within the next 10 years, then
\code{pred_horizon = 10}. \code{pred_horizon} can be \code{NULL} if \code{pred_type} is
\code{'mort'}, since mortality predictions are aggregated over all
event times}

\item{pred_type}{(\emph{character}) the type of predictions to compute. Valid
options are
\itemize{
\item 'risk' : probability of having an event at or before \code{pred_horizon}.
\item 'surv' : 1 - risk.
\item 'chf': cumulative hazard function
\item 'mort': mortality prediction
}}

\item{expand_grid}{(\emph{logical}) if \code{TRUE}, partial dependence will be
computed at all possible combinations of inputs in \code{pred_spec}. If
\code{FALSE}, partial dependence will be computed for each variable
in \code{pred_spec}, separately.}

\item{boundary_checks}{(\emph{logical}) if \code{TRUE}, \code{pred_spec} will be checked
to make sure the requested values are between the 10th and 90th
percentile in the object's training data. If \code{FALSE}, these checks are
skipped.}

\item{n_thread}{(\emph{integer}) number of threads to use while computing predictions. Default is one thread. To use the maximum number of threads that your system provides for concurrent execution, set \code{n_thread = 0}.}

\item{...}{Further arguments passed to or from other methods (not currently used).}

\item{new_data}{a \link{data.frame}, \link[tibble:tibble-package]{tibble}, or \link[data.table:data.table]{data.table} to compute predictions in.}

\item{na_action}{(\emph{character}) what should happen when \code{new_data} contains missing values (i.e., \code{NA} values). Valid options are:
\itemize{
\item 'fail' : an error is thrown if \code{new_data} contains \code{NA} values
\item 'omit' : rows in \code{new_data} with incomplete data will be dropped
}}
}
\value{
a \link[data.table:data.table]{data.table} containing
individual conditional expectations for the specified variable(s) at the
specified prediction horizon(s).
}
\description{
Compute individual conditional expectations for an ORSF model.
Unlike partial dependence, which shows the expected prediction as a function of one or multiple predictors, individual conditional expectations (ICE) show the prediction for an individual observation as a function of a predictor.
You can compute individual conditional expectations three ways using a random forest:
\itemize{
\item using in-bag predictions for the training data
\item using out-of-bag predictions for the training data
\item using predictions for a new set of data
}

See examples for more details
}
\section{Examples}{
Begin by fitting an ORSF ensemble

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(aorsf)

set.seed(329)

fit <- orsf(data = pbc_orsf, formula = Surv(time, status) ~ . - id)

fit
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## ---------- Oblique random survival forest
## 
##      Linear combinations: Accelerated
##           N observations: 276
##                 N events: 111
##                  N trees: 500
##       N predictors total: 17
##    N predictors per node: 5
##  Average leaves per tree: 21
## Min observations in leaf: 5
##       Min events in leaf: 1
##           OOB stat value: 0.84
##            OOB stat type: Harrell's C-statistic
##      Variable importance: anova
## 
## -----------------------------------------
}\if{html}{\out{</div>}}

Use the ensemble to compute ICE values using out-of-bag predictions:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pred_spec <- list(bili = seq(1, 10, length.out = 25))

ice_oob <- orsf_ice_oob(fit, pred_spec, boundary_checks = FALSE)

ice_oob
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##       id_variable id_row pred_horizon bili      pred
##    1:           1      1         1788    1 0.8976716
##    2:           1      2         1788    1 0.1202763
##    3:           1      3         1788    1 0.6842180
##    4:           1      4         1788    1 0.3865812
##    5:           1      5         1788    1 0.1184953
##   ---                                               
## 6896:          25    272         1788   10 0.3421749
## 6897:          25    273         1788   10 0.4296413
## 6898:          25    274         1788   10 0.4496536
## 6899:          25    275         1788   10 0.3186596
## 6900:          25    276         1788   10 0.5490316
}\if{html}{\out{</div>}}

Much more detailed examples are given in the
\href{https://docs.ropensci.org/aorsf/articles/pd.html#individual-conditional-expectations-ice}{vignette}
}

