% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orsf.R
\name{orsf}
\alias{orsf}
\alias{orsf_train}
\title{Oblique Random Survival Forest (ORSF)}
\usage{
orsf(
  data,
  formula,
  control = orsf_control_fast(),
  weights = NULL,
  n_tree = 500,
  n_split = 5,
  n_retry = 3,
  n_thread = 1,
  mtry = NULL,
  sample_with_replacement = TRUE,
  sample_fraction = 0.632,
  leaf_min_events = 1,
  leaf_min_obs = 5,
  split_rule = "logrank",
  split_min_events = 5,
  split_min_obs = 10,
  split_min_stat = switch(split_rule, logrank = 3.841459, cstat = 0.5),
  oobag_pred_type = "surv",
  oobag_pred_horizon = NULL,
  oobag_eval_every = n_tree,
  oobag_fun = NULL,
  importance = "anova",
  group_factors = TRUE,
  tree_seeds = NULL,
  attach_data = TRUE,
  no_fit = FALSE,
  na_action = "fail",
  verbose_progress = FALSE,
  ...
)

orsf_train(object)
}
\arguments{
\item{data}{a \link{data.frame}, \link[tibble:tibble-package]{tibble}, or \link[data.table:data.table]{data.table} that contains the
relevant variables.}

\item{formula}{(\emph{formula}) The response on the left hand side should
include a time variable, followed by a status variable, and may be
written inside a call to \link[survival:Surv]{Surv} (see examples).
The terms on the right are names of predictor variables.}

\item{control}{(\emph{orsf_control}) An object returned from one of the
\code{orsf_control} functions:
\itemize{
\item \link{orsf_control_fast} (the default) uses a single iteration of Newton
Raphson scoring to identify a linear combination of predictors.
\item \link{orsf_control_cph} uses Newton Raphson scoring until a convergence
criteria is met.
\item \link{orsf_control_net} uses \code{glmnet} to identify linear combinations of
predictors, similar to Jaeger (2019).
\item \link{orsf_control_custom} allows the user to apply their own function
to create linear combinations of predictors.
}}

\item{weights}{(\emph{numeric vector}) Optional. If given, this input should
have length equal to \code{nrow(data)}. Values in \code{weights} are treated like
replication weights, i.e., a value of 2 is the same thing as having 2
observations in \code{data}, each containing a copy of the corresponding
person's data.

\emph{Use} \code{weights} \emph{cautiously}, as \code{orsf} will count the number of
observations and events prior to growing a node for a tree, so higher
values in \code{weights} will lead to deeper trees.}

\item{n_tree}{(\emph{integer}) the number of trees to grow.
Default is \code{n_tree = 500.}}

\item{n_split}{(\emph{integer}) the number of cut-points assessed when splitting
a node in decision trees. Default is \code{n_split = 5}.}

\item{n_retry}{(\emph{integer}) when a node can be split, but the current
linear combination of inputs is unable to provide a valid split, \code{orsf}
will try again with a new linear combination based on a different set
of randomly selected predictors, up to \code{n_retry} times. Default is
\code{n_retry = 3}. Set \code{n_retry = 0} to prevent any retries.}

\item{n_thread}{(\emph{integer}) number of threads to use while growing trees, computing predictions, and computing importance. Default is one thread. To use the maximum number of threads that your system provides for concurrent execution, set \code{n_thread = 0}.}

\item{mtry}{(\emph{integer}) Number of predictors randomly included as candidates
for splitting a node. The default is the smallest integer greater than
the square root of the number of total predictors, i.e.,
\verb{mtry = ceiling(sqrt(number of predictors))}}

\item{sample_with_replacement}{(\emph{logical}) If \code{TRUE} (the default),
observations are sampled with replacement when an in-bag sample
is created for a decision tree. If \code{FALSE}, observations are
sampled without replacement and each tree will have an in-bag sample
containing \code{sample_fraction}\% of the original sample.}

\item{sample_fraction}{(\emph{double}) the proportion of observations that
each trees' in-bag sample will contain, relative to the number of
rows in \code{data}. Only used if \code{sample_with_replacement} is \code{FALSE}.
Default value is 0.632.}

\item{leaf_min_events}{(\emph{integer}) minimum number of events in a
leaf node. Default is \code{leaf_min_events = 1}}

\item{leaf_min_obs}{(\emph{integer}) minimum number of observations in a
leaf node. Default is \code{leaf_min_obs = 5}.}

\item{split_rule}{(\emph{character}) how to assess the quality of a potential
splitting rule for a node. Valid options are
\itemize{
\item 'logrank' : a log-rank test statistic.
\item 'cstat'   : Harrell's concordance statistic.
}}

\item{split_min_events}{(\emph{integer}) minimum number of events required
in a node to consider splitting it. Default is \code{split_min_events = 5}}

\item{split_min_obs}{(\emph{integer}) minimum number of observations required
in a node to consider splitting it. Default is \code{split_min_obs = 10}.}

\item{split_min_stat}{(double) minimum test statistic required to split
a node. Default is 3.841459 if \code{split_rule = 'logrank'} and 0.50 if
\code{split_rule = 'cstat'}. If no splits are found with a statistic
exceeding \code{split_min_stat}, the given node either becomes a leaf or
a retry occurs (up to \code{n_retry} retries).}

\item{oobag_pred_type}{(\emph{character}) The type of out-of-bag predictions
to compute while fitting the ensemble. Valid options are
\itemize{
\item 'none' : don't compute out-of-bag predictions
\item 'risk' : probability of event occurring at or before \code{oobag_pred_horizon}.
\item 'surv' : 1 - risk.
\item 'chf'  : cumulative hazard function at \code{oobag_pred_horizon}.
\item 'mort' : mortality, i.e., the number of events expected if all
observations in the training data were identical to a
given observation.
}}

\item{oobag_pred_horizon}{(\emph{numeric}) A numeric value indicating what time
should be used for out-of-bag predictions. Default is the median
of the observed times, i.e., \code{oobag_pred_horizon = median(time)}.}

\item{oobag_eval_every}{(\emph{integer}) The out-of-bag performance of the
ensemble will be checked every \code{oobag_eval_every} trees. So, if
\code{oobag_eval_every = 10}, then out-of-bag performance is checked
after growing the 10th tree, the 20th tree, and so on. Default
is \code{oobag_eval_every = n_tree}.}

\item{oobag_fun}{(\emph{function}) to be used for evaluating out-of-bag prediction accuracy every \code{oobag_eval_every}
trees. When \code{oobag_fun = NULL} (the default), Harrell's C-statistic (1982) is used to evaluate accuracy. if you use your own \code{oobag_fun} note the following:
\itemize{
\item \code{oobag_fun} should have two inputs: \code{y_mat} and \code{s_vec}
\item \code{y_mat} is a two column matrix with first column named 'time', second named 'status'
\item \code{s_vec} is a numeric vector containing predicted survival probabilities.
\item \code{oobag_fun} should return a numeric output of length 1
}

For more details, see the out-of-bag \href{https://docs.ropensci.org/aorsf/articles/oobag.html#user-supplied-out-of-bag-evaluation-functions}{vignette}.}

\item{importance}{(\emph{character}) Indicate method for variable importance:
\itemize{
\item 'none': no variable importance is computed.
\item 'anova': compute analysis of variance (ANOVA) importance
\item 'negate': compute negation importance
\item 'permute': compute permutation importance
}

For details on these methods, see \link{orsf_vi}.}

\item{group_factors}{(\emph{logical}) Only relevant if variable importance is
being estimated. if \code{TRUE}, the importance of factor variables will be reported overall by aggregating the importance of individual levels of the factor. If \code{FALSE}, the importance of individual factor levels will be returned.}

\item{tree_seeds}{(\emph{integer vector}) Optional. if specified, random seeds
will be set using the values in \code{tree_seeds[i]}  before growing tree \code{i}.
Two forests grown with the same number of trees and the same seeds will
have the exact same out-of-bag samples, making out-of-bag error
estimates of the forests more comparable. If \code{NULL} (the default),
no seeds are set during the training process.}

\item{attach_data}{(\emph{logical}) if \code{TRUE}, a copy of the training
data will be attached to the output. This is helpful if you
plan on using functions like \link{orsf_pd_oob} or \link{orsf_summarize_uni}
to interpret the forest using its training data. Default is \code{TRUE}.}

\item{no_fit}{(\emph{logical}) if \code{TRUE}, model fitting steps are defined and
saved, but training is not initiated. The object returned can be
directly submitted to \code{orsf_train()} so long as \code{attach_data} is \code{TRUE}.}

\item{na_action}{(\emph{character}) what should happen when \code{data} contains missing values (i.e., \code{NA} values). Valid options are:
\itemize{
\item 'fail' : an error is thrown if \code{data} contains \code{NA} values
\item 'omit' : rows in \code{data} with incomplete data will be dropped
\item 'impute_meanmode' : missing values for continuous and categorical variables in \code{data} will be imputed using the mean and mode, respectively. Note that is this
option is selected and \code{attach_data} is \code{TRUE}, the data attached
to the output will be the imputed version of \code{data}.
}}

\item{verbose_progress}{(\emph{logical}) if \code{TRUE}, progress messages are
printed in the console. If \code{FALSE} (the default), nothing is printed.}

\item{...}{Further arguments passed to or from other methods (not currently used).}

\item{object}{an untrained 'aorsf' object, created by setting
\code{no_fit = TRUE} in \code{orsf()}.}
}
\value{
an accelerated oblique RSF object (\code{aorsf})
}
\description{
Fit an oblique random survival forest
}
\details{
This function is based on and similar to the \code{ORSF} function
in the \code{obliqueRSF} R package. The primary difference is that this
function runs much faster. The speed increase is attributable to better
management of memory (i.e., no unnecessary copies of inputs) and using
a Newton Raphson scoring algorithm to identify linear combinations of
inputs rather than performing penalized regression using routines in
\code{glmnet}.The modified Newton Raphson scoring algorithm that this
function applies is an adaptation of the C++ routine developed by
Terry M. Therneau that fits Cox proportional hazards models
(see \code{\link[survival:coxph]{survival::coxph()}} and more specifically \code{\link[survival:agreg.fit]{survival::coxph.fit()}}).
}
\section{Details on inputs}{


\strong{formula}:
\itemize{
\item The response in \code{formula} can be a survival
object as returned by the \link[survival:Surv]{Surv} function,
but can also just be the time and status variables. I.e.,
\code{Surv(time, status) ~ .} works just like \code{time + status ~ .}
\item A \code{.} symbol on the right hand side is short-hand for using all
variables in \code{data} (omitting those on the left hand side of
\code{formula}) as predictors.
\item The order of variables in the left hand side matters. i.e.,
writing \code{status + time ~ .} will make \code{orsf} assume your
\code{status} variable is actually the \code{time} variable.
\item The response variable can be a survival object stored in \code{data}.
For example, y ~ . is a valid formula if \code{data$y} inherits
from the \code{Surv} class.
\item Although you can fit an oblique random survival forest with 1 predictor
variable, your formula should have at least 2 predictors. The reason for
this recommendation is that a linear combination of predictors is trivial
if there is only one predictor.
}

\strong{mtry}:

The \code{mtry} parameter may be temporarily reduced to ensure there
are at least 2 events per predictor variable. This occurs when using
\link{orsf_control_cph} because coefficients in the Newton Raphson scoring
algorithm may become unstable when the number of covariates is
greater than or equal to the number of events. This reduction does not
occur when using \link{orsf_control_net}.

\strong{oobag_fun}:

If \code{oobag_fun} is specified, it will be used in to compute negation
importance or permutation importance, but it will not have any role
for ANOVA importance.

\strong{n_thread}:

If an R function must be called from C++ (i.e., user-supplied function to
compute out-of-bag error or identify linear combinations of variables),
\code{n_thread} will automatically be set to 1 because attempting to run R
functions in multiple threads will cause the R session to crash.
}

\section{What is an oblique decision tree?}{


Decision trees are developed by splitting a set of training data into two
new subsets, with the goal of having more similarity within the new subsets
than between them. This splitting process is repeated on the resulting
subsets of data until a stopping criterion is met. When the new subsets of
data are formed based on a single predictor, the decision tree is said to
be axis-based because the splits of the data appear perpendicular to the
axis of the predictor. When linear combinations of variables are used
instead of a single variable, the tree is oblique because the splits of
the data are neither parallel nor at a right angle to the axis

\emph{Figure} : Decision trees for classification with axis-based splitting
(left) and oblique splitting (right). Cases are orange squares; controls
are purple circles. Both trees partition the predictor space defined by
variables X1 and X2, but the oblique splits do a better job of separating
the two classes.

\if{html}{\figure{tree_axis_v_oblique.png}{options: width=95\%}}
}

\section{What is a random forest?}{


Random forests are collections of de-correlated decision trees.
Predictions from each tree are aggregated to make an ensemble
prediction for the forest. For more details, see Breiman at el, 2001.
}

\section{Training, out-of-bag error, and testing}{


In random forests, each tree is grown with a bootstrapped version of
the training set. Because bootstrap samples are selected with replacement,
each bootstrapped training set contains about two-thirds of instances in
the original training set. The 'out-of-bag' data are instances that are
\emph{not} in the bootstrapped training set. Each tree in the random forest
can make predictions for its out-of-bag data, and the out-of-bag
predictions can be aggregated to make an ensemble out-of-bag prediction.
Since the out-of-bag data are not used to grow the tree, the accuracy of
the ensemble out-of-bag predictions approximate the generalization error
of the random forest. Generalization error refers to the error of a
random forest's predictions when it is applied to predict outcomes for
data that were not used to train it, i.e., testing data.
}

\section{Missing data}{


Data passed to aorsf functions are not allowed to have missing values.
A user should impute missing values using an R package with that purpose,
such as \code{recipes} or \code{mlr3pipelines}.
}

\section{Examples}{
First we load some relevant packages

\if{html}{\out{<div class="sourceCode r">}}\preformatted{set.seed(329730)
suppressPackageStartupMessages(\{
 library(aorsf)
 library(survival)
 library(tidymodels)
 library(tidyverse)
 library(randomForestSRC)
 library(ranger)
 library(riskRegression) 
 library(obliqueRSF)
\})
}\if{html}{\out{</div>}}

The entry-point into \code{aorsf} is the standard call to \code{orsf()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit <- orsf(pbc_orsf, Surv(time, status) ~ . - id)
}\if{html}{\out{</div>}}

printing \code{fit} provides quick descriptive summaries:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## ---------- Oblique random survival forest
## 
##      Linear combinations: Accelerated
##           N observations: 276
##                 N events: 111
##                  N trees: 500
##       N predictors total: 17
##    N predictors per node: 5
##  Average leaves per tree: 25
## Min observations in leaf: 5
##       Min events in leaf: 1
##           OOB stat value: 0.84
##            OOB stat type: Harrell's C-statistic
##      Variable importance: anova
## 
## -----------------------------------------
}\if{html}{\out{</div>}}
\subsection{Model control}{

For these examples we will make use of the \code{orsf_control_} functions to
build and compare models based on their out-of-bag predictions. We will
also standardize the out-of-bag samples using the input argument
\code{tree_seeds}
\subsection{Accelerated linear combinations}{

The accelerated ORSF ensemble is the default because it has a nice
balance of computational speed and prediction accuracy. It runs a single
iteration of Newton Raphson scoring on the Cox partial likelihood
function to find linear combinations of predictors.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_accel <- orsf(pbc_orsf, 
                  control = orsf_control_fast(),
                  formula = Surv(time, status) ~ . - id,
                  tree_seeds = 329)
}\if{html}{\out{</div>}}
}

\subsection{Linear combinations with Cox regression}{

\code{orsf_control_cph} runs Cox regression in each non-terminal node of each
survival tree, using the regression coefficients to create linear
combinations of predictors:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_cph <- orsf(pbc_orsf, 
                control = orsf_control_cph(),
                formula = Surv(time, status) ~ . - id,
                tree_seeds = 329)
}\if{html}{\out{</div>}}
}

\subsection{Linear combinations with penalized cox regression}{

\code{orsf_control_net} runs penalized Cox regression in each non-terminal
node of each survival tree, using the regression coefficients to create
linear combinations of predictors. This can be really helpful if you
want to do feature selection within the node, but it is a lot slower
than the other options.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# select 3 predictors out of 5 to be used in
# each linear combination of predictors.
fit_net <- orsf(pbc_orsf, 
                control = orsf_control_net(df_target = 3),
                formula = Surv(time, status) ~ . - id,
                tree_seeds = 329)
}\if{html}{\out{</div>}}
}

\subsection{Linear combinations with your own function}{

Let’s make two customized functions to identify linear combinations of
predictors.
\itemize{
\item The first uses random coefficients

\if{html}{\out{<div class="sourceCode r">}}\preformatted{f_rando <- function(x_node, y_node, w_node)\{
 matrix(runif(ncol(x_node)), ncol=1) 
\}
}\if{html}{\out{</div>}}
\item The second derives coefficients from principal component analysis.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{f_pca <- function(x_node, y_node, w_node) \{ 

 # estimate two principal components.
 pca <- stats::prcomp(x_node, rank. = 2)
 # use the second principal component to split the node
 pca$rotation[, 1L, drop = FALSE]

\}
}\if{html}{\out{</div>}}
\item The third uses \code{orsf()} inside of \code{orsf()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# This approach is known as reinforcement learning trees.  
# some special care is taken to prevent your R session from crashing.
# Specifically, random coefficients are used when n_obs <= 10
# or n_events <= 5. 

f_aorsf <- function(x_node, y_node, w_node)\{

 colnames(y_node) <- c('time', 'status')
 colnames(x_node) <- paste("x", seq(ncol(x_node)), sep = '')

 data <- as.data.frame(cbind(y_node, x_node))

 if(nrow(data) <= 10 || sum(y_node[,'status']) <= 5) 
  return(matrix(runif(ncol(x_node)), ncol = 1))

 fit <- orsf(data, time + status ~ ., 
             weights = as.numeric(w_node),
             n_tree = 25,
             importance = 'permute')

 out <- orsf_vi(fit)

 # drop the least two important variables
 n_vars <- length(out)
 out[c(n_vars, n_vars-1)] <- 0

 # ensure out has same variable order as input
 out <- out[colnames(x_node)]

 matrix(out, ncol = 1)

\}
}\if{html}{\out{</div>}}
}

We can plug these functions into \code{orsf_control_custom()}, and then pass
the result into \code{orsf()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_rando <- orsf(pbc_orsf,
                  Surv(time, status) ~ . - id,
                  control = orsf_control_custom(beta_fun = f_rando),
                  tree_seeds = 329)

fit_pca <- orsf(pbc_orsf,
                Surv(time, status) ~ . - id,
                control = orsf_control_custom(beta_fun = f_pca),
                tree_seeds = 329)

fit_rlt <- orsf(pbc_orsf, time + status ~ . - id, 
                control = orsf_control_custom(beta_fun = f_aorsf),
                tree_seeds = 329)
}\if{html}{\out{</div>}}

So which fit seems to work best in this example? Let’s find out by
evaluating the out-of-bag survival predictions.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{risk_preds <- list(
 accel = 1 - fit_accel$pred_oobag,
 cph   = 1 - fit_cph$pred_oobag,
 net   = 1 - fit_net$pred_oobag,
 rando = 1 - fit_rando$pred_oobag,
 pca   = 1 - fit_pca$pred_oobag,
 rlt   = 1 - fit_rlt$pred_oobag
)

sc <- Score(object = risk_preds, 
            formula = Surv(time, status) ~ 1, 
            data = pbc_orsf, 
            summary = 'IPA',
            times = fit_accel$pred_horizon)
}\if{html}{\out{</div>}}

The AUC values, from highest to lowest:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sc$AUC$score[order(-AUC)]
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##    model times       AUC         se     lower     upper
## 1:   net  1788 0.9134593 0.02079935 0.8726933 0.9542253
## 2:   cph  1788 0.9109155 0.02111657 0.8695278 0.9523032
## 3: accel  1788 0.9099638 0.02122647 0.8683607 0.9515669
## 4:   rlt  1788 0.9069752 0.02132529 0.8651783 0.9487720
## 5: rando  1788 0.9023489 0.02218936 0.8588586 0.9458393
## 6:   pca  1788 0.8994220 0.02201713 0.8562692 0.9425748
}\if{html}{\out{</div>}}

And the indices of prediction accuracy:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sc$Brier$score[order(-IPA), .(model, times, IPA)]
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##         model times       IPA
## 1:        net  1788 0.4916815
## 2:        cph  1788 0.4833913
## 3:      accel  1788 0.4749974
## 4:        rlt  1788 0.4630984
## 5:        pca  1788 0.4371223
## 6:      rando  1788 0.4258456
## 7: Null model  1788 0.0000000
}\if{html}{\out{</div>}}

From inspection,
\itemize{
\item \code{net}, \code{accel}, and \code{rlt} have high discrimination and index of
prediction accuracy.
\item \code{rando} and \code{pca} do less well, but they aren’t bad.
}
}

}

\subsection{tidymodels}{

This example uses \code{tidymodels} functions but stops short of using an
official \code{tidymodels} workflow. I am working on getting \code{aorsf} pulled
into the \code{censored} package and I will update this with real workflows
if that happens!
\subsection{Comparing ORSF with other learners}{

Start with a recipe to pre-process data

\if{html}{\out{<div class="sourceCode r">}}\preformatted{imputer <- recipe(pbc_orsf, formula = time + status ~ .) \%>\% 
 step_impute_mean(all_numeric_predictors()) \%>\%
 step_impute_mode(all_nominal_predictors()) 
}\if{html}{\out{</div>}}

Next create a 10-fold cross validation object and pre-process the data:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# 10-fold cross validation; make a container for the pre-processed data
analyses <- vfold_cv(data = pbc_orsf, v = 10) \%>\%
 mutate(recipe = map(splits, ~prep(imputer, training = training(.x))),
        train = map(recipe, juice),
        test = map2(splits, recipe, ~bake(.y, new_data = testing(.x))))

analyses
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## #  10-fold cross-validation 
## # A tibble: 10 x 5
##    splits           id     recipe   train               test              
##    <list>           <chr>  <list>   <list>              <list>            
##  1 <split [248/28]> Fold01 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  2 <split [248/28]> Fold02 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  3 <split [248/28]> Fold03 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  4 <split [248/28]> Fold04 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  5 <split [248/28]> Fold05 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  6 <split [248/28]> Fold06 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  7 <split [249/27]> Fold07 <recipe> <tibble [249 x 20]> <tibble [27 x 20]>
##  8 <split [249/27]> Fold08 <recipe> <tibble [249 x 20]> <tibble [27 x 20]>
##  9 <split [249/27]> Fold09 <recipe> <tibble [249 x 20]> <tibble [27 x 20]>
## 10 <split [249/27]> Fold10 <recipe> <tibble [249 x 20]> <tibble [27 x 20]>
}\if{html}{\out{</div>}}

Define functions for a ‘workflow’ with \code{randomForestSRC}, \code{ranger}, and
\code{aorsf}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rfsrc_wf <- function(train, test, pred_horizon)\{
 
 # rfsrc does not like tibbles, so cast input data into data.frames
 train <- as.data.frame(train)
 test <- as.data.frame(test)
 
 rfsrc(formula = Surv(time, status) ~ ., data = train) \%>\% 
  predictRisk(newdata = test, times = pred_horizon) \%>\% 
  as.numeric()
 
\}

ranger_wf <- function(train, test, pred_horizon)\{
 
 ranger(Surv(time, status) ~ ., data = train) \%>\% 
  predictRisk(newdata = test, times = pred_horizon) \%>\% 
  as.numeric()
 
\}

aorsf_wf <- function(train, test, pred_horizon)\{
 
 train \%>\% 
  orsf(Surv(time, status) ~ .,) \%>\% 
  predict(new_data = test, pred_horizon = pred_horizon) \%>\% 
  as.numeric()
 
\}
}\if{html}{\out{</div>}}

Run the ‘workflows’ on each fold:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# 5 year risk prediction
ph <- 365.25 * 5

results <- analyses \%>\% 
 transmute(test, 
           pred_aorsf = map2(train, test, aorsf_wf, pred_horizon = ph),
           pred_rfsrc = map2(train, test, rfsrc_wf, pred_horizon = ph),
           pred_ranger = map2(train, test, ranger_wf, pred_horizon = ph))
}\if{html}{\out{</div>}}

Next unnest each column to get back a \code{tibble} with all of the testing
data and predictions.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{results <- results \%>\% 
 unnest(everything())

glimpse(results)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## Rows: 276
## Columns: 23
## $ id          <int> 3, 39, 43, 48, 50, 54, 64, 66, 78, 80, 83, 114, 131, 141, ~
## $ trt         <fct> d_penicill_main, d_penicill_main, d_penicill_main, placebo~
## $ age         <dbl> 70.07255, 55.39220, 48.87064, 49.13621, 53.50856, 39.19781~
## $ sex         <fct> m, f, f, m, f, f, f, m, f, m, f, m, f, f, f, f, m, f, f, f~
## $ ascites     <fct> 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0~
## $ hepato      <fct> 0, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1, 1, 0, 1, 1~
## $ spiders     <fct> 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0~
## $ edema       <fct> 0.5, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0.5, 0, 0, 0, 0, 0, 0, 0, ~
## $ bili        <dbl> 1.4, 0.7, 1.1, 1.9, 1.1, 1.3, 2.1, 1.4, 6.3, 7.2, 1.3, 3.2~
## $ chol        <int> 176, 282, 361, 259, 257, 288, 373, 427, 436, 247, 250, 259~
## $ albumin     <dbl> 3.48, 3.00, 3.64, 3.70, 3.36, 3.40, 3.50, 3.70, 3.02, 3.72~
## $ copper      <int> 210, 52, 36, 281, 43, 262, 52, 105, 75, 269, 48, 208, 74, ~
## $ alk.phos    <dbl> 516.0, 9066.8, 5430.2, 10396.8, 1080.0, 5487.2, 1009.0, 19~
## $ ast         <dbl> 96.10, 72.24, 67.08, 188.34, 106.95, 73.53, 150.35, 182.90~
## $ trig        <int> 55, 111, 89, 178, 73, 125, 188, 171, 104, 91, 100, 78, 104~
## $ platelet    <int> 151, 563, 203, 214, 128, 254, 178, 123, 236, 360, 81, 268,~
## $ protime     <dbl> 12.0, 10.6, 10.6, 11.0, 10.6, 11.0, 11.0, 11.0, 10.6, 11.2~
## $ stage       <ord> 4, 4, 2, 3, 4, 4, 3, 3, 4, 4, 4, 3, 4, 2, 3, 4, 2, 3, 4, 3~
## $ time        <int> 1012, 2297, 4556, 4427, 2598, 1434, 1487, 4191, 1690, 890,~
## $ status      <dbl> 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 0, 0, 0, 1, 0~
## $ pred_aorsf  <dbl> 0.76027848, 0.25291419, 0.06284001, 0.59437152, 0.15286015~
## $ pred_rfsrc  <dbl> 0.47891074, 0.16833427, 0.05141013, 0.46526027, 0.06438684~
## $ pred_ranger <dbl> 0.61304990, 0.13930022, 0.03715869, 0.48395613, 0.04959462~
}\if{html}{\out{</div>}}

And finish by aggregating the predictions and computing performance in
the testing data. Note that I am computing one statistic for all
predictions instead of computing one statistic for each fold. This
approach is fine when you have smaller testing sets and/or small event
counts.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Score(
 object = list(aorsf = results$pred_aorsf,
               rfsrc = results$pred_rfsrc,
               ranger = results$pred_ranger),
 formula = Surv(time, status) ~ 1, 
 data = results, 
 summary = 'IPA',
 times = ph
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## 
## Metric AUC:
## 
## Results by model:
## 
##     model times  AUC lower upper
## 1:  aorsf  1826 90.9  86.7  95.1
## 2:  rfsrc  1826 90.0  85.8  94.3
## 3: ranger  1826 90.1  86.0  94.3
## 
## Results of model comparisons:
## 
##    times  model reference delta.AUC lower upper   p
## 1:  1826  rfsrc     aorsf      -0.9  -2.2   0.5 0.2
## 2:  1826 ranger     aorsf      -0.8  -2.1   0.6 0.3
## 3:  1826 ranger     rfsrc       0.1  -0.8   1.0 0.8

## 
## NOTE: Values are multiplied by 100 and given in \%.

## NOTE: The higher AUC the better.

## 
## Metric Brier:
## 
## Results by model:
## 
##         model   times Brier lower upper  IPA
## 1: Null model 1826.25  20.5  18.1  22.9  0.0
## 2:      aorsf 1826.25  10.8   8.5  13.0 47.4
## 3:      rfsrc 1826.25  11.8   9.6  13.9 42.6
## 4:     ranger 1826.25  11.7   9.6  13.8 42.7
## 
## Results of model comparisons:
## 
##      times  model  reference delta.Brier lower upper            p
## 1: 1826.25  aorsf Null model        -9.7 -12.4  -7.0 2.820785e-12
## 2: 1826.25  rfsrc Null model        -8.7 -11.0  -6.4 5.857526e-14
## 3: 1826.25 ranger Null model        -8.7 -11.1  -6.4 1.380943e-13
## 4: 1826.25  rfsrc      aorsf         1.0   0.2   1.8 1.507974e-02
## 5: 1826.25 ranger      aorsf         1.0   0.3   1.7 8.236836e-03
## 6: 1826.25 ranger      rfsrc        -0.0  -0.5   0.4 9.336601e-01

## 
## NOTE: Values are multiplied by 100 and given in \%.

## NOTE: The lower Brier the better, the higher IPA the better.
}\if{html}{\out{</div>}}

From inspection,
\itemize{
\item \code{aorsf} obtained slightly higher discrimination (AUC)
\item \code{aorsf} obtained higher index of prediction accuracy (IPA)
}
}

}

\subsection{mlr3 pipelines}{

\strong{Warning}: this code may or may not run depending on your current
version of \code{mlr3proba}. First we load some additional \code{mlr3} libraries.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{suppressPackageStartupMessages(\{
 library(mlr3verse)
 library(mlr3proba)
 library(mlr3extralearners)
 library(mlr3viz)
 library(mlr3benchmark)
\})
}\if{html}{\out{</div>}}

Next we’ll define some tasks for our learners to engage with.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Mayo Clinic Primary Biliary Cholangitis Data
task_pbc <- 
 TaskSurv$new(
  id = 'pbc',  
  backend = select(pbc_orsf, -id) \%>\% 
   mutate(stage = as.numeric(stage)),  
  time = "time", 
  event = "status"
 )

# Veteran's Administration Lung Cancer Trial
data(veteran, package = "randomForestSRC")

task_veteran <- 
 TaskSurv$new(
  id = 'veteran',  
  backend = veteran,  
  time = "time", 
  event = "status"
 )

# NKI 70 gene signature
data_nki <- OpenML::getOMLDataSet(data.id = 1228)

task_nki <- 
 TaskSurv$new(
  id = 'nki',  
  backend = data_nki$data,  
  time = "time", 
  event = "event"
 )

# Gene Expression-Based Survival Prediction in Lung Adenocarcinoma
data_lung <- OpenML::getOMLDataSet(data.id = 1245)

task_lung <- 
 TaskSurv$new(
  id = 'nki',  
  backend = data_lung$data \%>\% 
   mutate(OS_event = as.numeric(OS_event) -1),  
  time = "OS_years", 
  event = "OS_event"
 )


# Chemotherapy for Stage B/C colon cancer
# (there are two rows per person, one for death 
#  and the other for recurrence, hence the two tasks)

task_colon_death <-
 TaskSurv$new(
  id = 'colon_death',  
  backend = survival::colon \%>\%
   filter(etype == 2) \%>\% 
   drop_na() \%>\% 
   # drop id, redundant variables
   select(-id, -study, -node4, -etype),
   mutate(OS_event = as.numeric(OS_event) -1),  
  time = "time", 
  event = "status"
 )

task_colon_recur <-
 TaskSurv$new(
  id = 'colon_death',  
  backend = survival::colon \%>\%
   filter(etype == 1) \%>\% 
   drop_na() \%>\% 
   # drop id, redundant variables
   select(-id, -study, -node4, -etype),
   mutate(OS_event = as.numeric(OS_event) -1),  
  time = "time", 
  event = "status"
 )

# putting them all together
tasks <- list(task_pbc,
              task_veteran,
              task_nki,
              task_lung,
              task_colon_death,
              task_colon_recur,
              # add a few more pre-made ones
              tsk("actg"),
              tsk('gbcs'),
              tsk('grace'),
              tsk("unemployment"),
              tsk("whas"))
}\if{html}{\out{</div>}}

Now we can make a benchmark designed to compare our three favorite
learners:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Learners with default parameters
learners <- lrns(c("surv.ranger", "surv.rfsrc", "surv.aorsf"))

# Brier (Graf) score, c-index and training time as measures
measures <- msrs(c("surv.graf", "surv.cindex", "time_train"))

# Benchmark with 5-fold CV
design <- benchmark_grid(
  tasks = tasks,
  learners = learners,
  resamplings = rsmps("cv", folds = 5)
)

benchmark_result <- benchmark(design)

bm_scores <- benchmark_result$score(measures, predict_sets = "test")
}\if{html}{\out{</div>}}

Let’s look at the overall results:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{bm_scores \%>\%
 select(task_id, learner_id, surv.graf, surv.cindex, time_train) \%>\%
 group_by(learner_id) \%>\% 
 filter(!is.infinite(surv.graf)) \%>\% 
 summarize(
  across(
   .cols = c(surv.graf, surv.cindex, time_train),
   .fns = mean, 
   na.rm = TRUE
  )
 )
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## # A tibble: 3 x 4
##   learner_id  surv.graf surv.cindex time_train
##   <chr>           <dbl>       <dbl>      <dbl>
## 1 surv.aorsf      0.152       0.733      1.41 
## 2 surv.ranger     0.166       0.712      1.95 
## 3 surv.rfsrc      0.155       0.723      0.745
}\if{html}{\out{</div>}}

From inspection,
\itemize{
\item \code{aorsf} has a higher expected value for ‘surv.cindex’ (higher is
better)
\item \code{aorsf} has a lower expected value for ‘surv.graf’ (lower is better)
}
}
}

\references{
Harrell FE, Califf RM, Pryor DB, Lee KL, Rosati RA. Evaluating the Yield of Medical Tests. \emph{JAMA} 1982; 247(18):2543-2546. DOI: 10.1001/jama.1982.03320430047030

Breiman L. Random forests. \emph{Machine learning} 2001 Oct; 45(1):5-32. DOI: 10.1023/A:1010933404324

Ishwaran H, Kogalur UB, Blackstone EH, Lauer MS. Random survival forests. \emph{Annals of applied statistics} 2008 Sep; 2(3):841-60. DOI: 10.1214/08-AOAS169

Jaeger BC, Long DL, Long DM, Sims M, Szychowski JM, Min YI, Mcclure LA, Howard G, Simon N. Oblique random survival forests. \emph{Annals of applied statistics} 2019 Sep; 13(3):1847-83. DOI: 10.1214/19-AOAS1261

Jaeger BC, Welden S, Lenoir K, Speiser JL, Segar MW, Pandey A, Pajewski NM. Accelerated and interpretable oblique random survival forests. \emph{Journal of Computational and Graphical Statistics} Published online 08 Aug 2023. DOI: 10.1080/10618600.2023.2231048
}
