#' Generate a random Gaussian vector
#'
#' @param hdobj An S3 object of class 'no_nbd' or 'nbd' generated by [ts_hdchange()].
#'
#' @return The Gaussian random vector \eqn{\mathcal{Z}}.
#' @references Li, J., Chen, L., Wang, W. and Wu, W.B., 2022. \eqn{\ell^2} Inference for Change Points in High-Dimensional Time Series via a Two-Way MOSUM.
#' \emph{arXiv preprint arXiv:2208.13074}.
#' @export
#'
#' @examples
#' # generate data
#' data_no_nbd <- sim_hdchange_no_nbd(n = 200,
#' p = 30,
#' S = 30,
#' tau = c(40, 100, 160),
#' dist_info =
#'   list(dist = "normal", dependence = "MA_inf", param = 1),
#' jump_max = c(2, 2, 1.5))
#'
#' # construct no_nbd object
#' ts_no_nbd <- ts_hdchange(data_no_nbd,
#' window_size = 30,
#' m = 8,
#' h = 1,
#' N_rep = 999,
#' alpha = 1e-5,
#' quantiles = c(0.01, 0.05, 0.1))
#'
#' Z <- genZ(ts_no_nbd)
#'
genZ <- function(hdobj) {
  n <- hdobj$n # number of observations
  p <- hdobj$p # cross-sectional dimension
  b <- hdobj$b # bandwidth parameter
  N_rep <- hdobj$N_rep


  #--------covariance matrix of X ---------#
  cov_x_MAinf <- get_cov_x_MAinf(n, b)

  #--------Gaussian vector Z by true covariance matrix---------#

  R <- chol(cov_x_MAinf)
  z_MAinf <- sapply(c(1:(N_rep * p)), function(k) {
    t(R) %*% matrix(stats::rnorm(nrow(cov_x_MAinf)), nrow = nrow(cov_x_MAinf))
  })
  z_MAinf <- t(z_MAinf)


  return(z_MAinf)
}

#' The covariance matrix for generating random Gaussian vector
#'
#' @param n Number of time series observations.
#' @param b Bandwith parameter \eqn{b = window\_size/n}.
#'
#' @return The covariance matrix. See section 2.2 of Li et al. (2023).
#' @references Li, J., Chen, L., Wang, W. and Wu, W.B., 2022. \eqn{\ell^2} Inference for Change Points in High-Dimensional Time Series via a Two-Way MOSUM.
#' \emph{arXiv preprint arXiv:2208.13074}.
#' @export
#'
#' @examples
#'
#' # generate data
#' data_no_nbd <- sim_hdchange_no_nbd(n = 200,
#' p = 30,
#' S = 30,
#' tau = c(40, 100, 160),
#' dist_info =
#'   list(dist = "normal", dependence = "MA_inf", param = 1),
#' jump_max = c(2, 2, 1.5))
#'
#' # construct no_nbd object
#' ts_no_nbd <- ts_hdchange(data_no_nbd,
#' window_size = 30,
#' m = 8,
#' h = 1,
#' N_rep = 999,
#' alpha = 1e-5,
#' quantiles = c(0.01, 0.05, 0.1))
#'
#' Cov_x_MAinf <- get_cov_x_MAinf(ts_no_nbd$n, ts_no_nbd$b)
#'
get_cov_x_MAinf <- function(n, b) {

  cov_x_MAinf <- matrix(0, nrow = n - 2 * b * n, ncol = n - 2 * b * n)

  index_grid <- expand.grid(j = (1:(n - 2 * b * n)), i = (1:(n - 2 * b * n)))
  index <- index_grid[index_grid$i <= index_grid$j, ]

  cov_x_MAinf_i_j <- function(i, j) {
    c <- (j - i) / (b * n)

    if (c <= 1) {
      18 * c^2 - 24 * c + 8
    } else if (c > 1 & c <= 2) {
      2 * c^2 - 8 * c + 8
    } else {
      0
    }
  }

  cov_x_MAinf_vec <- mapply(cov_x_MAinf_i_j, index$i, index$j)

  cov_x_MAinf_mat <- matrix(0, nrow = n - 2 * b * n, ncol = n - 2 * b * n)

  cov_x_MAinf_mat[lower.tri(cov_x_MAinf_mat, diag = TRUE)] <- cov_x_MAinf_vec

  cov_x_MAinf_mat_trans <- t(cov_x_MAinf_mat)

  cov_x_MAinf_mat_trans[lower.tri(cov_x_MAinf_mat_trans, diag = TRUE)] <- cov_x_MAinf_vec

  return(cov_x_MAinf_mat_trans)
}

#' Compute the long-run variance of the gap vector
#'
#' @param hdobj An S3 object of class 'no_nbd' or 'nbd' generated
#' by [ts_hdchange()].
#'
#' @return The covariance matrix of the gap vectors \eqn{\hat{J}(.)}.
#' @references Li, J., Chen, L., Wang, W. and Wu, W.B., 2022. \eqn{\ell^2} Inference for Change Points in High-Dimensional Time Series via a Two-Way MOSUM.
#' \emph{arXiv preprint arXiv:2208.13074}.
#'
#' @export
#'
#' @examples
#'
#' # generate data
#' data_no_nbd <- sim_hdchange_no_nbd(n = 200,
#' p = 30,
#' S = 30,
#' tau = c(40, 100, 160),
#' dist_info =
#'   list(dist = "normal", dependence = "MA_inf", param = 1),
#' jump_max = c(2, 2, 1.5))
#'
#' # construct no_nbd object
#' ts_no_nbd <- ts_hdchange(data_no_nbd,
#' window_size = 30,
#' m = 8,
#' h = 1,
#' N_rep = 999,
#' alpha = 1e-5,
#' quantiles = c(0.01, 0.05, 0.1))
#'
#' lr_var <- get_lr_var(ts_no_nbd)
#'
get_lr_var <- function(hdobj) {
  ynew <- t(hdobj$data)
  n <- hdobj$n # number of observations
  p <- hdobj$p # cross-sectional dimension
  m <- hdobj$m
  h <- hdobj$h

  size <- floor(n / m - 1)
  xi <- matrix(0, size, p)
  sigmam <- array(0, dim = c(p, p, size))
  xi[1, ] <- colSums(ynew[1:m, ]) / m

  for (l in 1:(size - 1))
  {
    xi[l + 1, ] <- colSums(ynew[(l * m + 1):(l * m + m), ]) / m
    sigmam[, , l] <-
      m * (xi[(l + 1), ] - xi[l, ]) %*% t((xi[(l + 1), ] - xi[l, ])) / (2)
  }

  aaa <- apply(sigmam, c(1, 2), varest, size = size)

  index <- expand.grid(j = 1:p, i = 1:p)

  get_root <- function(i, j, n, m, h, sigmam, aaa) {
    obj <- stats::uniroot(
      robustfun2,
      x = sigmam[i, j, ],
      alpha = 10 * (aaa[i, i] * aaa[j, j])^{
        1 / 2
      } * (m / n * h)^{
        1 / 2
      },
      interval = c(-4e+11, 9e+11)
    )

    return(obj$root)
  }

  varestt_vec <- mapply(get_root, index$i, index$j,
                        MoreArgs = list(n = n, m = m, h = h, sigmam = sigmam, aaa = aaa)
  )

  varestt <- matrix(varestt_vec, nrow = p, ncol = p)


  # varmatrix       <- diag(1, p, p)
  # diag(varmatrix) <- diag(varestt)
  # corrlatema      <- solve(sqrt(varmatrix))%*%varestt%*%solve(sqrt(varmatrix))


  return(varestt)
}


