#include <math.h>
#include <R.h>

	/* Standard defines */
#define MIN(a,b) ((a)<(b)? (a):(b))
#define MAX(a,b) ((a)<(b)? (b):(a))

#define OI 16
#define HOI 8
#define OIM1 15
/*      OI is the order of interpolation to be used
HOI is half the order of interpolation to be used
OIM1 is OI-1    */

#define FALSE 0
#define TRUE 1

/*====================================================================== */
double millsratio(double x)
{
/* Computes Mill's ratio, which is the ratio of the right tail */
/*	probability to the density for a standard normal distribution. */
/* Note that this function should not be called for negative arguments. */

double val,t;
double rhodiff[26]={1.,3.94766755064487339,
	.262699373075432297,-4.79530994886257864,-.472387694632788044,
	-4.64178612251027508,.888029797897972282E-1,3.34506626993363627,
	-.253018663557510120,-3.22271919537673200,.367907414060725404,
	.515135649489030960,-.782205852853700406,-1.66658903030166903,
	-.317783299853388699,3.44266858016113165,.110601063267646032,
	-113.675951225046565,-.475674678528123486E-2,891.504346794291090,
	.155839071476740257E-3,-52392.5264048076612,-.145796580254323180E-4,
	5331522.33187226848,.169353718097630593E-6,-83341339.7482781260};
double xdata[25]={0.,1.,.2704,.5776,.0784,.04,.0576,.0256,.1024,
	.1296,.16,.1936,.2304,.0064,.3136,.36,.4096,.4624,
	.5184,.0144,.64,.7056,.7744,.8464,.9216};

if(x<0.){
	return 0.;
}

/* Thiele interpolation by Abramowitz and Stegun 25.2.50*/
t=1./(x+1.);
val=t*(rhodiff[0]+(t-xdata[0])/(rhodiff[1]+(t-xdata[1])/(rhodiff[2]+
	(t-xdata[2])/(rhodiff[3]+(t-xdata[3])/(rhodiff[4]+
	(t-xdata[4])/(rhodiff[5]+(t-xdata[5])/(rhodiff[6]+
	(t-xdata[6])/(rhodiff[7]+(t-xdata[7])/(rhodiff[8]+
	(t-xdata[8])/(rhodiff[9]+(t-xdata[9])/(rhodiff[10]+
	(t-xdata[10])/(rhodiff[11]+(t-xdata[11])/(rhodiff[12]+
	(t-xdata[12])/(rhodiff[13]+(t-xdata[13])/(rhodiff[14]+
	(t-xdata[14])/(rhodiff[15]+(t-xdata[15])/(rhodiff[16]+
	(t-xdata[16])/(rhodiff[17]+(t-xdata[17])/(rhodiff[18]+
	(t-xdata[18])/(rhodiff[19]+(t-xdata[19])/(rhodiff[20]+
	(t-xdata[20])/(rhodiff[21]+(t-xdata[21])/(rhodiff[22]+
	(t-xdata[22])/(rhodiff[23]+(t-xdata[23])/(rhodiff[24]+
	(t-xdata[24])/(rhodiff[25]))))))))))))))))))))))))));
return val;
}
/*=====================================================================*/
double lognormaltail(double z)
{
/* Finds the logarithm of right hand tail probability for a standard normal distribution. */
/* Underflow will affect the result if the argument z is large negative. */

double val;
const double dnorm0=.39894228040143267793994605993438187;
if(z<0.) val=log1p(-dnorm0*exp(-.5*z*z)*millsratio(-z));
else val=log(dnorm0*millsratio(z))-.5*z*z;
return val;
}
/*=====================================================================*/

double normaltail(double z)
{
/* Finds the right hand tail probability for a standard normal distribution */
/* Note that precision is poor if the argument z is negative. */
/* Will return zero if z is large. */

double val;
double const dnorm0=.39894228040143267793994605993438187;
if(z<0.) val=1.-dnorm0*exp(-.5*z*z)*millsratio(-z);
else val=dnorm0*exp(-.5*z*z)*millsratio(z);
return val;
}

/*=====================================================================*/

double LogGamma(double x)
{
/*  Abramowitz & Stegun equation 6.1.48	 */

double Z,val,c0=0.9189385332046727417803297364056177,
	a0=1/12.,a1=1/30.,a2=53/210.,a3=195/371.,a4=22999/22737.,
	a5=29944523./19733142.,a6=109535241009./48264275462.;
if(x>10) val=(x-.5)*log(x)+c0-x+a0/(x+a1/(x+a2/(x+a3/(x+a4/(x+a5/(x+a6/x))))));
else {
	Z=x+9;
	val=(Z-.5)*log(Z)+c0-Z+a0/(Z+a1/(Z+a2/(Z+a3/(Z+a4/(Z+a5/(Z+a6/Z
		))))))-log(x*(x+1)*(x+2)*(x+3)*(x+4)*(x+5)*(x+6)*(x+7)*(x+8));
}
return val;
}

/*====================================================================== */
void calc_recip_denom(int nx,double x[],double denom[])
/* Calculates reciprocals of denominators for use in later interpolation */
{
double product;
int i,j,k,offset;
for (i=0; i<nx-OIM1; i++){
	offset=i;
	for (j=0; j<OI; j++){
		product=1;
		for (k=0; k<OI; k++){
			if(k != j)product=product*(x[j+offset]-x[k+offset]);
		}
		denom[i*OI+j]=1./product;
	}
}
}
/*========================================================================= */
void interpolate(double x,double *f,double *d,int nxn,double xn[],
	double fn[],double dn[],double xdenomn[])
{
/* Find interpolated values for x using vectors xn, fn and dn with */
/* reciprocals of denominators specified by array xdenomn */
double difference[OI],product,weight;
int low,high,mid,offset,start,k;

/* First find the smallest index of a larger value of xn by binary chop */
low=0;
high=nxn-1;
if(xn[high]<x){
	return;
}
do{
	mid=(low+high)/2;
	if(xn[mid] >= x) high=mid;
	else{
		if(low == mid)break;
		low=mid;
	}
} while(TRUE);

start=MIN(MAX(0,high-HOI),nxn-OI);
offset=start;
product=1;
for (k=0; k<OI; k++){
	difference[k]=x-xn[k+offset];
	product=product*difference[k];
}
if(product == 0){
	for (k=0; k<OI; k++){
/*Use appropriate alpha value */
		if(x == xn[k+offset]){
			*f=fn[k+offset];
			*d=dn[k+offset];
			break;
		}
	}
}
else{
	*f=0;
	*d=0;
	for (k=0; k<OI; k++){
		weight=product*xdenomn[start*OI+k]/difference[k];
		*f+=weight*fn[k+offset];
		*d+=weight*dn[k+offset];
		}
	}
}
/*=================================================================== */
void interpolate_over_alpha(int nx,int nalpha,double alphalist[],
	double thisalpha,double tablef[],double tabled[],
	double thisf[],double thisd[],double denom[])
{
/* To interpolate the tables tablef and tabled over alpha */
double weight,product,difference[OI];
int i,j,k,start,offset;

/* First find the smallest index of a larger value of alpha */
/* (This could be made faster with a binary chop algorithm) */
for (j=0; j<nalpha; j++){
	if(alphalist[j] > thisalpha)break;
}
start=MIN(MAX(0,j-HOI),nalpha-OI);
offset=start;
product=1;
for (k=0; k<OI; k++){
	difference[k]=thisalpha-alphalist[k+offset];
	product=product*difference[k];
}

/* Minor option: when thisalpha is a tabulated value */
if(product == 0){
	for (k=0; k<OI; k++){
		if(thisalpha == alphalist[k+offset]){
			for (i=0; i<nx; i++){
				thisf[i]=tablef[i*nalpha+(k+offset)];
				thisd[i]=tabled[i*nalpha+(k+offset)];
			}
			break;
		}
	}
}

/* Major option: need to interpolate */
else{
	for (i=0; i<nx; i++){
		thisf[i]=0;
		thisd[i]=0;
	}
	for (k=0; k< OI; k++){
		weight=product*denom[start*OI+k]/difference[k];
		for (i=0; i<nx; i++){
			thisf[i]+=weight*tablef[i*nalpha+(k+offset)];
			thisd[i]+=weight*tabled[i*nalpha+(k+offset)];
		}
	}
}
}
/*====================================================================== */
static const double pi=3.141592653589793238462643383279502884197;
static const double hpi=1.57079632679489661923132169163975144209858;
static const double EulersC=.5772156649015328606065120900824024310422;
static const double neglarge=-1.7E308;
static const double poslarge=1.7E308;
static const double logposlarge=709.7;

static double previous_alpha=-999.;
static double previous_oneminusalpha=-999.;
static double previous_twominusalpha=-999.;

static double absam1,a1,sa2,nu,Calpha_C,Calpha_M,midpoint,xi,ximid,xlowlimit;
static double Clogd,alphastar,ratio,angle,C1,C2,C3,C4,eta;
static double ffound,dfound,logapprox,logmidpoint,logscalef;

/* Boolean variables stored as integers. */
static int distributiontabulated, alphasmall, alphalarge;

/* Interpolation table i gives values for density as di and for distribution */
/* function (or its complement) as fi for nxi values of x and nyi values of y*/

/* First tables are for alpha < 1, alpha*xi > 1/5 in C parametrization */
/* Vy1 is alpha, Vx1 is proportional to 1/(alpha*xi)	*/ 
#define nx1 70
#define ny1 20
static double f1[nx1],d1[nx1];
static double xdenom1[(nx1-OIM1)*OI],ydenom1[(ny1-OIM1)*OI];

/* Second tables are for alpha < .5, alpha*xi < 1/5 and x<1 in C parametrization */
/* Vy2 is alpha, Vx2 is proportional to x**(-1/alpha) */
#define nx2 20
#define ny2 20
static double f2[nx2],d2[nx2];
static double xdenom2[(nx2-OIM1)*OI],ydenom2[(ny2-OIM1)*OI];

/* Third tables are for alpha < .5, x > 1 in C parametrization */
/* Vy3 is alpha, Vx3 is proportional to x**(-1/alpha) */
#define nx3 20
#define ny3 20
static double f3[nx3],d3[nx3];
static double xdenom3[(nx3-OIM1)*OI],ydenom3[(ny3-OIM1)*OI];

/* Fourth tables are for 1.7 <alpha < 2, -1.3 < x (M=S0) < 20 */
/* Use the difference from the alpha=2 (normal) distribution for all larger x, 
scaling so that the asymptotic behaviour as x tends to infinity is handled
accurately. */
/* Vy4 is alpha, Vx4 is based on x in the S0 parametrization */
#define nx4 100
#define ny4 17
static double f4[nx4],d4[nx4];
static double xdenom4[(nx4-OIM1)*OI],ydenom4[(ny4-OIM1)*OI];
static double f4_alpha2[nx4],d4_alpha2[nx4];

/* Fifth tables are for 1.7 <alpha < 2, x (M=S0) > 20 */
/* Use Zolotarev 2.5.6 */
/* Vy5 is alpha, Vx5 is proportional to y**(-1/alpha) where x=y+eta y**(1-alpha) */
#define nx5 20
#define ny5 17
static double f5[nx5],d5[nx5];
static double xdenom5[(nx5-OIM1)*OI],ydenom5[(ny5-OIM1)*OI];

/* Sixth tables are for 0.5 <alpha < 1.7, x (M=S0) > 5 */
/* Use Zolotarev 2.5.6 */
/* Vy6 is alpha, Vx6 is proportional to y**(-1/alpha) where x=y+eta y**(1-alpha) */
#define nx6 20
#define ny6 40
static double f6[nx6],d6[nx6];
static double xdenom6[(nx6-OIM1)*OI],ydenom6[(ny6-OIM1)*OI];

/* Seventh tables are for 0.5 <alpha < 1.7, from xi=2/5 to x=7.3 */
/* 	Table 1 goes from xi=.2/alpha to xi=infinity, so regions overlap. */
/* Vy7 is alpha, Vx7 is x (S0 parametrization) */
#define nx7 60
#define ny7 40
static double f7[nx7],d7[nx7];
static double xdenom7[(nx7-OIM1)*OI],ydenom7[(ny7-OIM1)*OI];

static double Vx1 [ 70 ]= {
.0000000000000000, .45990742405315702E-03, .18333067643362053E-02, 
.40948494741676056E-02, .71947340420536812E-02, .11056395086954524E-01, 
.15595905748776201E-01, .20672025722893478E-01, .25594501142759296E-01, 
.31245398792875995E-01, .37087285542121984E-01, .43126427968428027E-01, 
.49369312012549849E-01, .55822650287644937E-01, .62493389618247408E-01, 
.69388718813952477E-01, .76516076683197376E-01, .83883160292598458E-01, 
.91497933477380111E-01, .99368635608504396E-01, .10750379062218374, 
.11591221631753668, .12460303392821592, .13358567797391729, .14286990639774855, 
.15246581099551482, .16238382814304608, .17263474982777582, .18322973499084361, 
.19418032118607487, .20549843656226696, .21719641217527808, .22928699463649704, 
.24178335910434123, .25469912262550709, .26804835783277259, .28184560700621741, 
.29610589650481867, .31084475157542890, .32607821154624439, .34182284541192187, 
.35809576781759844, .37491465544911939, .39229776383687964, .41026394458073590, 
.42883266300353062, .44802401624085197, .46785875177470526, .48835828641887336, 
.50954472576378951, .53144088408885182, .55407030475014718, .57745728105166361, 
.60162687760811018, .62660495220756474, .65241817818222980, .67909406729565092, 
.70666099315483777, .73514821515578355, .76458590297097739, .79500516158754930, 
.82643805690479155, .85559084019345732, .88755296168977793, .91792530156133001, 
.94521252885087126, .96815777279823845, .98552017573978112, .99633259962233911, 
1.0000000000000000
};
static double Vy1 [ 20 ]= {
.17517132627621161E-02, .15629641247743881E-01, .42933637439770855E-01, 
.82487632228667507E-01, .13288049721334105, .19221839463378387, 
.25805643818551127, .32821819512532135, .39285714285714279, .46428571428571441, 
.53571428571428559, .60714285714285676, .67178180487467909, .74194356181448828, 
.80778160536621613, .86711950278665917, .91751236777133283, .95706636256022914, 
.98437035875225654, .99824828673723776
};
static double Vx2 [ 20 ]= {
.0000000000000000, .75098428300898815E-02, .29791037091918493E-01, 
.66016460423343193E-01, .11476091580842640, .17407448001180459, 
.24190142749515875, .31545656747747763, .38461538461538458, .46153846153846168, 
.53846153846153832, .61538461538461497, .68454343252252237, .75809857250484125, 
.82592551998819541, .88523908419157404, .93398353957665670, .97020896290808167, 
.99249015716991007, 1.0000000000000000
};
static double Vy2 [ 20 ]= {
.90829577264398432E-03, .81042584851375166E-02, .22261886245745155E-01, 
.42771365177980580E-01, .68900999068418178E-01, .99668797960107969E-01, 
.13380704301905721, .17018721355519717, .20370370522141457, .24074074253439903, 
.27777777984738350, .31481481716036797, .34811350341623859, .38352894144936878, 
.41618636085133742, .44474474382870355, .46821429659638314, .48565616721924343, 
.49638414972153511, .50000000000000000
};
static double Vx3 [ 20 ]= {
.0000000000000000, .75098428300898815E-02, .29791037091918493E-01, 
.66016460423343193E-01, .11476091580842640, .17407448001180459, 
.24190142749515875, .31545656747747763, .38461538461538458, .46153846153846168, 
.53846153846153832, .61538461538461497, .68454343252252237, .75809857250484125, 
.82592551998819541, .88523908419157404, .93398353957665670, .97020896290808167, 
.99249015716991007, 1.0000000000000000
};
static double Vy3 [ 20 ]= {
.90829577264398432E-03, .81042584851375166E-02, .22261886245745155E-01, 
.42771365177980580E-01, .68900999068418178E-01, .99668797960107969E-01, 
.13380704301905721, .17018721355519717, .20370370522141457, .24074074253439903, 
.27777777984738350, .31481481716036797, .34811350341623859, .38352894144936878, 
.41618636085133742, .44474474382870355, .46821429659638314, .48565616721924343, 
.49638414972153511, .50000000000000000
};
static double Vx4 [ 100 ]= {
-1.2999999523162842, -1.2900227383977363, -1.2603961181158390, 
-1.2119463888758943, -1.1460931661069296, -1.0649174596912374, 
-0.97083421311929197, -0.86716067172983902, -0.62038851443989262, 
-0.50561347756205421, -0.38883536304392674, -0.27002865572374146, 
-0.14916758733786950, -0.26226136520824922E-01, .98821971194736680E-01, 
.22600326337801757, .35534452070007827, .48687277693383724, .62061531895407107, 
.75659968673741496, .89485367336236088, 1.0354053250092612, 1.1782829409603224, 
1.3235150735996140, 1.4711305284130614, 1.6211583639884442, 1.7736278920154085, 
1.9285686772854511, 2.0860105376919300, 2.2459835442300609, 2.4085180209969153, 
2.5736445451914278, 2.7413939471143891, 2.9117973101684456, 3.0848859708581031, 
3.2606915187897236, 3.4392457966715355, 3.6205809003136125, 3.8047291786278983, 
3.9917232336281856, 4.1815959204301336, 4.3743803472512468, 4.5701098754108997, 
4.7688181193303265, 4.9705389465326135, 5.1753064776426996, 5.3831550863873900, 
5.5941193995953427, 5.8082342971970888, 6.0255349122249910, 6.2460566308132996, 
6.4698350921980960, 6.6969061887173353, 6.9273060658108250, 7.1610711220202390, 
7.3982380089891038, 7.6388436314627981, 7.8829251472885602, 8.1305199674155091, 
8.3816657558945735, 8.6364004298785915, 8.8947621596222319, 9.1567893684820376, 
9.4225207329163823, 9.6919951824855275, 9.9652518998515660, 10.242330320778464, 
10.523270134132062, 10.808111281880031, 11.096893959091915, 11.389658613939091, 
11.686445947694850, 11.987296914734273, 12.292252722534343, 12.601354831673888, 
12.914644955833595, 13.232165061796024, 13.553957369445556, 13.880064351768468, 
14.210528734852886, 14.545393497888767, 14.884701873167970, 15.228497346084168, 
15.576823655132927, 15.929724791911653, 16.287245001119629, 16.649428780557940, 
17.016320881129616, 17.387966306839502, 17.764410314794247, 18.145698415202475, 
18.531876371374551, 18.373407583023493, 18.744892715314990, 19.091529022485730, 
19.398204744539669, 19.652361577986795, 19.842518040457691, 19.960151188643124, 
20.000000556310056
};
static double Vy4 [ 17 ]= {
1.6999999999999993, 1.7027979890699996, 1.7110776170479802, 1.7245258889918809, 
1.7426256856204851, 1.7646598646851874, 1.7898265427562645, 1.8171129946147158, 
1.8428571428571434, 1.8687937736827926, 1.8967749516115102, 1.9231345371971944, 
1.9467763201756920, 1.9669541572284253, 1.9828325939152442, 1.9937419617298957, 
1.9992993146948947
};
static double Vx5 [ 20 ]= {
.0000000000000000, .75330474961529531E-02, .29824353590714753E-01, 
.66031239593527524E-01, .11476146128592268, .17408425107550407, 
.24184069203609582, .31530421627038896, .38461538461538458, .46153846153846168, 
.53846153846153832, .61538461538461497, .68469578372961060, .75815930796390418, 
.82591574892449593, .88523853871407709, .93396876040647214, .97017564640928544, 
.99246695250384676, 1.0000000000000000
};
static double Vy5 [ 17 ]= {
1.6999999999999993, 1.7027979890699996, 1.7110776170479802, 1.7245258889918809, 
1.7426256856204851, 1.7646598646851874, 1.7898265427562645, 1.8171129946147158, 
1.8428571428571434, 1.8687937736827926, 1.8967749516115102, 1.9231345371971944, 
1.9467763201756920, 1.9669541572284253, 1.9828325939152442, 1.9937419617298957, 
1.9992993146948947
};
static double Vx6 [ 20 ]= {
.0000000000000000, .75330474961529531E-02, .29824353590714753E-01, 
.66031239593527524E-01, .11476146128592268, .17408425107550407, 
.24184069203609582, .31530421627038896, .38461538461538458, .46153846153846168, 
.53846153846153832, .61538461538461497, .68469578372961060, .75815930796390418, 
.82591574892449593, .88523853871407709, .93396876040647214, .97017564640928544, 
.99246695250384676, 1.0000000000000000
};
static double Vy6 [ 40 ]= {
.50000000000000000, .50356107699818153, .51409878533379239, .53121476780784960, 
.55425087260789052, .58229437323569311, .61432469078070007, .64905290223691114, 
.68181818181818166, .71818181818181781, .75454545454545485, .79090909090909101, 
.82727272727272716, .86363636363636331, .90000000000000036, .93636363636363651, 
.97272727272727266, 1.0090909090909097, 1.0454545454545450, 1.0818181818181820, 
1.1181818181818191, 1.1545454545454543, 1.1909090909090914, 1.2272727272727266, 
1.2636363636363637, 1.3000000000000007, 1.3363636363636360, 1.3727272727272730, 
1.4090909090909083, 1.4454545454545453, 1.4818181818181824, 1.5181818181818176, 
1.5509470977630890, 1.5856753092193010, 1.6177056267643071, 1.6457491273921097, 
1.6687852321921515, 1.6859012146662078, 1.6964389230018178, 1.6999999999999993
};
static double Vx7 [ 60 ]= {
.0000000000000000, .82247814183373201E-03, .32711707587513954E-02, 
.72961622120137071E-02, .12807480769521498E-01, .19664684878409028E-01, 
.27698823081422741E-01, .36657933726714886E-01, .45352326007332966E-01, 
.55282138904670930E-01, .65513515354532648E-01, .76053158229517304E-01, 
.86907883055544977E-01, .98084618011856972E-01, .10959040393101516, 
.12143239429890273, .13361785525472381, .14615416559100347, .15904881675358795, 
.17230941284164403, .18594367060766004, .19995941945744500, .21436460145012903, 
.22916727129816294, .24437559636731909, .25999785667669029, .27604244489869068, 
.29251786635905530, .30943273903684032, .32679579356442279, .34461587322750020, 
.36290193396509229, .38166304436953835, .40090838568649989, .42064725181495888, 
.44088904930721817, .46164329736890153, .48291962785895404, .50472778528964213, 
.52707762682655179, .54997912228859214, .57344235414799183, .59747751753029998, 
.62209492021438884, .64730498263244840, .67311823786999359, .69954533166585797, 
.72659702241219559, .75428418115448181, .78261779159151601, .81160895007541356, 
.84126886561161474, .86857593045082560, .89813099268938767, .92595691436868410, 
.95077239619515552, .97147728427734759, .98705021247965163, .99671868778078920, 
1.0000000000000000
};
static double Vy7 [ 40 ]= {
.50000000000000000, .50356107699818153, .51409878533379239, .53121476780784960, 
.55425087260789052, .58229437323569311, .61432469078070007, .64905290223691114, 
.68181818181818166, .71818181818181781, .75454545454545485, .79090909090909101, 
.82727272727272716, .86363636363636331, .90000000000000036, .93636363636363651, 
.97272727272727266, 1.0090909090909097, 1.0454545454545450, 1.0818181818181820, 
1.1181818181818191, 1.1545454545454543, 1.1909090909090914, 1.2272727272727266, 
1.2636363636363637, 1.3000000000000007, 1.3363636363636360, 1.3727272727272730, 
1.4090909090909083, 1.4454545454545453, 1.4818181818181824, 1.5181818181818176, 
1.5509471454468038, 1.5856753569030158, 1.6177056744480236, 1.6457491750758244, 
1.6687852798758662, 1.6859012623499225, 1.6964389706855343, 1.7000000476837158
};
static double tablef1 [ 1400 ]= {
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
.99980721622909974, .99979787351837557, .99977927901906039, .99975184063617184, 
.99971602437137730, .99967261729077439, .99962289543500615, .99956810379820860, 
.99951597874238918, .99945654372048820, .99939518405247563, .99933190150168227, 
.99927297833405504, .99920724163368102, .99914387312195985, .99908536615467214, 
.99903464138905762, .99899415975094474, .99896587327746222, .99895138890697566, 
.99923175236050044, .99919464561891647, .99912081605676484, .99901192670444505, 
.99886988675355703, .99869788784557390, .99850105752062224, .99828438801804564, 
.99807848195979887, .99784395360573430, .99760210742390210, .99735296949374386, 
.99712125004830987, .99686302044509301, .99661437138966325, .99638503523887456, 
.99618638442388874, .99602796667118831, .99591733416317718, .99586070314336739, 
.99828493509947391, .99820255111350531, .99803871766319041, .99779728112012522, 
.99748268843611076, .99710225764555749, .99666757995605137, .99618990076834046, 
.99573671685583975, .99522142050912743, .99469100274124733, .99414557938687231, 
.99363916323690793, .99307577528951996, .99253422618564802, .99203553422968904, 
.99160417000683687, .99126056398514351, .99102080750027577, .99089814423636924, 
.99698876111795276, .99684520152391909, .99655990274111772, .99613992559656939, 
.99559349888895554, .99493390237632873, .99418179567895493, .99335712312956792, 
.99257646055794369, .99169077014928442, .99078120406237957, .98984806823839611, 
.98898356714780056, .98802389106771005, .98710341921524714, .98625747866222524, 
.98552701853987912, .98494599119449366, .98454099657149319, .98433392877061410, 
.99537676730489955, .99515841728647547, .99472483934389777, .99408742799546290, 
.99325956672328153, .99226239263080318, .99112812766122493, .98988768550044759, 
.98871646512106948, .98739109660852598, .98603364419806439, .98464470511222946, 
.98336113797459845, .98193974775097903, .98057975319010127, .97933265909800760, 
.97825788809213809, .97740433017003536, .97681006376407198, .97650644047502655, 
.99348579560888828, .99318151414427269, .99257786064008269, .99169174218889089, 
.99054315417312666, .98916298058835617, .98759730321295303, .98589001316842373, 
.98428253834160062, .98246858807296034, .98061608857983618, .97872599315047548, 
.97698392407339263, .97505978717653174, .97322348174482254, .97154352827519530, 
.97009861599105141, .96895296603499292, .96815629355579635, .96774955506440552, 
.99137659739233508, .99097873046177076, .99019019978705680, .98903454366593113, 
.98753975413984263, .98574813655529203, .98372147969479085, .98151816924112056, 
.97944971621913879, .97712227847115918, .97475236942979038, .97234130020818721, 
.97012498617435239, .96768336311595371, .96535912963038095, .96323767355496226, 
.96141663241654018, .95997505656225401, .95897377386448912, .95846293792349035, 
.98933680943547764, .98885066999645377, .98788808155487640, .98647941564921815, 
.98466092652209358, .98248640013552446, .98003294979939282, .97737289990726417, 
.97488219034063561, .97208679892676386, .96924781169257579, .96636681142943337, 
.96372470956923817, .96082056041039621, .95806214194076300, .95554936755367337, 
.95339608297197742, .95169382368739808, .95051265979568333, .94991041961563494, 
.98700221892280460, .98641769559341785, .98526147169116751, .98357217120523843, 
.98139603365447403, .97880038215601717, .97587991376533101, .97272271720032721, 
.96977471040749430, .96647501398675617, .96313301416813246, .95975049310030347, 
.95665597839310845, .95326245184961300, .95004653671845474, .94712295491547527, 
.94462198056667102, .94264761013229637, .94127902809256880, .94058166169415414, 
.98459696058044377, .98391397166215044, .98256432025759222, .98059553443725622, 
.97806461985800119, .97505316461563885, .97167392331412317, .96803093706357157, 
.96463831974845560, .96085068518520922, .95702437295191767, .95316123749375026, 
.94963503144385986, .94577645819948541, .94212753258127213, .93881651955756473, 
.93598863992480108, .93375905066882248, .93221499964958165, .93142866906492172, 
.98211964400358487, .98133824620270782, .97979563499728928, .97754885411308223, 
.97466639518416343, .97124475766883311, .96741513791748912, .96329763934587687, 
.95947278746272946, .95521295626418223, .95092009672450195, .94659599821483242, 
.94265740473250426, .93835626959333096, .93429679465010895, .93061965795355162, 
.92748371956274356, .92501416645852352, .92330540592828669, .92243564993813754, 
.97956891084545394, .97868930233429741, .97695446232228600, .97443151821074026, 
.97120110466103904, .96737519961333962, .96310374262198728, .95852294131679083, 
.95427795181503383, .94956110620136325, .94481863530505850, .94005212240011715, 
.93571921065544961, .93099640390502580, .92654710535166807, .92252339196036193, 
.91909660156872708, .91640096052178943, .91453725123808993, .91358908647243398, 
.97694344004667077, .97596596286197679, .97403989074912367, .97124295613837386, 
.96766852961835037, .96344455775190496, .95873994833083498, .95370699591497510, 
.94905371388442283, .94389453298175596, .93871864662381010, .93352729204051943, 
.92881704691966216, .92369206904394086, .91887217072615868, .91451991347893680, 
.91081807493031319, .90790904681313678, .90589930109870465, .90487730139632361, 
.97424195302754879, .97316709478318830, .97105105425800209, .96798264089078201, 
.96406848897720632, .95945292923663084, .95432399212334129, .94884999032744588, 
.94380003230828302, .93821274027016088, .93261896991013626, .92701947690641762, 
.92194792422197525, .91643905475080667, .91126646963722402, .90660238723789610, 
.90264009250593868, .89952936692856977, .89738176887014387, .89629012862081103, 
.97146321880476316, .97029161370536843, .96798713556018379, .96465009113674705, 
.96040084050172747, .95540044189310092, .94985613760059551, .94395214483349399, 
.93851691889817734, .93251532635911361, .92651860368185235, .92052689630504814, 
.91510920803080387, .90923364630409764, .90372513450314629, .89876479589339997, 
.89455558122049339, .89125396979777283, .88897607275132562, .88781865818222705, 
.96860605899699392, .96733848814852652, .96484736925993952, .96124487323171859, 
.95666548200966717, .95128725502145317, .94533667524791665, .93901371186125282, 
.93320443495629590, .92680197492345429, .92041668749561145, .91404798766838635, 
.90829857115594803, .90207255480449455, .89624385565060327, .89100181658619282, 
.88655829157074217, .88307583766892306, .88067464432988263, .87945503555005355, 
.96566935268760057, .96430674370940872, .96163104489877338, .95776660315075446, 
.95286235253906437, .94711356017158455, .94076592280600968, .93403497521544321, 
.92786268815938011, .92107244722447668, .91431248668014253, .90758138049661763, 
.90151395469531914, .89495286020649956, .88881880367742416, .88330872130799343, 
.87864267671168683, .87498874677781069, .87247077564409381, .87119230149014482, 
.96265204111592806, .96119546706654868, .95833750987102839, .95421494833691312, 
.94899143346789572, .94287958188968390, .93614422564389321, .92901624944285643, 
.92249182990531908, .91532657548791940, .90820537946083546, .90112587455196003, 
.89475353557230231, .88787196429098769, .88144656575207547, .87568129557407792, 
.87080379416842624, .86698715444226859, .86435849550986255, .86302426270554478, 
.95955313217234472, .95800380980808975, .95496617220121305, .95058962946054337, 
.94505274958363206, .93858557843329649, .93147195712754260, .92395787930831230, 
.91709205303899088, .90956425724019507, .90209484601993739, .89468042146211624, 
.88801569931904734, .88082755148773972, .87412409284211723, .86811577136715645, 
.86303722510353520, .85906610658796800, .85633246842120048, .85494538618953686, 
.95637170467548294, .95473099206669387, .95151650317452341, .94689042208513896, 
.94104637009956016, .93423184245212998, .92674951897889746, .91886023935864880, 
.91166358989025331, .90378545043237091, .89598045913687374, .88824410908526163, 
.88129901707890368, .87381755596622035, .86684865561452340, .86060877134479874, 
.85533900738067170, .85122116127878666, .84838791109524503, .84695071209343153, 
.95310691241332490, .95137630594810307, .94798803981305280, .94311715823562559, 
.93697240961469297, .92981870163182512, .92197734162044620, .91372373355607839, 
.90620671056961355, .89799016921570984, .88986187612830925, .88181614813093279, 
.87460222603391280, .86684013378180624, .85961780728989468, .85315726203505138, 
.84770557959431403, .84344832493120558, .84052052297350610, .83903578122451572, 
.94975798793272037, .94793911874198367, .94438038719110307, .93926972786512619, 
.93283102901407700, .92534651929792666, .91715588450106666, .90854879496526519, 
.90072172147697471, .89217848025897428, .88373883186457269, .87539586063804664, 
.86792421263615172, .85989363913619432, .85242935212621918, .84575851427607596, 
.84013373390590329, .83574399868717419, .83272642788027473, .83119657422939863, 
.94632424606461085, .94441887590555185, .94069322058383509, .93534808021642835, 
.92862243630630270, .92081569497731741, .91228563639918470, .90333588548088495, 
.89520896398664251, .88635049951820499, .87761113268283819, .86898266999243445, 
.86126399815186638, .85297660401338338, .84528131849946653, .83841006954677622, 
.83262057601791251, .82810493300034382, .82500212468686962, .82342946020174712, 
.94280508717468048, .94081510381217426, .93692628744424722, .93135222507454407, 
.92434688739504045, .91622666491441018, .90736711569963990, .89808549558439388, 
.88966881327798064, .88050638938581371, .87147865105144107, .86257609222864351, 
.85462072612652296, .84608772060582382, .83817193576726789, .83110971112741072, 
.82516349098248620, .82052818892056223, .81734444530914996, .81573115295822962, 
.93920000013134619, .93712741225865037, .93307940920414700, .92728223390692455, 
.92000468678146019, .91157990253943189, .90240087064092478, .89279814412032465, 
.88410167728608502, .87464635615859720, .86534132086634108, .85617572840917544, 
.84799365145532268, .83922582606250462, .83109961426833179, .82385543924993998, 
.81776011381615366, .81301110488421202, .80975051872428405, .80809867360417675, 
.93550856498497659, .93335549672625817, .92915248289540775, .92313824088807261, 
.91559618819445043, .90687591888619612, .89738747952971509, .88747437808375640, 
.87850799575089167, .86877064777450064, .85919913328182385, .84978125791229964, 
.84138213080321922, .83238988918131018, .82406292793811053, .81664544956665286, 
.81040830410146869, .80555126806305200, .80221773996454526, .80052931829648655, 
.93173045535397803, .92949914039186121, .92514548258837315, .91892044380547777, 
.91112179514562353, .90211526295681754, .89232755091982419, .88211477241169700, 
.87288823934642124, .86287955177618736, .85305213299433935, .84339243248960272, 
.83478561416520936, .82557899873866958, .81706059912046547, .80947811439602579, 
.80310612391758662, .79814648951236122, .79474374325418728, .79302063033029591, 
.92786544051496855, .92555821588649179, .92105846064481989, .91462910484400450, 
.90658196140619296, .89729852203091065, .88722172375292152, .87671992977204716, 
.86724290887459043, .85697339346616086, .84690041491190293, .83700907097839405, 
.82820363739471359, .81879235320520038, .81009148523284735, .80235196630651373, 
.79585181856708964, .79079478250510693, .78732637861671229, .78557037584534606, 
.92391338719565497, .92153268679982325, .91689154878339529, .91026455124607519, 
.90197719140292776, .89242632191690197, .88207066745854779, .87129048034460155, 
.86157253451026283, .85105253422365690, .84074412115243724, .83063105457333919, 
.82163581555745191, .81202925164026729, .80315456700395049, .79526568367904238, 
.78864379966466502, .78349434355304126, .77996369140583810, .77817652258023640, 
.91987426107028714, .91742260892985428, .91264495895594089, .90582717584522054, 
.89730804053051871, .88749932714319613, .87687508201113473, .86582708158921429, 
.85587767508609636, .84511736995802289, .83458343832336390, .82425832257727738, 
.81508183699203984, .80528908559376755, .79624893805175700, .78821807795251597, 
.78148063023101066, .77624353570578219, .77265390431279279, .77083722120693865, 
.91574812795869232, .91322813127797264, .90831898403359068, .90131743747082338, 
.89257511537785950, .88251824108705579, .87163569794192519, .86033041799684851, 
.85015891740733140, .83916832967704502, .82841859504210358, .81789086856389515, 
.80854145797783905, .79857133187199580, .78937379560918230, .78120808230747907, 
.77436101149736170, .76904087379066144, .76539540148147966, .76355078886066474, 
.91153515473090863, .90894949679033132, .90391399830200059, .89673586122214377, 
.88777907386591692, .87748380603919185, .86635327630385672, .85480120081973943, 
.84441687558800216, .83320587415179725, .82224985966319819, .81152873689534499, 
.80201449792634594, .79187554604760202, .78252843223629409, .77423474158399830, 
.76728377117583946, .76188501131361930, .75818671442622576, .75631569454635272, 
.90723560992036356, .90458704284719949, .89943045776552966, .89208303861000537, 
.88292062529506843, .87239680320221213, .86102860858765196, .84924016777736477, 
.83865219040120875, .82723049466220577, .81607753818281981, .80517201954650930, 
.79550083502534397, .78520135661258494, .77571222838391618, .76729720326302486, 
.76024785299111586, .75477472878783736, .75102650949775862, .74913054615635211, 
.90284986404938428, .90014120150262056, .89486890026065358, .88735962756482589, 
.87800053030000036, .86725805262123679, .85566251658753656, .84364808273532066, 
.83286552863707186, .82124271180973873, .80990197229564331, .79882085319475382, 
.78900040227571966, .77854845968881425, .76892464569458607, .76039470936758313, 
.75325230730167370, .74770892329451144, .74391357668437852, .74199407887712210, 
.89837838967160555, .89561249947735788, .89022994537935463, .88256635230999803, 
.87301960071049889, .86206841304516990, .85025585221518363, .83802573535458291, 
.82705758246285388, .81524307438550636, .80372353758260939, .79247541653997611, 
.78251318386971891, .77191661422343538, .76216522094425321, .75352658916236426, 
.74629628266580150, .74068659911098655, .73684681956903830, .73490514479803992, 
.89382176113656164, .89100155790871716, .88551429420369787, .87770400309997254, 
.86797869931672211, .85682878171730814, .84480949726066978, .83237394070897119, 
.82122906878048307, .80923215828367512, .79754264181110501, .78613592782476727, 
.77603921186683078, .76530563760719539, .75543356054253108, .74669225254866101, 
.73937901822957786, .73370685926682611, .72982524629075418, .72786270356437455, 
.88918065408241020, .88630909186141926, .88072272885326375, .87277343582274547, 
.86287873953780814, .85154009409416387, .83932436309941150, .82669353886895469, 
.81538072857737232, .80321056545147385, .79135972333163163, .77980264252871845, 
.76957856312969197, .75871540166265272, .74872933552131471, .73989118406680510, 
.73249983683240649, .72676889790932497, .72284796138164709, .72086581394037508, 
.88445584466337590, .88153590960426886, .87585611184758694, .86777557146683548, 
.85772068499296761, .84620332349160599, .83380139034424905, .82098539445023799, 
.80951332626702510, .79717892286827696, .78517524955721940, .77347585121449303, 
.76313135648766073, .75214582895673232, .74205227695162679, .73312293743098433, 
.72565813874091545, .71987199337739971, .71591415836991701, .71391362616825882, 
.87964820851907810, .87668291165790424, .87091538528621282, .86271139545321174, 
.85250554897450359, .84081948065763568, .82824154844203346, .81525039612584349, 
.80362764901643358, .79113788154728726, .77898971551370311, .76715587750632552, 
.75669775010013562, .74559688939839774, .73540217173698064, .72638713053194970, 
.71885339593476572, .71301550189730756, .70902311305488441, .70700537502540450, 
.87475871949347728, .87175108961944647, .86590156984946498, .85758195683304062, 
.84723439382251964, .83538961327136063, .82264583521427848, .80948945610067857, 
.79772450605775880, .78508811555425539, .77280364245060240, .76084307618421987, 
.75027793899548623, .73906859708757533, .72877885873868575, .71968344085209923, 
.71208514687863467, .70619885182585029, .70217417737351784, .70014037349590941, 
.86978844811168943, .86674152477037791, .86081576362348489, .85238836735251844, 
.84190833020139622, .82991480536796403, .81701527634163840, .80370350954782732, 
.79180472798220436, .77903032103847436, .76661757650373308, .75453783137935759, 
.74387215276467689, .73256100738600705, .72218222519454500, .71301160124499263, 
.70535299172366184, .69942153837698218, .69536677378903322, .69331800698430979, 
.86473855982342940, .86165538647447748, .85565914075358240, .84713180038647973, 
.83652851627845337, .82439617668972942, .81135092479218407, .79789351400605735, 
.78586916601439682, .77296521527199979, .76043208740190527, .74824055485816277, 
.73748065339145186, .72607421418461160, .71561220339754728, .70637139603779286, 
.69865658788931151, .69268311877644118, .68860039014160657, .68653772800911350, 
.85961031302230484, .85649393037311838, .85043294993039698, .84181348974288017, 
.83109615680555204, .81883488196342713, .80565386019365282, .79206044873826187, 
.77991869126593338, .76689353569367458, .75424776721111919, .74195168438413805, 
.73110373320331712, .71960834734525070, .70906876760553894, .69976265742060562, 
.69199564598310492, .68598320779639355, .68187457490924230, .67979905132213259, 
.85440505685063628, .85125849638569717, .84513851271383977, .83643472834068222, 
.82561250210473380, .81323211010463869, .79992518815006453, .78620531405079186, 
.77395419396710008, .76081603895510042, .74806522911060769, .73567168214804679, 
.72474171292963152, .71316357029768263, .70255193115662939, .69318526209138354, 
.68536992602124269, .67932147362838369, .67518893283368298, .67310154940671729, 
.84912422879989358, .84595050652340709, .83977722170023550, .83099686676408846, 
.82007884695934785, .80758908334986035, .79416603950330544, .78032913057385156, 
.76797658267606617, .75473349996620542, .74188510619589720, .72940103325828487, 
.71839493985490066, .70674007777495051, .69606174376832808, .68663912812908590, 
.67877923391789763, .67269763405826533, .66854312087207124, .66644484831406614, 
.84376935211722659, .84057146252498538, .83435053853852359, .82550131169648078, 
.81449652941245421, .80190705631748860, .78837756954048954, .77443293850334005, 
.76198678346513937, .74864671093850410, .73570805030478048, .72314024428439005, 
.71206378605692211, .70033809367260269, .68959828900120135, .68012421207124729, 
.67222341821504661, .66611145291140450, .66193684444008516, .65982862380198759, 
.83834203302893806, .83512294332347137, .82885999180181635, .81994952423784406, 
.80886692947465166, .79618731499906503, .78256095714812657, .76851779680473575, 
.75598573908394373, .74255648042456557, .72953473086274834, .71688984184762439, 
.70574864672079496, .69395786901900824, .68316168187027027, .67364050617511140, 
.66570236702825136, .65956273674044086, .65536985391658265, .65325259774497868, 
.83284395779206755, .82960660235337691, .82330717472101134, .81434301810984355, 
.80319146774380668, .79043117568241950, .77671740391432309, .76258478237981098, 
.74997440809961624, .73646363235256418, .72336583374498975, .71065037125340835, 
.69944993852098769, .68759968004564920, .67675206658946419, .66718803584407305, 
.65921600518683965, .65305133173131724, .64884194138349616, .64671653478834223, 
.82727688958556522, .82402416470802853, .81769374278756013, .80868335775312583, 
.79747160393951033, .78463998380861488, .77084813318045953, .75663498919716310, 
.74395376401435698, .73036900505513436, .71720206015259158, .70442239516115801, 
.69316809806470958, .68126382634769767, .67036961443628806, .66076685720343431, 
.65276429154959748, .64657712080628293, .64235293757795597, .64022023922241533, 
.82164266525278984, .81837742415714432, .81202141123285554, .80297215632178076, 
.79170883535540759, .77881511276485160, .76495438904397650, .75066952738772497, 
.73792479436087610, .72427345029204260, .71104412550101292, .69820649228773191, 
.68690358039073596, .67495062912641779, .66401452172544917, .65437705481143316, 
.64634721647937421, .64014002090514777, .63590270903639401, .63376355205587509, 
.81594319190730702, .81266824003501126, .80629195239205476, .79721107358027088, 
.78590469523287010, .77295796261574889, .75903743531410928, .74468952230659979, 
.73188849977648651, .71817783226648124, .70489275831932841, .69200325614129943, 
.68065685751865967, .66866042950603266, .65768700788159962, .64801873949322997, 
.63996479946202289, .63373998042833246, .62949115541283196, .62734634826962399, 
.81018044341414752, .80689853400986444, .80050719296046668, .79140181370846374, 
.78006075105978334, .76706995877565387, .75309855442258944, .73869611356272757, 
.72584589305676961, .71208302663502310, .69874869915908833, .68581329378295708, 
.67442841704423862, .66239358691864358, .65138731360256774, .64169204628702503, 
.63361708685683804, .62737697682721449, .62311820695565845, .62096853423494169, 
.80435645675885048, .80107028674530145, .79466901115092903, .78554612302073201, 
.77417860279851658, .76115255062487219, .74713904629152184, .73269045401805311, 
.71979799818992085, .70598991951151902, .69261269951197757, .67963722461389664, 
.66821876077715092, .65615047755164080, .64511569910553412, .63539713249280172, 
.62730414976719906, .62105101432899745, .61678382212906790, .61463004528152743, 
.79847332831672324, .79518553446473317, .78877933376084886, .77964578760537306, 
.76825988104741183, .75520721007293123, .74116022716081398, .72667370875800996, 
.71374584937303487, .69989940646541271, .68648552073574065, .67347567918633100, 
.66202840341801572, .64993149285278073, .63887244244955210, .62913417581534170, 
.62102608202146448, .61476212178483192, .61048798536694449, .60833084340272825, 
.79253321003483213, .78924636543002158, .78284013315781409, .77370263089087044, 
.76230624514039313, .74923543007220161, .73516342837769788, .72064705403527773, 
.70769049001173634, .69381239151514285, .68036793298810283, .66732929803676133, 
.65585787127203432, .64373703808876237, .63265783792864594, .62290337259416817, 
.61478299825534921, .60851035063123893, .60423070494840425, .60207091508672850, 
.78653830553926707, .78325491634555044, .77685342419286496, .76771851114576439, 
.75631938118952857, .74323872308539640, .72914999515104262, .71461167618889676, 
.70163297170469852, .68772978611746349, .67426071416864841, .66119873054048639, 
.64970770099704145, .63756753095371299, .62647219453045511, .61670493611397403, 
.60857503208805308, .60229577295605152, .59801201098545442, .59585026926377227, 
.78049086618021257, .77721336869903801, .77082126105066862, .76169531891910069, 
.75030100007559397, .73721861951065171, .72312128527328989, .70856877054094303, 
.69557435321470718, .68165250815366318, .66816464886881954, .65508463378654735, 
.64357843838413287, .63142340022450139, .62031583445603655, .61053909498986814, 
.60240233438532242, .59611847966110532, .59183195351432871, .58966893536062537, 
.77439318702730942, .77112394505041770, .76474573404595747, .75563497442862015, 
.74425283539188514, .73117666606805876, .71707866781298268, .70251954027307317, 
.68951569942803792, .67558148091379866, .66208052733038569, .64898767147255221, 
.63747063716938468, .62530508446024780, .61418909169701341, .60440609162247172, 
.59626507160343512, .58997857871481407, .58569060068302414, .58352696145449001, 
.76824760282768723, .76498890528008534, .75862896637565991, .74953942490398517, 
.73817664134668581, .72511442415166105, .71102352178096462, .69646519528534334, 
.68345808030400623, .66951763208016823, .65600914441288483, .64290851281904826, 
.63138485787546650, .61921303074379264, .60809231066675140, .59830618071850306, 
.59016342420880430, .58387619348855491, .57958803702841688, .57742441251946441, 
.76205648393891867, .75881054280775229, .75247311083619683, .74341064189246930, 
.73207419062995083, .71903346815106906, .70495723477344108, .69040695103979277, 
.67740256981663716, .66346189271135803, .64995129857068445, .63684783150330604, 
.62532166668221989, .61314769346323739, .60202584488269628, .59223962787302942, 
.58409758516852450, .57781146117148885, .57352436183708644, .57136136875942278, 
.75582223224895340, .75259118079302401, .74628034651540653, .73725061853460483, 
.72594727224987576, .71293538374694843, .69888120159517053, .68434602739134964, 
.67135024489046335, .65741519622827227, .64390779084042282, .63080630461254594, 
.61928163432549610, .60710953313197269, .59599005569738706, .58620670821003318, 
.57806775850773373, .57178453125905371, .56749968758463776, .56533792402187155, 
.75018241864562651, .74696650894177452, .74068295870806677, .73168734933636170, 
.72041937613982410, .70743950528619648, .69341129026923543, .67889558692100460, 
.66591242310819132, .65198731631103446, .63848718927570458, .62539149398397742, 
.61387149905788019, .60170442143966241, .59058998294494858, .58081166951410024, 
.57267747423535942, .56639833544793938, .56211651306976695, .55995635047856307, 
.74414998206226768, .74095193140515681, .73470077612329665, .72574615613658056, 
.71452145362459252, .70158178895651013, .68758740621667425, .67309837593031019, 
.66013358894146068, .64622400617186027, .63273609250210205, .61965061953996559, 
.60813897093830349, .59598061653412060, .58487447683927929, .57510399725838979, 
.56697685161795131, .56070355909856040, .55642595226181379, .55426799228572199, 
.73855799580067194, .73537795549946083, .72915964498829422, .72024697246467806, 
.70906717149540921, .69617004388867298, .68221238730331801, .66775326711128180, 
.65480984495172001, .64091899224376836, .62744632902530639, .61437390422715055, 
.60287295759828996, .59072565662509247, .57962978479468896, .56986873385878822, 
.56174985987813952, .55548328880513453, .55121045906785759, .54905497392209801, 
.73364620408335735, .73048308021710306, .72429574489410342, .71542290841271061, 
.70428618421745892, .69143042900358065, .67750912390760409, .66308023696717955, 
.65015889154774531, .63628780946373542, .62283154600226009, .60977331280285818, 
.59828403173760258, .58614870538002806, .57506381757097891, .56531270109631671, 
.55720241670682069, .55094274179199010, .54667477043966550, .54452178896944492, 
.72959534268988857, .72644690745255092, .72028654925986713, .71144858522249266, 
.70034984881696549, .68753093474492355, .67364239207056242, .65924113623671632, 
.64634025374915016, .63248773887566856, .61904704813235478, .60600237944756397, 
.59452426204876974, .58240035378024968, .57132586183185741, .56158406254218818, 
.55348176234781055, .54722844924425929, .54296493477109486, .54081424225242192, 
.72657691131230318, .72343983823472735, .71730039259861367, .70848955950603454, 
.69742052998466775, .68463061919383728, .67076807888196777, .65638895497292260, 
.64350461156187300, .62966722710719358, .61623931267106757, .60320581656777106, 
.59173690070842611, .57962238299057511, .56855640215649450, .55882218811765960, 
.55072635176839169, .54447816446731301, .54021822811899534, .53806936921508319, 
.72471712321437931, .72158722325950830, .71546099828699627, .70666735676782200, 
.69561721250438791, .68284581714404946, .66899996009709406, .65463511579591060, 
.64176150142307187, .62793397983298860, .61451442657422306, .60148825868937694, 
.59002538057955867, .57791701674349294, .56685659875920547, .55712732614313476, 
.54903568682189319, .54279081828833908, .53853319368106756, .53638551787046862, 
.72408974202511756, .72096229098605047, .71484058261365890, .70605282091171218, 
.69500914595879737, .68224410606279418, .66840398937180912, .65404407143385646, 
.64117416794012194, .62735006534880844, .61393341580547045, .60090979555088708, 
.58944901681370432, .57734279146200962, .56628430421783382, .55655674404319522, 
.54846655696383562, .54222283525179071, .53796600867800259, .53581874102836080
};
static double tabled1 [ 1400 ]= {
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
.99980922912917836, .99981583309268696, .99982861008423296, .99984661173313005, 
.99986867607888208, .99989340700502061, .99991926516262153, .99994499018420868, 
.99996701824876966, .99998949544413085, 1.0000100152542795, 1.0000285780410891, 
1.0000436898301583, 1.0000582795548432, 1.0000702538882305, 1.0000796225302369, 
1.0000865193311734, 1.0000912511131292, 1.0000941677571060, 1.0000955407423047, 
.99923976243125068, .99926610743368449, .99931707371864820, .99938887004051880, 
.99947685179031021, .99957544145677890, .99967849395965069, .99978098126654480, 
.99986870940610673, .99995819320692159, 1.0000398510072499, 1.0001136885811932, 
1.0001737719615240, 1.0002317513360433, 1.0002793108759818, 1.0003164993748563, 
1.0003438594692353, 1.0003626197250937, 1.0003741774441708, 1.0003796161458078, 
.99830277539634593, .99836169169764477, .99847565247426373, .99863614995886985, 
.99883276336115800, .99905299030598138, .99928307344501344, .99951176876729764, 
.99970741867320756, .99990686489000991, 1.0000887478983511, 1.0002530966680230, 
1.0003867346355948, 1.0005155923385192, 1.0006211991439735, 1.0007037000035901, 
1.0007643388618608, 1.0008058788007703, 1.0008314494273989, 1.0008434751537398, 
.99701998458319141, .99712367300311455, .99732419448521381, .99760650530318062, 
.99795218437330091, .99833915935054751, .99874318444805066, .99914447417612795, 
.99948751778798062, .99983693875803237, 1.0001553099798886, 1.0004427213979454, 
1.0006762038889534, 1.0009011043440712, 1.0010852117167266, 1.0012288630641455, 
1.0013343161460904, 1.0014064674970973, 1.0014508338123649, 1.0014716831857655, 
.99542451602241078, .99558417724516612, .99589286421630430, .99632727674649968, 
.99685889523113025, .99745360313597498, .99807400635357990, .99868965015963429, 
.99921544832081066, .99975050628788464, 1.0002375056390893, 1.0006766597043699, 
1.0010330100732403, 1.0013758475071899, 1.0016561195188896, 1.0018744921357756, 
1.0020345624967790, 1.0021439266502177, 1.0022110906158186, 1.0022426252673855, 
.99355276319651131, .99377849087405767, .99421477723607077, .99482845562673117, 
.99557895155695242, .99641782346506991, .99729211349801772, .99815879339658953, 
.99889821251452315, .99964983507457250, 1.0003331382296929, 1.0009485472588042, 
1.0014472934726415, 1.0019264856722612, 1.0023176428398171, 1.0026219326788881, 
1.0028446223845227, 1.0029965306100448, 1.0030896928664463, 1.0031333911361338, 
.99146479230822704, .99176472010752492, .99234422285393531, .99315889841000482, 
.99415446370281824, .99526625631403209, .99642379101968559, .99756995205490573, 
.99854670326854278, .99953841828049583, 1.0004388559743138, 1.0012487626982809, 
1.0019042702570786, 1.0025331956134889, 1.0030457755997733, 1.0034438702598023, 
1.0037347205426532, 1.0039327995925624, 1.0040541016219990, 1.0041109404827278, 
.98944532555507536, .98981751214873970, .99053639131253968, .99154646522644307, 
.99277993434081524, .99415621417574329, .99558771195120876, .99700363518603730, 
.99820899522818185, .99943149811334697, 1.0005401933071891, 1.0015362207651608, 
1.0023413962350332, 1.0031129352234498, 1.0037408546656608, 1.0042278038098551, 
1.0045830303649166, 1.0048245922631360, 1.0049723282467562, 1.0050414885477443, 
.98713374606750437, .98758922406949001, .98846864774098697, .98970354509442160, 
.99121034057956681, .99288995401717006, .99463503659983732, .99635909042934112, 
.99782503872118822, .99931007228557256, 1.0006551521233400, 1.0018619658821695, 
1.0028362674031459, 1.0037685840243529, 1.0045261949674371, 1.0051127822627328, 
1.0055399944010874, 1.0058300433958667, 1.0060071819171288, 1.0060900227561116, 
.98475192335936235, .98529385359539745, .98633978974942993, .98780759042996102, 
.98959709353104985, .99158987218535055, .99365804169534033, .99569887691816372, 
.99743216592319273, .99918596851158903, 1.0007725156029996, 1.0021941634798264, 
1.0033404544991018, 1.0044358878018702, 1.0053247345979288, 1.0060118708413643, 
1.0065115209097830, 1.0068502247574214, 1.0070567930412537, 1.0071533021087591, 
.98229846047861524, .98293008957083350, .98414866088927255, .98585763948028005, 
.98793943720637301, .99025540674108570, .99265633453975188, .99502274396866697, 
.99703023262098844, .99905914203539936, 1.0008923261094156, 1.0025329354945396, 
1.0038541475325984, 1.0051151077873954, 1.0061367997123325, 1.0069254526362710, 
1.0074980399184119, 1.0078856020822680, 1.0081216513053448, 1.0082318282197884, 
.97977199166273499, .98049664885036758, .98189412703507450, .98385274588143312, 
.98623662426083458, .98888599907545116, .99162952253833103, .99433043976943269, 
.99661909357037992, .99892954775391829, 1.0010146262531132, 1.0028784040577103, 
1.0043775357716600, 1.0058065022761626, 1.0069627102615439, 1.0078539005859051, 
1.0084999672468857, 1.0089366233041426, 1.0092022259264868, 1.0093260808246676, 
.97717118756335464, .97799228084387402, .97957507982977265, .98179198095377784, 
.98448791736123775, .98748109466267031, .99057721359659434, .99362171157451762, 
.99619860257436610, .99879714022500421, 1.0011394589078066, 1.0032306916373033, 
1.0049108081362821, 1.0065103274772085, 1.0078027814645778, 1.0087975796786677, 
1.0095177074529826, 1.0100037221601870, 1.0102989697473526, 1.0104365221347356, 
.97449476043826433, .97541577203410990, .97719044011438516, .97967443599839199, 
.98269259056745561, .98604014383115290, .98949901653128836, .99289630590455735, 
.99576861256648319, .99866187367840464, 1.0012668672204816, 1.0035899211471322, 
1.0054541535102071, 1.0072268381896681, 1.0086573249775406, 1.0097568486942026, 
1.0105516561522703, 1.0110873210205495, 1.0114123224436735, 1.0115636002442425, 
.97174146927209126, .97276595043724845, .97473916132524430, .97749922458721683, 
.98084993072588900, .98456260255197636, .98839454149512729, .99215396875595396, 
.99532897569915779, .99852370202927543, 1.0013968946166592, 1.0039562160303768, 
1.0060077609886362, 1.0079562883452970, 1.0095266498316007, 1.0107320615971609, 
1.0116022018695601, 1.0121878331407954, 1.0125427130727478, 1.0127077518348631, 
.96891012479014182, .97004168997922813, .97222023284305070, .97526548484003017, 
.97895923887237279, .98304793324396889, .98726340041506089, .99139444581814928, 
.99487954343695273, .99838257889401416, 1.0015295848016024, 1.0043297003221796, 
1.0065718200749578, 1.0086989314461527, 1.0104110631919416, 1.0117235686639352, 
1.0126697275327059, 1.0133056644757017, 1.0136905621227030, 1.0138694043487539, 
.96599959433329818, .96724191476328314, .96963268327928864, .97297238168164268, 
.97701983164199513, .98149560559444726, .98610520744378405, .99061748269896088, 
.99442016665460742, .99823845760820384, 1.0016649817580505, 1.0047104986949900, 
1.0071465208362873, 1.0094550209188495, 1.0113108709757863, 1.0127317174008597, 
1.0137546116887617, 1.0144412151540121, 1.0148562831769243, 1.0150489777525078, 
.96300880656537313, .96436560320654607, .96697558368674130, .97061910907296978, 
.97503104268318486, .97990509739454379, .98491957942413610, .98982282515793052, 
.99395069573978545, .99809129124650875, 1.0018031297409336, 1.0050987364894732, 
1.0077320540252557, 1.0102248104014913, 1.0122263783578953, 1.0137568532971803, 
1.0148572295013965, 1.0155948806871524, 1.0160402842772915, 1.0162468859813198, 
.95993675598780814, .96141179202607518, .96424805068198782, .96820489220985340, 
.97299222407331909, .97827589538794502, .98370613636597892, .98901021934760069, 
.99347098070044970, .99794103264439082, 1.0019440732694289, 1.0054945397331791, 
1.0083286111737824, 1.0110085539755809, 1.0131578901850586, 1.0147993204454007, 
1.0159779535736828, 1.0167670529663606, 1.0172429690484563, 1.0174635381297521, 
.95678250723995595, .95837958005701362, .96144924946882404, .96572898968370846, 
.97090274773298813, .97660749613175157, .98246450193507451, .98817941206260795, 
.99298087127678714, .99778763442154339, 1.0020878571166418, 1.0058980351486901, 
1.0089363846633064, 1.0118065063524233, 1.0141057113161154, 1.0158594620538994, 
1.0171171546297848, 1.0179581210894877, 1.0184647376302076, 1.0186993394392083, 
.95354519916634828, .95526813188779114, .95857839675253642, .96319069559823556, 
.96876200683594527, .97489940686805243, .98119430395342366, .98733015099644628, 
.99248021705759848, .99763104900695232, 1.0022345262971419, 1.0063093501526295, 
1.0095555677750085, 1.0126189230214764, 1.0150701469003622, 1.0169376208710874, 
1.0182752020813897, 1.0191684720491603, 1.0197059874539764, 1.0199546921204978, 
.95022404863529342, .95207668129931955, .95563476353586030, .96058934163659715, 
.96656941721165857, .97315114640469802, .97989517491045464, .98646218500572269, 
.99196886760106973, .99747122866550986, 1.0023841260525435, 1.0067286128466291, 
1.0101863547228445, 1.0134460603665523, 1.0160515026044792, 1.0180341395361900, 
1.0194524644990293, 1.0203984913069435, 1.0209671138914740, 1.0212299960412452, 
.94681835409583104, .94880453449700386, .95261767778828299, .95792429907359122, 
.96432441873727592, .97136224600364529, .97856675248438818, .98557526438169074, 
.99144667255982866, .99730812552611492, 1.0025367018352853, 1.0071559520011613, 
1.0108289406716586, 1.0142881757545901, 1.0170500847960171, 1.0191493608685980, 
1.0206493100041871, 1.0216485632728880, 1.0222485107974291, 1.0225256493014847, 
.94332749886252176, .94545107312609489, .94952652698111129, .95519498072748155, 
.96202647671571540, .96953225027514001, .97720868007303652, .98466914112881532, 
.99091348181018335, .99714169161120081, 1.0026922992907412, 1.0075914970329727, 
1.0114835217422300, 1.0151455276008146, 1.0180662006899173, 1.0202836281053280, 
1.0218661065948467, 1.0229190717058592, 1.0235505709638579, 1.0238420487159434, 
.93975095411974452, .94201575706244989, .94636076048142070, .95240084284626747, 
.95967508323649864, .96766071807589771, .97582060733322473, .98374356925008610, 
.99036914558542799, .99697187886763339, 1.0028509642377728, 1.0080353779767321, 
1.0121502950047654, 1.0160183754133878, 1.0191001584632973, 1.0214372850943008, 
1.0231032224146637, 1.0242104000470025, 1.0248736864998222, 1.0251795902178151, 
.93608828163915514, .93849812697217183, .94311989179864941, .94954138692329748, 
.95726975851586549, .96574722340934649, .97440219072795908, .98279830503875876, 
.98981351461308764, .99679863919893130, 1.0030127426478217, 1.0084877254514126, 
1.0128294584621043, 1.0169069798201109, 1.0201522673427899, 1.0226106764496974, 
1.0243610259740168, 1.0255229316963410, 1.0262182491479581, 1.0265386691959826, 
.93233913620572062, .93489780663485422, .93980350067819773, .94661616143725003, 
.95481005221262916, .96379135632589541, .97295309408039654, .98183310737617080, 
.98924644025596409, .99662192449875342, 1.0031776806226256, 1.0089486706218413, 
1.0135212110236855, 1.0178116025792925, 1.0212228376679591, 1.0238041476745408, 
1.0256398863296932, 1.0268570502406664, 1.0275846505469675, 1.0279196807754214, 
.92850326774932457, .93121450502631409, .93641123503695911, .94362476351163793, 
.95229554471617672, .96179272382110759, .97147298913360836, .98084773803524028, 
.98866777465682509, .99644168668560684, 1.0033458243706352, 1.0094183451557999, 
1.0142257524711689, 1.0187325065765604, 1.0223121809337306, 1.0250180452547550, 
1.0269401732287751, 1.0282131396394287, 1.0289732824476172, 1.0293230200487742, 
.92458052318035022, .92744801815773481, .93294281273624291, .94056684048913164, 
.94972584840296892, .95975095072956917, .96996155611506385, .97984196198922158, 
.98807737088657099, .99625787773872276, 1.0035172201822036, 1.0098968811769997, 
1.0149432834164729, 1.0196699558091058, 1.0234206098142868, 1.0262527167282469, 
1.0282622572213505, 1.0295915843741894, 1.0303845368884976, 1.0307490822657015, 
.92057084792890775, .92359823066909696, .92939802318806064, .93744209141614032, 
.94710060885785985, .95766568061217205, .96841848430470046, .97881554772525325, 
.98747508309569343, .99607044973504966, 1.0036919144036096, 1.0103844112142231, 
1.0156740052528843, 1.0206242153586367, 1.0245484381705046, 1.0275085107319839, 
1.0296065097459202, 1.0309927695662771, 1.0318188063367391, 1.0321982629855120, 
.91647428718851320, .91966511717565769, .92577672879124306, .93425026843326409, 
.94441950605654279, .95553657663445712, .96684347260538761, .97776826756220392, 
.98686076666882705, .99587935488731150, 1.0038699534099824, 1.0108810681468929, 
1.0164181200998117, 1.0215955513541306, 1.0256959810426916, 1.0287857770295892, 
1.0309733031907582, 1.0324170810665454, 1.0332764837980639, 1.0336709581976828, 
.91229098686604737, .91564874336705215, .92207886619434409, .93099117806740128, 
.94168225550542450, .95336332243360711, .96523623011453297, .97669989797228085, 
.98623427838217992, .99568454558307418, 1.0040513835771798, 1.0113869851473027, 
1.0171758307416878, 1.0225842309253301, 1.0268635546301410, 1.0300848665215969, 
1.0323630109339934, 1.0338649055205382, 1.0347579628998584, 1.0351675644141698, 
.90802119424076055, .91154926685935411, .91830444738276140, .92766468242149824, 
.93888860933524954, .95114562297162364, .96359647669553117, .97561021990583918, 
.98559547656361257, .99548597442476505, 1.0042362512526848, 1.0119022956197257, 
1.0179473405614701, 1.0235905221478121, 1.0280514762588031, 1.0314061312402083, 
1.0337760073647779, 1.0353366304118743, 1.0362636379504100, 1.0366884787368158, 
.90366525833593503, .90736693780115196, .91445356058798555, .92427070025814395, 
.93603835734482743, .94888320537218651, .96192394354770239, .97449901911879155, 
.98494422125512160, .99528359427058344, 1.0044246027255730, 1.0124271331366073, 
1.0187328534691451, 1.0246146939803538, 1.0292600643382188, 1.0327499243301446, 
1.0352126678875768, 1.0368326440862647, 1.0377939039769994, 1.0382340989026941, 
.89922363000760452, .90310209923537299, .91052637101736111, .92080920797344579, 
.93313132799127407, .94657581973865712, .96021837377332386, .97336608650199197, 
.98428037437746529, .99507735827624255, 1.0046164841956091, 1.0129616313720271, 
1.0195325738256082, 1.0256570161952527, 1.0304896383087110, 1.0341166000169792, 
1.0366733689113441, 1.0383533357582522, 1.0393491567451634, 1.0398048233098240, 
.89469686175545782, .89875518721924585, .90652312140322644, .91728024045788104, 
.93016738932324539, .94422323995067337, .95847952294032146, .97221121841193525, 
.98360379989666025, .99486721993747473, 1.0048119417415295, 1.0135059240326187, 
1.0203467063622567, 1.0267177593021788, 1.0317405185797230, 1.0355065135641743, 
1.0381584878251220, 1.0398990955024823, 1.0409297927611447, 1.0414010510253775, 
.89008560726172159, .89432673070540591, .90244413237077747, .91368389184109156, 
.92714644985373562, .94182526443676984, .95670715963914565, .97103421700208725, 
.98291436399205678, .99465313313323411, 1.0050110212885692, 1.0140601447861188, 
1.0211754560966171, 1.0277971944660922, 1.0330130264600978, 1.0369200212198924, 
1.0396684029614214, 1.0414703142310973, 1.0425362092592856, 1.0430231817782198, 
.88539062066446406, .88981735118776106, .89828980262449132, .91002031611788681, 
.92406845936912674, .93938171692046735, .95490106603233582, .96983489055413519, 
.98221193522569294, .99443505216952377, 1.0052137685752893, 1.0146244271877183, 
1.0220190282443058, 1.0288955934197155, 1.0343074840810083, 1.0383574801545512, 
1.0412034935485805, 1.0430673836586530, 1.0441688041759087, 1.0446716159374103, 
.88061275557234553, .88522776211630838, .89406060895343464, .90628972765303084, 
.92093340967129877, .93689244713728315, .95306103839524514, .96861305380842555, 
.98149638471261191, .99421293182377612, 1.0054202291197676, 1.0151989046043861, 
1.0228776281276080, 1.0300132283710066, 1.0356242143121910, 1.0398192483899826, 
1.0427641396531704, 1.0446906962558016, 1.0458279761110583, 1.0463467544780927, 
.87575296382841472, .88055876808566846, .88975710605628222, .90249240156272770, 
.91774133524976775, .93435733152015177, .95118688764638493, .96736852829283282, 
.98076758629181482, .99398672738970706, 1.0056304481852092, 1.0157837101373295, 
1.0237514610809468, 1.0311503719060575, 1.0369635406720792, 1.0413056847209752, 
1.0443507221134078, 1.0463406451928503, 1.0475141242793182, 1.0480489989360529, 
.87081229403107818, .87581126380264274, .88537992618736483, .89862867397106283, 
.91449231388098307, .93177627385077810, .94927843986583593, .96610114264928748, 
.98002541669751142, .99375639472256683, 1.0058444707450425, 1.0163789765427558, 
1.0246407323535078, 1.0323072968878104, 1.0383257872323821, 1.0428171486299194, 
1.0459636224644482, 1.0480176242742001, 1.0492276484508023, 1.0497787513520891, 
.86579188982087840, .87098623283864551, .88092977862557653, .89469894214003109, 
.91118646715210372, .92914920587450689, .94733553680016769, .96481073295717557, 
.97926975573032071, .99352189028470517, 1.0060623414475605, 1.0169848361511016, 
1.0255456470092756, 1.0334842763509697, 1.0397112785176230, 1.0443540001952061, 
1.0476032228563439, 1.0497220278645645, 1.0509689488833043, 1.0515364142072203, 
.86069298794219218, .86608474617336884, .87640744896846889, .89070366447215799, 
.90782396090677797, .92647608787635416, .94535803635231008, .96349714305280554, 
.97850048642806309, .99328317119136633, 1.0062841045801740, 1.0176014207848938, 
1.0264664098247314, 1.0346815833934677, 1.0411203394001181, 1.0459165999939875, 
1.0492699059653996, 1.0514542508077989, 1.0527384262465054, 1.0533223903496700, 
.85551691608941383, .86110796053654659, .87181379825436478, .88664336038511582, 
.90440500561067239, .92375690921592901, .94334581305483234, .96216022484413555, 
.97771749523577922, .99304019525662746, 1.0065098040333393, 1.0182288616754058, 
1.0274032251844583, 1.0358994910648243, 1.0425532949908436, 1.0475053089998627, 
1.0509640548995896, 1.0532146883390956, 1.0545364815390579, 1.0551370829144788, 
.85026509054761457, .85605711655516725, .86714976191582727, .88251861005814447, 
.90092985663473502, .92099168881906901, .94129875852509870, .96079983861994045, 
.97692067217460354, .99279292103939398, 1.0067394832642318, 1.0188672893782769, 
1.0283562969748981, 1.0371382722517386, 1.0440104705266251, 1.0491204884760195, 
1.0526860530986635, 1.0550037359912450, 1.0563635159993019, 1.0569808952365254, 
.84493901363805834, .85093353671393990, .86241634856830784, .87833005405049291, 
.89739881445442671, .91818047562411131, .93921678190079472, .95941585335260038, 
.97610991100911471, .99254130788935901, 1.0069731852602271, 1.0195168336882559, 
1.0293258284764975, 1.0383981995612293, 1.0454921912540553, 1.0507624998643286, 
1.0544362842295167, 1.0568217894956096, 1.0582199310103095, 1.0588542307576816, 
.83954027097933426, .84573862313727133, .85761463863829268, .87407839279252508, 
.89381222476342603, .91532334898084800, .93709981025435241, .95800814699368941, 
.97528510941278113, .99228531599283831, 1.0072109525022623, 1.0201776235532427, 
1.0303120222544788, 1.0396795452016392, 1.0469987823105336, 1.0524317046708618, 
1.0562151320773725, 1.0586692446784187, 1.0601061279999064, 1.0607574929287680, 
.83407052857519115, .84047385520141971, .85274578283573610, .86976438595055106, 
.89017047850058439, .91242041900033588, .93494778898483522, .95657660676154777, 
.97444616913111703, .99202490641838548, 1.0074528269281477, 1.0208497869877879, 
1.0313150800484756, 1.0409825808618096, 1.0485305686028044, 1.0541284643482849, 
1.0580229804332892, 1.0605464973529475, 1.0620225083362749, 1.0626910851069375, 
.82853152974047728, .83514078698588978, .84781100047603830, .86538885166687773, 
.88647401178920426, .90947182685387651, .93276068218589536, .95512112942002958, 
.97359299614216466, .99176004116209682, 1.0076988498958945, 1.0215334509862217, 
1.0323352026612582, 1.0423075775887216, 1.0500878746833584, 1.0558531401755491, 
1.0598602129784769, 1.0624539432081173, 1.0639694732197040, 1.0646554104490689, 
.82292509187684815, .82974104457349229, .84281157765726900, .86095266567698847, 
.88272330578800440, .90647774501962874, .93053847298846526, .95364162154762477, 
.97272550081391440, .99149068319250866, 1.0079490621471325, 1.0222287414355788, 
1.0333725898467878, 1.0436548056638986, 1.0516710246250500, 1.0576060931352953, 
1.0617272131658875, 1.0643919776940185, 1.0659474235710242, 1.0666508718017216, 
.81725310311015809, .82427632320883149, .83774886529877290, .85645676030619466, 
.87891888645344718, .90343837747548228, .92828116387690829, .95213799979617419, 
.97184359805828147, .99121679649499483, 1.0082035037706887, 1.0229357830284875, 
1.0344274401978217, 1.0450245344788551, 1.0532803418942682, 1.0593876837893610, 
1.0636243640995169, 1.0663609959048364, 1.0679567599172279, 1.0686778715881677, 
.81151751880163558, .81874838432527908, .83262427704769237, .85190212334850468, 
.87506132421340832, .90035395983698430, .92598877697741577, .95061019113840772, 
.97094720748125696, .99093834611556886, 1.0084622141663959, 1.0236546991761886, 
1.0354999510333012, 1.0464170324098772, 1.0549161492229997, 1.0611982721527708, 
1.0655520484118395, 1.0683613924596298, 1.0699978822747549, 1.0707368116929923, 
.80572035794511787, .81315905245076681, .82743928706033942, .84728979683087580, 
.87115123355248980, .89722475943929414, .92366135431751717, .94905813310356235, 
.97003625352885747, .99065529820399745, 1.0087252320092035, 1.0243856119218515, 
1.0365903182857428, 1.0478325666924024, 1.0565787684800974, 1.0630382175665753, 
1.0675106481397778, 1.0703935613814022, 1.0720711900308946, 1.0728280933447263, 
.79986369946273550, .80751021200295647, .82219542766569409, .84262087566639943, 
.86718927250958810, .89405107536232287, .92129895805564399, .94748177400035971, 
.96911066562850334, .99036762005613066, 1.0089925952136605, 1.0251286418543553, 
1.0376987363888581, 1.0492714032952806, 1.0582685205420772, 1.0649078785698860, 
1.0695005445995931, 1.0724578959748687, 1.0741770818237404, 1.0749521169969514, 
.79394967841152109, .80180380398454076, .81689428691864752, .83789650620035983, 
.86317614208864901, .89083323839840722, .91890167067977613, .94588107312664838, 
.96817037832546360, .99007528015535762, 1.0092643408988418, 1.0258839080226936, 
1.0388253981656184, 1.0507338067951708, 1.0599857251637381, 1.0668076127714543, 
1.0715221182610666, 1.0745547887033229, 1.0763159554211024, 1.0771092822083002, 
.78798048211345606, .79604182258958511, .81153750605089803, .83311788465346892, 
.85911258558385550, .88757161096206305, .91646959517429116, .94425600096505224, 
.96721533141401483, .98977824821309590, 1.0095405053537851, 1.0266515278511710, 
1.0399704947169823, 1.0522200402513413, 1.0617307008489090, 1.0687377767211144, 
1.0735757486213280, 1.0766846310649734, 1.0784882075987754, 1.0792999875217468, 
.78195834622146720, .79022631173192870, .80612677682768075, .82828625546692913, 
.85499938782080120, .88426658694156346, .91400285515422997, .94260653936399752, 
.96624547006297001, .98947649520822545, 1.0098211240035095, 1.0274316170555482, 
1.0411342153114971, 1.0537303650811218, 1.0635037647216055, 1.0706987257814111, 
1.0756618140786709, 1.0788478134691146, 1.0806942340185328, 1.0815246303435739, 
.77588555073383580, .78435936150673979, .80066383881874259, .82340290955430440, 
.85083737431551443, .88091859149229046, .91150159496629557, .94093268170352828, 
.96526074493524949, .98916999342537870, 1.0101062313756803, 1.0282242895602927, 
1.0423167472759827, 1.0552650409362578, 1.0653052323978784, 1.0726908139997191, 
1.0777806918066857, 1.0810447251124773, 1.0829344291062053, 1.0837836068223750, 
.76976441596939460, .77844310459634647, .79515047659216564, .81846918246548206, 
.84662741035348577, .87752808077201261, .90896597975600291, .93923443304536347, 
.96426111230117562, .98885871649200319, 1.0103958610679893, 1.0290296574170927, 
1.0435182758874999, 1.0568243255804080, 1.0671354178586208, 1.0747143939811437, 
1.0799327576290216, 1.0832757538560871, 1.0852091859301876, 1.0860773117284415, 
.76359729851575475, .77247971263145476, .78958851683981091, .81348645246829321, 
.84237039999114827, .87409554161844727, .90639619550050921, .93751181026668795, 
.96324653414518990, .98854263941411247, 1.0106900457163106, 1.0298478307247823, 
1.0447389842668067, 1.0584084747680185, 1.0689946333235993, 1.0767698167624924, 
1.0821183858950838, 1.0855412861029503, 1.0875188960806934, 1.0884061383338643, 
.75738658716262874, .76647139251880558, .78397982544326617, .80845613855360887, 
.83806728498252947, .87062149116966936, .90379244900676203, .93576484217721867, 
.96221697826570540, .98822173861064722, 1.0109888169636996, 1.0306789175508244, 
1.0459790532734887, 1.0600177421248025, 1.0708831891269595, 1.0788574316875863, 
1.0843379493569498, 1.0878417066768638, 1.0898639495500742, 1.0907704782936634, 
.75176753395157392, .76103301579618643, .77889892350537815, .80389414105716395, 
.83415995372351726, .86746310814386329, .90142270672442881, .93417345504195004, 
.96127853283405473, .98792909178866994, 1.0112613724428772, 1.0314372299259780, 
1.0471106307427391, 1.0614862280551940, 1.0726064248208502, 1.0807621012924005, 
1.0863627566400829, 1.0899400333832048, 1.0920027967861621, 1.0929268118455637, 
.74575674787232937, .75521290735763114, .77345684595265796, .79900244431841849, 
.82996503502427788, .86406808317157524, .89887260129979796, .93245938971004927, 
.96026710607845889, .98761352655272461, 1.0115553735859901, 1.0322553803103600, 
1.0483316160753098, 1.0630707697903121, 1.0744657526237125, 1.0828169929607128, 
1.0885470091600893, 1.0922033448879498, 1.0943096169920798, 1.0952523835560584, 
.74018432539918466, .74981483100268043, .76840513937705657, .79445655453751096, 
.82606175817877070, .86090512818037956, .89649413347183193, .93085920088669782, 
.95932226095412065, .98731857992931094, 1.0118302636480370, 1.0330205187057944, 
1.0494736081221774, 1.0645528386475467, 1.0762047606391483, 1.0847387362477185, 
1.0905895057085866, 1.0943195437251094, 1.0964663167083977, 1.0974265159984455, 
.73528932100727007, .74507103114512885, .76396230978171631, .79045447464133967, 
.82262144311271845, .85811410502489471, .89439316235416011, .92944448300550242, 
.95848642212313084, .98705753262269680, 1.0120736420685597, 1.0336980914542224, 
1.0504850116362914, 1.0658654748240317, 1.0777449114190691, 1.0864405963434776, 
1.0923981290281699, 1.0961932544426420, 1.0983757468188471, 1.0993513029182875, 
.73125202810443712, .74115706524673863, .76029423320282397, .78714736219120962, 
.81977569918971227, .85580312485040166, .89265197077731173, .92827114136690048, 
.95779282393731324, .98684081566589110, 1.0122757513343092, 1.0342608775711835, 
1.0513251555883809, 1.0669558823018279, 1.0790242856672121, 1.0878542126543302, 
1.0939003072029794, 1.0977493703789394, 1.0999614299833196, 1.1009496853425988, 
.72824354594763308, .73823966399536056, .75755867131101184, .78467925962049989, 
.81765022165061008, .85407567771187953, .89134949229208593, .92739290354447190, 
.95727344834380412, .98667847822135191, 1.0124271834778322, 1.0346826163136666, 
1.0519547904184654, 1.0677730997090502, 1.0799831116736889, 1.0889135974868422, 
1.0950259933022766, 1.0989154053810750, 1.1011495635859223, 1.1021473030117077, 
.72638981990832219, .73644170882345261, .75587216414019276, .78315689913508446, 
.81633846934859022, .85300897734213565, .89054480262153402, .92685008534950453, 
.95695233814014935, .98657808642618605, 1.0125208474020158, 1.0349434995743960, 
1.0523442985653022, 1.0682786633027899, 1.0805762747904348, 1.0895689497636525, 
1.0957223320603098, 1.0996366739837042, 1.1018844776388685, 1.1028880701479138, 
.72576447202562844, .73583511462542406, .75530306371574940, .78264306011036400, 
.81589559053775906, .85264873080964765, .89027297171281483, .92666667679527182, 
.95684382399286394, .98654415629025938, 1.0125525064861385, 1.0350316851025216, 
1.0524759667210426, 1.0684495646837464, 1.0807767876230476, 1.0897904816491941, 
1.0959577139504708, 1.0998804777999798, 1.1021328897999061, 1.1031384584733370
};
static double tablef2 [ 400 ]= {
1.0000000000315712, 1.0000000229651955, 1.0000004974673784, 1.0000037399980137, 
1.0000167065020968, 1.0000541654970072, 1.0001401132280043, 1.0003068928840446, 
1.0005540452926909, 1.0009628190592030, 1.0015503319211347, 1.0023576819070414, 
1.0033085415893246, 1.0045959986575392, 1.0060769205685118, 1.0076378289064938, 
1.0091303009099846, 1.0103748302726607, 1.0112020542675741, 1.0114919652519547, 
.99999763725565949, .99999082770070014, .99997872467694648, .99996563144309147, 
.99996065262016376, .99998045653043810, 1.0000502971730199, 1.0002028168923240, 
1.0004384772983046, 1.0008352839411558, 1.0014104778778292, 1.0022040356137869, 
1.0031403317461192, 1.0044091345105941, 1.0058691603954488, 1.0074082822964909, 
1.0088800177044761, 1.0101072959455035, 1.0109230704290093, 1.0112089720262425, 
.99997337235016182, .99994700891648702, .99989890586075714, .99983904566637793, 
.99978267374189478, .99975188158156880, .99977549714421976, .99988697583044950, 
1.0000894066051458, 1.0004513847108227, 1.0009904544171930, 1.0017432926955743, 
1.0026363801522415, 1.0038496483392563, 1.0052473188701105, 1.0067213269257169, 
1.0081310201669551, 1.0093066466721270, 1.0100881239780626, 1.0103620126208677, 
.99987920073821934, .99982331163893967, .99972083704719861, .99959031495560535, 
.99945623655754934, .99934883607289082, .99930225190661370, .99935102336504511, 
.99950217725730406, .99980967906328622, 1.0002913660585253, 1.0009786385864503, 
1.0018014745013020, 1.0029238494561352, 1.0042189965273647, 1.0055856243105767, 
1.0068927938205987, 1.0079829469087105, 1.0087076182102397, 1.0089616011481382, 
.99964672914967811, .99955545673916335, .99938780303178465, .99917252495690456, 
.99894514278931879, .99874555171651036, .99861406903302030, .99858622740115521, 
.99867370020940804, .99891205757924373, .99931914072070305, .99991941711467136, 
1.0006477114325778, 1.0016465977010143, 1.0028015535540065, 1.0040207301163317, 
1.0051867154179117, 1.0061589340552028, 1.0068051277186074, 1.0070315939840000, 
.99920264998358677, .99907487971065001, .99883997023054416, .99853717670773656, 
.99821331768582633, .99791808487434142, .99769784063315703, .99758860677977246, 
.99760670226274806, .99776727116242181, .99808755649649483, .99858383948542562, 
.99919705621319044, 1.0000438370845466, 1.0010247421605065, 1.0020599171421309, 
1.0030491140383038, 1.0038733098985038, 1.0044208563665624, 1.0046127084024148, 
.99848194428559700, .99832098876477338, .99802491012315231, .99764246744221541, 
.99723064514679061, .99684777639244149, .99654519077606707, .99635850225347833, 
.99630798682807198, .99638800191803056, .99661429713969096, .99699411390119586, 
.99747565708852226, .99814601031549433, .99892322539615441, .99974183759577484, 
1.0005221684084300, 1.0011710303695347, 1.0016015360700117, 1.0017522865619384, 
.99744913171055472, .99726213233895400, .99691804726887367, .99647306620138465, 
.99599201382804670, .99553960807617159, .99517027074226239, .99491800941306385, 
.99480576020504508, .99480835261635536, .99493894377551984, .99519533947477934, 
.99553395522109833, .99600996200604874, .99656067337193044, .99713703366042561, 
.99768279933419568, .99813422790747697, .99843271716651605, .99853706851833079, 
.99624503117157159, .99604262192807136, .99567015549968385, .99518821479844843, 
.99466610796761801, .99417182237113000, .99376056125051959, .99346372044163322, 
.99330519338236556, .99324422416925850, .99329061240743743, .99343362151540848, 
.99363768828394861, .99392808603312843, .99426051457585751, .99460210208914490, 
.99491970223229020, .99517860467546082, .99534815537628374, .99540715430823190, 
.99464686675505476, .99443723215677382, .99405152766461036, .99355242761924101, 
.99301109435519874, .99249615763377075, .99206133143112940, .99173385802650293, 
.99153694439868156, .99141555863354247, .99137472304966845, .99139455159365864, 
.99144873442560666, .99152939557673991, .99161299698168093, .99168551838921914, 
.99174070385511404, .99177765770656479, .99179834331795591, .99180494276292241, 
.99278287033293633, .99257601120172001, .99219558761261339, .99170358552791593, 
.99116988520116411, .99066073056371595, .99022592924808717, .98988726216291778, 
.98966541371692973, .98949419098177760, .98937274620627302, .98927239034624370, 
.98917640383351113, .98904385385344840, .98887229033897661, .98866743060419815, 
.98845116175735914, .98825799431236838, .98812411013325718, .98807626369525514, 
.99066003970277972, .99046550985639803, .99010808872623557, .98964645870383172, 
.98914624639900484, .98866840941117939, .98825650138092946, .98792554284703404, 
.98769186948022780, .98748113292002759, .98728552879218179, .98706788698449177, 
.98682140252792283, .98647216031071727, .98603912181765896, .98554861746211653, 
.98505191779182877, .98462051665458394, .98432640126002635, .98422207825136598, 
.98853624333756862, .98836059298312828, .98803833564300836, .98762308288938605, 
.98717423299336615, .98674561865216748, .98637322413970641, .98606489145016119, 
.98583149683257931, .98559383611594120, .98533692849865118, .98501623377251768, 
.98463412745015877, .98408709072845058, .98341360986663517, .98265928655279677, 
.98190285724429127, .98125039959988514, .98080743355021760, .98065061567550271, 
.98605962004969983, .98591178608212948, .98564132812054805, .98529441287472830, 
.98492144914569590, .98456633200120303, .98425503163457451, .98398662137299497, 
.98376457157960095, .98350699619273985, .98319033001941891, .98276243916517547, 
.98223572986465024, .98147523543372694, .98054049144759114, .97949835462390167, 
.97845785118863726, .97756321209437658, .97695700023506071, .97674258158490601, 
.98358193919619574, .98346651215737735, .98325641732188696, .98298919349855862, 
.98270491296384133, .98243624199110624, .98219817469859272, .98198067229420472, 
.98177892523214594, .98151076382859748, .98114384338606564, .98061923427109277, 
.97995882689896108, .97899867865253654, .97781799262927938, .97650391058956476, 
.97519436401652859, .97407000480601014, .97330879883719557, .97303966569031697, 
.98126650435998763, .98118450156616355, .98103668976833602, .98085175780381595, 
.98065921221569019, .98048034296730513, .98031921493946772, .98015702152260964, 
.97998058004402344, .97970913629987022, .97930199354084368, .97869442920368721, 
.97791680531409420, .97677985006051382, .97538016692352514, .97382316962514381, 
.97227282501721102, .97094259791745963, .97004238055632452, .96972415611296414, 
.97926205114724163, .97921093094229128, .97912068992237555, .97901188089143876, 
.97890431098193909, .97880887406770212, .97871974759342872, .97861036614806340, 
.97845988044382226, .97818982612151317, .97775217450127228, .97707753940097686, 
.97620336509366501, .97491956602259622, .97333718560566307, .97157702233271281, 
.96982496367175786, .96832210629619764, .96730524744847524, .96694581901656362, 
.97771411277550768, .97768787395046608, .97764403298126634, .97759661458629943, 
.97755764711500603, .97752970577878606, .97749910214969749, .97743318390425549, 
.97730494790395839, .97703827310891156, .97657940989431224, .97585555935947654, 
.97490949742332045, .97351566934995681, .97179593152175781, .96988272965861433, 
.96797854599190380, .96634539797857044, .96524046406392450, .96484991581401180, 
.97673767632336517, .97672753867311868, .97671373176814658, .97670606814325323, 
.97671155733445725, .97672741300760846, .97673489296306748, .97669746596764595, 
.97658415151229139, .97632053633105664, .97584923727374639, .97509537777086719, 
.97410504142299537, .97264315774903232, .97083826836411857, .96883007283016787, 
.96683138828470883, .96511726211239923, .96395757090322809, .96354767223125248, 
.97640442763332080, .97639985159420073, .97639642035321728, .97640249366715381, 
.97642335003763248, .97645435461460572, .97647502519295026, .97644750007547343, 
.97633942527908157, .97607700832041533, .97560162173552278, .97483769223498695, 
.97383242307012108, .97234753623283954, .97051383249824918, .96847347170854781, 
.96644277623996211, .96470121290267330, .96352296588477460, .96310650938180786
};
static double tabled2 [ 400 ]= {
1.0000005410689554, 1.0000430565124399, 1.0003246681168620, 1.0011977386863391, 
1.0031080322545169, 1.0065103776699897, 1.0117653018175377, 1.0191228475268701, 
1.0275674488891993, 1.0388555420580654, 1.0523362220191059, 1.0681959024445291, 
1.0846827856169521, 1.1047885169665650, 1.1259584447770190, 1.1468013249961471, 
1.1657541808072056, 1.1810177103039237, 1.1909457183045717, 1.1943893577432980, 
.99999996096075972, 1.0000494329106249, 1.0003439388072153, 1.0012340000393274, 
1.0031634594053567, 1.0065855225829066, 1.0118597214603082, 1.0192360405913199, 
1.0276974909452283, 1.0390051025733089, 1.0525078657900870, 1.0683941131868913, 
1.0849105887463364, 1.1050560652094683, 1.1262727309309346, 1.1471667618146735, 
1.1661705973169301, 1.1814782543961062, 1.1914364172957610, 1.1948907809457405, 
.99998096092408806, 1.0000516084208670, 1.0003853782539440, 1.0013271992104684, 
1.0033150578947923, 1.0067971780266269, 1.0121299901971952, 1.0195632097402301, 
1.0280754898855213, 1.0394416671573454, 1.0530103384750706, 1.0689755396335394, 
1.0855797109686587, 1.1058427574888789, 1.1271975270603158, 1.1482426145366964, 
1.1673969620922195, 1.1828348886141564, 1.1928820704682508, 1.1963680927364551, 
.99989527198078620, 1.0000020338040909, 1.0004028117299555, 1.0014330382202696, 
1.0035206897553208, 1.0071053672038923, 1.0125379866912589, 1.0200675188238206, 
1.0286651000750897, 1.0401285710672916, 1.0538056245093756, 1.0698995979154962, 
1.0866460073550352, 1.1070990663924762, 1.1286765476125988, 1.1499649554853333, 
1.1693616160813565, 1.1850092238805983, 1.1951996838335470, 1.1987366615854753, 
.99967406423674554, .99983263762924912, 1.0003296366084373, 1.0014869547144425, 
1.0037181723920141, 1.0074502888884476, 1.0130258903570102, 1.0206923470995091, 
1.0294098393764626, 1.0410082282893274, 1.0548334724685913, 1.0711014755633526, 
1.0880385760146392, 1.1087450434569639, 1.1306185806189137, 1.1522298814016467, 
1.1719478314017180, 1.1878733687834071, 1.1982537245455407, 1.2018582364453332, 
.99924345939204806, .99946989319544310, 1.0000930549007951, 1.0014172437627717, 
1.0038371484091364, 1.0077629365677578, 1.0135256779072486, 1.0213698737720118, 
1.0302410747482531, 1.0420097094751821, 1.0560188254888596, 1.0724996791948465, 
1.0896676521340611, 1.1106788594959902, 1.1329069130070477, 1.1549039737754352, 
1.1750053289652209, 1.1912623664139876, 1.2018691837156752, 1.2055542313745062, 
.99853785145066010, .99884810071954933, .99962727452494393, 1.0011581380868733, 
1.0038120232630217, 1.0079779136337763, 1.0139718974426584, 1.0220339625577930, 
1.0310911923939807, 1.0430624889648941, 1.0572864928439474, 1.0740120754253299, 
1.0914423635019765, 1.1127970233515156, 1.1354225583588575, 1.1578508815442023, 
1.1783802310531297, 1.1950070701726794, 1.2058664493390607, 1.2096413082709583, 
.99752103913736134, .99792932152671088, .99889120084554350, 1.0006645073925939, 
1.0035932581652269, 1.0080412340737019, 1.0143061541279716, 1.0226216361772209, 
1.0318926543806202, 1.0440930871632654, 1.0585554961067722, 1.0755478923010782, 
1.0932604324101822, 1.1149812651093549, 1.1380280213150187, 1.1609117879375985, 
1.1818922989910434, 1.1989086079605582, 1.2100338803012847, 1.2139032790661890, 
.99633163101536870, .99683851585573224, .99798438330719874, 1.0000022365500922, 
1.0032086511333851, 1.0079416998949877, 1.0144798784302897, 1.0230494273846999, 
1.0325325709538089, 1.0449570632997633, 1.0596485174694472, 1.0768929424232174, 
1.0948685915087797, 1.1169275282485120, 1.1403607292857316, 1.1636607669134005, 
1.1850527794609680, 1.2024239416931473, 1.2137913543462560, 1.2177468316039599, 
.99474941083975854, .99537288372829302, .99673655839102038, .99904438654134309, 
1.0025805636015119, 1.0076523996264402, 1.0145148604616718, 1.0233844172472859, 
1.0331157748036588, 1.0457989800859500, 1.0607505155696841, 1.0782763724149005, 
1.0965418825193782, 1.1189695883801054, 1.1428213734977146, 1.1665704219958650, 
1.1884052432075232, 1.2061577644440602, 1.2177852383799146, 1.2218331413500350, 
.99290094274641993, .99364786540446615, .99524244595649192, .99785821264569852, 
1.0017451748635369, 1.0071777242395032, 1.0143851140861673, 1.0235724175308489, 
1.0335645085846434, 1.0465160952102857, 1.0617330555191458, 1.0795412840101581, 
1.0980935116820312, 1.1208820410289055, 1.1451402375897279, 1.1693231558619044, 
1.1915846464364304, 1.2097040427529977, 1.2215814938388414, 1.2257182077987769, 
.99079322764156352, .99167003556216260, .99350784672774595, .99644853515096976, 
1.0007062698840945, 1.0065204988113172, 1.0140926623073179, 1.0236148278627670, 
1.0338797514350819, 1.0471090409981052, 1.0625965001370448, 1.0806878064282985, 
1.0995233956061219, 1.1226645347429445, 1.1473166385278195, 1.1719178790769267, 
1.1945894392795149, 1.2130607800887088, 1.2251777924973271, 1.2293995795211810, 
.98868267137600785, .98968140512693636, .99174779325309059, .99499431222808077, 
.99960161750722132, 1.0057761168769475, 1.0136925199140341, 1.0235298472234458, 
1.0340500473452783, 1.0475366817421403, 1.0632714468651558, 1.0816176703413765, 
1.1007049824327606, 1.1241559127946801, 1.1491512239772965, 1.1741149648905638, 
1.1971406639417466, 1.2159153208811395, 1.2282385559036759, 1.2325335389883836, 
.98621976929054533, .98735318821253182, .98967234184646902, .99325757620919475, 
.99825268219168106, 1.0048273464374711, 1.0131261253737998, 1.0233127765393215, 
1.0341144993499007, 1.0478825170935659, 1.0638843537491034, 1.0825019337482205, 
1.1018536412018569, 1.1256263304507352, 1.1509750341465974, 1.1763098664404259, 
1.1996967417410471, 1.2187800231953578, 1.2313128086339138, 1.2356821234689463, 
.98375441428898414, .98501631183772531, .98757683405078223, .99148600283570365, 
.99685260299271017, 1.0038111068681026, 1.0124767525131930, 1.0229979746539855, 
1.0340681394671927, 1.0481023876310587, 1.0643539952517040, 1.0832218462791952, 
1.1028141688036488, 1.1268763186379298, 1.1525400434039996, 1.1782035909259039, 
1.2019090677630790, 1.2212638716659368, 1.2339807272732171, 1.2384152881004009, 
.98144950320766822, .98282690581188725, .98560462510153291, .98980568557540971, 
.99550753840951189, 1.0028128648672900, 1.0118101197411353, 1.0226333710000257, 
1.0339450073171659, 1.0482171949238508, 1.0646898269869487, 1.0837764483096231, 
1.1035765664209392, 1.1278860162857378, 1.1538165256661763, 1.1797567317879460, 
1.2037292326365916, 1.2233109817969396, 1.2361814350533824, 1.2406703984425238, 
.97945352165693333, .98092792060099399, .98388817616133351, .98833483134611888, 
.99431912778379072, 1.0019169647205908, 1.0111940142680205, 1.0222718713297874, 
1.0337865063013172, 1.0482576980891436, 1.0649136992470987, 1.0841799714743412, 
1.1041489631872909, 1.1286574953302380, 1.1548010797449950, 1.1809609865269350, 
1.2051447021885036, 1.2249054945439597, 1.2378969312794268, 1.2424287130847749, 
.97791177642921134, .97945941107145107, .98255757282978096, .98718994358603707, 
.99338803930616226, 1.0012074914819451, 1.0106966029320728, 1.0219673141251409, 
1.0336354061430721, 1.0482564600263262, 1.0650497033070827, 1.0844493710517644, 
1.1045427261556184, 1.1291967467487001, 1.1554950304926608, 1.1818136806516906, 
1.2061494938207853, 1.2260389235669105, 1.2391171611353458, 1.2436796427383034, 
.97693910588135746, .97853225427366541, .98171615675846180, .98646406515706764, 
.99279526439095434, 1.0007527715605709, 1.0103740248449771, 1.0217648650447242, 
1.0335284216279759, 1.0482424673018582, 1.0651205205943493, 1.0846021758173674, 
1.1047715313723896, 1.1295139826537610, 1.1559058568951008, 1.1823202092617981, 
1.2067474908983025, 1.2267141500478356, 1.2398444448656252, 1.2444253287701712, 
.97660711907044528, .97821568626310706, .98142864158229726, .98621571183847334, 
.99259204328902517, 1.0005963765446770, 1.0102624571971510, 1.0216940426022731, 
1.0334899580882244, 1.0482354847038486, 1.0651421897017759, 1.0846514671734786, 
1.1048463531445754, 1.1296184275820063, 1.1560415770831141, 1.1824878516743940, 
1.2069456032529728, 1.2269379658424964, 1.2400855765165657, 1.2446725799230260
};
static double tablef3 [ 400 ]= {
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
.99625642648432255, .99627250465082517, .99630678226870073, .99636281401509086, 
.99644550862169057, .99655980766028707, .99670902582467880, .99689516055413585, 
.99709242588054692, .99734020439918369, .99762017729233243, .99793306858348463, 
.99824286307222632, .99860208745446377, .99896037810370575, .99929467458518484, 
.99958375539010291, .99980678107267096, .99994734912578220, .99999530020826832, 
.98525898763496722, .98532182046497907, .98545573458723079, .98567454807314947, 
.98599735873902361, .98644341669691071, .98702568883675179, .98775210391936386, 
.98852222196482864, .98949009960072909, .99058466300701498, .99180927672510055, 
.99302334163350814, .99443325457074760, .99584195353624361, .99715865528214920, 
.99829917463404594, .99918033934842243, .99973629355047666, .99992604609840585, 
.96772292233211488, .96785881746815616, .96814830249135042, .96862100581192012, 
.96931793261234234, .97028049912127079, .97153678901693552, .97310436752941121, 
.97476715299654537, .97685883516372751, .97922748484331723, .98188223586705048, 
.98451955530044788, .98758973013485452, .99066580469090448, .99354918805889725, 
.99605347246091680, .99799270976351817, .99921827385205686, .99963693759242755, 
.94478097310113685, .94500963933657667, .94549641984405486, .94629058978186398, 
.94746046988085431, .94907524907112073, .95118224631895865, .95381191303170554, 
.95660322121111574, .96011866491313791, .96410657383214715, .96858643747145257, 
.97304889128489191, .97826020946777248, .98350060851905652, .98843113985842290, 
.99272858041900180, .99606643096312109, .99818054062910544, .99890357756772885, 
.91784154213460645, .91817497837492779, .91888421715037316, .92004008935669901, 
.92174101997637614, .92408699485733370, .92714707264156460, .93096713703758094, 
.93502520822785284, .94014302871535149, .94596049704205054, .95251314339947424, 
.95906092994176206, .96673594457041512, .97448693207790238, .98181181305144349, 
.98822289059736568, .99322024480831061, .99639372333888376, .99748056807309760, 
.88829387051363895, .88873690292937724, .88967839562506368, .89121092261484414, 
.89346344441263581, .89656741160491815, .90061457910179299, .90566797331730997, 
.91104067051969873, .91782637461342043, .92555675290190675, .93428944159128213, 
.94304569507139790, .95335108718530948, .96380750970465257, .97373711840417487, 
.98246812768413316, .98930080772274287, .99365237086490188, .99514496826812560, 
.85769499157270479, .85824547306561172, .85941415646382191, .86131401590055168, 
.86410285282272692, .86794207738726958, .87294556226036868, .87919412541674524, 
.88584293320222973, .89425279795921564, .90385495679316685, .91473448147072911, 
.92568192206512545, .93862009724880313, .95181177083964619, .96440192961008497, 
.97552552892995981, .98426657949060192, .98985043641160531, .99176879052998279, 
.83022637447002390, .83086790437418023, .83222863986597051, .83443801570833109, 
.83767727700074046, .84213238941682367, .84793577156588684, .85518406415651793, 
.86290196357137217, .87267667413797656, .88385921841232018, .89656279778898285, 
.90938578149931226, .92459704676157251, .94017375074596654, .95510728683651092, 
.96835828258789858, .97880987487821691, .98550475987463926, .98780815681609035, 
.80107787443834222, .80181024069192341, .80336208560051214, .80587837695204034, 
.80956266864839677, .81462449638563450, .82121448972795164, .82944572879532119, 
.83821624252730353, .84933874834745782, .86208923954375183, .87661387204555008, 
.89132314993632034, .90884031491823380, .92686041074134695, .94421882382635358, 
.95969198756078344, .97194463396519470, .97981614441844389, .98252856515273168, 
.77332954701130718, .77414267557380967, .77586394128858382, .77865123361006705, 
.78272685334829217, .78832033255887124, .79559812252103157, .80468842671623442, 
.81438005663700985, .82668547811180848, .84081869881237759, .85695967106249293, 
.87335599306557332, .89295393038353355, .91320136670196583, .93279312980021523, 
.95033278495476281, .96427406242397595, .97325532720307739, .97635473339201695, 
.74690492198296055, .74778961066731464, .74966052338474032, .75268615203773370, 
.75710436400987680, .76316146501774562, .77103750051300057, .78087442106298099, 
.79136741767443852, .80470484339949722, .82005001373568909, .83761671460036868, 
.85551233914506675, .87697577443114449, .89924011721369657, .92087471826256251, 
.94032246447883128, .95583540009231394, .96585553000720203, .96931830340106384, 
.72421802969773785, .72515986230043605, .72714988944597103, .73036437348091587, 
.73505274836745182, .74147391608034219, .74981837444249888, .76023923493643386, 
.77135945154505836, .78550663596056691, .80180688860190785, .82050388874344815, 
.83959660269925929, .86256188700306979, .88646526518592916, .90977571942076030, 
.93080258247770609, .94762587040780533, .95851679692266545, .96228500976353681, 
.70114388457754295, .70213964457370925, .70424169611519472, .70763293728303467, 
.71257284673497001, .71933139366713872, .72810848639117376, .73906785881416392, 
.75076675795532699, .76566274790765576, .78284987444284337, .80260259337800877, 
.82282118535762661, .84721042453074236, .87268203688169377, .89761063947554531, 
.92017516943489153, .93828349149261348, .95003277146522447, .95410286795784949, 
.68078280049955109, .68182253720466068, .68401560053950905, .68754968018778839, 
.69269166801274396, .69971981353309431, .70884125694248501, .72022840010887512, 
.73238704612851058, .74787921222107878, .76577535984269363, .78637705874918640, 
.80750740763088391, .83305919694866803, .85982286257353524, .88609677301817047, 
.90995039387490440, .92914353707102892, .94162107172240317, .94594796720416385, 
.66366398581195285, .66473800875500977, .66700176475824624, .67064619953217830, 
.67594337948980698, .68317742243000878, .69256073797998596, .70427243981022047, 
.71677984365155778, .73272507752100435, .75116207484183251, .77241491670517548, 
.79424910847034869, .82070510332579272, .84848212564104475, .87581989009325678, 
.90070060349723591, .92076349696729953, .93382752067686869, .93836170770034266, 
.65005543802780696, .65115492568426581, .65347100241089096, .65719669888735643, 
.66260750268984414, .66999147579939311, .67956467035550905, .69151112463924858, 
.70427067569277946, .72054394560555479, .73937381552568249, .76110192897055928, 
.78345282170382879, .81057701681956440, .83910839660619430, .86724379286645625, 
.89289957524198216, .91362231288405671, .92713297490563951, .93182533584042049, 
.64020163038635347, .64131854864277904, .64367033142646302, .64745124104872576, 
.65293885859648859, .66042370012837182, .67012414542572085, .68222755572510074, 
.69515559899307000, .71164830938844292, .73074168911970927, .75278984704240870, 
.77549033778099341, .80306909995612263, .83211672482271093, .86080114303186650, 
.88699318971715013, .90817440939137695, .92199634484989978, .92679911292321648, 
.63424804513029920, .63537507901031720, .63774753736727019, .64156031358583498, 
.64709209722100325, .65463472255051780, .66440784731008939, .67660073691706701, 
.68962478646567438, .70624262275149940, .72548654018712377, .74771805625105969, 
.77061952274688417, .79846062625379621, .82780745576078196, .85681137793436357, 
.88331664242540486, .90476644352982172, .91877110599122258, .92363875639853732, 
.63225951076933262, .63338985299403383, .63576906535717437, .63959223306793933, 
.64513838965080950, .65269978197925449, .66249647286288715, .67471834588386426, 
.68777348332780319, .70443185114397699, .72372457962196213, .74601563803612370, 
.76898248536012503, .79690910769934931, .82635368922836583, .85546221704751924, 
.88207022224168152, .90360825698064606, .91767301279399927, .92256201282558492
};
static double tabled3 [ 400 ]= {
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
.99252220753007248, .99255428262151735, .99262266107603825, .99273442794162259, 
.99289936851774585, .99312733570880551, .99342494336361720, .99379618700543093, 
.99418965361252365, .99468392273924033, .99524249397309583, .99586685807598585, 
.99648517891734190, .99720234204968050, .99791785274994038, .99858564855816034, 
.99916328501719909, .99960903886848140, .99989003678856969, .99998590066456339, 
.97066355679105543, .97078796536145895, .97105305909304984, .97148609958315435, 
.97212478635667576, .97300715213958200, .97415888782675208, .97559585436450724, 
.97711962151719656, .97903541708598030, .98120320123173443, .98363034765562785, 
.98603867316042326, .98883833666579102, .99163884729482721, .99425958015146099, 
.99653218497825202, .99828966878195358, .99939928695949976, .99977814814001120, 
.93614797062317123, .93641375450534980, .93697966605090999, .93790318457973898, 
.93926396236112497, .94114260151686047, .94359409043990694, .94665352602465858, 
.94990037845970343, .95398814124939235, .95862290654692583, .96382589406673658, 
.96900450904297608, .97504644059330292, .98111540019524801, .98681899491227354, 
.99178483739157619, .99563823820743858, .99807721091151702, .99891105007504466, 
.89163311269038747, .89207294918316182, .89300849965163109, .89453318400579029, 
.89677682330083819, .89987131309707769, .90390778786030268, .90894686237560796, 
.91430009098088239, .92105163505181198, .92872666641452284, .93737225616680064, 
.94601205668624666, .95613990788150449, .96636855957547152, .97603517747268276, 
.98449588005378919, .99109078589378230, .99527866566696210, .99671289744067981, 
.84031126199742867, .84093963494357715, .84227454434779037, .84444648886861229, 
.84763744080562176, .85203315031850912, .85776385019166690, .86492018297330908, 
.87253141566834946, .88215021871606991, .89311786788470393, .90552185759550641, 
.91797589550527869, .93265578246998659, .94757659059416965, .96177020223613241, 
.97427043490857268, .98406584099626226, .99031023492471604, .99245314048963762, 
.78523347649329267, .78604887844874072, .78777863401762438, .79058766621663301, 
.79470686478680186, .80037312744031086, .80775514981905616, .81697593427717519, 
.82679452150162303, .83922967942470694, .85345494438143787, .86961284438540976, 
.88591890956372865, .90525527287175245, .92504626735258910, .94400820188246770, 
.96082196783768881, .97407464720316883, .98255917080131117, .98547740987234533, 
.72957679370497308, .73056400742985872, .73265497855768336, .73604355737085991, 
.74100231707716935, .74781240995234144, .75667718310897791, .76775172673382652, 
.77955769371001038, .79454200592795354, .81173996484846522, .83136048138562024, 
.85126434053257083, .87501308023625313, .89949469476949484, .92312477395935855, 
.94422585599739173, .96095874243100554, .97171901379267289, .97542870080157451, 
.68085010477218311, .68197254847811817, .68434645420327711, .68818597553154148, 
.69379348975351873, .70148236892501237, .71148232956041801, .72397535236477029, 
.73730583558392904, .75425625102418969, .77376667496641482, .79611145792374460, 
.81888331313665685, .84620275165103420, .87454467077533904, .90208143343211988, 
.92682617480909646, .94655504373298549, .95929258774736648, .96369324217784758, 
.63046618745954319, .63171244090226109, .63434387896088840, .63859052628943969, 
.64477870369531154, .65324828038623362, .66425182183393783, .67799717472431775, 
.69267666342913947, .71137683925963879, .73296450844954897, .75778695429410092, 
.78320469390035308, .81387223695108180, .84589939405725822, .87723213074865636, 
.90557469057097384, .92830153015892136, .94303658011751912, .94813873938794768, 
.58381075004427441, .58515603234348679, .58799189635625315, .59255824235775778, 
.59919704458531786, .60826613759436954, .62003474030375683, .63473215278028305, 
.65043930458407584, .67048122041913993, .69367945474216952, .72045152344185348, 
.74798669192472478, .78138484228422733, .81648004184491041, .85103652301519217, 
.88248959049427034, .90784650280948853, .92435217536410186, .93007950513107551, 
.54060788018038031, .54203042301734794, .54502420259355566, .54983397451785898, 
.55681036284262220, .56632185055885098, .57864875452221620, .59403740313338582, 
.61049184331532436, .63151696899474874, .65591147791751503, .68415809781085513, 
.71332763554822110, .74888100396333535, .78645567045358522, .82367618443497337, 
.85775052999391155, .88535905389651193, .90339737584459046, .90966893240253588, 
.50450195876099091, .50597761439337741, .50907854532274299, .51405029150584292, 
.52124621112949243, .53103901395869701, .54371476376477945, .55953140565176884, 
.57644887598440719, .59808915378688876, .62324557011096793, .65245374844160675, 
.68271690331511914, .71975236109204255, .75908028285269991, .79823292809240909, 
.83424988383249976, .86355560562113798, .88276275897810574, .88945182965250147, 
.46874326710582565, .47026047954511069, .47344366915679953, .47853612082303371, 
.48588970577615448, .49587689855726952, .50878598153728516, .52488357795820673, 
.54210471490116952, .56415468108059497, .58983347578895551, .61972595786262863, 
.65079814445790285, .68897340743018998, .72970101879468618, .77044670469031529, 
.80810798587660582, .83887926483459307, .85910935483879847, .86616629576835935, 
.43801969823924075, .43956321238016915, .44279682036421392, .44795939032885307, 
.45539806046713271, .46548141464784581, .47849664286223759, .49471519812006909, 
.51206618330128595, .53429825763449723, .56022636699979267, .59047373204015197, 
.62199920583734414, .66085909085161776, .70248000073054806, .74429244343880574, 
.78309588374559425, .81491242380649420, .83588472303238093, .84321084366649579, 
.41280753157188049, .41436569628940761, .41762578390007055, .42282128257608670, 
.43029298215142736, .44040361236815750, .45343723751230530, .46966720269175660, 
.48702881356669059, .50928528764048930, .53526997052368752, .56563381544405167, 
.59734765476513363, .63654213273771598, .67865307954283327, .72109864662271173, 
.76061745542959081, .79311270105383791, .81457777594465375, .82208452639771146, 
.39317806148454032, .39474304910851504, .39801392962595711, .40321888407146322, 
.41069217950735609, .42079026450469381, .43379329087561225, .44997463970537355, 
.46728156580200769, .48947460975863244, .51540513365937879, .54574274312277082, 
.57747913601210321, .61677864699382068, .65910236083848650, .70186986519410832, 
.74178640263451234, .77467959745331660, .79644256758241938, .80406005843257811, 
.37919712283488138, .38076440465688077, .38403748534083254, .38924015945922108, 
.39670118684139183, .40677159545203739, .41972792264618319, .43584274368502729, 
.45307572165757842, .47517761130514996, .50101451287850829, .53126747134991203, 
.56294937351340280, .60223448294014315, .64461182096155745, .68750820745847163, 
.72761337007593263, .76071157953223933, .78263472670527467, .79031288266233890, 
.37084592612527945, .37241352375247644, .37568566639396966, .38088328512838993, 
.38833150313047976, .39837772685375411, .41129599481359752, .42735805180752706, 
.44453254887310867, .46656114779707436, .49231948904608475, .52249463359519310, 
.55411468483878668, .59335390168294588, .63572202503416986, .67865274629079197, 
.71883004152279273, .75201687170139631, .77401312936961331, .78171959413145098, 
.36807281352488208, .36964033776464444, .37291178857208201, .37810710566192945, 
.38555014670041399, .39558705133175276, .40849097094910564, .42453331487861695, 
.44168598244977070, .46368706640262813, .48941549493830272, .51956018277108851, 
.55115471444741027, .59037232882262869, .63273032217448244, .67566509833177069, 
.71585932961519683, .74906970468814404, .77108630336038131, .77880078307140488
};
static double tablef4 [ 1700 ]= {
.15821626304596489, .15839713933313340, .15893839258000933, .15983698224801138, 
.16108534434254088, .16266750905051527, .16456218012416732, .16672749740764808, 
.16888245111285621, .17117016471375074, .17377786713539509, .17637588767317047, 
.17883061325784000, .18102466679410811, .18281880456118613, .18408747712451123, 
.18474536796960755, .15857220108029879, .15875334201585553, .15929539020629552, 
.16019530862687914, .16144553231625064, .16303007672257142, .16492761880921045, 
.16709623506023197, .16925448156440359, .17154569330268368, .17415737831207809, 
.17675935486633973, .17921780280622140, .18141516794630949, .18321200150605305, 
.18448257320895009, .18514144646661906, .15958291248533041, .15976489205777611, 
.16030945377497458, .16121355633448178, .16246960982531097, .16406155953895671, 
.16596797701052782, .16814672595574159, .17031502825627079, .17261686995788789, 
.17524060129950456, .17785449412368842, .18032411583441194, .18253139424403322, 
.18433628055098192, .18561251682441393, .18627431774108608, .16108752630847575, 
.16127103961107708, .16182018183593802, .16273185851192587, .16399836971697693, 
.16560346346530547, .16752545809721978, .16972179298763063, .17190737198967496, 
.17422731546525103, .17687138962448101, .17950525263331774, .18199348065230708, 
.18421718108224885, .18603535414894667, .18732091042603005, .18798752018905654, 
.16284198360720006, .16302787260340648, .16358408238786051, .16450735978479500, 
.16578972607772899, .16741450760219742, .16935952888186973, .17158151274634148, 
.17379197794172949, .17613770695392808, .17881045009384408, .18147217808528554, 
.18398614847040312, .18623241148927278, .18806874070480056, .18936698345023956, 
.19004012393973255, .16455905882215696, .16474825108476204, .16531425320692544, 
.16625349493870326, .16755747302176741, .16920878078657320, .17118442766734349, 
.17344006517335384, .17568276463258545, .17806149963122536, .18077051474767430, 
.18346710961393717, .18601298226182333, .18828698311555972, .19014549175778261, 
.19145916151391371, .19214022167803343, .16597173539815158, .16616509187367626, 
.16674340195556847, .16770259394569356, .16903335874226497, .17071721047882968, 
.17272999489387103, .17502593305271236, .17730676337252338, .17972409435001077, 
.18247503437867971, .18521147676515573, .18779344366255343, .19009855945199625, 
.19198177732702840, .19331254771406664, .19400236003393570, .16688487186651152, 
.16708292693163646, .16767509002674544, .16865662802136089, .17001718638773158, 
.17173691503224578, .17379021748358420, .17612962422445078, .17845111356954607, 
.18090914616438439, .18370379524879751, .18648130916553987, .18910009379241705, 
.19143666227012510, .19334467343282885, .19469250893576467, .19539102596317445, 
.16700874501247459, .16721616060967509, .16783581647525558, .16886136862199763, 
.17027996108781979, .17206857313937579, .17419833309998453, .17661816133242045, 
.17901334879654160, .18154366121378707, .18441424731692668, .18726151343733202, 
.18994149677140951, .19232934494371540, .19427712492419927, .19565199690041973, 
.19636419896361668, .16647818621601176, .16668796791476040, .16731451518524387, 
.16835091504357763, .16978344802114140, .17158804184714377, .17373477496976988, 
.17617152163918737, .17858130649177506, .18112502814653655, .18400865924066118, 
.18686689253670075, .18955563547915225, .19195015832722667, .19390267103061501, 
.19528052799229434, .19599416666542530, .16583999420444789, .16605027507087131, 
.16667818999974060, .16771646591229852, .16915085869028121, .17095670475879540, 
.17310353452094854, .17553880703957300, .17794571497662037, .18048509039200877, 
.18336240531362616, .18621313474882267, .18889384901479261, .19128053085439145, 
.19322622001858289, .19459904887281010, .19531001809793216, .16530084660414518, 
.16550936862720664, .16613197350465864, .16716128961026097, .16858297122082264, 
.17037233529977433, .17249898595857649, .17491074093157782, .17729388183766409, 
.17980773797856153, .18265571058280825, .18547702889918774, .18812985422332928, 
.19049156256570393, .19241681469227181, .19377518940863223, .19447866340601763, 
.16506988827689056, .16527414991529815, .16588404533494241, .16689239007383574, 
.16828520005888004, .17003840111032126, .17212235620365451, .17448609827206632, 
.17682224524698673, .17928702982979700, .18208003667424699, .18484755853487059, 
.18745038429396627, .18976804387754487, .19165770155432454, .19299112732602544, 
.19368173328822980, .16534406219912332, .16554150446992447, .16613111608557837, 
.16710617798820002, .16845352156571078, .17015030247863205, .17216832411893024, 
.17445869923940552, .17672374200731597, .17911494040222564, .18182623763056988, 
.18451444324423111, .18704407527350919, .18929763596708082, .19113574178775627, 
.19243315722310786, .19310522837150973, .16629253704994243, .16648075031487550, 
.16704293815270324, .16797308729302721, .16925924033179623, .17088033212068554, 
.17281021511210309, .17500287376515011, .17717354007977612, .17946737408030813, 
.18207089044204242, .18465477725431764, .18708838851901582, .18925804055426626, 
.19102877863657014, .19227919510658645, .19292709402815084, .16804145824114267, 
.16821838684862506, .16874705027882092, .16962231896485272, .17083374390479750, 
.17236246113441900, .17418484578131113, .17625839619348743, .17831408200269630, 
.18048934028419106, .18296163569361926, .18541854697439522, .18773530314188425, 
.18980285307718103, .19149162410712134, .19268486027874238, .19330335179856464, 
.17066039940208580, .17082451890058731, .17131512280420694, .17212806573260736, 
.17325457968713337, .17467825789335470, .17637829120826221, .17831612212804790, 
.18024063014665390, .18228043646577644, .18460262613845871, .18691407720486991, 
.18909678406044786, .19104706672782776, .19264160221852641, .19376905095897595, 
.19435369317353698, .17415191345115841, .17430235597802671, .17475230580761619, 
.17549862196673852, .17653425825534397, .17784534006424482, .17941396526999767, 
.18120570141357292, .18298866719269280, .18488196714416577, .18704137954870603, 
.18919466855102551, .19123125508267408, .19305342667272812, .19454482339489079, 
.19560017076082023, .19614768381784614, .17844545252808416, .17858206053207795, 
.17899086693615568, .17966968500861377, .18061311246366687, .18180971248768665, 
.18324438717875061, .18488675537866109, .18652454735832713, .18826709570591035, 
.19025842720609279, .19224780642416109, .19413244299769888, .19582098497277839, 
.19720452642600172, .19818433184770387, .19869289837964013, .18339663709609400, 
.18351993353219495, .18388913039073726, .18450289563855998, .18535730597651418, 
.18644313603941018, .18774783204341255, .18924476576745206, .19074068544622580, 
.19233534895155183, .19416110568113343, .19598830805243239, .19772197008462325, 
.19927723874912662, .20055287193138391, .20145691834014953, .20192636887074955, 
.18879242001957364, .18890349370053011, .18923630436210931, .18979025435569333, 
.19056269250243554, .19154630424469529, .19273072383746290, .19409261616878323, 
.19545629607421050, .19691255237208158, .19858263308960805, .20025660079720697, 
.20184692881725716, .20327511521246089, .20444747137389671, .20527880650455049, 
.20571064902395081, .19436214699120863, .19446246348135218, .19476324461054215, 
.19526451731002828, .19596470479448713, .19685809737382612, .19793614221414002, 
.19917826463154051, .20042425928906302, .20175686626970721, .20328723562404888, 
.20482298171286462, .20628335875937401, .20759577903241105, .20867366784564112, 
.20943828441159348, .20983555031020576, .19979391590463141, .19988507542923253, 
.20015860096406995, .20061507324871114, .20125384969833995, .20207058335278791, 
.20305822058986059, .20419844329561551, .20534412029489643, .20657104240251656, 
.20798156283434749, .20939820583958524, .21074606774691029, .21195778202328780, 
.21295314777670904, .21365928969697154, .21402618797391018, .20475505451242532, 
.20483852699239111, .20508920080608758, .20550819960669531, .20609576982212449, 
.20684880665289204, .20776157430440953, .20881763562746111, .20988058406610435, 
.21102037657370543, .21233199367857677, .21365012609026249, .21490462892510792, 
.21603247907688450, .21695887954902449, .21761601044504766, .21795740691160229, 
.20891504327817592, .20899191521509500, .20922301543252236, .20961006916522185, 
.21015428195482450, .21085382901803534, .21170429275802852, .21269095112147934, 
.21368621479693076, .21475516599938374, .21598675702930695, .21722542331487754, 
.21840469986261182, .21946495117583553, .22033569567891353, .22095321849684774, 
.22127398364205098, .21196887857571065, .21203964612151960, .21225270360387430, 
.21261050322216773, .21311539501643850, .21376704055123683, .21456254222245996, 
.21548898719784398, .21642648246207161, .21743587824613131, .21860115567924387, 
.21977480450484282, .22089313026401222, .22189898365843982, .22272513458569540, 
.22331099950089597, .22361529378573874, .21365875346833962, .21372319044617233, 
.21391758806253203, .21424530315176346, .21471011124970060, .21531351721928030, 
.21605453343290842, .21692242109051271, .21780490127852184, .21875879157330941, 
.21986368503837209, .22097950443657835, .22204474512833575, .22300403939046481, 
.22379252871318353, .22435190159549623, .22464248567483378, .21379205606716783, 
.21384917568861339, .21402202564465655, .21431507438083261, .21473386434351660, 
.21528220933828349, .21596156642905878, .21676397573355133, .21758585015483178, 
.21847965423957807, .21952056285071719, .22057659270221086, .22158830555363315, 
.22250173946163612, .22325386655759297, .22378804400510988, .22406570945776047, 
.21225404336231438, .21230215943450256, .21244847755294743, .21269878694482650, 
.21306075481610159, .21354102019121168, .21414409855808642, .21486558725834493, 
.21561279717112525, .21643298621636090, .21739619782368130, .21838053624860448, 
.21932903915626153, .22018920334993872, .22089975572910464, .22140549920290847, 
.22166870963711666, .20901410231296216, .20905098190957874, .20916413645604459, 
.20936086096425136, .20965127681449988, .21004535130917465, .21055127475483237, 
.21116906200088598, .21182007289782187, .21254502543926124, .21340744941825338, 
.21429873473901567, .21516534358868727, .21595676164266245, .21661393412869220, 
.21708335328131523, .21732816503652050, .20412519154952005, .20414827865927651, 
.20422065374808943, .20435125719024733, .20455292722968946, .20483939030991966, 
.20522305651654654, .20570919926283873, .20623701027834290, .20683897081084845, 
.20757014795439394, .20833932346978304, .20909780192079697, .20979804886600859, 
.21038423567564102, .21080528296186010, .21102558052815309, .19771677829288709, 
.19772345883125941, .19774724461296442, .19779880239070449, .19789385712987234, 
.19805020787585279, .19828489781190473, .19860918562563265, .19898410919986376, 
.19943201879930050, .19999718489591509, .20061036674487265, .20122945777937334, 
.20181127080242353, .20230468232055626, .20266223615273932, .20285027412794895, 
.18998225160957416, .18997012574113434, .18993813365385159, .18989868713739680, 
.18987046381363948, .18987550553806753, .18993573213994069, .19006892458150140, 
.19026177811607325, .19052461680809071, .19088843772254790, .19131053831145464, 
.19175728186908517, .19219142697702862, .19256835822311139, .19284578338399819, 
.19299298086954006, .18116231876738620, .18112944723776439, .18103585024230243, 
.18089564582911014, .18073043000460329, .18056650942071256, .18043074172838217, 
.18034773092791462, .18033302647508290, .18038323299819856, .18051372714765865, 
.18071236829432349, .18095578400071993, .18121432088344214, .18145179357597302, 
.18163279065252180, .18173068332865727, .17152621197563755, .17147130867907553, 
.17131222247756842, .17106469297866320, .17075311577763008, .17040794964347103, 
.17006088028389263, .16974338480698362, .16950211217380229, .16931864110186642, 
.16919077914910485, .16913998279843670, .16915463520561180, .16921415367373052, 
.16929258847452100, .16936309331460850, .16940432070322461, .16135261571544504, 
.16127516781922080, .16104901604732072, .16069140606365589, .16022935282220827, 
.15969724668503224, .15913135062316616, .15856982928285251, .15809149950396414, 
.15766213892600203, .15726066531512745, .15694384497652034, .15671282384957125, 
.15655722364735716, .15646278739993491, .15641266769078394, .15639186411772749, 
.15091207388566710, .15081238177241615, .15052002091559158, .15005361229141850, 
.14944257391399055, .14872493275065909, .14794117763355250, .14713577701935379, 
.14641951263989900, .14574220527832746, .14506335154824068, .14447523853995067, 
.14399217708291853, .14361463069038269, .14334094933954405, .14316527248352057, 
.14307974251965599, .14045228899480974, .14033142918517066, .13997604182684609, 
.13940599056974651, .13865286717073669, .13775799909738520, .13676569151933646, 
.13572618516725013, .13478078848953945, .13386385246772969, .13291578436550888, 
.13206308893088037, .13133299227884015, .13073685608026669, .13028588034971195, 
.12998559108232319, .12983568222078498, .13018724680352964, .13004697020337508, 
.12963376305464142, .12896860654325557, .12808504132408124, .12702734625313727, 
.12584321260652551, .12458802749326758, .12343110173266103, .12229240598983886, 
.12109443326964496, .11999523068740751, .11903406778451696, .11823266768791091, 
.11761460820765664, .11719654456779346, .11698567605171067, .12029056747720017, 
.12013315156231973, .11966891388043766, .11891984032455168, .11792121050346158, 
.11671991236964228, .11536660006076743, .11392113530773880, .11257745237000891, 
.11124273842867485, .10982348451925827, .10850549582966833, .10733868661604634, 
.10635407009095285, .10558645753324826, .10506273325358247, .10479709184012177, 
.11089297691039471, .11072105326942344, .11021364001864212, .10939362782461577, 
.10829784374966950, .10697546017030943, .10547969408667784, .10387420263711257, 
.10237358764120365, .10087422281273700, .99269073632705943E-01, 
.97767148717936819E-01, .96427223652532215E-01, .95288111165742234E-01, 
.94394159599901759E-01, .93781031381317506E-01, .93468992817074492E-01, 
.10208337507877069, .10189975319316762, .10135755590740286, .10048048907410548, 
.99306752269335746E-01, .97887498201480669E-01, .96278135732898645E-01, 
.94545457119355122E-01, .92920486178858258E-01, .91290972269885862E-01, 
.89539182855583127E-01, .87892369701998305E-01, .86416166761396573E-01, 
.85155427954604540E-01, .84161964706492218E-01, .83478368117369861E-01, 
.83129738593408678E-01, .93912695358652820E-01, .93720200256754482E-01, 
.93151658951213454E-01, .92231508980911106E-01, .90999158899198726E-01, 
.89507455761442659E-01, .87813656292358877E-01, .85987061689283079E-01, 
.84270848321645483E-01, .82546394856061495E-01, .80688226279579139E-01, 
.78936817533233367E-01, .77362602632771588E-01, .76014613719127724E-01, 
.74949873835067723E-01, .74215841472012295E-01, .73841030586978676E-01, 
.86399616228748852E-01, .86200952692337476E-01, .85614147293175194E-01, 
.84664287283033504E-01, .83391831868440791E-01, .81851049478044460E-01, 
.80100721494351490E-01, .78212072609801689E-01, .76436360311151391E-01, 
.74650772631753515E-01, .72724969892868127E-01, .70907863585738640E-01, 
.69272717778723875E-01, .67870927193663055E-01, .66762493501955109E-01, 
.65997661122443180E-01, .65606895216334823E-01, .79537195339422703E-01, 
.79334845744381141E-01, .78737188625894364E-01, .77769876747764408E-01, 
.76474261482703282E-01, .74905767464828188E-01, .73124391280907572E-01, 
.71202737740293759E-01, .69396437549862175E-01, .67580484873973723E-01, 
.65622308177866115E-01, .63774922008327128E-01, .62112648752156836E-01, 
.60687600729428817E-01, .59560710402115767E-01, .58783076416971509E-01, 
.58385741039943893E-01, .73299616188509623E-01, .73095773787072812E-01, 
.72493809048523025E-01, .71519851542282631E-01, .70215982778864955E-01, 
.68638522681479880E-01, .66848381960875253E-01, .64919043322698808E-01, 
.63107272380919455E-01, .61287607979360302E-01, .59327518045716530E-01, 
.57480379756574572E-01, .55820071076189158E-01, .54398032156118943E-01, 
.53274384166524996E-01, .52499426577695227E-01, .52103595117897750E-01, 
.67648428618800757E-01, .67444963815116402E-01, .66844266020648802E-01, 
.65872847805645479E-01, .64573354314884035E-01, .63002743030584618E-01, 
.61222543703151291E-01, .59306654865108055E-01, .57510259260743796E-01, 
.55708881249338364E-01, .53771875457410787E-01, .51949899302183696E-01, 
.50315185101346041E-01, .48917409688098427E-01, .47814514288231448E-01, 
.47054701098456629E-01, .46666870666913239E-01, .62537882844850942E-01, 
.62336337142986573E-01, .61741493513949822E-01, .60780149549045134E-01, 
.59495344382352211E-01, .57944419934503048E-01, .56189236733340885E-01, 
.54303687196793946E-01, .52539179141421928E-01, .50773357569549524E-01, 
.48878873513393692E-01, .47101242917884048E-01, .45510153005256862E-01, 
.44152731073179657E-01, .43083760335511485E-01, .42348428131680931E-01, 
.41973450934192302E-01, .57919165250995067E-01, .57720766114090417E-01, 
.57135421766982025E-01, .56190114694922792E-01, .54928108242013593E-01, 
.53406883571098504E-01, .51688353933831088E-01, .49846024911853734E-01, 
.48125834157794485E-01, .46408393348369804E-01, .44570647615299525E-01, 
.42851167162743967E-01, .41316472974954568E-01, .40010639093761097E-01, 
.38984679211249838E-01, .38280209876441491E-01, .37921383922389439E-01, 
.53743516265721915E-01, .53549205922216021E-01, .52976151107131046E-01, 
.52051419547180018E-01, .50818332787076634E-01, .49334278277038963E-01, 
.47660958997473940E-01, .45871155906908011E-01, .44204086985085177E-01, 
.42543920614847290E-01, .40772533455038396E-01, .39120290897585286E-01, 
.37650166732717025E-01, .36402919464314065E-01, .35425496180394755E-01, 
.34755698132410882E-01, .34414967930214403E-01, .49964332594343012E-01, 
.49774804525518392E-01, .49216084922242755E-01, .48315230304857792E-01, 
.47115462224069156E-01, .45673858114878924E-01, .44051667888341139E-01, 
.42320654622704811E-01, .40712450093587531E-01, .39115155151256714E-01, 
.37415920449111892E-01, .35836106280566354E-01, .34434953783454562E-01, 
.33249835186897975E-01, .32323581766547309E-01, .31690179103649248E-01, 
.31368394831543689E-01, .46538426221300180E-01, .46354164286036337E-01, 
.45811199111644801E-01, .44936485224200129E-01, .43772997385911741E-01, 
.42377307137840783E-01, .40809993170298500E-01, .39141552742162203E-01, 
.37595481827112767E-01, .36064019059777956E-01, .34439704756571002E-01, 
.32934463594637642E-01, .31603769076792819E-01, .30481677237967290E-01, 
.29607033072196368E-01, .29010178890430834E-01, .28707369059288289E-01, 
.43426637887748316E-01, .43247954445784297E-01, .42721651386225953E-01, 
.41874498049016717E-01, .40749089757508836E-01, .39401319373248499E-01, 
.37890903988769056E-01, .36286871963479919E-01, .34804287390504215E-01, 
.33339602782937516E-01, .31790700282439896E-01, .30359924516003663E-01, 
.29099055640454778E-01, .28038998369844362E-01, .27214856573354523E-01, 
.26653608854656754E-01, .26369233177617551E-01, .40593995067834829E-01, 
.40421065797848169E-01, .39911925822677920E-01, .39093085009216599E-01, 
.38006642475310028E-01, .36707664093408095E-01, .35254844295942292E-01, 
.33715582479533539E-01, .32296411172271680E-01, .30897979440352948E-01, 
.29423353772090526E-01, .28065366685401834E-01, .26872253426689353E-01, 
.25871988802178975E-01, .25096244242779321E-01, .24568967442015331E-01, 
.24302129892897466E-01, .38009576825530307E-01, .37842471045391471E-01, 
.37350679531568898E-01, .36560387980249215E-01, .35513099171020170E-01, 
.34262928295144983E-01, .32867415373758917E-01, .31392210701034172E-01, 
.30035362918358528E-01, .28701636088738679E-01, .27299056554525247E-01, 
.26011163335581629E-01, .24882841097155859E-01, .23939383232836835E-01, 
.23209361675657240E-01, .22714039277510104E-01, .22463653303911507E-01, 
.35646211787230299E-01, .35484917908437108E-01, .35010420995485597E-01, 
.34248527602251361E-01, .33240060566282738E-01, .32038084420550103E-01, 
.30698881037717240E-01, .29286267359046717E-01, .27989962690027920E-01, 
.26718722864081567E-01, .25385274160602413E-01, .24164181865649195E-01, 
.23097198707931386E-01, .22207191178790751E-01, .21519961917345287E-01, 
.21054421788924575E-01, .20819328657443299E-01, .33480099229938616E-01, 
.33324545521557739E-01, .32867112261770547E-01, .32133181321615065E-01, 
.31162827756584386E-01, .30007987503957077E-01, .28723606325526521E-01, 
.27371614857168314E-01, .26133632107935217E-01, .24922257431595693E-01, 
.23654643671822146E-01, .22496765013662684E-01, .21487469538648416E-01, 
.20647443465670261E-01, .20000030991211663E-01, .19562095456215362E-01, 
.19341142223805570E-01, .31490412004338351E-01, .31340483148904599E-01, 
.30899754940225087E-01, .30193147855971159E-01, .29259935875794806E-01, 
.28150868277243796E-01, .26919499707808391E-01, .25625848069977263E-01, 
.24443710403748764E-01, .23289368091829962E-01, .22084128006123582E-01, 
.20985790138055495E-01, .20030523077274286E-01, .19237063256391934E-01, 
.18626579229300534E-01, .18214156151549751E-01, .18006242138644990E-01, 
.29658915821182526E-01, .29514465876977099E-01, .29089995058715720E-01, 
.28409933779033700E-01, .27512714713631179E-01, .26447859882469164E-01, 
.25267497704206266E-01, .24029728798022343E-01, .22900836708910525E-01, 
.21800618141959844E-01, .20654270898057575E-01, .19611851319309637E-01, 
.18707064585015927E-01, .17956907533298993E-01, .17380624631943935E-01, 
.16991755979677548E-01, .16795856707664494E-01, .27969622407227699E-01, 
.27830484808089230E-01, .27421764343300306E-01, .26767379842459964E-01, 
.25904893925327490E-01, .24882575772300925E-01, .23751109273861190E-01, 
.22566690794905770E-01, .21488414790106207E-01, .20439426634526359E-01, 
.19348566040172346E-01, .18358575518803419E-01, .17500899780631674E-01, 
.16790984612753784E-01, .16246370988694435E-01, .15879253089148154E-01, 
.15694429386690710E-01, .26408483104629104E-01, .26274478279324397E-01, 
.25880964292751271E-01, .25251333120420723E-01, .24422258465545910E-01, 
.23440743749668990E-01, .22356023946422353E-01, .21222417953130287E-01, 
.20192161142159409E-01, .19191584288503379E-01, .18152936139404729E-01, 
.17212064890553706E-01, .16398342570788305E-01, .15725831216790685E-01, 
.15210560148515972E-01, .14863546594860000E-01, .14688944780840419E-01, 
.24963123135721701E-01, .24834064249569691E-01, .24455192944945994E-01, 
.23849364441114004E-01, .23052353051857762E-01, .22109894015220684E-01, 
.21069780277975143E-01, .19984490525956078E-01, .18999729333110352E-01, 
.18044854993725341E-01, .17055309032583371E-01, .16160448494435344E-01, 
.15387744744352113E-01, .14750023531802011E-01, .14261968706400018E-01, 
.13933563637581083E-01, .13768410902601523E-01, .23622613516421247E-01, 
.23498310751085872E-01, .23133510965273447E-01, .22550527270749618E-01, 
.21784231129333149E-01, .20879095728820487E-01, .19881487846665602E-01, 
.18842090926864244E-01, .17900400247681407E-01, .16988650263337117E-01, 
.16045275209418702E-01, .15193524289432991E-01, .14459124732032501E-01, 
.13853795510787142E-01, .13391018846657910E-01, .13079865609937211E-01, 
.12923463233711583E-01, .22377276286919875E-01, .22257539992701716E-01, 
.21906242394795723E-01, .21345152956177067E-01, .20608242605207421E-01, 
.19738735480648268E-01, .18781595035075468E-01, .17785759894389394E-01, 
.16884827396645423E-01, .16013763882236701E-01, .15113811527937603E-01, 
.14302473415104879E-01, .13603874485162424E-01, .13028740528368971E-01, 
.12589476807487609E-01, .12294345002707817E-01, .12146060952901144E-01, 
.21218518488515335E-01, .21103161472766918E-01, .20764805200908110E-01, 
.20224677098508344E-01, .19515854584285931E-01, .18680331169267653E-01, 
.17761695147083846E-01, .16807194384859749E-01, .15944827363880278E-01, 
.15112155325006378E-01, .14253057817860315E-01, .13479631394137100E-01, 
.12814526970231055E-01, .12267576811396515E-01, .11850217571155962E-01, 
.11569990119514687E-01, .11429251850344735E-01, .20138690604067859E-01, 
.20027529761024107E-01, .19701567112483873E-01, .19181492236856525E-01, 
.18499499845212205E-01, .17696375439552527E-01, .16814364262878051E-01, 
.15899079678305388E-01, .15073206895959172E-01, .14276772334657116E-01, 
.13456135403128703E-01, .12718303837553313E-01, .12084570407126452E-01, 
.11563961408152865E-01, .11167039383372199E-01, .10900700421628181E-01, 
.10766988247222666E-01, .19130965678298622E-01, .19023823118946046E-01, 
.18709722767718158E-01, .18208822630217867E-01, .17552448500549812E-01, 
.16780203667301603E-01, .15933025245787977E-01, .15054949421631457E-01, 
.14263619637659880E-01, .13501404866105269E-01, .12716998798703747E-01, 
.12012616930831775E-01, .11408298587157990E-01, .10912341183910747E-01, 
.10534515898891740E-01, .10281139817006265E-01, .10153980986883188E-01, 
.18189235875677508E-01, .18085939679656327E-01, .17783188785182157E-01, 
.17300617561138390E-01, .16668698998233832E-01, .15925882300646940E-01, 
.15111833281236973E-01, .14269068482212533E-01, .13510446868285592E-01, 
.12780564188865990E-01, .12030313731875755E-01, .11357395213614759E-01, 
.10780689196546209E-01, .10307832267332831E-01, .99478769925308505E-02, 
.97066186777541759E-02, .95855821563413397E-02, .17308023740962075E-01, 
.17208408424071942E-01, .16916513909148428E-01, .16451460168498945E-01, 
.15842885268920588E-01, .15128114101233653E-01, .14345579172400428E-01, 
.13536334468356108E-01, .12808697727806181E-01, .12109382251347884E-01, 
.11391356161233882E-01, .10748060919997042E-01, .10197304058560419E-01, 
.97461215786376933E-02, .94029116663028288E-02, .91729978951763692E-02, 
.90576897916684884E-02, .16482405872241031E-01, .16386312641205700E-01, 
.16104801851337603E-01, .15656489321736237E-01, .15070197377986226E-01, 
.14382157448347432E-01, .13629607321550698E-01, .12852194580385950E-01, 
.12153925338276028E-01, .11483527448153712E-01, .10795928152404311E-01, 
.10180550484542444E-01, .96542066924293554E-02, .92233856866320799E-02, 
.88958882049875957E-02, .86766100756683930E-02, .85666696515295411E-02, 
.15707947093901546E-01, .15615223943205659E-01, .15343644849253119E-01, 
.14911332473008064E-01, .14346313502781113E-01, .13683757380849437E-01, 
.12959745900289423E-01, .12212575105222081E-01, .11542155952698675E-01, 
.10899133743337069E-01, .10240287379550636E-01, .96512448267206379E-02, 
.91478936842575766E-02, .87362234142971760E-02, .84234879617590564E-02, 
.82141942464680311E-02, .81092904263641245E-02, .14980643532487797E-01, 
.14891145225172923E-01, .14629066292570473E-01, .14212047775132804E-01, 
.13667341435612811E-01, .13029086468924327E-01, .12332247175042145E-01, 
.11613821384509060E-01, .10969828835294073E-01, .10352740732430614E-01, 
.97210877164903408E-02, .91569107829893946E-02, .86752371751469184E-02, 
.82815994645056915E-02, .79827500362133919E-02, .77828413377410343E-02, 
.76826696554767255E-02, .14296873260340764E-01, .14210461224730724E-01, 
.13957471043044407E-01, .13555074042373751E-01, .13029768123702379E-01, 
.12414693946187579E-01, .11743736326456235E-01, .11052646497370835E-01, 
.10433745026411412E-01, .98412427149095691E-02, .92353289128093968E-02, 
.86946516258497412E-02, .82334363731736415E-02, .78567969614290675E-02, 
.75710247447960139E-02, .73799483517813539E-02, .72842282709612632E-02, 
.13653353388809344E-01, .13569895555524486E-01, .13325602299534353E-01, 
.12937187368859168E-01, .12430416011292810E-01, .11837461807282790E-01, 
.11191167400605150E-01, .10526087225918494E-01, .99310234991505479E-02, 
.93618452294776389E-02, .87803137590318492E-02, .82618650418613496E-02, 
.78199764466578246E-02, .74593772656286125E-02, .71859342518540851E-02, 
.70031796001608273E-02, .69116521575888542E-02, .13047102672439711E-01, 
.12966473270217159E-01, .12730504046428108E-01, .12355463414908304E-01, 
.11866405156774673E-01, .11294566799143990E-01, .10671785271008385E-01, 
.10031466132672136E-01, .94590634945325488E-02, .89120278026404813E-02, 
.83536114612667318E-02, .78562072718099795E-02, .74325934976651794E-02, 
.70871457679952546E-02, .68253390785765243E-02, .66504337383921450E-02, 
.65628594658018909E-02, .12475408834449612E-01, .12397488158799062E-01, 
.12169488278046828E-01, .11807244532524963E-01, .11335120267882999E-01, 
.10783447415719854E-01, .10183092685333423E-01, .95663587882272583E-02, 
.90155120443679328E-02, .84895118957980433E-02, .79530261924738929E-02, 
.74755623720547792E-02, .70692445762097246E-02, .67381226316635167E-02, 
.64873094710371998E-02, .63198155904858446E-02, .62359726768074298E-02, 
.11935799947443213E-01, .11860474112720169E-01, .11640106319400278E-01, 
.11290111034195083E-01, .10834181937832001E-01, .10301775154290973E-01, 
.97228216286928612E-02, .91285653558368547E-02, .85982358685336938E-02, 
.80922332221334912E-02, .75765699813290432E-02, .71180157554541351E-02, 
.67280818979854233E-02, .64105176541003816E-02, .61701008107973472E-02, 
.60096119566515147E-02, .59292946186668052E-02, .11426019306204283E-01, 
.11353179989033639E-01, .11140123669714692E-01, .10801856019054616E-01, 
.10361421480424012E-01, .98474294130423030E-02, .92889083641186244E-02, 
.87160858756631342E-02, .82052969759302541E-02, .77183177529086449E-02, 
.72224392530515578E-02, .67818313269350341E-02, .64074305865305875E-02, 
.61027085794756993E-02, .58721324082847909E-02, .57182707528071640E-02, 
.56412877868211196E-02, .10944003315194821E-01, .10873547495142803E-01, 
.10667497883378479E-01, .10340463261673460E-01, .99148588573114583E-02, 
.94184755094497245E-02, .88794716159442764E-02, .83270987019459421E-02, 
.78349314134313897E-02, .73660608517335923E-02, .68889944587562984E-02, 
.64654316528397621E-02, .61057693852938581E-02, .58132223153865657E-02, 
.55919691419294686E-02, .54443829520977611E-02, .53705564432767611E-02, 
.10487861985087209E-01, .10419691686917637E-01, .10220359076521740E-01, 
.99040877445296920E-02, .94926832694197175E-02, .90131453814263629E-02, 
.84927934480811818E-02, .79599416364424594E-02, .74855307004365740E-02, 
.70339090713162503E-02, .65747423278696400E-02, .61673807021499208E-02, 
.58217138840157574E-02, .55407186061729825E-02, .53283055030521704E-02, 
.51866668945686938E-02, .51158310639825046E-02, .10055861692722616E-01, 
.99898837333757991E-02, .97969927089484824E-02, .94910384784770586E-02, 
.90932360500417305E-02, .86298206013416152E-02, .81273024601913480E-02, 
.76130953752851505E-02, .71556255626562398E-02, .67204442249836236E-02, 
.62783203571831871E-02, .58863687771593346E-02, .55540018830100308E-02, 
.52839757934045250E-02, .50799516838032606E-02, .49439546075179322E-02, 
.48759547797733874E-02, .96464099093280502E-02, .95825356518440593E-02, 
.93958243426872368E-02, .90997633083759726E-02, .87149955517076549E-02, 
.82670173896142642E-02, .77815589834469160E-02, .72851689470874751E-02, 
.68438716409911417E-02, .64243694086587215E-02, .59984832142340348E-02, 
.56211993153852452E-02, .53014805827057369E-02, .50418783570892745E-02, 
.48458214074967162E-02, .47151798403152504E-02, .46498715170747032E-02, 
.92580416442831718E-02, .91961867601917380E-02, .90154061217375941E-02, 
.87288354453810679E-02, .83565637645645011E-02, .79233733617080845E-02, 
.74542420062372761E-02, .69748868702273039E-02, .65490369025334493E-02, 
.61444967008895783E-02, .57340907849922100E-02, .53707772967015541E-02, 
.50630953375081084E-02, .48134059821046396E-02, .46249212508286050E-02, 
.44993675669573567E-02, .44366155936676433E-02, .88894073875542684E-02, 
.88294916287216946E-02, .86544047540741496E-02, .83769415046364513E-02, 
.80166544422521548E-02, .75976357817291338E-02, .71441376006888425E-02, 
.66810777987606519E-02, .62699905229640180E-02, .58797363112345607E-02, 
.54840976386996377E-02, .51340990323311539E-02, .48378797551841745E-02, 
.45976239358199705E-02, .44163412482443433E-02, .42956247498431898E-02, 
.42353025647229982E-02, .85392623642182360E-02, .84812093447394227E-02, 
.83115908077817675E-02, .80428708585352018E-02, .76940825434207317E-02, 
.72886511288190597E-02, .68501286546866777E-02, .64026644609622239E-02, 
.60056930446458516E-02, .56290869825998445E-02, .52475437180102055E-02, 
.49102431473708108E-02, .46249469575291293E-02, .43936745768273881E-02, 
.42192466020683651E-02, .41031321911310153E-02, .40451211469272625E-02, 
.82064569401941456E-02, .81501939286155942E-02, .79858291592816728E-02, 
.77255061422389507E-02, .73877548241441217E-02, .69953558102726146E-02, 
.65711857424241658E-02, .61386547233289101E-02, .57551876438640490E-02, 
.53916274824092822E-02, .50235460917901695E-02, .46983625974392059E-02, 
.44234818465333416E-02, .42007698426294437E-02, .40328703494305366E-02, 
.39211373250616060E-02, .38653260753311182E-02, .78899280401393580E-02, 
.78353857620741315E-02, .76760704537484660E-02, .74238147706409512E-02, 
.70966614394033980E-02, .67167678786896456E-02, .63063589903247022E-02, 
.58881336370451461E-02, .55175923650151882E-02, .51665090418853621E-02, 
.48112916322099204E-02, .44976775840152184E-02, .42327342438323338E-02, 
.40181845872294293E-02, .38565068596833787E-02, .37489478269554697E-02, 
.36952317696919488E-02, .75886914570719661E-02, .75358039081023006E-02, 
.73813434566545617E-02, .71368413450571778E-02, .68198684312401319E-02, 
.64519796302407312E-02, .60547708152596169E-02, .56502563442199514E-02, 
.52920931999658142E-02, .49529486231244414E-02, .46100304980078947E-02, 
.43074692529979372E-02, .40520127908135022E-02, .38452506587562944E-02, 
.36895060548164040E-02, .35859259331202857E-02, .35342067054572514E-02, 
.73018349491692320E-02, .72505392178131688E-02, .71007481914445343E-02, 
.68637008441974848E-02, .65565109976910277E-02, .62001509778881539E-02, 
.58156094289682318E-02, .54242417383353805E-02, .50779379079768840E-02, 
.47502229107775863E-02, .44190703227685651E-02, .41270740777124993E-02, 
.38806795131128930E-02, .36813516232635851E-02, .35312682610616338E-02, 
.34314833813518441E-02, .33816683998540301E-02, .70285120327881366E-02, 
.69787481332224564E-02, .68334497720745312E-02, .66035725078543404E-02, 
.63057874506830735E-02, .59605035076541130E-02, .55881230169872557E-02, 
.52093667878363847E-02, .48744304860889173E-02, .45576629395748689E-02, 
.42377710213138860E-02, .39558788415952366E-02, .37181449669493587E-02, 
.35259180542504234E-02, .33812396130194779E-02, .32850768946427510E-02, 
.32370789363713299E-02, .67679363924179952E-02, .67196471065927785E-02, 
.65786728509198865E-02, .63556943336299122E-02, .60669537830864212E-02, 
.57323151381767582E-02, .53716145126461263E-02, .50049614440882293E-02, 
.46809262121397270E-02, .43746492811243612E-02, .40655401393361436E-02, 
.37933161475937355E-02, .35638638962845028E-02, .33784233187290048E-02, 
.32389079426763225E-02, .31462041415817035E-02, .30999409616993493E-02, 
.65193768383024191E-02, .64725075670461060E-02, .63356966126603461E-02, 
.61193581172149089E-02, .58393187754137172E-02, .55149153141777580E-02, 
.51654368971318634E-02, .48104040653915705E-02, .44968271930144214E-02, 
.42006077237555801E-02, .39018286815818847E-02, .36388603912942802E-02, 
.34173313395438705E-02, .32383798001807129E-02, .31037990950018735E-02, 
.30144001160948763E-02, .29697940982789366E-02, .66195841162630751E-02, 
.65721407591901029E-02, .64336467416681647E-02, .62146241984182430E-02, 
.59310681173821442E-02, .56025286548494549E-02, .52485162964537929E-02, 
.48887897454730940E-02, .45709893877622781E-02, .42707091765062126E-02, 
.39677602005932285E-02, .37010564621634515E-02, .34763303440455105E-02, 
.32947608595027137E-02, .31581895680122714E-02, .30674575932746577E-02, 
.30221832007679213E-02, .63884389415257911E-02, .63423231367270511E-02, 
.62077209683967204E-02, .59949043765242472E-02, .57194785548654054E-02, 
.54004995181302059E-02, .50569655399331009E-02, .47080844975433090E-02, 
.44000412010267382E-02, .41091403348119887E-02, .38158210777580817E-02, 
.35577420688070673E-02, .33403963651233125E-02, .31648695138847558E-02, 
.30328922682590228E-02, .29452364285115504E-02, .29015043528528941E-02, 
.61841433679657347E-02, .61392108451065082E-02, .60080770690094107E-02, 
.58007895839993469E-02, .55326042929431151E-02, .52221349979680665E-02, 
.48879221952981373E-02, .45486825476069755E-02, .42493073628302631E-02, 
.39667338375609209E-02, .36819587499574914E-02, .34315273785412064E-02, 
.32207222347932184E-02, .30505477667497520E-02, .29226381698177800E-02, 
.28377051980073083E-02, .27953380744018892E-02, .60119424856849863E-02, 
.59680148182825275E-02, .58398260526346347E-02, .56372321343913998E-02, 
.53751909578506271E-02, .50719394727210829E-02, .47456285119296029E-02, 
.44145574129966104E-02, .41225219353207268E-02, .38469956811653118E-02, 
.35694472853644507E-02, .33254807078104192E-02, .31202017252989797E-02, 
.29545475581982490E-02, .28300720437536299E-02, .27474372388296611E-02, 
.27062219863195573E-02, .58749484058703994E-02, .58318251594123398E-02, 
.57059937176477636E-02, .55071559505411802E-02, .52500302604871485E-02, 
.49525510335056214E-02, .46325565561762773E-02, .43080119655777857E-02, 
.40218379385797351E-02, .37519370022993767E-02, .34801544223407536E-02, 
.32413433732151392E-02, .30404695226957522E-02, .28784173652392662E-02, 
.27566774984198695E-02, .26758730549496521E-02, .26355750820568282E-02, 
.57756751076692694E-02, .57331376237514439E-02, .56090225910334635E-02, 
.54129192932029519E-02, .51593712178566453E-02, .48660919347728437E-02, 
.45506917994700413E-02, .42308925264075261E-02, .39489786699715744E-02, 
.36831648180782441E-02, .34155698688685455E-02, .31805018511004354E-02, 
.29828251801096986E-02, .28233865699558085E-02, .27036311803902121E-02, 
.26241542831897942E-02, .25845215373436351E-02, .57155947805321572E-02, 
.56734129773566514E-02, .55503401050043298E-02, .53558969216966614E-02, 
.51045204893074049E-02, .48137900910432173E-02, .45011775396483393E-02, 
.41842567145903313E-02, .39049263630477954E-02, .36415904464784329E-02, 
.33765337585782759E-02, .31437339549695090E-02, .29479942881996363E-02, 
.27901387847879288E-02, .26715852665908546E-02, .25929122870194895E-02, 
.25536823521331293E-02, .56954675804620604E-02, .56534051324317446E-02, 
.55306819657567939E-02, .53367958161632230E-02, .50861479957800141E-02, 
.47962726907323748E-02, .44845951993257235E-02, .41686397878632833E-02, 
.38901758158022865E-02, .36276707731874068E-02, .33634650794002736E-02, 
.31314256368299956E-02, .29363352278290187E-02, .27790103086405937E-02, 
.26608595702102863E-02, .25824559895615577E-02, .25433610348864198E-02
};
static double tabled4 [ 1700 ]= {
-0.36068645816262769E-01, -0.36094714230189895E-01, -0.36173065457728505E-01, 
-0.36304177981874131E-01, -0.36488144546297355E-01, -0.36723714793936685E-01, 
-0.37008426413166630E-01, -0.37336141344075910E-01, -0.37663741388330521E-01, 
-0.38012225150597351E-01, -0.38409448870327018E-01, -0.38804449405676199E-01, 
-0.39176451640435206E-01, -0.39507640899221617E-01, -0.39777390603876923E-01, 
-0.39967497273643994E-01, -0.40065860452209827E-01, -0.35281637805292643E-01, 
-0.35308618040260593E-01, -0.35389612996058348E-01, -0.35524848839059188E-01, 
-0.35714039471801601E-01, -0.35955481344722651E-01, -0.36246267375157654E-01, 
-0.36579835380692073E-01, -0.36912282964864498E-01, -0.37265008397786348E-01, 
-0.37666104167831865E-01, -0.38064097902246394E-01, -0.38438252726622164E-01, 
-0.38770884793654606E-01, -0.39041515515703368E-01, -0.39232097746498502E-01, 
-0.39330662521137698E-01, -0.32949955468751971E-01, -0.32979569436207723E-01, 
-0.33068197264820305E-01, -0.33215327045201012E-01, -0.33419565375637394E-01, 
-0.33677885921149425E-01, -0.33986085124085137E-01, -0.34336369235525362E-01, 
-0.34682595718267144E-01, -0.35047306235616471E-01, -0.35459251372241873E-01, 
-0.35865530710755142E-01, -0.36245543145081542E-01, -0.36582007387129141E-01, 
-0.36854900644733395E-01, -0.37046653093716769E-01, -0.37145693540622630E-01, 
-0.29168870249354901E-01, -0.29202502473946016E-01, -0.29302763510427161E-01, 
-0.29467981610605817E-01, -0.29695026262906276E-01, -0.29978802252361551E-01, 
-0.30313084099207566E-01, -0.30688174104415128E-01, -0.31054602793189036E-01, 
-0.31436615224761037E-01, -0.31863875476186007E-01, -0.32281462046673585E-01, 
-0.32669074893841528E-01, -0.33010139065804980E-01, -0.33285433373500727E-01, 
-0.33478213083938828E-01, -0.33577581624823077E-01, -0.24141760767213926E-01, 
-0.24180122843073353E-01, -0.24294047308053637E-01, -0.24480414100207622E-01, 
-0.24733930021220884E-01, -0.25046938234214115E-01, -0.25410722614886704E-01, 
-0.25813279700849848E-01, -0.26201447293410597E-01, -0.26601371907480922E-01, 
-0.27043564507313544E-01, -0.27471117130574173E-01, -0.27864319900731661E-01, 
-0.28207662165813639E-01, -0.28483135574410019E-01, -0.28675212899843330E-01, 
-0.28773964447047894E-01, -0.18224906832224125E-01, -0.18267632221688083E-01, 
-0.18394096399084920E-01, -0.18599662736336200E-01, -0.18876785915051285E-01, 
-0.19215169309674356E-01, -0.19603555136571683E-01, -0.20027668001304802E-01, 
-0.20431424346171220E-01, -0.20842496245916609E-01, -0.21291672177979851E-01, 
-0.21721070758490935E-01, -0.22112045216620906E-01, -0.22450581036913147E-01, 
-0.22720384621525456E-01, -0.22907597930109075E-01, -0.23003566928987799E-01, 
-0.11919173316195047E-01, -0.11964494656710203E-01, -0.12098277467447729E-01, 
-0.12314587994875998E-01, -0.12603978248137674E-01, -0.12953974458387904E-01, 
-0.13351270635640448E-01, -0.13779920163131586E-01, -0.14183162880772412E-01, 
-0.14589076484473997E-01, -0.15027508066298021E-01, -0.15441880408761247E-01, 
-0.15815317785718305E-01, -0.16135825916659637E-01, -0.16389441431223797E-01, 
-0.16564500295231314E-01, -0.16653951713314616E-01, -0.58718323768999162E-02, 
-0.59165134948625155E-02, -0.60480986448959283E-02, -0.62598810739211948E-02, 
-0.65413247414236892E-02, -0.68788149967480231E-02, -0.72580693754607222E-02, 
-0.76626717124924073E-02, -0.80389670784434195E-02, -0.84135459675751655E-02, 
-0.88134299947951189E-02, -0.91869296230202031E-02, -0.95198798501555729E-02, 
-0.98029107876325579E-02, -0.10025103181840687E-01, -0.10177568525015147E-01, 
-0.10255191792085859E-01, .36130872602185287E-02, .35854722934360226E-02, 
.35047091675578803E-02, .33765109348626965E-02, .32096307604818856E-02, 
.30149095858541322E-02, .28033318579859258E-02, .25863464355412110E-02, 
.23928996787582315E-02, .22085809049737235E-02, .20211516426393503E-02, 
.18550425540591611E-02, .17144483075494700E-02, .16006087649927004E-02, 
.15149688307523454E-02, .14581335285319679E-02, .14298081039563716E-02, 
.53375255413218624E-02, .53245022365252682E-02, .52869209629566594E-02, 
.52288880911662838E-02, .51565475256157645E-02, .50771890162370215E-02, 
.49979205707410381E-02, .49252944869537102E-02, .48691624601575239E-02, 
.48245335793986625E-02, .47896198523473274E-02, .47692037178641133E-02, 
.47611903027984489E-02, .47621075655242852E-02, .47679093678492457E-02, 
.47745216362928937E-02, .47787211869559807E-02, .52958238275959274E-02, 
.53007091299355257E-02, .53160789664352800E-02, .53439360449398050E-02, 
.53870292406743405E-02, .54480735999273147E-02, .55292163019292628E-02, 
.56308618373808244E-02, .57397766703258888E-02, .58623356164649761E-02, 
.60091480087450019E-02, .61615488033656294E-02, .63101478067384177E-02, 
.64461299888095118E-02, .65592334062689875E-02, .66401319553680803E-02, 
.66823605296516897E-02, .35000179420722195E-02, .35249158260655545E-02, 
.35994546604669574E-02, .37232765855986253E-02, .38953186358096936E-02, 
.41131570434989979E-02, .43733613002577918E-02, .46694304170551747E-02, 
.49623549928105928E-02, .52711731156210410E-02, .56202665622963506E-02, 
.59648459092051928E-02, .62874341042124292E-02, .65733292189210334E-02, 
.68054344021332147E-02, .69686678587183078E-02, .70530290374242431E-02, 
.77316768874734974E-04, .12282405696222246E-03, .25832540984581089E-03, 
.48111214114906808E-03, .78630853048089575E-03, .11663174823964319E-02, 
.16120649946133690E-02, .21099916974092687E-02, .25943244427887786E-02, 
.30972313365922456E-02, .36574748322867889E-02, .42029887552089938E-02, 
.47077510665830304E-02, .51507937152243631E-02, .55077555031336984E-02, 
.57574292981301253E-02, .58860404327169590E-02, -0.47279860172759827E-02, 
-0.46629276033270540E-02, -0.44695419202364431E-02, -0.41526214408667429E-02, 
-0.37204472282583835E-02, -0.31852906804241362E-02, -0.25613663593366141E-02, 
-0.18687936441803307E-02, -0.11991288187337655E-02, -0.50754976481125712E-03, 
.25878441385740919E-03, .10012040899453508E-02, .16851308039209807E-02, 
.22832011227422728E-02, .27636407236322062E-02, .30989548115165095E-02, 
.32714543732923402E-02, -0.10562014423478859E-01, -0.10480086611822830E-01, 
-0.10236688097150505E-01, -0.98382162901072753E-02, -0.92956058515408120E-02, 
-0.86248374865958519E-02, -0.78442512307776990E-02, -0.69794088796955216E-02, 
-0.61446255760412016E-02, -0.52838640978306864E-02, -0.43314860092407417E-02, 
-0.34101194068372901E-02, -0.25623729370994842E-02, -0.18218016374008364E-02, 
-0.12273730548116482E-02, -0.81274981943243020E-03, -0.59952804540404933E-03, 
-0.16978725722935992E-01, -0.16884009728333271E-01, -0.16602618795387398E-01, 
-0.16141940567706979E-01, -0.15514580577265782E-01, -0.14738931390466791E-01, 
-0.13836059939817515E-01, -0.12835348164914323E-01, -0.11868954199973672E-01, 
-0.10871943583999691E-01, -0.97681141452095905E-02, -0.86994893933279634E-02, 
-0.77155938785831097E-02, -0.68555699506673024E-02, -0.61649180422710644E-02, 
-0.56830003597928697E-02, -0.54351165477360850E-02, -0.23466758217683011E-01, 
-0.23364331017785130E-01, -0.23059940208505592E-01, -0.22561308413133922E-01, 
-0.21881645350366886E-01, -0.21040308347382995E-01, -0.20059495957520011E-01, 
-0.18970472070298904E-01, -0.17916820686502744E-01, -0.16827718602538989E-01, 
-0.15619456858131830E-01, -0.14447252572882666E-01, -0.13365851660124876E-01, 
-0.12418942566956357E-01, -0.11657417772320287E-01, -0.11125477151615035E-01, 
-0.10851683138662895E-01, -0.29484162680112821E-01, -0.29379552485413660E-01, 
-0.29068528769746950E-01, -0.28558553066249812E-01, -0.27862447086100313E-01, 
-0.26999142227007663E-01, -0.25990396867479608E-01, -0.24867344733201353E-01, 
-0.23777684804480750E-01, -0.22648124868620220E-01, -0.21391096882876451E-01, 
-0.20167675744116188E-01, -0.19035633441761048E-01, -0.18041736356909789E-01, 
-0.17240654877668474E-01, -0.16680160366199251E-01, -0.16391376675143607E-01, 
-0.34498441106415957E-01, -0.34397004314819679E-01, -0.34095251535303142E-01, 
-0.33599934227484897E-01, -0.32922713002882686E-01, -0.32080956839613553E-01, 
-0.31094656816241163E-01, -0.29992993869276993E-01, -0.28920346520236968E-01, 
-0.27804440647686564E-01, -0.26557767002026948E-01, -0.25339486286673835E-01, 
-0.24207857609515781E-01, -0.23210898648486945E-01, -0.22405037605772941E-01, 
-0.21839983262305046E-01, -0.21548462502174622E-01, -0.38028470514734354E-01, 
-0.37934768482021111E-01, -0.37655879759061317E-01, -0.37197605302210765E-01, 
-0.36569997616540818E-01, -0.35788138695160128E-01, -0.34869357109473170E-01, 
-0.33839494179446698E-01, -0.32832892384098852E-01, -0.31781479340823551E-01, 
-0.30601611467588004E-01, -0.29443152793866553E-01, -0.28362196397116023E-01, 
-0.27405958639980330E-01, -0.26630333839629657E-01, -0.26085058254035041E-01, 
-0.25803285303432195E-01, -0.39684362424874435E-01, -0.39601621680865048E-01, 
-0.39355269121556083E-01, -0.38950148230021464E-01, -0.38394634048389019E-01, 
-0.37701300413029062E-01, -0.36884472066118237E-01, -0.35965889836638634E-01, 
-0.35064675749037363E-01, -0.34119485780080186E-01, -0.33053835220496951E-01, 
-0.32002157295402046E-01, -0.31015895329589274E-01, -0.30139378130097823E-01, 
-0.29425601871767466E-01, -0.28922294003268258E-01, -0.28661718740490488E-01, 
-0.39201208984499011E-01, -0.39130941195882153E-01, -0.38921730606058080E-01, 
-0.38577674489662865E-01, -0.38105766696589875E-01, -0.37516364958801354E-01, 
-0.36821037847378987E-01, -0.36037392217164163E-01, -0.35266339266865863E-01, 
-0.34454841785741916E-01, -0.33535976189414458E-01, -0.32624637752924547E-01, 
-0.31765612101793260E-01, -0.30998456939335586E-01, -0.30371083332670103E-01, 
-0.29927247014394656E-01, -0.29696985832178262E-01, -0.36463053420246964E-01, 
-0.36404891668648465E-01, -0.36231853368028206E-01, -0.35947667519026658E-01, 
-0.35558536351103114E-01, -0.35073333761013174E-01, -0.34501649892438320E-01, 
-0.33857644783252053E-01, -0.33223632608949893E-01, -0.32555324430595403E-01, 
-0.31796566488753980E-01, -0.31041217411274601E-01, -0.30326160124257401E-01, 
-0.29684753015982786E-01, -0.29158086866015349E-01, -0.28784288134912166E-01, 
-0.28589958266848004E-01, -0.31514289051777382E-01, -0.31466067363773761E-01, 
-0.31322867626737687E-01, -0.31088517710371723E-01, -0.30769192742282014E-01, 
-0.30373313068332231E-01, -0.29909654842079757E-01, -0.29390222404533838E-01, 
-0.28881033641897251E-01, -0.28345794437849074E-01, -0.27738953248896650E-01, 
-0.27134724554218368E-01, -0.26561827584319599E-01, -0.26046680519488281E-01, 
-0.25622532798828170E-01, -0.25320765418531878E-01, -0.25163623787255833E-01, 
-0.24556959017961934E-01, -0.24515027043903119E-01, -0.24390881122657831E-01, 
-0.24188915971746907E-01, -0.23916077192269136E-01, -0.23581474581372726E-01, 
-0.23194418471759324E-01, -0.22766443426318176E-01, -0.22351984486933325E-01, 
-0.21920879626859114E-01, -0.21436606762675545E-01, -0.20957937025137704E-01, 
-0.20506259748145426E-01, -0.20101169410983961E-01, -0.19767974268224592E-01, 
-0.19530939243609491E-01, -0.19407479648349865E-01, -0.15933953055484167E-01, 
-0.15893683085843515E-01, -0.15774847116772264E-01, -0.15582792687899244E-01, 
-0.15325927255277483E-01, -0.15015110897986980E-01, -0.14661493385475084E-01, 
-0.14277952772127254E-01, -0.13913860345519795E-01, -0.13542437372349917E-01, 
-0.13133355117107160E-01, -0.12736485370178068E-01, -0.12367784241931882E-01, 
-0.12041066356033447E-01, -0.11774629337200762E-01, -0.11586130949546081E-01, 
-0.11488248020313126E-01, -0.60996840809338726E-02, -0.60560991413736148E-02, 
-0.59277322003915651E-02, -0.57211248371982050E-02, -0.54466194910394805E-02, 
-0.51176349330154273E-02, -0.47482122735243524E-02, -0.43542328619606552E-02, 
-0.39874664341431592E-02, -0.36212189811951254E-02, -0.32275628808086615E-02, 
-0.28555527913529440E-02, -0.25184840152947027E-02, -0.22263052682455794E-02, 
-0.19922419858595043E-02, -0.18287625455993769E-02, -0.17445208732466497E-02, 
.44187393130219960E-02, .44703047776420615E-02, .46221705624145066E-02, 
.48665374434848025E-02, .51909399337938707E-02, .55789620437655119E-02, 
.60130310991886402E-02, .64730032458013920E-02, .68973028089270392E-02, 
.73159784702126647E-02, .77588612321738129E-02, .81691385614269165E-02, 
.85328571874723307E-02, .88413430183690335E-02, .90836673176239429E-02, 
.92503222361872087E-02, .93353618716464914E-02, .15065750818926932E-01, 
.15129049201214452E-01, .15315703923308504E-01, .15616771737765619E-01, 
.16017774431477387E-01, .16499278170443353E-01, .17040031610525724E-01, 
.17614943254959117E-01, .18146267918479439E-01, .18670558768157410E-01, 
.19223837078777457E-01, .19733606389079467E-01, .20181822551338303E-01, 
.20558130673099296E-01, .20850598331030635E-01, .21049867048131925E-01, 
.21150902512215904E-01, .25302316075999490E-01, .25379723808078718E-01, 
.25608381596901292E-01, .25978462732549674E-01, .26473847449988729E-01, 
.27072457493135568E-01, .27749704427973681E-01, .28475512822802578E-01, 
.29151485561256536E-01, .29823143718358154E-01, .30536419129790156E-01, 
.31196926271665985E-01, .31779456818608623E-01, .32269050166411052E-01, 
.32649365161191386E-01, .32908140083184184E-01, .33039179825639954E-01, 
.34647045336395677E-01, .34739295258842798E-01, .35012274278040177E-01, 
.35455628318278989E-01, .36052151822406876E-01, .36777805211912304E-01, 
.37605410855517038E-01, .38500492372621105E-01, .39341966647524533E-01, 
.40185787756161953E-01, .41090457728654425E-01, .41936060055956494E-01, 
.42687950332541261E-01, .43324085874524569E-01, .43820678592752065E-01, 
.44159680428451246E-01, .44331658357286446E-01, .42709419029560831E-01, 
.42815552607073093E-01, .43130111292128806E-01, .43642604637431089E-01, 
.44335384906936601E-01, .45183320771315767E-01, .46157661427469084E-01, 
.47220611868712875E-01, .48229041799534668E-01, .49249618020370284E-01, 
.50354642749422793E-01, .51398077509649157E-01, .52334708957066878E-01, 
.53133758357734713E-01, .53761769806678898E-01, .54192616459577812E-01, 
.54411844743456533E-01, .49212226677825523E-01, .49329757191998530E-01, 
.49678564225430592E-01, .50248389173063741E-01, .51021776393192428E-01, 
.51973423388581373E-01, .53074127020045568E-01, .54284174435749358E-01, 
.55441565836821992E-01, .56622735804987383E-01, .57913397681011523E-01, 
.59143917184144690E-01, .60258695070870456E-01, .61217650432761797E-01, 
.61976610378398973E-01, .62500030318975108E-01, .62767227722681951E-01, 
.54001751209310826E-01, .54126999323007866E-01, .54499124876369626E-01, 
.55108401764889008E-01, .55938111121452583E-01, .56963628618316151E-01, 
.58156367972691157E-01, .59476213536955327E-01, .60747561042792421E-01, 
.62054553832673456E-01, .63494327395748465E-01, .64878942631077310E-01, 
.66143900077553316E-01, .67240447743628542E-01, .68114013380808858E-01, 
.68719489597411754E-01, .69029541953781470E-01, .57045766950749512E-01, 
.57174308862364347E-01, .57556559769273895E-01, .58183533236755396E-01, 
.59039650483083274E-01, .60101653609631066E-01, .61342467672439220E-01, 
.62723013335359568E-01, .64060752872071122E-01, .65444606860347032E-01, 
.66979769315571119E-01, .68467352346565258E-01, .69836545622419269E-01, 
.71031687044672484E-01, .71989487451205814E-01, .72656396993495617E-01, 
.72998894039520826E-01, .58420835690809067E-01, .58547994293573935E-01, 
.58926384872389624E-01, .59547868552114136E-01, .60398249408385354E-01, 
.61456115625989509E-01, .62696549833447912E-01, .64082725277646913E-01, 
.65432451989355683E-01, .66835959377441551E-01, .68402158665211751E-01, 
.69929710902965403E-01, .71344840454931807E-01, .72587633376467550E-01, 
.73588928395750292E-01, .74289003449649949E-01, .74649472281549528E-01, 
.58291486182744150E-01, .58412797700275118E-01, .58773952744799782E-01, 
.59367680111738368E-01, .60181259455225185E-01, .61195390819734671E-01, 
.62387692209169357E-01, .63724495750862475E-01, .65031061561327486E-01, 
.66395303339386383E-01, .67925038256739928E-01, .69425120216468766E-01, 
.70822476270148579E-01, .72056139799282701E-01, .73054714562018411E-01, 
.73755439446239981E-01, .74117082770694995E-01, .56884514935598388E-01, 
.56996115702872531E-01, .57328437823676033E-01, .57875033203182651E-01, 
.58624631706819745E-01, .59560125863763996E-01, .60661804251967502E-01, 
.61899728989566236E-01, .63112856860730412E-01, .64383378112564629E-01, 
.65813276530659576E-01, .67221470862105906E-01, .68539134904534046E-01, 
.69707572161485709E-01, .70657106439576248E-01, .71325522506742456E-01, 
.71671191766650766E-01, .54461797835386530E-01, .54560694884741599E-01, 
.54855184986457237E-01, .55339561834552194E-01, .56003904766093036E-01, 
.56833243811002188E-01, .57810487544316704E-01, .58909690696707130E-01, 
.59988407199134008E-01, .61120229715833668E-01, .62397138484937463E-01, 
.63658497582740073E-01, .64842758951865323E-01, .65896535299684547E-01, 
.66755645044414155E-01, .67361990107040220E-01, .67676094925704394E-01, 
.51294685339173203E-01, .51378896100318275E-01, .51629580891512969E-01, 
.52041683551435854E-01, .52606499746779264E-01, .53311062447295013E-01, 
.54140733720226285E-01, .55073575964704330E-01, .55989025973641769E-01, 
.56949962781635562E-01, .58035147916876959E-01, .59108832823558909E-01, 
.60118988435947714E-01, .61019958840722970E-01, .61756210326920156E-01, 
.62276878180202909E-01, .62546959435769572E-01, .47642373049693654E-01, 
.47710939771744745E-01, .47914921648426978E-01, .48249831095642931E-01, 
.48708049673748081E-01, .49278449502414806E-01, .49948630115679293E-01, 
.50700504278885311E-01, .51437012313595210E-01, .52209074674003465E-01, 
.53080204673769643E-01, .53941890645438383E-01, .54752933369053486E-01, 
.55476999682629133E-01, .56069418168858201E-01, .56488871735505340E-01, 
.56706639172737328E-01, .43735745571792731E-01, .43788645897886440E-01, 
.43945838735810622E-01, .44203345669359524E-01, .44554529107228770E-01, 
.44989944047048924E-01, .45499213217224652E-01, .46067842201031627E-01, 
.46622364234511916E-01, .47201368394309981E-01, .47852336698617007E-01, 
.48494361181614522E-01, .49097452288904588E-01, .49635284491978744E-01, 
.50075168221274335E-01, .50386642676845936E-01, .50548382319389046E-01, 
.39767250655815364E-01, .39805235347059077E-01, .39917879141141734E-01, 
.40101679233830428E-01, .40350908593213192E-01, .40657676484754504E-01, 
.41013446956786152E-01, .41407017933940971E-01, .41787337849964530E-01, 
.42181068998227496E-01, .42620030343936223E-01, .43049610899289735E-01, 
.43450593350647999E-01, .43806492730410235E-01, .44096636851849162E-01, 
.44301682465245740E-01, .44408051299299478E-01, .35886521942979352E-01, 
.35910913733220820E-01, .35982970510371817E-01, .36099654300398902E-01, 
.36256111185158438E-01, .36445908581366343E-01, .36662212756650880E-01, 
.36896808095963143E-01, .37118947772006625E-01, .37344386586282745E-01, 
.37590585862852681E-01, .37826677990700400E-01, .38043153336185138E-01, 
.38232503663437688E-01, .38385178552270138E-01, .38492273381218342E-01, 
.38547595706267736E-01, .32200839938143676E-01, .32213324821059403E-01, 
.32249850503173812E-01, .32307848500402657E-01, .32383323986422408E-01, 
.32471232563273285E-01, .32566341080845974E-01, .32663127684404582E-01, 
.32748455851447852E-01, .32828609453742906E-01, .32908651342739195E-01, 
.32978119648222844E-01, .33035743920108884E-01, .33081646090066412E-01, 
.33115819434318745E-01, .33138397111704826E-01, .33149640800359037E-01, 
.28779155382033485E-01, .28781591015522318E-01, .28788158108919083E-01, 
.28796768848190924E-01, .28804284206324034E-01, .28806991910025618E-01, 
.28801180830835528E-01, .28783747695728661E-01, .28756207311920804E-01, 
.28717026334806335E-01, .28661181172681918E-01, .28594926115431001E-01, 
.28523574979693614E-01, .28453279698299149E-01, .28391606124335680E-01, 
.28345879376612891E-01, .28321512445283782E-01, .25658295358167556E-01, 
.25652552746119106E-01, .25634773023623453E-01, .25603358457880209E-01, 
.25556020956415553E-01, .25490324980385639E-01, .25404009036645886E-01, 
.25296127047832257E-01, .25179883571366822E-01, .25047621831318371E-01, 
.24886622469628056E-01, .24716170842393932E-01, .24546507356653147E-01, 
.24388167667742953E-01, .24254204470098326E-01, .24157125807874463E-01, 
.24106035883942011E-01, .22850084580878314E-01, .22837930625122602E-01, 
.22801101028399898E-01, .22738489575569051E-01, .22648643258732705E-01, 
.22530337908276563E-01, .22382681084391520E-01, .22206630185127859E-01, 
.22024307069750778E-01, .21823527813465571E-01, .21586129355759352E-01, 
.21341006833276110E-01, .21101801451755550E-01, .20881901698534541E-01, 
.20697875380440812E-01, .20565483382845098E-01, .20496095191860898E-01, 
.20348370416786086E-01, .20331391706247620E-01, .20280264762014853E-01, 
.20194364845033490E-01, .20073041830400744E-01, .19916201392750063E-01, 
.19724222390602508E-01, .19499715070800925E-01, .19271256880201502E-01, 
.19023544075599440E-01, .18734937207321656E-01, .18440944520261311E-01, 
.18157296629097205E-01, .17898924435810620E-01, .17684210649536569E-01, 
.17530492509834520E-01, .17450156576074630E-01, .18135261596067799E-01, 
.18114824677836625E-01, .18053489139291671E-01, .17951089317135311E-01, 
.17807728092720996E-01, .17624341874403295E-01, .17402463030630626E-01, 
.17146105971294644E-01, .16888213068792044E-01, .16611532000581242E-01, 
.16292554037923505E-01, .15970903313421346E-01, .15663339048540472E-01, 
.15385294953992016E-01, .15155622455408423E-01, .14991909852372635E-01, 
.14906574721895521E-01, .16186208421777339E-01, .16163448947585833E-01, 
.16095297677025375E-01, .15982014463904532E-01, .15824389570685000E-01, 
.15624275950642613E-01, .15384224040200631E-01, .15109408005328383E-01, 
.14835428053740168E-01, .14544007980868802E-01, .14211000877545512E-01, 
.13878157724295155E-01, .13562456742417930E-01, .13279071162648285E-01, 
.13046346138572442E-01, .12881173763941043E-01, .12795306221821371E-01, 
.14473818113604472E-01, .14449650392457625E-01, .14377409850016462E-01, 
.14257739674987157E-01, .14092040151350399E-01, .13882956552698942E-01, 
.13633910159006973E-01, .13350999521447908E-01, .13071138525195865E-01, 
.12775721026027991E-01, .12440844202838447E-01, .12108869164068968E-01, 
.11796411875837900E-01, .11517872558003534E-01, .11290455702226827E-01, 
.11129760737611606E-01, .11046449999169951E-01, .12970488141723547E-01, 
.12945627217855325E-01, .12871424688595198E-01, .12748859399106294E-01, 
.12579857482518325E-01, .12367728369863256E-01, .12116612598515306E-01, 
.11833312442154740E-01, .11555033797893535E-01, .11263335113941439E-01, 
.10935143608599552E-01, .10612324988494034E-01, .10310744724765350E-01, 
.10043725494639315E-01, .98269787220692066E-02, .96745051296155612E-02, 
.95956782300499460E-02, .11650049802688659E-01, .11625039612874592E-01, 
.11550488652569687E-01, .11427661367427172E-01, .11258923620584820E-01, 
.11048122823829502E-01, .10799970143493192E-01, .10521770827617438E-01, 
.10250274113580123E-01, .99675359633763453E-02, .96516657966568414E-02, 
.93432683170359063E-02, .90572239018635991E-02, .88056326846546028E-02, 
.86025735519046007E-02, .84603590860094710E-02, .83870415733987898E-02, 
.10488655730195931E-01, .10463898642785519E-01, .10390188529023427E-01, 
.10269025709373367E-01, .10103131878128687E-01, .98967731707628546E-02, 
.96550918120911038E-02, .93857189913362487E-02, .91244207548051119E-02, 
.88539595110174573E-02, .85538065835122062E-02, .82628090771648885E-02, 
.79947435436005659E-02, .77604559417643283E-02, .75723995583550346E-02, 
.74412539890080075E-02, .73738262013339400E-02, .94651379880378965E-02, 
.94409227021901437E-02, .93689027661501747E-02, .92507670711656810E-02, 
.90895146097846463E-02, .88897214504806230E-02, .86568346953933960E-02, 
.83986601355632261E-02, .81496288942064822E-02, .78933281722115972E-02, 
.76106518120684858E-02, .73383974327158397E-02, .70892018733775325E-02, 
.68726993567838502E-02, .66998145949321913E-02, .65797330750289517E-02, 
.65181513907095739E-02, .85610283825091286E-02, .85375545485838599E-02, 
.84678084585776848E-02, .83536241195117499E-02, .81982067155612869E-02, 
.80063460310262503E-02, .77836837342242192E-02, .75380747609212605E-02, 
.73023971196908791E-02, .70611178965146910E-02, .67965374604928297E-02, 
.65432621303992263E-02, .63128090105032597E-02, .61136864035603927E-02, 
.59554345439145435E-02, .58459216593608875E-02, .57898913245507709E-02, 
.77603868286456271E-02, .77377857585515276E-02, .76706938819979522E-02, 
.75610515369967258E-02, .74122075015416056E-02, .72290824719978336E-02, 
.70174190996354479E-02, .67850206109064219E-02, .65630929863665687E-02, 
.63369960239044298E-02, .60903754084699908E-02, .58556102348702524E-02, 
.56431528442051522E-02, .54604925867345637E-02, .53159465026824081E-02, 
.52162496571040379E-02, .51653484594967125E-02, .70495388321026968E-02, 
.70278903177602011E-02, .69636802529322592E-02, .68589215202901725E-02, 
.67170527030758608E-02, .65430558758777903E-02, .63426974329924427E-02, 
.61236503457302444E-02, .59154015220665435E-02, .57041893255709445E-02, 
.54749200201732228E-02, .52577811377045869E-02, .50622348519944928E-02, 
.48948647290350221E-02, .47629243022719739E-02, .46721887028894895E-02, 
.46259482470277884E-02, .64167858377449285E-02, .63961314547514945E-02, 
.63349179050958080E-02, .62352017931723534E-02, .61004662657016349E-02, 
.59356976919874684E-02, .57466240960493600E-02, .55407291004448333E-02, 
.53457840839052076E-02, .51488755199684631E-02, .49360760392711056E-02, 
.47354631468364940E-02, .45555922841399189E-02, .44022513439224420E-02, 
.42817780452596267E-02, .41991411246686712E-02, .41570955189685144E-02, 
.58521245219187697E-02, .58324779479992763E-02, .57742935663468963E-02, 
.56796475665170281E-02, .55520303402535843E-02, .53963866982969545E-02, 
.52183576233344083E-02, .50251947106285857E-02, .48429918212101880E-02, 
.46596439350349225E-02, .44622948599191943E-02, .42770208363262334E-02, 
.41115556700833380E-02, .39709935029388216E-02, .38608872792903395E-02, 
.37855304537441005E-02, .37472422308574014E-02, .53469925318811234E-02, 
.53283474328969481E-02, .52731663256421757E-02, .51835251159278737E-02, 
.50628912248969933E-02, .49161318788815767E-02, .47487639214278225E-02, 
.45677777131629578E-02, .43976501217438723E-02, .42270405383771359E-02, 
.40440725141456879E-02, .38729439095800192E-02, .37206491784037747E-02, 
.35916801033229312E-02, .34909181497427004E-02, .34220909374638266E-02, 
.33871624127731299E-02, .48940468662914162E-02, .48763828561549805E-02, 
.48241382902827921E-02, .47393727045566313E-02, .46255065402427895E-02, 
.44873018103371281E-02, .43301238370768375E-02, .41606833296977336E-02, 
.40019144983034710E-02, .38431954992114211E-02, .36735442834900088E-02, 
.35154084322825373E-02, .33751199748331341E-02, .32566486103866358E-02, 
.31643007618869607E-02, .31013284321900272E-02, .30694045557389483E-02, 
.44869749135469589E-02, .44702620187159247E-02, .44208596401766349E-02, 
.43407981415672587E-02, .42334322904198900E-02, .41033982952345099E-02, 
.39558905642007756E-02, .37973301957091588E-02, .36491906696232562E-02, 
.35015235154752305E-02, .33441627304806351E-02, .31979331029193792E-02, 
.30685745839616157E-02, .29596043756367498E-02, .28748352431173367E-02, 
.28171175388832887E-02, .27878842834847912E-02, .41203352554807287E-02, 
.41045372337559175E-02, .40578648567920327E-02, .39823093180455637E-02, 
.38811454972114772E-02, .37588690124130985E-02, .36204904213931366E-02, 
.34721377461061141E-02, .33339088908874255E-02, .31964846933771507E-02, 
.30504439261705230E-02, .29151118179703999E-02, .27956996339470897E-02, 
.26953319175825941E-02, .26173962045024195E-02, .25644018152096142E-02, 
.25375825662235000E-02, .37894243158673349E-02, .37745011051456114E-02, 
.37304359054180216E-02, .36591731052292129E-02, .35638972079781533E-02, 
.34489531791228331E-02, .33191599274471290E-02, .31803538168927847E-02, 
.30513425610300868E-02, .29233941752830200E-02, .27877676982090054E-02, 
.26624056579882083E-02, .25520472153083172E-02, .24594754895883369E-02, 
.23877102996817329E-02, .23389697563525056E-02, .23143210764743692E-02, 
.34901648730157353E-02, .34760743831743295E-02, .34344881369381178E-02, 
.33672980525619408E-02, .32775908321090812E-02, .31695545226602456E-02, 
.30478126160617324E-02, .29179148273845031E-02, .27974622903551303E-02, 
.26782703193504141E-02, .25522199615288133E-02, .24359804328307330E-02, 
.23338689907624736E-02, .22483711236139718E-02, .21821873596299113E-02, 
.21372857473004616E-02, .21145932058343453E-02, .32190128223332642E-02, 
.32057122331807840E-02, .31664750919633976E-02, .31031368065443103E-02, 
.30186813459575704E-02, .29171365893248878E-02, .28029299605012107E-02, 
.26813321034858278E-02, .25688181190877714E-02, .24577131286249560E-02, 
.23404676115436037E-02, .22325789646169998E-02, .21379869295625954E-02, 
.20589168008136923E-02, .19977907206908587E-02, .19563608577889205E-02, 
.19354352227349686E-02, .29728790791662945E-02, .29603257681344768E-02, 
.29233089393255881E-02, .28636047795964409E-02, .27840916031430653E-02, 
.26886362395280592E-02, .25814717670619115E-02, .24675991084277464E-02, 
.23624440313019229E-02, .22588063818031697E-02, .21496586089103657E-02, 
.20494198904303586E-02, .19616916347582722E-02, .18884709908661397E-02, 
.18319364032380924E-02, .17936527340689202E-02, .17743266209066530E-02, 
.27490640174512352E-02, .27372162105377094E-02, .27022938231154280E-02, 
.26460121887286634E-02, .25711426494983277E-02, .24813919392811506E-02, 
.23808022884809666E-02, .22741154066117327E-02, .21757801535445360E-02, 
.20790383821656451E-02, .19773416287204399E-02, .18841168213698733E-02, 
.18026616102555320E-02, .17347724947471958E-02, .16824137653683295E-02, 
.16469870116226188E-02, .16291119862636740E-02, .25452022910952203E-02, 
.25340195076152777E-02, .25010698730568736E-02, .24480071023935206E-02, 
.23774955162908589E-02, .22930841084756741E-02, .21986290571476327E-02, 
.20986240580297224E-02, .20066090329579427E-02, .19162374005826466E-02, 
.18214009963322118E-02, .17346132472192525E-02, .16588985702434165E-02, 
.15958764694592208E-02, .15473223562955287E-02, .15144948840966578E-02, 
.14979389783332949E-02, .23592162679766287E-02, .23486596123051310E-02, 
.23175660385144170E-02, .22675275684725573E-02, .22011024422114727E-02, 
.21216853199634814E-02, .20329520440507871E-02, .19391598354572559E-02, 
.18530031784128632E-02, .17685187825480207E-02, .16800036518272374E-02, 
.15991297248739089E-02, .15286751652347761E-02, .14701028925473412E-02, 
.14250211576899384E-02, .13945629836043098E-02, .13792085700229178E-02, 
.21892766235651132E-02, .21793089639246298E-02, .21499602406852736E-02, 
.21027612525673272E-02, .20401658624724910E-02, .19654185731841308E-02, 
.18820212317512562E-02, .17940062019945876E-02, .17132816587478580E-02, 
.16342413708639327E-02, .15515556486456693E-02, .14761207326950332E-02, 
.14104924154894230E-02, .13559947002421931E-02, .13140874202978582E-02, 
.12857927731130029E-02, .12715347472566280E-02, .20337689008708203E-02, 
.20243549653163741E-02, .19966456101899472E-02, .19521113049005001E-02, 
.18931038160004167E-02, .18227222094541562E-02, .17443010709884672E-02, 
.16616594123665636E-02, .15859740229930983E-02, .15119714140165430E-02, 
.14346662633157139E-02, .13642391939306397E-02, .13030448056649871E-02, 
.12522835281098728E-02, .12132830191097551E-02, .11869673788492637E-02, 
.11737116017785589E-02, .18912650552305561E-02, .18823714675052818E-02, 
.18562017988252462E-02, .18141674080038857E-02, .17585206729943750E-02, 
.16922203097951883E-02, .16184405916012078E-02, .15407984338341786E-02, 
.14697901708128245E-02, .14004525254886096E-02, .13281182225152745E-02, 
.12623071321022294E-02, .12051914784841000E-02, .11578614837507652E-02, 
.11215267604598865E-02, .10970243047937055E-02, .10846862656904571E-02, 
.17604991749986336E-02, .17520944472901498E-02, .17273705341392556E-02, 
.16876811464705906E-02, .16351822871152962E-02, .15726976346045012E-02, 
.15032481765136318E-02, .14302596427532572E-02, .13635950828534314E-02, 
.12985805620091193E-02, .12308428784390439E-02, .11692912646115515E-02, 
.11159323223097544E-02, .10717577466348835E-02, .10378714437596535E-02, 
.10150328414343008E-02, .10035365061287949E-02, .16403467098965378E-02, 
.16324012052599925E-02, .16090347316919588E-02, .15715449931313697E-02, 
.15219948392029149E-02, .14630783391262770E-02, .13976701963379118E-02, 
.13290154572200706E-02, .12663875397443962E-02, .12053825220671316E-02, 
.11418994109403861E-02, .10842825996765391E-02, .10343880140743439E-02, 
.99311906072516555E-03, .96148475217818516E-03, .94017525200052019E-03, 
.92945206881190397E-03, .15298066536428862E-02, .15222925276342839E-02, 
.15002005988858473E-02, .14647743301803950E-02, .14179867706623273E-02, 
.13624078387333360E-02, .13007728523971890E-02, .12361562275266598E-02, 
.11772821298984915E-02, .11199987469845236E-02, .10604573250158537E-02, 
.10064792986912406E-02, .95978328019123292E-03, .92119338838576024E-03, 
.89163324922829801E-03, .87173102102667643E-03, .86171905999482524E-03, 
.14279862214482189E-02, .14208773500756040E-02, .13999822613432068E-02, 
.13664920247197074E-02, .13222933108609717E-02, .12698373106884899E-02, 
.12117266959997834E-02, .11508748339867389E-02, .10954939810885629E-02, 
.10416678480937847E-02, .98578165932220363E-03, .93517221735242931E-03, 
.89143279149765274E-03, .85531614793134508E-03, .82766891135080281E-03, 
.80906360212446677E-03, .79970680763778463E-03, .13340876399853053E-02, 
.13273595392436731E-02, .13075885221933124E-02, .12759151604654455E-02, 
.12341431890566267E-02, .11846104099801436E-02, .11297933880506251E-02, 
.10724535434861223E-02, .10203257574083943E-02, .96971389450157424E-03, 
.91722043610302062E-03, .86973265599930101E-03, .82872922638670913E-03, 
.79489857510547208E-03, .76901774362472905E-03, .75160921784439203E-03, 
.74285675792996994E-03, .12473967307305427E-02, .12410264715943589E-02, 
.12223114297000913E-02, .11923435945931952E-02, .11528471915180909E-02, 
.11060518504585253E-02, .10543143405772673E-02, .10002527575326758E-02, 
.95115654799102559E-03, .90353548356146302E-03, .85419397291779933E-03, 
.80960194098672416E-03, .77113312800146968E-03, .73941784002564466E-03, 
.71517011607491129E-03, .69886735393556677E-03, .69067305218054314E-03, 
.11672730196907379E-02, .11612391413866182E-02, .11435163821270230E-02, 
.11151500637941324E-02, .10777882817601700E-02, .10335575623290610E-02, 
.98470094437517574E-03, .93370134999643190E-03, .88743234158228769E-03, 
.84259638581495738E-03, .79618574767503715E-03, .75428253072700358E-03, 
.71816425315116158E-03, .68840862234198898E-03, .66567252841605458E-03, 
.65039265989747257E-03, .64271449820823320E-03, .10931411495567233E-02, 
.10874235731458046E-02, .10706335427829070E-02, .10437716088849583E-02, 
.10084130487008184E-02, .96658618583930084E-03, .92042613778444619E-03, 
.87228834551441117E-03, .82865783543891391E-03, .78641751178866253E-03, 
.74273456498602270E-03, .70333039685754038E-03, .66939416721005670E-03, 
.64145590335940505E-03, .62012056632587524E-03, .60578802230935457E-03, 
.59858770456994225E-03, .10244834059347180E-02, .10190633496372588E-02, 
.10031503745738422E-02, .97770212461791256E-03, .94422428605648982E-03, 
.90465170089956768E-03, .86101711298273368E-03, .81555573230506818E-03, 
.77438937087612170E-03, .73456999270477691E-03, .69342781703067894E-03, 
.65634847623478106E-03, .62443988418840908E-03, .59818877823167641E-03, 
.57815285604639020E-03, .56469862057609505E-03, .55794118880709404E-03, 
.96083319885107278E-03, .95569309609768414E-03, .94060513351672172E-03, 
.91648587216102946E-03, .88477453801479803E-03, .84731702512864612E-03, 
.80604898999287519E-03, .76309223793559922E-03, .72422882332236324E-03, 
.68666920320858916E-03, .64789566843301543E-03, .61298102569064023E-03, 
.58295838705316302E-03, .55827512670308607E-03, .53944585891998575E-03, 
.52680680907604542E-03, .52046030497651164E-03, .90176936530007629E-03, 
.89689278615892663E-03, .88258118568721182E-03, .85971181725902302E-03, 
.82966047243464682E-03, .79418843978348787E-03, .75513931633799220E-03, 
.71452792500900208E-03, .67781830365258415E-03, .64236958344152161E-03, 
.60580602397200707E-03, .57290874066951922E-03, .54464189221079473E-03, 
.52141700924310155E-03, .50370937536090205E-03, .49182769720717255E-03, 
.48586286303098223E-03, .84691117909574203E-03, .84228275547592940E-03, 
.82870203290268162E-03, .80700867830605792E-03, .77851796454489247E-03, 
.74491072530855453E-03, .70794327310437033E-03, .66952948704359239E-03, 
.63483555119683410E-03, .60136014158922732E-03, .56686016176068859E-03, 
.53584452248771840E-03, .50921374535910842E-03, .48734667830655138E-03, 
.47068265024664650E-03, .45950532099346073E-03, .45389533460535146E-03, 
.79591397141759803E-03, .79151932626496141E-03, .77862694982442770E-03, 
.75804058624139839E-03, .73101779216445104E-03, .69916290669922744E-03, 
.66414948702471662E-03, .62779614399749656E-03, .59498991820240200E-03, 
.56336053750093737E-03, .53078893394033081E-03, .50152979817473878E-03, 
.47642485483009915E-03, .45582311301305996E-03, .44013098993503521E-03, 
.42960931787012390E-03, .42432955757638592E-03, .74846527979481679E-03, 
.74429096036203435E-03, .73204714971066009E-03, .71250327294822020E-03, 
.68686185851229422E-03, .65665452421595277E-03, .62347636381528170E-03, 
.58905605273149660E-03, .55801886148843969E-03, .52811766396347159E-03, 
.49734945283795454E-03, .46973131256208389E-03, .44605058397983370E-03, 
.42662900066802996E-03, .41184281595431295E-03, .40193203088778449E-03, 
.39695986626702887E-03, .70428145533101569E-03, .70031487050691086E-03, 
.68868240825540998E-03, .67012071720601210E-03, .64577987125022450E-03, 
.61712235777521723E-03, .58566887098475504E-03, .55306316083152606E-03, 
.52368487022404870E-03, .49540265539417994E-03, .46632219361372247E-03, 
.44023832517741773E-03, .41788803683164764E-03, .39956810093557088E-03, 
.38562709256025572E-03, .37628598053873714E-03, .37160058602986453E-03, 
.66310466812588933E-03, .65933402974028166E-03, .64827808521215717E-03, 
.63064218763489921E-03, .60752641693957363E-03, .58032754403612035E-03, 
.55049570915958351E-03, .51959444312824207E-03, .49177276970327564E-03, 
.46500826462525995E-03, .43750845522414389E-03, .41286015512874256E-03, 
.39175368159277923E-03, .37446293866293367E-03, .36131107533959839E-03, 
.35250165224063939E-03, .34808384069476208E-03, .62470025942032469E-03, 
.62111452739354472E-03, .61060249215395205E-03, .59383963081554160E-03, 
.57187837894814657E-03, .54605303416607851E-03, .51774682002503618E-03, 
.48844746935712818E-03, .46208735009776271E-03, .43674655751159977E-03, 
.41072813047830293E-03, .38742402595726816E-03, .36748126527219832E-03, 
.35115277977147723E-03, .33873833628271304E-03, .33042555562834419E-03, 
.32625762941267730E-03, .64007964687947663E-03, .63641961088012496E-03, 
.62568902247022167E-03, .60857540484699965E-03, .58615039433571763E-03, 
.55977344726801707E-03, .53085471706615568E-03, .50091240711257223E-03, 
.47396583343427252E-03, .44805392508769293E-03, .42144141042289900E-03, 
.39759837842021442E-03, .37718916135461635E-03, .36047504717128898E-03, 
.34776512706787970E-03, .33925337007127920E-03, .33498533610794556E-03, 
.60481661151200259E-03, .60132746728705131E-03, .59109949932731516E-03, 
.57479244380807067E-03, .55343366564470405E-03, .52832464920350751E-03, 
.50081353185150903E-03, .47234841502916537E-03, .44674887458193488E-03, 
.42214845530525288E-03, .39689985357650080E-03, .37429380749513450E-03, 
.35495501244294561E-03, .33912568079531863E-03, .32709360329456012E-03, 
.31903826403649628E-03, .31499984584608523E-03, .57427747746391334E-03, 
.57093786510445659E-03, .56114960615073024E-03, .54554788193479848E-03, 
.52512099276100916E-03, .50111937700927651E-03, .47483667389669259E-03, 
.44765957960276231E-03, .42323357456931310E-03, .39977481080711973E-03, 
.37571254165303573E-03, .35418158052161327E-03, .33577245584584064E-03, 
.32071110822039472E-03, .30926711232118074E-03, .30160761407602052E-03, 
.29776829479400839E-03, .54899930264058045E-03, .54578458914057942E-03, 
.53636353961196968E-03, .52135065006957799E-03, .50170135381165588E-03, 
.47862321999121618E-03, .45336419856791232E-03, .42725964976790835E-03, 
.40381013242441356E-03, .38130066545520895E-03, .35822414652049157E-03, 
.33758588267209189E-03, .31994823720006533E-03, .30552386266738781E-03, 
.29456742339860337E-03, .28723600250342001E-03, .28356166818028822E-03, 
.52919529661217069E-03, .52607917640872068E-03, .51694795436326595E-03, 
.50239968313608432E-03, .48336369616534325E-03, .46101357095306941E-03, 
.43656108143135604E-03, .41130103454503950E-03, .38861991302446217E-03, 
.36685701241857805E-03, .34455527450586365E-03, .32461825291682977E-03, 
.30758632051866623E-03, .29366183738024497E-03, .28308789571187637E-03, 
.27601379272502698E-03, .27246883970244602E-03, .51501520217250277E-03, 
.51197009090835175E-03, .50304758526372112E-03, .48883383151154098E-03, 
.47023927953855675E-03, .44841294169698084E-03, .42454047676139197E-03, 
.39988746837821818E-03, .37775841812532517E-03, .35653166530621213E-03, 
.33478608646280496E-03, .31535221395227671E-03, .29875471664228586E-03, 
.28518865028142443E-03, .27488888167895883E-03, .26799918728564917E-03, 
.26454694524016316E-03, .50650363437249220E-03, .50350131610842324E-03, 
.49470458227198648E-03, .48069237787872659E-03, .46236373771266703E-03, 
.44085283599893630E-03, .41732956880731166E-03, .39304190215139727E-03, 
.37124500533409100E-03, .35034072776737944E-03, .32892955778161267E-03, 
.30979812955598233E-03, .29346170264054293E-03, .28011097508323483E-03, 
.26997590111629984E-03, .26319696547093393E-03, .25980040262652438E-03, 
.50366411039359554E-03, .50067609701156862E-03, .49192140467405531E-03, 
.47797656677103654E-03, .45973679631727951E-03, .43833130168065764E-03, 
.41492470561588739E-03, .39075908382255991E-03, .36907312709574558E-03, 
.34827654057223233E-03, .32697702957085493E-03, .30794657208369489E-03, 
.29169729150338029E-03, .27841843662583819E-03, .26833832698051245E-03, 
.26159635288842270E-03, .25821837011773496E-03
};
static double tablef5 [ 340 ]= {
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0001463819130838, 1.0001463725948467, 1.0001463131563000, 
1.0001461164680732, 1.0001456603434127, 1.0001448174105128, 1.0001434822058022, 
1.0001416043780427, 1.0001394402941715, 1.0001368918048061, 1.0001337464595024, 
1.0001304242966114, 1.0001271611264289, 1.0001241721694281, 1.0001216923375567, 
1.0001199249888073, 1.0001190051179458, 1.0005797318116811, 1.0005796985353583, 
1.0005794735642741, 1.0005787105716089, 1.0005769237543187, 1.0005736064217525, 
1.0005683387221993, 1.0005609193341805, 1.0005523612456904, 1.0005422770937229, 
1.0005298258607400, 1.0005166704884860, 1.0005037458547830, 1.0004919055006698, 
1.0004820809636532, 1.0004750786660204, 1.0004714339739695, 1.0012841970821003, 
1.0012841364667397, 1.0012836757773569, 1.0012820433069283, 1.0012781560957269, 
1.0012708836379940, 1.0012592880492743, 1.0012429164673797, 1.0012240043921248, 
1.0012016986214667, 1.0011741375139454, 1.0011450026376856, 1.0011163684075817, 
1.0010901298168178, 1.0010683546712369, 1.0010528330898889, 1.0010447536398492, 
1.0022334804342214, 1.0022334058205151, 1.0022326931804879, 1.0022299897062823, 
1.0022233958714346, 1.0022109264423575, 1.0021909317837281, 1.0021626080110462, 
1.0021298232237665, 1.0020911049800123, 1.0020432186680894, 1.0019925622793895, 
1.0019427519082866, 1.0018970933802493, 1.0018591931715219, 1.0018321735417335, 
1.0018181078902300, 1.0033908977021702, 1.0033908417231618, 1.0033899245790601, 
1.0033860726673904, 1.0033763723379245, 1.0033577741838120, 1.0033277393590547, 
1.0032850170849656, 1.0032354428089434, 1.0031768023457655, 1.0031041907471323, 
1.0030273121066220, 1.0029516717911531, 1.0028823074589783, 1.0028247134062760, 
1.0027836464379398, 1.0027622660490483, 1.0047152542294746, 1.0047152680685003, 
1.0047142564407174, 1.0047093043800845, 1.0046963127726565, 1.0046709846578386, 
1.0046297346344166, 1.0045707748029609, 1.0045021599796584, 1.0044208452692414, 
1.0043200194385629, 1.0042131613642997, 1.0041079511848263, 1.0040114240730429, 
1.0039312505039779, 1.0038740716706129, 1.0038442997219656, 1.0061540378057710, 
1.0061541866683179, 1.0061532426743780, 1.0061473566526966, 1.0061311111388873, 
1.0060988174299779, 1.0060457176316340, 1.0059694087533604, 1.0058803177031923, 
1.0057745190811282, 1.0056431367231387, 1.0055037402545457, 1.0053663883477744, 
1.0052403057281221, 1.0051355467639076, 1.0050608171189368, 1.0050219019774553, 
1.0075142339253131, 1.0075145676228379, 1.0075138521805547, 1.0075073358984052, 
1.0074883256651508, 1.0074497819546371, 1.0073858040820676, 1.0072933769374051, 
1.0071851319111615, 1.0070563324290926, 1.0068961560352772, 1.0067260296027762, 
1.0065582757404192, 1.0064042079861524, 1.0062761535018854, 1.0061847864326174, 
1.0061372018087562, 1.0090269103862917, 1.0090275155384794, 1.0090272440189203, 
1.0090203182708075, 1.0089985929250964, 1.0089534858512665, 1.0088777858717064, 
1.0087677616959343, 1.0086384520903591, 1.0084842428367629, 1.0082921537379979, 
1.0080878894538956, 1.0078863079670495, 1.0077010675894864, 1.0075470451861332, 
1.0074371234883646, 1.0073798678965622, 1.0105428375986216, 1.0105437850825574, 
1.0105441609068169, 1.0105371359841098, 1.0105130738959791, 1.0104618036558899, 
1.0103747585204785, 1.0102474515099824, 1.0100972863225446, 1.0099177952958385, 
1.0096938430257147, 1.0094554081935838, 1.0092199084183082, 1.0090033751631198, 
1.0088232641918735, 1.0086946924188469, 1.0086277134666195, 1.0120620035081467, 
1.0120633653530120, 1.0120645953314877, 1.0120577869467235, 1.0120317735569586, 
1.0119747484893959, 1.0118767436554215, 1.0117324760833950, 1.0115616706911481, 
1.0113570309959665, 1.0111012690426417, 1.0108286331966589, 1.0105591252542714, 
1.0103111786605852, 1.0101048577691789, 1.0099575397282039, 1.0098807845641673, 
1.0134336040715168, 1.0134354019271912, 1.0134375814965970, 1.0134312431579497, 
1.0134038076057947, 1.0133419537534334, 1.0132344110387705, 1.0130751440794459, 
1.0128859308024183, 1.0126587493626797, 1.0123743746345459, 1.0120708979543622, 
1.0117706708288523, 1.0114943160718910, 1.0112642743421185, 1.0110999820577238, 
1.0110143724350966, 1.0148902243927767, 1.0148925502043333, 1.0148959253952194, 
1.0148903744493671, 1.0148617854100481, 1.0147951895634160, 1.0146778940548375, 
1.0145030289114143, 1.0142945070815874, 1.0140435604531044, 1.0137289132362208, 
1.0133927250911106, 1.0130598602295933, 1.0127532883409160, 1.0124979959346356, 
1.0123156259575496, 1.0122205837786871, 1.0162362745587441, 1.0162391482321114, 
1.0162438012789474, 1.0162392445006532, 1.0162099198302550, 1.0161392997418148, 
1.0160133409249901, 1.0158243691893440, 1.0155982282503808, 1.0153254825928208, 
1.0149829674950965, 1.0146165899693818, 1.0142535522132896, 1.0139190128567817, 
1.0136403316716478, 1.0134412088709681, 1.0133374231045260, 1.0174168135529171, 
1.0174202155512522, 1.0174261273107537, 1.0174226551851926, 1.0173929491708875, 
1.0173190857685592, 1.0171858084469245, 1.0169847112793926, 1.0167432974039290, 
1.0164515636712559, 1.0160846935200352, 1.0156918706373537, 1.0153023604895259, 
1.0149432576692314, 1.0146440199527103, 1.0144301664814170, 1.0143186906616135, 
1.0183879719226216, 1.0183918422988265, 1.0183988869381839, 1.0183964573523019, 
1.0183666241296911, 1.0182902950846313, 1.0181511952963576, 1.0179402981783792, 
1.0176864478368550, 1.0173791866533866, 1.0169923425972511, 1.0165777877570459, 
1.0161664939284383, 1.0157871597570007, 1.0154709806617870, 1.0152449824844121, 
1.0151271652753981, 1.0191103727795163, 1.0191146114446288, 1.0191225561742503, 
1.0191209906539378, 1.0190911728035286, 1.0190131291683375, 1.0188698153066162, 
1.0186517319003154, 1.0183887062309385, 1.0180699499742021, 1.0176682848255718, 
1.0172375783929952, 1.0168100770332642, 1.0164156793385128, 1.0160868802237566, 
1.0158518324367491, 1.0157292891683023, 1.0195554780202980, 1.0195599520894141, 
1.0195684758988321, 1.0195674805781050, 1.0195377191772523, 1.0194586701650252, 
1.0193128098461348, 1.0190903429047402, 1.0188216962677381, 1.0184958809836289, 
1.0180850995865900, 1.0176444473787050, 1.0172069582942753, 1.0168032731609045, 
1.0164666905447552, 1.0162260601621962, 1.0161006011092333, 1.0197059553088008, 
1.0197105104284210, 1.0197192342440662, 1.0197184382318893, 1.0196887040441689, 
1.0196093239821122, 1.0194626114438559, 1.0192386702268517, 1.0189681289166737, 
1.0186399312777854, 1.0182260706708386, 1.0177820571256042, 1.0173411912244845, 
1.0169343652242819, 1.0165951498971200, 1.0163526310399620, 1.0162261855803645
};
static double tabled5 [ 340 ]= {
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0002755493055400, 1.0002757912719689, 1.0002764290870949, 
1.0002772197933207, 1.0002778166965103, 1.0002778440382429, 1.0002769744899533, 
1.0002749973115386, 1.0002721939693631, 1.0002684948132374, 1.0002635689731021, 
1.0002580857118788, 1.0002525062099390, 1.0002472709889065, 1.0002428561494248, 
1.0002396766003299, 1.0002380119880825, 1.0010914393092010, 1.0010924089360931, 
1.0010949675434462, 1.0010981489595707, 1.0011005741652161, 1.0011007479066882, 
1.0010973669454220, 1.0010895906106738, 1.0010785264193500, 1.0010639026746684, 
1.0010444100043985, 1.0010226973754099, 1.0010005944596574, 1.0009798497695541, 
1.0009623527847773, 1.0009497501587259, 1.0009431518264176, 1.0024182554543544, 
1.0024204443242728, 1.0024262298884510, 1.0024334576405083, 1.0024390512951316, 
1.0024396729978677, 1.0024324109839389, 1.0024153811222980, 1.0023910106112546, 
1.0023587125878526, 1.0023155902331646, 1.0022675056273582, 1.0022185233779415, 
1.0021725307041103, 1.0021337273111441, 1.0021057732654701, 1.0020911360121596, 
1.0042071108989261, 1.0042110143849658, 1.0042213543902196, 1.0042343502429512, 
1.0042446010243606, 1.0042462412037398, 1.0042341479795664, 1.0042049927263452, 
1.0041629349777013, 1.0041069893813868, 1.0040321263862440, 1.0039485274581090, 
1.0038632890773800, 1.0037832049051101, 1.0037156126183662, 1.0036669072554751, 
1.0036414009008834, 1.0063896246627904, 1.0063957310213685, 1.0064119470959023, 
1.0064324707681604, 1.0064490082285431, 1.0064525421110533, 1.0064351855827653, 
1.0063917884058198, 1.0063285496251044, 1.0062440391121588, 1.0061306370174168, 
1.0060037745474293, 1.0058742764878081, 1.0057525186320804, 1.0056497037524952, 
1.0055755958702226, 1.0055367803205397, 1.0088888553134341, 1.0088976353634092, 
1.0089210154935297, 1.0089508281538555, 1.0089753899794385, 1.0089819820779649, 
1.0089594619138647, 1.0089005122194261, 1.0088135657013144, 1.0086967393107744, 
1.0085394630231126, 1.0083631514780282, 1.0081829386143502, 1.0080133515307972, 
1.0078700690533071, 1.0077667571244500, 1.0077126353002692, 1.0116063169850054, 
1.0116181892509761, 1.0116498921316801, 1.0116906245889459, 1.0117249241713646, 
1.0117359345428458, 1.0117088622154655, 1.0116339338959814, 1.0115218850100871, 
1.0113704090710474, 1.0111657478560478, 1.0109357865555257, 1.0107003932523708, 
1.0104786695068853, 1.0102912226953271, 1.0101560156768585, 1.0100851705792702, 
1.0141775092156713, 1.0141924867345300, 1.0142325816452853, 1.0142844420306731, 
1.0143289395991479, 1.0143451943026884, 1.0143148505234849, 1.0142257138509247, 
1.0140905749186657, 1.0139067935152906, 1.0136576137899282, 1.0133770079602413, 
1.0130893706544060, 1.0128181907659248, 1.0125887990920357, 1.0124232771419657, 
1.0123365308467784, 1.0170393600250474, 1.0170580014101074, 1.0171080355280453, 
1.0171732034501664, 1.0172301930607972, 1.0172535201259594, 1.0172207395754456, 
1.0171168490178488, 1.0169567820015797, 1.0167376042273051, 1.0164392452404678, 
1.0161024104408063, 1.0157565872779208, 1.0154302166121838, 1.0151539566448480, 
1.0149545349145059, 1.0148499995420885, 1.0199098747415798, 1.0199324133254095, 
1.0199930577875829, 1.0200725615316331, 1.0201433058383809, 1.0201750543304784, 
1.0201411265687241, 1.0200235818042793, 1.0198393462473614, 1.0195852733981068, 
1.0192379915442014, 1.0188449134056716, 1.0184406930555721, 1.0180588121935401, 
1.0177353480385889, 1.0175017549081491, 1.0173792798154646, 1.0227889973524995, 
1.0228156710837581, 1.0228876106520368, 1.0229825004291977, 1.0230682909158539, 
1.0231098434065649, 1.0230760933505827, 1.0229460279851981, 1.0227384105245261, 
1.0224499658503907, 1.0220540348315230, 1.0216047090881988, 1.0211418841973117, 
1.0207041734237401, 1.0203331666675481, 1.0200651276730042, 1.0199245604330800, 
1.0253905408083013, 1.0254211481491116, 1.0255038629983928, 1.0256135341457391, 
1.0257140122750026, 1.0257656109991955, 1.0257331857335774, 1.0255928384500093, 
1.0253648496552743, 1.0250459013364648, 1.0246063845074889, 1.0241063822204457, 
1.0235905779095442, 1.0231022878828075, 1.0226881511774459, 1.0223888374534882, 
1.0222318363233271, 1.0281554673600384, 1.0281904628264145, 1.0282852289391606, 
1.0284115357916184, 1.0285287762848232, 1.0285923070401466, 1.0285625223175121, 
1.0284123417676541, 1.0281635065076269, 1.0278127300428876, 1.0273272899054007, 
1.0267735896696237, 1.0262014556615471, 1.0256592726486073, 1.0251991179343165, 
1.0248664076749059, 1.0246918499195714, 1.0307124182176692, 1.0307516640715693, 
1.0308581307355875, 1.0310006815142643, 1.0311344943837837, 1.0312102275956199, 
1.0311840339960479, 1.0310257822693856, 1.0307584216777201, 1.0303787662923856, 
1.0298512343390744, 1.0292480295067961, 1.0286237844909329, 1.0280316365457500, 
1.0275287594396088, 1.0271650185230023, 1.0269741405955306, 1.0329564420467758, 
1.0329995719491187, 1.0331167519348263, 1.0332742470123069, 1.0334234626649648, 
1.0335108427446544, 1.0334887250857967, 1.0333242139195935, 1.0330412034014329, 
1.0326366529333086, 1.0320724872888216, 1.0314259634830558, 1.0307559725647002, 
1.0301198751190024, 1.0295793718400350, 1.0291882809408188, 1.0289830127556432, 
1.0348034816120464, 1.0348499173101060, 1.0349762301811867, 1.0351465134600302, 
1.0353090159190996, 1.0354066483694520, 1.0353885429360135, 1.0352194666734609, 
1.0349240089624596, 1.0344992891796376, 1.0339051909863171, 1.0332231039802198, 
1.0325154526455658, 1.0318431110540691, 1.0312715445660992, 1.0308578593156132, 
1.0306406988572907, 1.0361779968144937, 1.0362269568551501, 1.0363602520640507, 
1.0365403400908351, 1.0367130902824726, 1.0368187464720338, 1.0368040164532641, 
1.0366318914078041, 1.0363274289603830, 1.0358878913395448, 1.0352716498767676, 
1.0345631534093824, 1.0338274726622163, 1.0331281184408649, 1.0325333805645425, 
1.0321028323505523, 1.0318767937705693, 1.0370251464125597, 1.0370756897329707, 
1.0372133677995408, 1.0373996219600810, 1.0375788421702574, 1.0376896124220556, 
1.0376771297769638, 1.0375032752075701, 1.0371933738027730, 1.0367447860006782, 
1.0361149538619281, 1.0353902047274197, 1.0346372474240741, 1.0339212273138083, 
1.0333121848288060, 1.0328712226372516, 1.0326397001665435, 1.0373115854487325, 
1.0373626688627520, 1.0375018426218383, 1.0376902030182191, 1.0378716375691150, 
1.0379841662541343, 1.0379724724589179, 1.0377980590498326, 1.0374863378795043, 
1.0370347043518562, 1.0364002868085547, 1.0356700465319166, 1.0349112474633739, 
1.0341895892843762, 1.0335757060510438, 1.0331312190582496, 1.0328978402781572
};
static double tablef6 [ 800 ]= {
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
.99831792143082476, .99832676995450176, .99835286639588428, .99839501282287157, 
.99845137841710141, .99851965282596211, .99859747604275173, .99868201959981029, 
.99876223693906208, .99885205943055044, .99894295261946897, .99903509639439914, 
.99912861429098276, .99922357969308262, .99932002112220630, .99941792673233043, 
.99951724810977904, .99961790346303958, .99971978027484643, .99982273747813499, 
.99992660720824456, 1.0000311961757471, 1.0001362866972805, 1.0002416374155716, 
1.0003469837342742, 1.0004520379881745, 1.0005564893645829, 1.0006600035872169, 
1.0007622223694370, 1.0008627626392035, 1.0009612155334098, 1.0010571451541743, 
1.0011410369518545, 1.0012269050185281, 1.0013029346385214, 1.0013666990907006, 
1.0014169235791304, 1.0014528835596783, 1.0014744150842816, 1.0014815831470976, 
.99336798524726422, .99340269558017558, .99350505529763455, .99367033937610660, 
.99389133121691686, .99415893579999470, .99446387754336041, .99479506870549839, 
.99510926096717256, .99546105265010998, .99581705643759866, .99617802137795752, 
.99654447705196370, .99691675729775808, .99729502040636919, .99767926623147807, 
.99806935059062074, .99846499727875382, .99886580796721192, .99927127022009188, 
.99968076382479731, 1.0000935656028720, 1.0005088528404902, 1.0009257054543448, 
1.0013431069875580, 1.0017599445110850, 1.0021750074883951, 1.0025869856445302, 
1.0029944658644954, 1.0033959281298745, 1.0037897404860758, 1.0041741530151556, 
1.0045109252957629, 1.0048562507346159, 1.0051625738071936, 1.0054199227990652, 
1.0056229329396871, 1.0057684643245426, 1.0058556792404789, 1.0058847271244491, 
.98541501890035188, .98549073723832115, .98571399567599760, .98607439752915194, 
.98655607944894974, .98713908852027477, .98780312576576368, .98852402627517533, 
.98920773746579449, .98997317933173157, .99074783380590981, .99153348289700172, 
.99233143600201601, .99314257971693198, .99396742011657313, .99480611841347000, 
.99565852076779870, .99652418290322275, .99740239008308007, .99829217291496108, 
.99919231937692732, 1.0001013833934345, 1.0010176902318051, 1.0019393389394367, 
1.0028642019966743, 1.0037899223193956, 1.0047139077079972, 1.0056333228048879, 
1.0065450785900820, 1.0074458194134945, 1.0083319075324524, 1.0091994050932707, 
1.0099616173402992, 1.0107455081381476, 1.0114429886541088, 1.0120306357874720, 
1.0124953751177628, 1.0128292178974081, 1.0130295772074718, 1.0130963596716558, 
.97487878704350039, .97500783102957245, .97538825029280696, .97600212870737912, 
.97682216215751250, .97781409124194618, .97894316841674744, .98016824755190080, 
.98132967971052532, .98262970783288528, .98394543574388982, .98528022156857586, 
.98663663349601272, .98801653076198792, .98942113219567984, .99085107374496806, 
.99230645617617574, .99378688395545223, .99529149615550039, .99681899008864894, 
.99836763824177555, .99993529897732991, 1.0015194213651930, 1.0031170444202626, 
1.0047247909387871, 1.0063388560511952, 1.0079549905394622, 1.0095684789021968, 
1.0111741120903024, 1.0127661547803886, 1.0143383070027419, 1.0158836598970553, 
1.0172468864029409, 1.0186546074921108, 1.0199124457256559, 1.0209764288894130, 
1.0218208505882125, 1.0224291863043495, 1.0227950335656939, 1.0229171053333779, 
.96230643559952689, .96249766981703255, .96306130162704739, .96397044124473685, 
.96518415992209894, .96665123904193029, .96831990276753821, .97012921406595554, 
.97184368451840852, .97376223707024223, .97570393953633383, .97767430786332055, 
.97967771793647629, .98171751962871667, .98379613318211756, .98591512977435758, 
.98807529782546555, .99027669633914828, .99251869634140510, .99480001127167028, 
.99711871699369448, .99947226192075211, 1.0018574675890475, 1.0042705198613631, 
1.0067069507974928, 1.0091616110866719, 1.0116286327984785, 1.0141013820715477, 
1.0165724012238073, 1.0190333396349536, 1.0214748726244332, 1.0238866074318831, 
1.0260246130334627, 1.0282436197957194, 1.0302368776503059, 1.0319314219582156, 
1.0332823269122360, 1.0342591307803459, 1.0348481147168014, 1.0350449092239087, 
.94827945506252442, .94853830151665330, .94930103073438132, .95053075101200202, 
.95217137671282315, .95415290720355139, .95640481586774828, .95884463535396869, 
.96115522002895737, .96373994654315898, .96635564966545739, .96901056195855195, 
.97171142350868578, .97446362871851433, .97727135045942226, .98013764372670975, 
.98306453058100762, .98605306783318325, .98910339863091379, .99221478882671699, 
.99538564874368340, .99861354070095054, 1.0018951724107716, 1.0052263761076983, 
1.0086020730128005, 1.0120162224669096, 1.0154617547816354, 1.0189304865506439, 
1.0224130168322432, 1.0258986022547625, 1.0293750087077553, 1.0328283368680162, 
1.0359071582800061, 1.0391215385179917, 1.0420269046337309, 1.0445116098323730, 
1.0465030847967647, 1.0479494622391414, 1.0488243576403953, 1.0491171679651852, 
.93345999847248440, .93378833322472854, .93475559123493879, .93631434039894812, 
.93839254610712008, .94090052233629995, .94374819001987031, .94683090337406706, 
.94974843521429297, .95301074663942009, .95631162927654156, .95966248470373194, 
.96307289450631872, .96655079805276523, .97010264345758035, .97373351398230168, 
.97744723172880088, .98124644010518935, .98513266618659046, .98910636374399741, 
.99316693736701311, .99731274775263146, 1.0015410978637427, 1.0058481992681105, 
1.0102291175402593, 1.0146776951323884, 1.0191864495815033, 1.0237464443013098, 
1.0283471284891395, 1.0329761418372438, 1.0376190787480088, 1.0422592055852826, 
1.0464217569458387, 1.0507958444682350, 1.0547767822042382, 1.0582041196247707, 
1.0609678270239864, 1.0629852401300717, 1.0642100037922837, 1.0646206931538229, 
.91983746588971538, .92022798788110538, .92137820822374505, .92323104441477810, 
.92569987344376192, .92867705822405289, .93205476096533879, .93570841354174172, 
.93916413729914150, .94302657564610370, .94693386728460949, .95090055056407985, 
.95493908358249968, .95906004795075406, .96327232297108023, .96758323242682942, 
.97199866578059713, .97652317518312726, .98116004930013967, .98591136456099504, 
.99077801401308763, .99575971351844972, 1.0008549845413786, 1.0060611122322891, 
1.0113740768941082, 1.0167884561993268, 1.0222972946782372, 1.0278919359838723, 
1.0335618122084724, 1.0392941830181937, 1.0450738155080144, 1.0508825933553452, 
1.0561238179683863, 1.0616653260401558, 1.0667421342855896, 1.0711412266933742, 
1.0747096035412711, 1.0773273991452814, 1.0789224054675448, 1.0794582648357964, 
.90511356501051332, .90556955364881841, .90691231019322194, .90907437969817795, 
.91195346640391561, .91542269123429676, .91935523746527481, .92360545736926580, 
.92762262428376907, .93211030021382646, .93664879153109815, .94125622062751291, 
.94594839189025026, .95073902145913880, .95563993562326122, .96066123986293694, 
.96581146017665576, .97109765794706071, .97652551919131447, .98209941860208516, 
.98782245830592796, .99369648073252193, .99972205438324369, 1.0058984305878014, 
1.0122234685130591, 1.0186935246995945, 1.0253033021978395, 1.0320456528893893, 
1.0389113247209116, 1.0458886432285377, 1.0529631137286821, 1.0601169266784001, 
1.0666128774896251, 1.0735277495512456, 1.0799095100560085, 1.0854796480609405, 
1.0900284556889765, 1.0933846085557679, 1.0954380397516104, 1.0961294375700223, 
.89079057683388119, .89130856796274538, .89283360644671406, .89528822209265180, 
.89855498618066398, .90248846917735859, .90694359642036249, .91175462681791997, 
.91629867186547675, .92137223211723329, .92650152721716628, .93170831643045848, 
.93701185231903050, .94242913405789941, .94797512894217317, .95366296380218774, 
.95950408775174811, .96550840735601506, .97168439491985070, .97803917015891406, 
.98457855501458780, .99130710079498141, .99822808614667213, 1.0053434835568282, 
1.0126538911144087, 1.0201584250708005, 1.0278545672660135, 1.0357379596345371, 
1.0438021356517855, 1.0520381755612132, 1.0604342683080887, 1.0689751579935329, 
1.0767789309667188, 1.0851417159303738, 1.0929164050748055, 1.0997519529680382, 
1.1053723491592511, 1.1095433824123826, 1.1121063642656275, 1.1129713015560008, 
.87685410044732093, .87743088003042657, .87912869638317592, .88186042155031108, 
.88549402781812236, .88986620427849461, .89481429809841762, .90015337515857008, 
.90519265109405367, .91081605547994215, .91649911085166447, .92226722037829978, 
.92814313617975130, .93414723411515888, .94029775757486511, .94661103164047922, 
.95310164879993775, .95978262715405982, .96666554173102870, .97376062913347428, 
.98107686527197280, .98862201537537620, .99640265479187695, 1.0044241582760485, 
1.0126906544564751, 1.0212049409424433, 1.0299683539853723, 1.0389805846670646, 
1.0482394311191987, 1.0577404731274911, 1.0674766514390333, 1.0774377289342574, 
1.0865941414659510, 1.0964708270188066, 1.1057193054254573, 1.1139096410187224, 
1.1206900900540930, 1.1257516833878201, 1.1288754468087649, 1.1299320842489831, 
.86461617922762635, .86524336288520687, .86708927659921176, .87005839558655551, 
.87400598967417131, .87875319220717418, .88412206457137243, .88991108930707918, 
.89537165052726686, .90146208958986196, .90761489188578715, .91385871973756216, 
.92021948867685244, .92672065995004238, .93338350377472323, .94022733435975558, 
.94726971766182411, .95452665271669879, .96201272716248865, .96974124726821931, 
.97772434239167449, .98597304330728075, .99449733325298251, 1.0033061698224104, 
1.0124074749446339, 1.0218080891082344, 1.0315136846488224, 1.0415286312685823, 
1.0518558049257405, 1.0624963287590730, 1.0734492317724202, 1.0847110076842004, 
1.0951175203350945, 1.1064070893881790, 1.1170458228994657, 1.1265279249234803, 
1.1344256736229730, 1.1403524763801818, 1.1440245639432587, 1.1452692859848543, 
.85196431174071208, .85264243322554756, .85463797968841093, .85784684040587900, 
.86211128326503639, .86723651362189025, .87302893763567513, .87927014304815532, 
.88515340123056785, .89171173193168429, .89833450607712151, .90505379100469152, 
.91189883784842996, .91889639121171318, .92607097241900915, .93344513696712250, 
.94103970693729821, .94887397915521399, .95696590981326590, .96533227610990691, 
.97398881522054681, .98295034059569175, .99223083518268240, 1.0018435206828400, 
1.0118009013800877, 1.0221147804063466, 1.0327962455484698, 1.0438556208787588, 
1.0553023796837282, 1.0671450135562437, 1.0793908525054376, 1.0920458323845323, 
1.1038025211059121, 1.1166318685217498, 1.1288007545637466, 1.1397187697820226, 
1.1488701663030194, 1.1557756231912120, 1.1600716481271749, 1.1615310762821243, 
.84057690814549080, .84129987768333392, .84342714366859195, .84684696038562057, 
.85139001904593170, .85684733196213747, .86301134465015339, .86964864431620859, 
.87590159018404856, .88286851545618827, .88990112506445140, .89703456490747902, 
.90430112714987354, .91173057428160909, .91935044018532464, .92718630846426997, 
.93526206861956018, .94360015087255222, .95222174053205666, .96114697282796935, 
.97039510909098403, .97998469506909514, .98993370205323772, 1.0002596513589646, 
1.0109797226116786, 1.0221108462642436, 1.0336697809331685, 1.0456731766435199, 
1.0581376262341065, 1.0710797095680606, 1.0845160399015610, 1.0984633308659417, 
1.1114821095727860, 1.1257628541514157, 1.1393873610380068, 1.1516839969688608, 
1.1620495440238237, 1.1699099659624739, 1.1748182244848548, 1.1764889593467227, 
.83082177254856138, .83158242053319043, .83382033365211250, .83741730294174843, 
.84219420632980091, .84793001240668542, .85440530715266039, .86137397595035547, 
.86793577139410638, .87524358219352272, .88261766620444460, .89009580582951131, 
.89771291585679802, .90550137897260807, .91349136187886049, .92171111194602118, 
.93018723486064336, .93894495411791457, .94800835349878565, .95740060389034287, 
.96714417598870228, .97726104060004326, .98777285846868346, .99870116186709734, 
1.0100675306634921, 1.0218937663662067, 1.0342020689401976, 1.0470152233510726, 
1.0603568064239226, 1.0742514307814102, 1.0887250532966182, 1.1038053942994495, 
1.1179379573261752, 1.1335086237249219, 1.1484368763146919, 1.1619779437881479, 
1.1734475467732342, 1.1821818624714393, 1.1876530168136759, 1.1895185181534336, 
.82295444050971289, .82374498293627763, .82607067615956087, .82980815505915551, 
.83477045871497024, .84072692432794793, .84744864225636463, .85467935085965463, 
.86148509628751135, .86906184702636169, .87670502957890473, .88445454602943052, 
.89234743164076757, .90041819897079067, .90869916593091227, .91722076745606995, 
.92601185115528295, .93509995787115130, .94451158853877526, .95427245914193393, 
.96440774596181045, .97494232375282708, .98590100002673220, .99730874937036940, 
1.0091909527976304, 1.0215736487475249, 1.0344838048177496, 1.0479496232103509, 
1.0620008990848406, 1.0766694611685055, 1.0919897409342383, 1.1079995457212906, 
1.1230505631821026, 1.1396909807667148, 1.1557071975703277, 1.1702932348082544, 
1.1826954051722165, 1.1921716279074157, 1.1981224588896380, 1.2001542651121737, 
.81719227082056523, .81800443168426351, .82039359994068195, .82423266792770533, 
.82932897867885147, .83544486357621872, .84234449815208545, .84976424254184157, 
.85674581704032104, .86451623120500787, .87235301579263613, .88029761899165737, 
.88838862941090024, .89666212611446305, .90515201528670819, .91389035300527632, 
.92290765443425937, .93223319044295572, .94189527325786659, .95192153331777885, 
.96233919007987034, .97317532018547803, .98445712722385936, .99621221845142538, 
1.0084688954101633, 1.0212564677155758, 1.0346056027867165, 1.0485487296813594, 
1.0631205236770704, 1.0783585118762651, 1.0943038625966459, 1.1110024594064420, 
1.1267367872474143, 1.1441764662065061, 1.1610093470390375, 1.1763834441835850, 
1.1894920040487627, 1.1995323252010990, 1.2058488918944850, 1.2080076886000428, 
.81367943737429504, .81450466341488459, .81693219163459529, .82083264746594442, 
.82600992099048887, .83222209332522323, .83922914134985871, .84676294910305505, 
.85385055400589160, .86173767988826555, .86969106571766619, .87775309882154640, 
.88596331408793483, .89435874744153898, .90297427768858851, .91184295609174626, 
.92099632395640996, .93046471928921848, .94027757428284175, .95046370604481112, 
.96105160368557196, .97206971568886502, .98354674250643459, .99551194068975601, 
1.0079954468018679, 1.0210286321618998, 1.0346445036707830, 1.0488781723768872, 
1.0637674214536383, 1.0793534212706817, 1.0956816654916586, 1.1128032464033359, 
1.1289582887066674, 1.1468914646134918, 1.1642301486130957, 1.1800938451339045, 
1.1936425435704432, 1.2040352237608782, 1.2105806659860723, 1.2128190109068042, 
.81249824440077101, .81332784423422766, .81576821390057552, .81968921914756987, 
.82489359241267546, .83113798925962723, .83818097767652244, .84575294031715298, 
.85287600221830073, .86080214450683901, .86879449723479973, .87689576285492266, 
.88514579424134532, .89358194929357171, .90223943451065047, .91115163685774869, 
.92035044419675893, .92986655536181995, .93972978168546106, .94996934248009346, 
.96061415771870751, .97169314201816459, .98323550511384306, .99527106547365496, 
1.0078305857511971, 1.0209461417578136, 1.0346515410744621, 1.0489828141954383, 
1.0639788116533255, 1.0796819574252132, 1.0961392365091174, 1.1134035410241490, 
1.1297007260566039, 1.1478008698918984, 1.1653109492206744, 1.1813408059845640, 
1.1950390949792447, 1.2055516807921729, 1.2121750793685826, 1.2144405321126552
};
static double tabled6 [ 800 ]= {
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 1.0000000000000000, 
.99496224769391617, .99498271468276978, .99504355981861792, .99514327727202367, 
.99527920552592919, .99544730230477396, .99564273559244887, .99585864115416201, 
.99606601777898784, .99630014487725782, .99653821814205634, .99677997158425080, 
.99702511864444143, .99727335298908182, .99752434893356323, .99777776153373599, 
.99803322638026382, .99829035912409048, .99854875475597593, .99880798665834780, 
.99906760544349005, .99932713758821889, .99958608387157188, .99984391761855151, 
1.0001000827495143, 1.0003539916312787, 1.0006050227223182, 1.0008525180003898, 
1.0010957801564764, 1.0013340695338093, 1.0015666007847897, 1.0017925392115679, 
1.0019897390828800, 1.0021913548783963, 1.0023698210721401, 1.0025195886828191, 
1.0026377059463145, 1.0027224111846356, 1.0027732029595855, 1.0027901265757961, 
.98023561389378944, .98031512498665907, .98055147749187255, .98093876679759873, 
.98146659152153537, .98211921368415445, .98287788230541129, .98371602136014458, 
.98452115056264722, .98543037115091703, .98635529195746585, .98729501841482725, 
.98824857534238597, .98921490898881870, .99019288768322633, .99118130121770764, 
.99217885906134373, .99318418748556003, .99419582566307607, .99521222078670990, 
.99623172223978418, .99725257483645458, .99827291113763350, .99929074283602559, 
1.0003039511918641, 1.0013102764889370, 1.0023073064681420, 1.0032924636827869, 
1.0042629917057786, 1.0052159401033067, 1.0061481480720875, 1.0070562266170575, 
1.0078507739557570, 1.0086651439218912, 1.0093878403011445, 1.0099957469556053, 
1.0104761660002883, 1.0108212541395096, 1.0110284173226571, 1.0110974842002083, 
.95688009629795234, .95705085994860439, .95755839444945856, .95838983237890374, 
.95952262281674550, .96092284359676550, .96255028472882120, .96434816528349943, 
.96607554267697004, .96802699755500177, .97001337549648825, .97203322264170411, 
.97408491249638429, .97616664727385793, .97827645643727279, .98041219257097356, 
.98257152467527831, .98475192894600305, .98695067706879169, .98916482202827860, 
.99139118140303450, .99362631808884561, .99586651836489093, .99810776718960548, 
1.0003457205853097, 1.0025756749429322, 1.0047925330504008, 1.0069907666206013, 
1.0091643750675163, 1.0113068402527228, 1.0134110768995721, 1.0154693783502333, 
1.0172777276378751, 1.0191388627200191, 1.0207974789490355, 1.0221981303683039, 
1.0233088474376755, 1.0241088911306071, 1.0245901060913245, 1.0247507008897703, 
.92651353992270102, .92679860683628790, .92764571011345376, .92903296598102569, 
.93092226417238133, .93325669657686805, .93596924259170411, .93896571810350538, 
.94184525704420841, .94509983627749583, .94841516074570709, .95178980061539342, 
.95522204721791182, .95870991026313113, .96225111080225179, .96584306986160573, 
.96948289263637122, .97316734809304017, .97689284378315799, .98065539561883164, 
.98445059230300114, .98827355404455974, .99211888511999724, .99598061976931429, 
.99985216083454498, 1.0037262104646136, 1.0075946921211918, 1.0114486630280661, 
1.0152782161137484, 1.0190723704076070, 1.0228189487699280, 1.0265044417755491, 
1.0297605358336646, 1.0331307906818830, 1.0361519651139779, 1.0387172836933578, 
1.0407614294095588, 1.0422395866001168, 1.0431311313986835, 1.0434290893027733, 
.89112722687521062, .89153935429394182, .89276376781312439, .89476815427994638, 
.89749662918082385, .90086642178166332, .90478076101415383, .90910442456905161, 
.91326013170364906, .91795936360499286, .92275020124953566, .92763221891621317, 
.93260461921120041, .93766622366846730, .94281545804413347, .94805033171893094, 
.95336841061063560, .95876678295519446, .96424201724476266, .96979011151461614, 
.97540643304936703, .98108564743147875, .98682163568027848, .99260739802541331, 
.99843494262237309, 1.0042951572464451, 1.0101776616925930, 1.0160706382601705, 
1.0219606373126497, 1.0278323544762080, 1.0336683755848199, 1.0394488850101198, 
1.0445912270486239, 1.0499516974953986, 1.0547924920582126, 1.0589316008072183, 
1.0622502789000177, 1.0646622220244237, 1.0661221922920703, 1.0666110277757432, 
.85272189021846501, .85326449685211014, .85487619046493213, .85751347410261147, 
.86110164319003957, .86553097565099424, .87067415640680423, .87635437907843139, 
.88181477585152801, .88799218844107763, .89429510002872159, .90072522849934733, 
.90728387009100087, .91397188396382933, .92078967132807588, .92773714776079275, 
.93481370737182823, .94201817743888239, .94934876202511220, .95680297292303196, 
.96437754603398507, .97206834098877504, .97987022143442870, .98777691294363268, 
.99578083493273573, 1.0038729022830679, 1.0120422915263740, 1.0202761654513923, 
1.0285593487835569, 1.0368739461476604, 1.0451988918054016, 1.0535094186279668, 
1.0609612327024076, 1.0687933271830652, 1.0759276571594811, 1.0820785829635280, 
1.0870471462272313, 1.0906803967365950, 1.0928892825561891, 1.0936305645046593, 
.81335987161787837, .81402822053613222, .81601295009375274, .81925930267987768, 
.82367384748920003, .82912045744710228, .83544233350559505, .84242309285520167, 
.84913439976127891, .85673008473447876, .86448586922977799, .87240664913989285, 
.88049691363780691, .88876072790057992, .89720171169630325, .90582301154055811, 
.91462726423615537, .92361654959108264, .93279232997369443, .94215537411284345, 
.95170566217842623, .96144226867221672, .97136321899963551, .98146531475129628, 
.99174392165923719, 1.0021927128539339, 1.0128033583653013, 1.0235651496931255, 
1.0344645456013428, 1.0454846219116287, 1.0566044037827671, 1.0677980534994462, 
1.0779211942927402, 1.0886569612992190, 1.0985303251709413, 1.1071220249173845, 
1.1141209690313072, 1.1192750124186572, 1.1224243794670605, 1.1234840865033338, 
.77825884795836575, .77903249981558575, .78132948265813806, .78508522374919898, 
.79019013159239970, .79648559740044711, .80379004979666036, .81185430776888434, 
.81960771387944920, .82838556493522009, .83735382101758749, .84652091703556709, 
.85589495751752309, .86548370333300839, .87529455698730918, .88533454339520634, 
.89561028325891057, .90612795621462805, .91689325078587569, .92791129789104820, 
.93918658418846723, .95072284088342385, .96252290273475027, .97458853083398500, 
.98692019121939634, .99951677943562145, 1.0123752786293292, 1.0254903355077063, 
1.0388537342421698, 1.0524537428487681, 1.0662742992694045, 1.0802939946882992, 
1.0930738741392156, 1.1067419149788695, 1.1194267002919505, 1.1305636175084934, 
1.1397106316008585, 1.1464931828719758, 1.1506584845572227, 1.1520637484278287, 
.74146590878204577, .74234298640565988, .74494652195583588, .74920194570963861, 
.75498334200055106, .76210965683954135, .77037487588307196, .77949788084241134, 
.78826941455112554, .79820269200066785, .80835720336928407, .81874567582753266, 
.82938065392183502, .84027449746737171, .85143938271301645, .86288730299556238, 
.87463006549871158, .88667928090188153, .89904634266497518, .91174239245272426, 
.92477826774629379, .93816442699555069, .95191084669013282, .96602688340741560, 
.98052109213946836, .99540098988161130, 1.0106727504056798, 1.0263408120968669, 
1.0424073753645351, 1.0588717589711649, 1.0757295749958935, 1.0929716691260669, 
1.1088241876533385, 1.1259349714273521, 1.1419744586494784, 1.1561970368924872, 
1.1679866758524853, 1.1767976291445243, 1.1822398123038020, 1.1840814692806261, 
.70679793470559049, .70776587983286654, .71063863340427236, .71533254206276076, 
.72170696080968466, .72956079019484055, .73866641625552087, .74871486113143937, 
.75837610274416067, .76931935101871085, .78051170381723489, .79197036215882166, 
.80371254878468334, .81575552370977134, .82811660943794285, .84081322165305661, 
.85386290184495028, .86728334871564361, .88109244536940834, .89530827923530787, 
.90994915140136823, .92503357153887178, .94058023382900824, .95660796821773275, 
.97313565983822814, .99018212744146927, 1.0077659490096167, 1.0259052191844467, 
1.0446172184428877, 1.0639179677246284, 1.0838216339927973, 1.1043397414303484, 
1.1233603187326484, 1.1440744585704039, 1.1636826078087759, 1.1812406974547083, 
1.1959298289498264, 1.2069947475329592, 1.2138690608008929, 1.2162025921480446, 
.67411042023916212, .67515809464497534, .67826697993017582, .68334521861717812, 
.69023893420423177, .69872916935296348, .70856923691200890, .71942584616577243, 
.72986380748476221, .74168892025808253, .75378814283281237, .76618317603703023, 
.77889599316827762, .79194887863724373, .80536448419227025, .81916589845271270, 
.83337672644634153, .84802117654613163, .86312415268553366, .87871135003270817, 
.89480935244420934, .91144573000755769, .92864913482113964, .94644939284281671, 
.96487758915436728, .98396614332558974, 1.0037488707168616, 1.0242610245574995, 
1.0455393125746073, 1.0676218810741376, 1.0905482592813638, 1.1143592587491011, 
1.1366064143952395, 1.1610425154688211, 1.1843935170147488, 1.2055033309529477, 
1.2233238960632102, 1.2368526949758762, 1.2453065119657196, 1.2481851106432160, 
.64624270401115369, .64735370269837300, .65065007751690052, .65603330173648111, 
.66333878370303256, .67233319436598415, .68275460200092286, .69425047812324137, 
.70530263762599843, .71782516374611582, .73064200547198332, .74377884589157839, 
.75726189830567026, .77111796931745635, .78537454776745408, .80005991544438892, 
.81520327681918126, .83083490611679278, .84698631093332155, .86369041238437017, 
.88098174249023486, .89889666021901671, .91747358838312743, .93675327449928070, 
.95677907989325917, .97759730293702618, .99925754463528904, 1.0218131283126974, 
1.0453215907204170, 1.0698452709481585, 1.0954520387615359, 1.1222162303815109, 
1.1473910475837112, 1.1752467180053969, 1.2020833744429757, 1.2265456416878513, 
1.2473593658745834, 1.2632687383596541, 1.2732608111198971, 1.2766724914020935, 
.61824099628669771, .61941121449737702, .62288288376284906, .62855112436411020, 
.63624116327836211, .64570607751852949, .65666959167273875, .66876123164632417, 
.68038561665654995, .69355793841379376, .70704374739108111, .72087280325864544, 
.73507569473009493, .74968393128972777, .76473007049706598, .78024787728921441, 
.79627251344550520, .81284075691994628, .82999125219039571, .84776479421950968, 
.86620465017806658, .88535692487242468, .90527097800761303, .92599990423222900, 
.94760109068155152, .97013687196332887, .99367531000509246, 1.0182911371753347, 
1.0440669177170856, 1.0710945084017569, 1.0994769407812681, 1.1293309160603595, 
1.1576014023779924, 1.1891140476786608, 1.2197243788757388, 1.2478605875084174, 
1.2719920582906892, 1.2905659820944835, 1.3022922045512348, 1.3063071535828028, 
.59373018779782838, .59494852417282285, .59856260110308734, .60446230321947003, 
.61246447843544605, .62231105043630808, .63371402126828114, .64628835789943484, 
.65837623947874924, .67207491815443809, .68610287221390698, .70049344653566681, 
.71528110847376924, .73050156763912177, .74619194094556807, .76239096002298901, 
.77913922028355387, .79647947297458899, .81445696360278985, .83311982231614068, 
.85251951435123829, .87271136170852903, .89375515109484094, .91571584828174479, 
.93866444597500420, .96267898199581114, .98784577847975008, 1.0142609732177974, 
1.0420324450099985, 1.0712822824466783, 1.1021500211870130, 1.1347969990281463, 
1.1658932329893050, 1.2007788882462984, 1.2349091952726066, 1.2665106575509604, 
1.2938040034972211, 1.3149398327232669, 1.3283442641697187, 1.3329450408229504, 
.57324696245285367, .57450277756604762, .57822776196833747, .58430768551704992, 
.59255281276667449, .60269631747194718, .61444102704655634, .62739057647367236, 
.63983867113921156, .65394644220972620, .66839593768684624, .68322348117137266, 
.69846678604548995, .71416510053968207, .73035940697908652, .74709267306993588, 
.76441015568602366, .78235976015381503, .80099246068894192, .82036279060053585, 
.84052941438859541, .86155579822091777, .88351100091438517, .90647061507124094, 
.93051789834126653, .95574514927941256, .98225540311208515, 1.0101645533907506, 
1.0396040517267662, 1.0707244092692203, 1.1036998372527842, 1.1387345504112172, 
1.1722664071507662, 1.2100848322321709, 1.2473045107940888, 1.2819755272758819, 
1.3120937951739904, 1.3355352426597245, 1.3504580786492002, 1.3555904010647495, 
.55706902851858697, .55835264264559936, .56215988593916522, .56837346220826745, 
.57679872797965881, .58716233566432052, .59916028262754317, .61238779966509216, 
.62510271499015435, .63951353782147713, .65427542349173395, .66942702105876439, 
.68500859426547434, .70106218757690253, .71763185394808049, .73476394290627611, 
.75250745048892920, .77091443554977113, .79004051013843390, .80994541531148956, 
.83069369814241989, .85235551126039810, .87500756351604958, .89873426014830760, 
.92362908430593293, .94979629078969752, .97735301029479582, 1.0064319028542830, 
1.0371845602063221, 1.0697859512979053, 1.1044403555733340, 1.1413894756901573, 
1.1768870196693544, 1.2170888818740426, 1.2568380373811050, 1.2940407292724860, 
1.3265046836401027, 1.3518715770113519, 1.3680677477078098, 1.3736468097588841, 
.54541148628109148, .54671412477927539, .55057766176372347, .55688269977742022, 
.56543122853358763, .57594542490508605, .58811660028635904, .60153423307686538, 
.61443162871992241, .62904972938100312, .64402534666070765, .65939878345259684, 
.67521212673602637, .69150942918217029, .70833695820178377, .72574351160755635, 
.74378080229610255, .76250391766165726, .78197186306899658, .80224820291044152, 
.82340181789799227, .84550780376435708, .86864854512410272, .89291500983674055, 
.91840832525075089, .94524172038834775, .97354295090371701, 1.0034573720564313, 
1.0351518981316448, 1.0688202002164797, 1.1046896751136592, 1.1430310154426175, 
1.1799651414434933, 1.2219179651451986, 1.2635356907090372, 1.3026187183431339, 
1.3368335599400356, 1.3636437022779586, 1.3807972401499049, 1.3867127292500803, 
.53838345588864611, .53969714939218838, .54359339883726410, .54995158601096650, 
.55857174277844990, .56917344163654493, .58144526684258113, .59497335329927281, 
.60797676456444993, .62271529716519569, .63781512457567002, .65331753584625529, 
.66926570914911354, .68570490292827724, .70268271804015914, .72024943044025846, 
.73845839737589980, .75736654356400379, .77703493771706478, .79752947432583188, 
.81892168119295305, .84128968035234064, .86471933943101031, .88930566327429295, 
.91515449335066679, .94238460751703046, .97113034898888728, 1.0015449669835119, 
1.0338049326557692, 1.0681156198762756, 1.1047189412772931, 1.1439038603755913, 
1.1817113081599264, 1.2247322465092840, 1.2674940118920501, 1.3077323916544181, 
1.3430265668806216, 1.3707288451497135, 1.3884753546980417, 1.3945994382900100, 
.53603363143442551, .53735095094381990, .54125793009428813, .54763355048075346, 
.55627720176485240, .56690760176206489, .57921243961508639, .59277675339740155, 
.60581493633749794, .62059297085432319, .63573354442005270, .65127827376871661, 
.66727070035664082, .68375648472721484, .70078367306993317, .71840303566216013, 
.73666848034463572, .75563754777344638, .77537199916752553, .79593851193571763, 
.81740950431147419, .83986411747685874, .86338939337121323, .88808169954982852, 
.91404847072737605, .94141036253166710, .97030395046819617, 1.0008851625309489, 
1.0333337178152531, 1.0678589737534125, 1.1047077924409023, 1.1441753784569226, 
1.1822759398561324, 1.2256559868718317, 1.2688030886404755, 1.3094313033855054, 
1.3450902529930551, 1.3730944489981287, 1.3910418301219749, 1.3972366125542910
};
static double tablef7 [ 2400 ]= {
-0.46377247451273951, -0.46211599547762228, -0.45729690457016242, 
-0.44972277281828027, -0.43999419882190427, -0.42881512729234972, 
-0.41685680307017914, -0.40476834508829512, -0.39411378239879252, 
-0.38305411503878978, -0.37271771524016239, -0.36303141296474606, 
-0.35393166930345199, -0.34536305723362254, -0.33727701655237202, 
-0.32963082803958110, -0.32238676388969884, -0.31551138062788264, 
-0.30897492779603469, -0.30275085117308481, -0.29681537356272986, 
-0.29114713952527854, -0.28572691306210868, -0.28053731934306042, 
-0.27556262322166514, -0.27078853860425850, -0.26620206379868405, 
-0.26179133882201139, -0.25754552133744207, -0.25345467845180902, 
-0.24950969206297127, -0.24570217582143677, -0.24238275249616742, 
-0.23897362751892520, -0.23592401083700362, -0.23332511878934131, 
-0.23123798287534084, -0.22971418440096877, -0.22878719240584020, 
-0.22847582194617522, -0.46553376300069221, -0.46389179656521778, 
-0.45911447578750791, -0.45160449956824672, -0.44195525708096023, 
-0.43086237813549289, -0.41898942837798953, -0.40697905731876011, 
-0.39638563685089329, -0.38538105040988702, -0.37508776539990707, 
-0.36543401786328145, -0.35635754497049288, -0.34780406662446850, 
-0.33972604641347764, -0.33208167452452059, -0.32483402812358719, 
-0.31795037447948216, -0.31140158956278474, -0.30516167057715188, 
-0.29920732530523497, -0.29351762459186342, -0.28807370697742485, 
-0.28285852660976796, -0.27785663723525000, -0.27305400639846711, 
-0.26843785504152295, -0.26399651854546097, -0.25971932594328673, 
-0.25559649459031342, -0.25161903803018837, -0.24777868516476276, 
-0.24442944440767414, -0.24098855247519152, -0.23790954536864339, 
-0.23528488746148793, -0.23317658379141321, -0.23163706509325246, 
-0.23070040056274033, -0.23038576267402516, -0.47074297306019136, 
-0.46914416291868227, -0.46449114002427394, -0.45717234790090977, 
-0.44776004801252145, -0.43692542894235231, -0.42530938518531680, 
-0.41353529333015127, -0.40312811361670697, -0.39229269952474349, 
-0.38213331920268989, -0.37258223147352110, -0.36358083669717978, 
-0.35507815910336443, -0.34702962303727986, -0.33939605897876524, 
-0.33214289059237784, -0.32523946546848842, -0.31865850072790108, 
-0.31237562106784195, -0.30636897168411475, -0.30061889221450672, 
-0.29510764070325446, -0.28981915879797565, -0.28473887111519419, 
-0.27985351306421058, -0.27515098248789399, -0.27062021132791014, 
-0.26625105419995720, -0.26203419130908745, -0.25796104357474098, 
-0.25402369819168380, -0.25058620967351436, -0.24705113557218628, 
-0.24388485366620422, -0.24118358289217395, -0.23901227225144514, 
-0.23742592477371935, -0.23646043188760446, -0.23613605312457886, 
-0.47919520547719918, -0.47766714123022519, -0.47321814417932945, 
-0.46621401169130627, -0.45719333945484208, -0.44678814488431084, 
-0.43560268836016497, -0.42422866895172795, -0.41414067792660142, 
-0.40359953816788618, -0.39367768724871849, -0.38431348478734573, 
-0.37545392191661120, -0.36705310045233058, -0.35907102722516837, 
-0.35147264987805621, -0.34422707935607444, -0.33730695800627558, 
-0.33068794219685299, -0.32434827572250624, -0.31826843572796398, 
-0.31243083697576640, -0.30681958337691190, -0.30142025805746259, 
-0.29621974504052734, -0.29120607701901208, -0.28636830478091174, 
-0.28169638470010577, -0.27718108137680558, -0.27281388304432885, 
-0.26858692778391707, -0.26449293893046716, -0.26091234682421471, 
-0.25722399174129273, -0.25391520381079535, -0.25108849500130481, 
-0.24881379975112169, -0.24715048968784348, -0.24613756718098684, 
-0.24579715381880646, -0.49055253446115066, -0.48912087685193700, 
-0.48495040318344235, -0.47837740086008329, -0.46989641017708106, 
-0.46008777584196436, -0.44950688765353636, -0.43870219426092211, 
-0.42907561948478318, -0.41896797634851951, -0.40940456541124898, 
-0.40033113470589282, -0.39170150399870346, -0.38347603480764053, 
-0.37562043949235135, -0.36810484528175486, -0.36090305189577732, 
-0.35399193758568664, -0.34735097999835579, -0.34096186664568912, 
-0.33480817587634287, -0.32887511375412631, -0.32314929559911021, 
-0.31761856346130424, -0.31227183269781228, -0.30709896227342492, 
-0.30209064451742285, -0.29723831093017766, -0.29253405130357762, 
-0.28797054394492200, -0.28354099520863868, -0.27923908686934062, 
-0.27546730170306399, -0.27157297166010064, -0.26807168956270305, 
-0.26507481127232929, -0.26265938567879937, -0.26089104634467980, 
-0.25981328928079678, -0.25945093897755367, -0.50434815979290515, 
-0.50303555702490166, -0.49920975592786754, -0.49317256920615282, 
-0.48536680754987382, -0.47631155412655490, -0.46650349359683224, 
-0.45643768499209153, -0.44742044263603603, -0.43789703639005850, 
-0.42882902813233759, -0.42016994656039580, -0.41188087164347609, 
-0.40392889184120062, -0.39628592257998324, -0.38892779227111274, 
-0.38183352857504049, -0.37498479606164881, -0.36836544943116500, 
-0.36196117574119735, -0.35575920577223463, -0.34974807953031473, 
-0.34391745446135852, -0.33825794760343436, -0.33276100488710308, 
-0.32741879229070375, -0.32222410469558826, -0.31717028915840828, 
-0.31225117999060126, -0.30746104355821385, -0.30279453112429888, 
-0.29824663837811594, -0.29424647509719487, -0.29010403628230991, 
-0.28636913613112858, -0.28316445510827010, -0.28057633835874596, 
-0.27867865201200320, -0.27752085929161330, -0.27713139678057414, 
-0.52006115239196737, -0.51888656960363244, -0.51546123810775635, 
-0.51004945845973015, -0.50303743804100820, -0.49487651187928128, 
-0.48599752827910991, -0.47683389130462228, -0.46857376340349272, 
-0.45979097867061886, -0.45136626162719229, -0.44326051208886458, 
-0.43544175119864348, -0.42788356854029924, -0.42056394758528232, 
-0.41346436807789444, -0.40656911344951219, -0.39986473166120856, 
-0.39333961202547035, -0.38698365053228184, -0.38078798331414121, 
-0.37474477300629438, -0.36884703648645994, -0.36308850521857883, 
-0.35746351145827865, -0.35196689509976720, -0.34659392709289522, 
-0.34134024623361914, -0.33620180680174389, -0.33117483503792572, 
-0.32625579285502093, -0.32144134749459596, -0.31719027735131733, 
-0.31277200113697806, -0.30877463662961901, -0.30533449376198224, 
-0.30254939004300702, -0.30050343248208172, -0.29925359800399531, 
-0.29883290633082310, -0.53704110112996006, -0.53601832454763643, 
-0.53303444732844762, -0.52831522864096059, -0.52218838175420412, 
-0.51503445822979366, -0.50721455367073698, -0.49909478640083998, 
-0.49172529318083064, -0.48383007645039217, -0.47619322383249429, 
-0.46878197763284313, -0.46157037240410387, -0.45453768187724847, 
-0.44766725456951872, -0.44094563123649566, -0.43436186919678021, 
-0.42790702025185173, -0.42157372388204661, -0.41535588784061972, 
-0.40924843564218033, -0.40324710570924882, -0.39734829074394012, 
-0.39154890866606828, -0.38584629850243496, -0.38023813613123314, 
-0.37472236592480712, -0.36929714519568313, -0.36396079900770828, 
-0.35871178341885462, -0.35354865561298726, -0.34847004968261919, 
-0.34396529498157574, -0.33926339078356504, -0.33499218210915266, 
-0.33130347924743414, -0.32830854296413248, -0.32610360039687880, 
-0.32475464750705491, -0.32430025403011703, -0.55300413135930715, 
-0.55212636881990562, -0.54956504953934793, -0.54551148803396821, 
-0.54024083497378372, -0.53406937532293486, -0.52729406890876451, 
-0.52021745260014940, -0.51375079972882642, -0.50676966092980213, 
-0.49995906317668404, -0.49329084875978874, -0.48674342233329260, 
-0.48030021000054574, -0.47394851089826018, -0.46767863128389436, 
-0.46148322456224661, -0.45535678325323520, -0.44929524433945178, 
-0.44329568012856374, -0.43735605426928220, -0.43147502788527209, 
-0.42565180461324742, -0.41988600610288745, -0.41417757156565294, 
-0.40852667645950425, -0.40293366651471080, -0.39739900414644713, 
-0.39192322493658238, -0.38650690235301610, -0.38115061924856242, 
-0.37585494497075939, -0.37113565173115770, -0.36618823485837387, 
-0.36167529604346621, -0.35776379507961442, -0.35457857339923105, 
-0.35222824272253339, -0.35078815513423867, -0.35030269171108813, 
-0.57065191720593245, -0.56993683173493348, -0.56785042320987245, 
-0.56454782720803365, -0.56024869706465596, -0.55520079049467552, 
-0.54963188539315844, -0.54377415406127083, -0.53837586179401986, 
-0.53249144714238739, -0.52668783414679277, -0.52094062765165572, 
-0.51523178538794800, -0.50954810406645212, -0.50388009745249696, 
-0.49822115450521698, -0.49256690025683902, -0.48691470527240155, 
-0.48126330526465699, -0.47561250327236598, -0.46996293436613573, 
-0.46431587818048092, -0.45867310838031039, -0.45303677091970129, 
-0.44740928495522703, -0.44179326175067540, -0.43619143800383536, 
-0.43060662084367030, -0.42504164236156272, -0.41949932200634327, 
-0.41398243552780367, -0.40849368942524911, -0.40357434427069611, 
-0.39838960283279551, -0.39363615438187843, -0.38949809988907730, 
-0.38611624380853704, -0.38361395324735209, -0.38207791536012820, 
-0.38155962382578124, -0.58822468711916787, -0.58767381891787185, 
-0.58606766322327508, -0.58352719506738249, -0.58021939677984657, 
-0.57632635344903538, -0.57200881809553150, -0.56742907731085468, 
-0.56316396025082349, -0.55845746002287688, -0.55375063499776467, 
-0.54902168378039240, -0.54425496102578230, -0.53943950226332071, 
-0.53456793688001647, -0.52963567665349320, -0.52464030249354177, 
-0.51958109551134180, -0.51445867437742898, -0.50927471178151817, 
-0.50403171034298744, -0.49873282362142640, -0.49338171165056516, 
-0.48798242313649504, -0.48253929843871712, -0.47705688890566234, 
-0.47153988921325601, -0.46599308015933044, -0.46042127997106597, 
-0.45482930263912086, -0.44922192213817774, -0.44360384165657879, 
-0.43853634283190768, -0.43316352233488320, -0.42820972646340405, 
-0.42387615876453100, -0.42032030895162081, -0.41768122676321467, 
-0.41605788144799810, -0.41550956334246963, -0.60571965461582100, 
-0.60533427367417325, -0.60421285158549864, -0.60244408516658043, 
-0.60014494722180112, -0.59743459708265989, -0.59440893892435698, 
-0.59116107163472920, -0.58808885665639565, -0.58463587756621860, 
-0.58111037747783401, -0.57749222124367383, -0.57376722156755089, 
-0.56992571071175324, -0.56596149449439015, -0.56187107622228860, 
-0.55765307376068129, -0.55330777645392515, -0.54883680440728722, 
-0.54424284341191198, -0.53952943624717520, -0.53470081632095801, 
-0.52976177332198326, -0.52471754323027597, -0.51957371697725951, 
-0.51433616347954247, -0.50901096383630440, -0.50360435428070427, 
-0.49812267608174909, -0.49257233105469006, -0.48695974169110856, 
-0.48129131519030677, -0.47614124509859607, -0.47064391687212204, 
-0.46554288865027688, -0.46105591544844615, -0.45735757411637140, 
-0.45460330131508064, -0.45290518254047224, -0.45233094188648731, 
-0.62313443255719370, -0.62291556396401282, -0.62228257366032669, 
-0.62129363254354508, -0.62001820245399986, -0.61851514403022301, 
-0.61681767551198119, -0.61495057298565670, -0.61312602960161444, 
-0.61099660730239558, -0.60873154999364043, -0.60631168546379588, 
-0.60372355658964461, -0.60095804432959543, -0.59800936587489217, 
-0.59487433662895406, -0.59155182022064934, -0.58804231413048202, 
-0.58434763413433688, -0.58047067137877140, -0.57641520321277429, 
-0.57218574401137590, -0.56778742584891462, -0.56322590148292784, 
-0.55850726400613282, -0.55363797892474342, -0.54862482547021918, 
-0.54347484474750984, -0.53819529293460411, -0.53279359822369188, 
-0.52727732056727905, -0.52165411358701907, -0.51650214704466202, 
-0.51095990200449892, -0.50577943509540935, -0.50119385467896027, 
-0.49739478650560165, -0.49455443179230174, -0.49279863910986454, 
-0.49220411234552297, -0.64046698792851708, -0.64041543489316588, 
-0.64027386379581042, -0.64007154825448953, -0.63983277840307529, 
-0.63955861548244430, -0.63922171612761780, -0.63877952086318412, 
-0.63825262106565606, -0.63751128696264991, -0.63658029366844848, 
-0.63544093144250402, -0.63407994367438836, -0.63248820221738733, 
-0.63065974988718065, -0.62859110046335509, -0.62628072180747629, 
-0.62372865079824169, -0.62093620414607609, -0.61790575953237448, 
-0.61464058864132265, -0.61114472860896019, -0.60742288191253124, 
-0.60348033722772720, -0.59932290560138834, -0.59495686763102718, 
-0.59038892835213852, -0.58562617730725264, -0.58067605187445726, 
-0.57554630241433768, -0.57024495818522825, -0.56478029329948298, 
-0.55972352084822374, -0.55423362675640020, -0.54905793719650875, 
-0.54444291930621636, -0.54059662906991668, -0.53770795772688984, 
-0.53591688337702659, -0.53530948827559766, -0.65771560179115196, 
-0.65783196658091014, -0.65818415690001963, -0.65877406062618959, 
-0.65958298101147717, -0.66055654678554254, -0.66160891893203844, 
-0.66263127439943159, -0.66344737499478956, -0.66415326800647334, 
-0.66462445153760608, -0.66484235675666659, -0.66479357859765087, 
-0.66446860187318546, -0.66386088584172376, -0.66296619976329552, 
-0.66178213681424769, -0.66030775642719286, -0.65854332015760308, 
-0.65649009629440713, -0.65415021533696027, -0.65152656323843672, 
-0.64862270266497823, -0.64544281489897992, -0.64199165672867553, 
-0.63827452792068640, -0.63429724580828661, -0.63006612424325070, 
-0.62558795472105311, -0.62086998794513559, -0.61591991447864647, 
-0.61074584346365302, -0.60589895398284344, -0.60057791949060668, 
-0.59550935992972298, -0.59095014816659408, -0.58712341030778348, 
-0.58423405192265765, -0.58243615275535554, -0.58182535396669399, 
-0.67487883379618696, -0.67516353677727559, -0.67601124530031520, 
-0.67739786253931411, -0.67926374103331755, -0.68150130951096899, 
-0.68396822587870587, -0.68649052888091222, -0.68869057028942726, 
-0.69089758507333177, -0.69283359432404557, -0.69448000233646912, 
-0.69582306664609217, -0.69685267162136444, -0.69756145107793854, 
-0.69794415506062124, -0.69799719020157203, -0.69771828529839808, 
-0.69710624842973679, -0.69616079175867840, -0.69488240685398273, 
-0.69327227795223456, -0.69133222377700643, -0.68906466077274331, 
-0.68647258220040919, -0.68355954868144919, -0.68032968660591842, 
-0.67678769143751150, -0.67293883342218408, -0.66878896358791340, 
-0.66434451824660268, -0.65961252050041330, -0.65510878026610586, 
-0.65009375902402788, -0.64525444489836770, -0.64085403412945308, 
-0.63712844066374119, -0.63429713407524502, -0.63252773657971517, 
-0.63192532413057878, -0.69195549072242968, -0.69240878767699132, 
-0.69375324029505503, -0.69594006412220899, -0.69887055489999583, 
-0.70238603964881474, -0.70628958142086706, -0.71034323366809082, 
-0.71396395003929780, -0.71772091409327043, -0.72117902606265183, 
-0.72431962419593998, -0.72712857729468738, -0.72959510161773788, 
-0.73171091660154297, -0.73346963745109583, -0.73486633624649000, 
-0.73589722494010823, -0.73655942791849016, -0.73685082135617752, 
-0.73676992305570399, -0.73631582089289271, -0.73548813103900990, 
-0.73428697924767405, -0.73271299996476391, -0.73076734904054153, 
-0.72845172653107737, -0.72576840656769415, -0.72272027161920582, 
-0.71931084872508677, -0.71554434547871792, -0.71142568371949100, 
-0.70741764301054721, -0.70286760141233295, -0.69840087901043024, 
-0.69428160966260888, -0.69075509273257296, -0.68805293890559349, 
-0.68635506092069987, -0.68577543642520111, -0.70894459856703951, 
-0.70956659644357123, -0.71140853787341274, -0.71439815029548404, 
-0.71839943111915061, -0.72320457155916984, -0.72856385607153862, 
-0.73417651224912606, -0.73925064854012201, -0.74460152146931335, 
-0.74963377262610176, -0.75432873931156064, -0.75867196486273769, 
-0.76265205460635999, -0.76625985961673959, -0.76948788942475932, 
-0.77232988766596034, -0.77478052585047008, -0.77683518435914323, 
-0.77848979905871973, -0.77974075820868383, -0.78058483862205552, 
-0.78101917299079082, -0.78104124231489847, -0.78064888876078886, 
-0.77984034520600709, -0.77861427833371033, -0.77696984251327095, 
-0.77490674191174858, -0.77242529837798579, -0.76952652267026689, 
-0.76621218659660428, -0.76287204061960101, -0.75896860396507160, 
-0.75504027872762436, -0.75134527738307333, -0.74813355129411629, 
-0.74564523587873599, -0.74407040293994142, -0.74353086636535326, 
-0.72584537777424463, -0.72663604901511580, -0.72897578813846697, 
-0.73276994266383078, -0.73784684170859349, -0.74395137730878502, 
-0.75078277477737154, -0.75797858497671489, -0.76453511735514268, 
-0.77151920643884109, -0.77817255596209711, -0.78447664878923218, 
-0.79041685802089201, -0.79598133825520123, -0.80116023286388516, 
-0.80594510160528543, -0.81032850492279385, -0.81430370191078150, 
-0.81786443247264717, -0.82100476323406146, -0.82371898289489953, 
-0.82600153689553379, -0.82784699416479868, -0.82925004071206931, 
-0.83020549618825834, -0.83070835044503377, -0.83075381768658407, 
-0.83033740611648921, -0.82945500109434456, -0.82810295978043935, 
-0.82627821510278032, -0.82397838666676415, -0.82149791575729125, 
-0.81844580063193151, -0.81524503298579820, -0.81213941573539716, 
-0.80937727076228749, -0.80720220789740793, -0.80581123563110291, 
-0.80533226321760720, -0.74265722123770495, -0.74361641681450430, 
-0.74645386802158376, -0.75105356530275524, -0.75720967820489704, 
-0.76462151102246134, -0.77293885000304019, -0.78173869488216030, 
-0.78980305144928364, -0.79845523843549149, -0.80677175659553513, 
-0.81473444134182882, -0.82232872116130273, -0.82954254139734385, 
-0.83636559158656743, -0.84278874385040858, -0.84880364099354555, 
-0.85440239300499343, -0.85957735381206790, -0.86432095893098004, 
-0.86862561063612609, -0.87248360139449121, -0.87588706918872317, 
-0.87882798036999721, -0.88129813708946703, -0.88328920731661631, 
-0.88479277606945650, -0.88580041682444477, -0.88630378219015227, 
-0.88629471285239625, -0.88576536355844548, -0.88470834453223657, 
-0.88329836019535501, -0.88132529674157500, -0.87906506325565203, 
-0.87673680551039246, -0.87457917247123640, -0.87283251011364593, 
-0.87169621664915697, -0.87130171799461664, -0.75937967475598667, 
-0.76050713603154474, -0.76384185692302853, -0.76924741403450481, 
-0.77648521182337626, -0.78521055788714222, -0.79502531937587362, 
-0.80544703683411112, -0.81504131622000231, -0.82539229101033185, 
-0.83540936668916210, -0.84507497993551428, -0.85437489068647055, 
-0.86329713686393117, -0.87183127970646201, -0.87996785033419378, 
-0.88769793839453659, -0.89501288305619331, -0.90190403936431474, 
-0.90836260150721426, -0.91437947037932776, -0.91994515689012168, 
-0.92504971535261149, -0.92968270336227787, -0.93383316608617783, 
-0.93748964397419954, -0.94064020367352597, -0.94327249242909161, 
-0.94537381651789165, -0.94693124430880027, -0.94793173436957978, 
-0.94836228866516203, -0.94825151801126706, -0.94760756579139971, 
-0.94652457671408718, -0.94518494162758415, -0.94380763286487790, 
-0.94262104791033285, -0.94182085208653610, -0.94153839151344643, 
-0.77601241965906886, -0.77730778918406418, -0.78113901495530746, 
-0.78735012882963273, -0.79567105738170352, -0.80571458745974955, 
-0.81703608770740910, -0.82909469020672077, -0.84023787607420464, 
-0.85231437363237350, -0.86406493570027509, -0.87547287426716935, 
-0.88652458923736244, -0.89720855382976528, -0.90751457738321928, 
-0.91743325924845498, -0.92695557668040818, -0.93607256840084874, 
-0.94477508780530095, -0.95305360804127703, -0.96089806685005730, 
-0.96829774305003757, -0.97524115941882639, -0.98171600886893895, 
-0.98770910244126264, -0.99320633890831600, -0.99819269677439748, 
-1.0026522502299111, -1.0065682111813061, -1.0099229998340857, 
-1.0126983464379151, -1.0148754266858735, -1.0163087791534013, 
-1.0172649459214225, -1.0176189077929290, -1.0175023165096535, 
-1.0171023365247631, -1.0166245358683801, -1.0162528883745097, 
-1.0161138525846527, -0.79255525735750199, -0.79401808869954860, 
-0.79834476350254679, -0.80536056900870212, -0.81476514063301120, 
-0.82613011094590627, -0.83896567318656690, -0.85267355513958598, 
-0.86538172505201993, -0.87920676247421381, -0.89271951042479680, 
-0.90590444151966365, -0.91874892078463866, -0.93124222271850543, 
-0.94337481258824607, -0.95513780861621789, -0.96652256991600094, 
-0.97752037308598216, -0.98812215219288968, -0.99831828480401263, 
-1.0080984121775334, -1.0174512855788804, -1.0263646335169903, 
-1.0348250468479061, -1.0428178804076251, -1.0503271712617401, 
-1.0573355748894411, -1.0638243217052281, -1.0697731972822861, 
-1.0751605504690558, -1.0799633342544417, -1.0841571846878837, 
-1.0873933624390932, -1.0902394478587012, -1.0923115637566292, 
-1.0936747862050156, -1.0944700952501325, -1.0948669277122682, 
-1.0950275137651446, -1.0950672046259899, -0.80900809559536324, 
-0.81063786228933632, -0.81545866784177456, -0.82327779095285425, 
-0.83376566868581625, -0.84645404213464970, -0.86080915752580789, 
-0.87617629240649386, -0.89046281987283282, -0.90605593109811455, 
-0.92135557098746590, -0.93634765761605621, -0.95102084936318743, 
-0.96536559574846403, -0.97937343963742929, -0.99303649040326925, 
-1.0063470147533147, -1.0192971092418213, -1.0318784297932966, 
-1.0440819611133807, -1.0558978140432562, -1.0673150425766176, 
-1.0783214749521786, -1.0889035553162854, -1.0990461941566882, 
-1.1087326271997837, -1.1179442838469096, -1.1266606675687358, 
-1.1348592520168961, -1.1425153979560452, -1.1496022974458722, 
-1.1560909529600141, -1.1613993905104829, -1.1664409980950106, 
-1.1705316102348402, -1.1736521573971057, -1.1758807658492443, 
-1.1773348410429074, -1.1781424856221181, -1.1784001139939324, 
-0.82537093621404434, -0.82716703991408380, -0.83248042160076963, 
-0.84110102806270848, -0.85267110321738526, -0.86668366169322686, 
-0.88256213983505660, -0.89959626685895422, -0.91547201567403202, 
-0.93284948179010296, -0.94995696412062147, -0.96678210096602368, 
-0.98331516405290065, -0.99954814615842147, -1.0154740878230260, 
-1.0310865656492638, -1.0463792907068084, -1.0613457821120948, 
-1.0759790916023657, -1.0902715620609724, -1.1042146078018975, 
-1.1177985078155901, -1.1310122056466625, -1.1438431114681877, 
-1.1562769034777229, -1.1682973271377426, -1.1798859921439337, 
-1.1910221684231443, -1.2016825840082004, -1.2118412293548551, 
-1.2214691745756778, -1.2305344081534306, -1.2381915576741418, 
-1.2457462490522069, -1.2521715509952651, -1.2573451611131385, 
-1.2612634339897086, -1.2639731397434848, -1.2655533410259785, 
-1.2660718952446350, -0.84164386425661031, -0.84360564216268041, 
-0.84940983285259641, -0.85882967273260413, -0.87148013621420839, 
-0.88681658454539214, -0.90422069399675209, -0.92292749437125965, 
-0.94040100462581433, -0.95957607814324275, -0.97850883487556006, 
-0.99718889047392456, -1.0156084326090307, -1.0337613490510704, 
-1.0516425833748386, -1.0692476437270162, -1.0865722150067383, 
-1.1036118405861206, -1.1203616498824118, -1.1368161147891578, 
-1.1529688224562511, -1.1688122549734288, -1.1843375686691420, 
-1.1995343673269419, -1.2143904648821948, -1.2288916342694418, 
-1.2430213401859245, -1.2567604547396116, -1.2700869563681076, 
-1.2829756141457105, -1.2953976617247667, -1.3073204677502863, 
-1.3176054833778750, -1.3279980898246135, -1.3370858702057888, 
-1.3446230049547852, -1.3505030674862948, -1.3546808799397840, 
-1.3571688950762781, -1.3579948566310316, -0.85782703826240403, 
-0.85995376988742989, -0.86624681166946482, -0.87646326013186582, 
-0.89019166799814864, -0.90685073007586769, -0.92578132931790080, 
-0.94616459218437365, -0.96524225753375026, -0.98622537936444421, 
-1.0069975577304060, -1.0275506193625401, -1.0478789459331574, 
-1.0679786466439030, -1.0878469479817168, -1.1074817290906143, 
-1.1268811550235105, -1.1460433751521579, -1.1649662636391140, 
-1.1836471851295314, -1.2020827729221286, -1.2202687095846845, 
-1.2381995017727698, -1.2558682422115501, -1.2732663526334385, 
-1.2903833020983650, -1.3072062956999502, -1.3237199293108652, 
-1.3399058068770597, -1.3557421179750990, -1.3712031750531115, 
-1.3862589121471720, -1.3994488279637678, -1.4130059843746081, 
-1.4250904118624792, -1.4353117166179941, -1.4434378765890616, 
-1.4493078695859176, -1.4528472814487694, -1.4540301612818340, 
-0.87392068161906622, -0.87621159495645007, -0.88299135897844661, 
-0.89400145360681949, -0.90880478731961767, -0.92678429492239556, 
-0.94724095424039056, -0.96930273252447929, -0.98998896848246154, 
-1.0127879766702699, -1.0354106678967838, -1.0578512861628950, 
-1.0801066553585241, -1.1021754005361837, -1.1240573770323732, 
-1.1457532390051741, -1.1672641017039442, -1.1885912660710700, 
-1.2097359833747012, -1.2306992434137596, -1.2514815735827842, 
-1.2720828384483243, -1.2925020309200284, -1.3127370468901420, 
-1.3327844355631351, -1.3526391177481147, -1.3722940642610701, 
-1.3917399263967192, -1.4109646103067557, -1.4299527872126494, 
-1.4486853318726316, -1.4671386828449897, -1.4835032223275102, 
-1.5005472464563550, -1.5159627663168762, -1.5291935028759580, 
-1.5398576464508540, -1.5476521325936752, -1.5523928390075370, 
-1.5539845115956974, -0.88992507485452288, -0.89237935200021978, 
-0.89964355657918915, -0.91144403153700249, -0.92731875331994706, 
-0.94661572813490380, -0.96859684289831705, -0.99233759935707833, 
-1.0146350025267983, -1.0392553320444071, -1.0637367934838604, 
-1.0880762240309094, -1.1122731045076084, -1.1363288333972499, 
-1.1602462006048080, -1.1840289957802241, -1.2076817077555972, 
-1.2312092852531080, -1.2546169376497469, -1.2779099600784631, 
-1.3010935706041993, -1.3241727493072861, -1.3471520702486850, 
-1.3700355177393060, -1.3928262782522460, -1.4155264988093030, 
-1.4381370018189787, -1.4606569452095366, -1.4830834153626047, 
-1.5054109389104779, -1.5276308980578719, -1.5497308329363693, 
-1.5695270289913983, -1.5903693301731341, -1.6094438141642013, 
-1.6260073447174723, -1.6395033216842720, -1.6494585992732820, 
-1.6555541899361299, -1.6576080098243680, -0.90584054876501485, 
-0.90845733104325554, -0.91620355819933097, -0.92879087549660799, 
-0.94573297918407850, -0.96634370849854712, -0.98984660431999882, 
-1.0152653481278275, -1.0391748463995865, -1.0656197195510519, 
-1.0919655890516595, -1.1182120293772282, -1.1443613572700475, 
-1.1704179622651081, -1.1963878300603326, -1.2222781969718837, 
-1.2480972944737383, -1.2738541558159751, -1.2995584649546195, 
-1.3252204332630553, -1.3508506927751609, -1.3764601966619294, 
-1.4020601186447126, -1.4276617433237884, -1.4532763390770671, 
-1.4789150043231549, -1.5045884765677582, -1.5303068917592877, 
-1.5560794790551871, -1.5819141731363686, -1.6078171227181919, 
-1.6337920699568680, -1.6572589097007473, -1.6821931710420310, 
-1.7052405398891168, -1.7254510252605835, -1.7420681171359971, 
-1.7544193569256801, -1.7620238796590764, -1.7645935772105428, 
-0.92166747828714846, -0.92444587092429187, -0.93267158147710483, 
-0.94604195958792003, -0.96404701732191367, -0.98596712383368379, 
-1.0109881540822305, -1.0380825683358046, -1.0636035621758573, 
-1.0918741693327483, -1.1200876709748806, -1.1482464906380541, 
-1.1763559232490988, -1.2044235254230147, -1.2324586928939547, 
-1.2604723668929947, -1.2884768311452051, -1.3164855736226191, 
-1.3445131951323621, -1.3725753519159629, -1.4006887226803240, 
-1.4288709924673886, -1.4571408468505700, -1.4855179703221886, 
-1.5140230425073737, -1.5426777250347765, -1.5715046304878724, 
-1.6005272627824569, -1.6297699154581919, -1.6592575105939669, 
-1.6890153561924737, -1.7190687937511751, -1.7464221261323736, 
-1.7757175554179647, -1.8030301696887834, -1.8271847372873071, 
-1.8472003945839859, -1.8621757773716514, -1.8714399451779680, 
-1.8745783154053668, -0.93740627703242352, -0.94034535341926651, 
-0.94904790077280743, -0.96319734082752628, -0.98226054593228895, 
-1.0054850521011578, -1.0320196882347379, -1.0607862487827175, 
-1.0879167438111437, -1.1180124143666663, -1.1480945549425576, 
-1.1781685178722168, -1.2082426818371887, -1.2383279046017255, 
-1.2684371582687670, -1.2985852925909966, -1.3287888909031897, 
-1.3590661952601704, -1.3894370851160796, -1.4199230989678115, 
-1.4505474917608410, -1.4813353230914597, -1.5123135726773540, 
-1.5435112804048561, -1.5749597085941633, -1.6066925239786247, 
-1.6387459962363558, -1.6711592086499305, -1.7039742744516717, 
-1.7372365494203268, -1.7709948270352105, -1.8053014965843575, 
-1.8367294492189834, -1.8706244288651710, -1.9024656102250816, 
-1.9308363386082694, -1.9545084748519805, -1.9723227814734787, 
-1.9833897336276669, -1.9871471537700760, -0.95305739241193221, 
-0.95615619799103857, -0.96533284072161296, -0.98025715047859940, 
-1.0003733568176816, -1.0248967441555343, -1.0529396593234532, 
-1.0833737453419883, -1.1121104764547940, -1.1440288400084764, 
-1.1759785958364721, -1.2079680737426788, -1.2400088059074470, 
-1.2721150440413145, -1.3043034554273463, -1.3365929472350633, 
-1.3690045867658820, -1.4015615969476130, -1.4342894140617928, 
-1.4672157999009474, -1.5003710042362395, -1.5337879761951281, 
-1.5675026252557431, -1.6015541342803781, -1.6359853284698731, 
-1.6708431053985069, -1.7061789324129508, -1.7420494186316098, 
-1.7785169694976537, -1.8156505321977940, -1.8535264400630098, 
-1.8922293630249452, -1.9278885037568176, -1.9665849816544987, 
-2.0031820713922706, -2.0360082163880207, -2.0635674448044763, 
-2.0844153959084220, -2.0974161954017401, -2.1018390320058877, 
-0.96862130128707635, -0.97187885709829391, -0.98152677044978520, 
-0.99722158623373192, -1.0183853443305610, -1.0442016080022445, 
-1.0737467543519128, -1.1058427510940798, -1.1361812984273567, 
-1.1699184363165722, -1.2037329301610451, -1.2376361063285848, 
-1.2716426859494070, -1.3057703677426427, -1.3400395869359172, 
-1.3744734036098578, -1.4090974913881338, -1.4439402087712034, 
-1.4790327430937065, -1.5144093225556563, -1.5501074959428029, 
-1.5861684830891474, -1.6226376022322724, -1.6595647834504719, 
-1.6970051805721400, -1.7350198975088864, -1.7736768490775745, 
-1.8130517812492588, -1.8532294816315840, -1.8943052181395199, 
-1.9363864525904837, -1.9795948868284412, -2.0196073339759009, 
-2.0632662793056844, -2.1048046540567524, -2.1422855916066830, 
-2.1739278704539089, -2.1979776015838985, -2.2130267194686297, 
-2.2181557113623707, -0.98409850608933003, -0.98751381200359190, 
-0.99763009838481764, -1.0140909051627547, -1.0362964953438532, 
-1.0633991944268746, -1.0944398745305666, -1.1281912686783477, 
-1.1601261657430209, -1.1956767531201926, -1.2313514211380967, 
-1.2671644841158656, -1.3031338553335094, -1.3392806960472847, 
-1.3756292384832638, -1.4122067401542069, -1.4490435438056268, 
-1.4861732284654126, -1.5236328448577507, -1.5614632342860069, 
-1.5997094348941907, -1.6384211835903415, -1.6776535263076664, 
-1.7174675540700006, -1.7579312878844968, -1.7991207422093776, 
-1.8411212051241677, -1.8840287839855747, -1.9279522791132542, 
-1.9730154660515220, -2.0193598907852559, -2.0671483142215634, 
-2.1115999482348866, -2.1603381467769300, -2.2069565843585766, 
-2.2492459536534302, -2.2851261505677671, -2.3125132720713153, 
-2.3297043650494307, -2.3355730906363581, -0.99948953135843943, 
-1.0030615690271729, -1.0136432675975642, -1.0308654173488905, 
-1.0541068801484426, -1.0824891838762196, -1.1150181166740281, 
-1.1504175847167588, -1.1839424190532359, -1.2212998577720189, 
-1.2588286065302781, -1.2965459334293825, -1.3344729172606955, 
-1.3726341625078256, -1.4110576867311360, -1.4497749417186092, 
-1.4888209461707413, -1.5282345187048716, -1.5680586079279883, 
-1.6083407226405897, -1.6491334708103291, -1.6904952214484881, 
-1.7324909094730576, -1.7751930105662248, -1.8186827215019044, 
-1.8630513921296215, -1.9084022690715718, -1.9548526294729422, 
-2.0025364076044352, -2.0516074502832642, -2.1022435826638923, 
-2.1546517294882652, -2.2035915916341362, -2.2574799755421373, 
-2.3092676933422251, -2.3564691800864539, -2.3966960584252939, 
-2.4275187719227316, -2.4469210893750417, -2.4535546400966455, 
-1.0147949206540587, -1.0185226561990166, -1.0295667516210978, 
-1.0475454801445294, -1.0718166441900578, -1.1014713744811686, 
-1.1354807561161218, -1.1725202461706319, -1.2076277528852795, 
-1.2467842955080497, -1.2861596492158915, -1.3257739784965099, 
-1.3656514738396304, -1.4058201318488370, -1.4463117064996629, 
-1.4871617969532815, -1.5284100531814766, -1.5701004915692729, 
-1.6122819208558057, -1.6550084856274232, -1.6983403410219196, 
-1.7423444790574274, -1.7870957347127794, -1.8326780092464812, 
-1.8791857600759279, -1.9267258219259980, -1.9754196443787142, 
-2.0254060585140232, -2.0768447230770963, -2.1299204530589995, 
-2.1848487075568475, -2.2418826197893096, -2.2953235068755724, 
-2.3543871109373486, -2.4113826885607699, -2.4635477890589259, 
-2.5081808495668807, -2.5424965585881241, -2.5641523065427911, 
-2.5715662881769510, -1.0300152338007940, -1.0338976202668564, 
-1.0454010506969666, -1.0641314929850121, -1.0894260005663901, 
-1.1203456711211091, -1.1558272310219837, -1.1944980384979225, 
-1.2311801870492372, -1.2721270523242394, -1.3133402905042151, 
-1.3548428842698061, -1.3966620576343158, -1.4388291196729812, 
-1.4813794783726462, -1.5243527939845309, -1.5677932566006321, 
-1.6117499835063993, -1.6562775403196965, -1.7014365973617358, 
-1.7472947400789822, -1.7939274604381086, -1.8414193658253502, 
-1.8898656539840120, -1.9393739180457377, -1.9900663662810991, 
-2.0420825689750517, -2.0955828829635879, -2.1507527575149510, 
-2.2078082004688171, -2.2670027917782383, -2.3286367900591949, 
-2.3865569762190519, -2.4507764971483987, -2.5129687544705281, 
-2.5700967129570791, -2.6191451915404080, -2.6569689468011464, 
-2.6808918811552234, -2.6890918473568521, -1.0451510444310219, 
-1.0491870240224674, -1.0611466884047873, -1.0806238927051470, 
-1.1069352232129690, -1.1391120754383535, -1.1760571279853971, 
-1.2163499654852299, -1.2545980400881604, -1.2973255202697491, 
-1.3403668061533136, -1.3837476020839401, -1.4274980659381247, 
-1.4716527144364278, -1.5162504976251445, -1.5613350157961110, 
-1.6069548669610656, -1.6531641237640091, -1.7000229474478104, 
-1.7475983545237090, -1.7959651601152424, -1.8452071314340766, 
-1.8954183964087381, -1.9467051672308380, -1.9991878579615268, 
-2.0530037013807712, -2.1083100058895754, -2.1652882428177187, 
-2.2241492244855534, -2.2851397338972963, -2.3485511138697857, 
-2.4147305422615979, -2.4770764880145135, -2.5463913088699051, 
-2.6137220569443063, -2.6757619875585057, -2.7291861264382202, 
-2.7704910913804386, -2.7966665114629676, -2.8056478990876044, 
-1.0602029377937494, -1.0643914439126370, -1.0768042086357987, 
-1.0970231493088527, -1.1243446407132272, -1.1577706767191660, 
-1.1961701688079545, -1.2380752306355389, -1.2778799046486988, 
-1.3223774650501172, -1.3672359650308001, -1.4124837181782328, 
-1.4581536979556247, -1.5042835021026868, -1.5509154852102898, 
-1.5980970365071410, -1.6458809942614321, -1.6943261989022769, 
-1.7434981959675362, -1.7934701086022800, -1.8443237085894284, 
-1.8961507257461292, -1.9490544490222232, -2.0031516901155411, 
-2.0585752036681466, -2.1154766896779642, -2.1740305473872392, 
-2.2344386111932084, -2.2969361866428994, -2.3617998317291180, 
-2.4293575169183064, -2.5000020816601358, -2.5666919344921180, 
-2.6410043768190472, -2.7133728123976977, -2.7802278277896075, 
-2.8379423268544040, -2.8826622552792567, -2.9110484325509138, 
-2.9207970192878778, -1.0751715088012399, -1.0795114679048692, 
-1.0923741728751746, -1.1133297621473865, -1.1416546306744344, 
-1.1763216435653532, -1.2161661983645637, -1.2596732199995555, 
-1.3010246246530002, -1.3472809958296517, -1.3939449903412702, 
-1.4410474050800473, -1.4886238950099338, -1.5367149937833209, 
-1.5853663013940362, -1.6346288195312362, -1.6845594291807642, 
-1.7352215156455235, -1.7866857554045745, -1.8390310883995653, 
-1.8923459094819033, -1.9467295249099561, -2.0022939351470797, 
-2.0591660253220527, -2.1174902717067101, -2.1774321095130511, 
-2.2391821587648177, -2.3029615778503216, -2.3690289192825889, 
-2.4376890160442293, -2.5093046569529980, -2.5843121609751454, 
-2.6552398159522759, -2.7344203094196478, -2.8116887039270715, 
-2.8832217023832656, -2.9451010522626153, -2.9931345669233189, 
-3.0236654929489855, -3.0341583419729830, -1.0900573602881840, 
-1.0945476935810785, -1.1078571577615917, -1.1295442564661358, 
-1.1588656146167746, -1.1947652162871232, -1.2360451734672013, 
-1.2811434863452491, -1.3240312741568314, -1.3720345371203924, 
-1.4204915233314501, -1.4694353758148431, -1.5189042838396272, 
-1.5689415565868345, -1.6195958624951858, -1.6709216184138380, 
-1.7229795260833742, -1.7758372639932862, -1.8295703521232405, 
-1.8842632167434632, -1.9400104934092359, -1.9969186196449735, 
-2.0551077858989557, -2.1147143359104619, -2.1758937381204806, 
-2.2388242917451375, -2.3037117899514312, -2.3707954463731507, 
-2.4403555126905303, -2.5127231944404618, -2.5882937427267647, 
-2.6675440165412314, -2.7425834923754353, -2.8264763129089081, 
-2.9084765688872500, -2.9845172061515699, -3.0504024374641019, 
-3.1016187208205572, -3.1342079198769257, -3.1454147234255796, 
-1.1048611014609324, -1.1095007264354149, -1.1232537528958666, 
-1.1456671802840210, -1.1759780533279367, -1.2131016999540845, 
-1.2558071526457453, -1.3024857345668632, -1.3468991377835700, 
-1.3966368026447533, -1.4468735893749054, -1.4976448408851886, 
-1.5489911230044884, -1.6009583478224469, -1.6535980610730381, 
-1.7069678809768349, -1.7611320888504109, -1.8161623821925958, 
-1.8721388105823362, -1.9291509248247403, -1.9872991814648402, 
-2.0466966592270599, -2.1074711625730571, -2.1697678123384634, 
-2.2337522570554555, -2.2996146851184385, -2.3675748834579053, 
-2.4378886821543997, -2.5108562610574574, -2.5868329974311801, 
-2.6662438415292136, -2.7496026850528494, -2.8286125797527735, 
-2.9170418043516424, -3.0035824245049190, -3.0839347319279184, 
-3.1536410124456791, -3.2078863989438644, -3.2424314489088046, 
-3.2543161427237752, -1.1195833465167147, -1.1243711783549002, 
-1.1385645588734190, -1.1616991015731730, -1.1929924426401348, 
-1.2313314580468635, -1.2754522867711411, -1.3236998082410478, 
-1.3696276926287101, -1.4210867710600203, -1.4730895663318009, 
-1.5256734679425197, -1.5788812523825024, -1.6327612526421844, 
-1.6873676898053285, -1.7427611572510899, -1.7990092603621655, 
-1.8561874248796959, -1.9143798967857575, -1.9736809670562174, 
-2.0341964669294947, -2.0960455947090928, -2.1593631551006998, 
-2.2243023187778208, -2.2910380462739437, -2.3597713708396282, 
-2.4307348062670080, -2.5041992481960893, -2.5804828873684120, 
-2.6599628769857802, -2.7430908372992220, -2.8304138128653848, 
-2.9132416300377710, -3.0060169860733703, -3.0968900212657324, 
-3.1813401335569527, -3.2546646195241191, -3.3117695314073186, 
-3.3481568891759945, -3.3606793896992468, -1.1342247134149179, 
-1.1391596662638150, -1.1537901855179293, -1.1776406057097926, 
-1.2099093095905902, -1.2494549066569369, -1.2949808104521172, 
-1.3447856772439366, -1.3922165915347140, -1.4453836634350008, 
-1.4991381550752609, -1.5535193440612822, -1.6085720457105703, 
-1.6643468251517763, -1.7209003692137661, -1.7782960115478270, 
-1.8366044162663731, -1.8959044354330050, -1.9562841655405874, 
-2.0178422388586346, -2.0806893983449712, -2.1449504209707708, 
-2.2107664754263374, -2.2782980284929448, -2.3477284530989881, 
-2.4192685449893530, -2.4931622312483808, -2.5696938638189413, 
-2.6491976523744145, -2.7320700321500957, -2.8187861313768772, 
-2.9099220814287561, -2.9964082580086804, -3.0933305991065487, 
-3.1883182034410189, -3.2766417200307711, -3.3533711167228635, 
-3.4131568148382190, -3.4512665542353926, -3.4643844727648663, 
-1.1487858227844259, -1.1538668109148145, -1.1689312502959339, 
-1.1934922931700302, -1.2267292089302400, -1.2674725091869519, 
-1.3143930341422239, -1.3657434263489083, -1.4146656476413628, 
-1.4695269223720955, -1.5250183520739449, -1.5811809405146608, 
-1.6380613660976158, -1.6957122329771219, -1.7541924793222698, 
-1.8135679389080146, -1.8739120635019928, -1.9353068233411035, 
-1.9978438128019977, -2.0616255993197745, -2.1267673668252356, 
-2.1933989217552594, -2.2616671517316469, -2.3317390566455027, 
-2.4038055125153059, -2.4780859851427506, -2.5548344909282763, 
-2.6343472181196930, -2.7169723921346569, -2.8031232241125310, 
-2.8932951736994483, -2.9880893730855922, -3.0780708856046264, 
-3.1789370951778619, -3.2778174048617430, -3.3697860122637890, 
-3.4497034054867695, -3.5119881193801636, -3.5516982522207194, 
-3.5653684622662101, -1.1632672969526958, -1.1684932358115205, 
-1.1839883768941533, -1.2092547774473059, -1.2434527199487579, 
-1.2853847715080275, -1.3336893368990415, -1.3865732447303403, 
-1.4369748201220038, -1.4935161926715654, -1.5507294239200526, 
-1.6086570799437137, -1.6673475244201057, -1.7268552052374035, 
-1.7872410952725768, -1.8485732860734891, -1.9109277439041719, 
-1.9743892471798368, -2.0390525340857701, -2.1050237002563641, 
-2.1724218999054568, -2.2413814210688545, -2.3120542283804593, 
-2.3846130974822342, -2.4592555072871001, -2.5362085151193483, 
-2.6157349232424787, -2.6981411658850747, -2.7837875233730052, 
-2.8731015365882158, -2.9665959045507408, -3.0648927931002459, 
-3.1582062667201063, -3.2628134635850530, -3.3653656161360432, 
-3.4607527236531932, -3.5436433834204326, -3.6082475170428974, 
-3.6494376639071531, -3.6636176332294794, -1.1776697590837970, 
-1.1830395662489668, -1.1989621939432571, -1.2249286831698474, 
-1.2600804435870143, -1.3031922375344249, -1.3528701597421777, 
-1.4072754163040020, -1.4591442010213691, -1.5173513034388131, 
-1.5762708836929704, -1.6359469058070450, -1.6964292404959693, 
-1.7577739838474971, -1.8200439268722193, -1.8833091770434658, 
-1.9476479430933225, -2.0131475036107900, -2.0799053896666053, 
-2.1480308228358265, -2.2176464637081672, -2.2888905435988498, 
-2.3619194754678889, -2.4369110715101683, -2.5140685381119856, 
-2.5936254792837783, -2.6758522255146526, -2.7610639290886647, 
-2.8496310497002364, -2.9419931290482655, -3.0386771757885001, 
-3.1403226489622313, -3.2368069366964969, -3.3449559261890616, 
-3.4509641334640153, -3.5495494004026718, -3.6352054041821007, 
-3.7019556571852856, -3.7445099421135808, -3.7591587811811995, 
-1.1919938324139643, -1.1975064284596870, -1.2138533338734114, 
-1.2405146443993198, -1.2766129998108195, -1.3208954851794710, 
-1.3719359995609974, -1.4278503108397198, -1.4811740031157195, 
-1.5410322515397576, -1.6016424690507047, -1.6630498539955509, 
-1.7253056069226222, -1.7884672780499973, -1.8525992620095206, 
-1.9177734432135880, -1.9840700047475806, -2.0515784226577343, 
-2.1203986770526453, -2.1906427225879450, -2.2624362747426470, 
-2.3359209861548051, -2.4112571109510195, -2.4886267870055701, 
-2.5682381100869995, -2.6503302354025324, -2.7351798295474210, 
-2.8231093224232620, -2.9144975952131928, -3.0097940211176000, 
-3.1095372074420022, -3.2143804694367741, -3.3138787050784302, 
-3.4253766774704841, -3.5346333493775344, -3.6362060922733810, 
-3.7244297460515097, -3.7931621218602816, -3.8369712650742462, 
-3.8520504813678116, -1.2062401395744355, -1.2118944488545147, 
-1.2286624318884640, -1.2560133030932947, -1.2930510252222542, 
-1.3384951226598316, -1.3908874035270555, -1.4482983757866528, 
-1.5030645487210506, -1.5645591863138970, -1.6268441219440393, 
-1.6899656264959895, -1.7539760554578001, -1.8189342220602400, 
-1.8849059138382274, -1.9519645580404692, -2.0201920502718913, 
-2.0896797693865694, -2.1605298110324673, -2.2328564833544697, 
-2.3067881222497171, -2.3824693015654912, -2.4600635375307913, 
-2.5397566190680205, -2.6217607401693379, -2.7063196728219818, 
-2.7937153075535868, -2.8842760168397734, -2.9783874855795407, 
-3.0765069372346416, -3.1791821221198764, -3.2870771259815781, 
-3.3894382816655182, -3.5041008041026296, -3.6164087835665697, 
-3.7207703367853804, -3.8113764736829721, -3.8819382482037694, 
-3.9269009097243961, -3.9423748871684721, -1.2204093019924085, 
-1.2262042533482969, -1.2433901250469457, -1.2714253077160023, 
-1.3093951708871039, -1.3559917851181291, -1.4097249639698934, 
-1.4686201287559189, -1.5248162593798564, -1.5879323954592108, 
-1.6518759698517218, -1.7166941669877691, -1.7824403258175971, 
-1.8491743356953023, -1.9169631716095259, -1.9858815761312703, 
-2.0560129038066552, -2.1274501520055762, -2.2002972114278392, 
-2.2746703804856305, -2.3507002016585021, -2.4285336959692176, 
-2.5083370957515210, -2.5902992084344185, -2.6746355889028695, 
-2.7615937607300420, -2.8514598158017632, -2.9445668509741489, 
-3.0413058908066009, -3.1421402320349778, -3.2476245869026039, 
-3.3584310999831453, -3.4635110978343437, -3.5811634757284416, 
-3.6963374877115958, -3.8033026466825519, -3.8961199494642544, 
-3.9683707415988474, -4.0143942202866993, -4.0302304626748651, 
-1.2345019393619301, -1.2404364667617593, -1.2580370514392860, 
-1.2867513119833436, -1.3256461003589448, -1.3733861315365785, 
-1.4284493136782823, -1.4888161506094635, -1.5464296463615705, 
-1.6111522920077241, -1.6767383084388527, -1.7432356382591829, 
-1.8106984367640866, -1.8791874878494788, -1.9487707550105646, 
-2.0195240766239690, -2.0915320224626162, -2.1648889363143544, 
-2.2397001985513154, -2.3160837533935998, -2.3941719594318180, 
-2.4741138400173552, -2.5560778341863943, -2.6402551814149473, 
-2.7268641184669353, -2.8161551295313858, -2.9084175803699490, 
-3.0039881967702589, -3.1032620387020455, -3.2067069093019827, 
-3.3148825810651514, -3.4284669221576878, -3.5361293599120049, 
-3.6566074599362741, -3.7744748996149089, -3.8838726045980381, 
-3.9787441317351169, -4.0525562499533168, -4.0995566676107984, 
-4.1157258543654764, -1.2471350129911296, -1.2531943964079966, 
-1.2711659489036178, -1.3004876229544522, -1.3402096440979243, 
-1.3889721564960056, -1.4452243913419491, -1.5069065219075193, 
-1.5657863044655314, -1.6319435951715493, -1.6989955884269710, 
-1.7669907804804248, -1.8359839233163624, -1.9060364564582872, 
-1.9772170699038552, -2.0496024089365790, -2.1232779387736262, 
-2.1983389945989100, -2.2748920512998465, -2.3530562579929670, 
-2.4329652961638120, -2.5147696382426932, -2.5986393074700396, 
-2.6847672725297822, -2.7733736554051900, -2.8647109938777806, 
-2.9590708896767977, -3.0567925029699254, -3.1582735451761677, 
-3.2639847101414956, -3.3744889275589824, -3.4904675241257221, 
-3.6003467204074440, -3.7232394107296733, -3.8433966298613473, 
-3.9548514562077854, -4.0514513628943085, -4.1265712995871082, 
-4.1743872968065592, -4.1908339346598007, -1.2604581444418093, 
-1.2666489291403595, -1.2850108862995209, -1.3149717605685030, 
-1.3555642102311949, -1.4054024114586927, -1.4629052697425555, 
-1.5259703565211378, -1.5861810665002698, -1.6538455762561393, 
-1.7224369099461851, -1.7920040207303147, -1.8626021541896975, 
-1.9342932949374310, -2.0071467400756129, -2.0812398118023143, 
-2.1566587281181584, -2.2334996578172430, -2.3118699944878909, 
-2.3918898948627367, -2.4736941404902439, -2.5574343996144999, 
-2.6432819901146849, -2.7314312769179998, -2.8221038822123106, 
-2.9155539496814402, -3.0120747934938401, -3.1120073923685907, 
-3.2157513802353588, -3.3237794729724865, -3.4366567144968956, 
-3.5550666271618239, -3.6671883503287472, -3.7925134632720510, 
-3.9149665471044415, -4.0284740723810062, -4.1267907208490255, 
-4.2032044624496150, -4.2518248809879573, -4.2685447328378867, 
-1.2726857899681516, -1.2789969200861881, -1.2977164235570857, 
-1.3282626611296320, -1.3696521346159369, -1.4204751334121990, 
-1.4791226530965698, -1.5434531035310484, -1.6048811052167682, 
-1.6739235338312766, -1.7439214187147916, -1.8149240198728969, 
-1.8869869161111994, -1.9601724636064000, -2.0345503773177751, 
-2.1101984489117322, -2.1872034209999738, -2.2656620443997109, 
-2.3456823534497726, -2.4273852048806304, -2.5109061392562322, 
-2.5963976418267931, -2.6840319035090521, -2.7740042151823761, 
-2.8665371732988310, -2.9618859375783329, -3.0603448709133581, 
-3.1622560210044837, -3.2680200941891990, -3.3781108595732663, 
-3.4930943649611064, -3.6136550472777196, -3.7277534754668942, 
-3.8552143123887226, -3.9796744706918532, -4.0949685955154296, 
-4.1947731859641131, -4.2723042327879570, -4.3216174602243953, 
-4.3385722778921600, -1.2833441579108145, -1.2897599899459933, 
-1.3087905539087887, -1.3398460168601145, -1.3819287684830932, 
-1.4336082319869370, -1.4932510121021887, -1.5586813337060368, 
-1.6211670473725643, -1.6914062695730203, -1.7626252346945739, 
-1.8348733842791034, -1.9082064964275540, -1.9826871536266715, 
-2.0583853298877264, -2.1353791120209293, -2.2137555755492637, 
-2.2936118423887475, -2.3750563555577662, -2.4582104165071970, 
-2.5432100440781324, -2.6302082318245703, -2.7193777042279512, 
-2.8109143047077647, -2.9050411930325200, -3.0020140923741057, 
-3.1021279154323955, -3.2057252282353756, -3.3132072008769593, 
-3.4250479817295849, -3.5418138745692476, -3.6641893986140897, 
-3.7799507558510714, -3.9092023939873677, -4.0353393780441182, 
-4.1521206894877007, -4.2531596032277408, -4.3316147714887911, 
-4.3814997682999968, -4.3986482752998679, -1.2920665550706416, 
-1.2985679322792334, -1.3178526369918111, -1.3493241199228450, 
-1.3919732001228629, -1.4443521919572349, -1.5048076949296658, 
-1.5711359239273301, -1.6344848625741111, -1.7057004740058259, 
-1.7779152341877364, -1.8511786749547310, -1.9255466758960293, 
-2.0010819390077881, -2.0778545797567682, -2.1559428502944487, 
-2.2354340158779733, -2.3164254119460073, -2.3990257172776404, 
-2.4833564888796489, -2.5695540175742737, -2.6577715809126681, 
-2.7481821937523194, -2.8409819891312602, -2.9363944066753107, 
-3.0346754282950714, -3.1361201899640112, -3.2410714273555992, 
-3.3499304035248181, -3.4631712537673742, -3.5813601252603191, 
-3.7051811890851116, -3.8222638760492802, -3.9529347206464549, 
-4.0803967732230939, -4.1983499345097695, -4.3003588503654369, 
-4.3795385895261809, -4.4298711395812465, -4.4471711222926968, 
-1.2985282777891731, -1.3050929474103978, -1.3245656794584195, 
-1.3563449421660188, -1.3994129694598840, -1.4523093814374914, 
-1.5133659309189991, -1.5803580625212936, -1.6443451040957959, 
-1.7162822686692880, -1.7892326840939037, -1.8632459159309949, 
-1.9383778839937493, -2.0146913416512700, -2.0922564693923005, 
-2.1711515990524806, -2.2514640901593830, -2.3332913860708642, 
-2.4167422854474743, -2.5019384747337861, -2.5890163805844644, 
-2.6781294187656322, -2.7694507397176342, -2.8631766031973069, 
-2.9595305589484848, -3.0587686727812047, -3.1611861263632570, 
-3.2671256478718109, -3.3769884208757430, -3.4912484055265613, 
-3.6104714482904940, -3.7353412560364783, -3.8533808364130873, 
-3.9850774566051648, -4.1134952173927006, -4.2322918084454422, 
-4.3349975200382884, -4.4146968945552354, -4.4653500278662674, 
-4.4827584440210124, -1.3024984746271806, -1.3091019978197784, 
-1.3286901531825981, -1.3606583445617715, -1.4039835251753430, 
-1.4571975070209426, -1.5186229095674570, -1.5860224074125207, 
-1.6504009182575188, -1.7227806621178989, -1.7961822052780396, 
-1.8706551202376887, -1.9462553370273408, -2.0230456252082401, 
-2.1010961888108022, -2.1804853910181761, -2.2613006302892780, 
-2.3436393957327906, -2.4276105373385857, -2.5133357967562375, 
-2.6009516575308020, -2.6906115912649287, -2.7824887997965391, 
-2.8767795856749299, -2.9737075277042853, -3.0735287007012522, 
-3.1765382674688833, -3.2830788997473430, -3.3935516750139438, 
-3.5084303825685499, -3.6282806143073278, -3.7537857149240881, 
-3.8724042537333654, -4.0047207137891343, -4.1337151748779783, 
-4.2530200207117108, -4.3561450970708044, -4.4361568818219403, 
-4.4870026050043821, -4.5044761359748140, -1.3038387641689417, 
-1.3104553979191598, -1.3300825008418120, -1.3621144417416098, 
-1.4055263901063988, -1.4588475205316456, -1.5203973672267761, 
-1.5879342923472155, -1.6524448560467535, -1.7249738787025923, 
-1.7985275665139844, -1.8731554919777191, -1.9489135861104634, 
-2.0258646213319755, -2.1040788068409864, -2.1836345134255706, 
-2.2646191494826753, -2.3471302161036185, -2.4312765768536715, 
-2.5171799879399046, -2.6049769476683604, -2.6948209416384370, 
-2.7868851837306035, -2.8813659851243294, -2.9784869280542865, 
-3.0785040833724873, -3.1817125998171463, -3.2884551216185391, 
-3.3991326811475995, -3.5142189998289051, -3.6342795724447803, 
-3.7599976092126668, -3.8788100865302559, -4.0113340476906760, 
-4.1405214361176341, -4.2599961977641430, -4.3632613865898646, 
-4.4433774956851693, -4.4942875070860167, -4.5117827696862713
};
static double tabled7 [ 2400 ]= {
-1.6536538601759874, -1.6448362995260364, -1.6197653047527809, 
-1.5820975288572190, -1.5367977900486816, -1.4889813049606280, 
-1.4428097067765013, -1.4013488139548225, -1.3691603044100960, 
-1.3400946891563090, -1.3169805749749952, -1.2989308495181207, 
-1.2852202617288417, -1.2752509225024931, -1.2685262032967415, 
-1.2646307527414911, -1.2632150305831100, -1.2639832183946309, 
-1.2666836832185049, -1.2711013916167579, -1.2770518283642487, 
-1.2843760864718732, -1.2929368768432747, -1.3026152657559242, 
-1.3133079927514799, -1.3249252547466701, -1.3373888672706375, 
-1.3506307328523484, -1.3645915612675741, -1.3792197977284761, 
-1.3944707239784355, -1.4103057042449481, -1.4250463927027746, 
-1.4411351721370038, -1.4563803638830666, -1.4700366548994102, 
-1.4814641801931943, -1.4900749552192853, -1.4954265471529329, 
-1.4972436387420147, -1.6587489931738117, -1.6499232141393281, 
-1.6248224830506866, -1.5870902704896117, -1.5416756497312731, 
-1.4936821817216908, -1.4472680148664179, -1.4055066222595551, 
-1.3730082871857276, -1.3435781690152819, -1.3200858610248236, 
-1.3016509292407310, -1.2875534126464439, -1.2771995736770809, 
-1.2700960072352809, -1.2658298322580610, -1.2640533699139331, 
-1.2644721725735728, -1.2668355840299506, -1.2709292322110024, 
-1.2765690118277431, -1.2835962263279121, -1.2918736396785968, 
-1.3012822479917028, -1.3117186250670381, -1.3230927288780787, 
-1.3353260808934637, -1.3483502490597581, -1.3621055798062673, 
-1.3765401356838716, -1.3916088040302754, -1.4072725489638378, 
-1.4218680547044138, -1.4378120432514270, -1.4529316941964407, 
-1.4664841609084687, -1.4778305260188116, -1.4863833475689499, 
-1.4917002369299408, -1.4935057676756069, -1.6738297010675449, 
-1.6649811205175598, -1.6397971224646541, -1.6018825882230485, 
-1.5561403556811015, -1.5076394250385463, -1.4605269855893583, 
-1.4178978559904087, -1.3845025893515135, -1.3540148593188748, 
-1.3294228172361180, -1.3098654535355960, -1.2946382630213502, 
-1.2831597050689112, -1.2749459091096380, -1.2695913618116730, 
-1.2667539940410071, -1.2661435453317735, -1.2675123986059298, 
-1.2706482970233408, -1.2753685094128000, -1.2815151211661138, 
-1.2889512073223202, -1.2975577029467005, -1.3072308290402352, 
-1.3178799643972632, -1.3294258780671409, -1.3417992554941036, 
-1.3549394655264411, -1.3687935263974289, -1.3833152372849740, 
-1.3984644487426781, -1.4126231748804403, -1.4281310434003023, 
-1.4428721051551436, -1.4561111301308232, -1.4672122435151587, 
-1.4755898069772075, -1.4808016959417247, -1.4825722371907997, 
-1.6983336062237295, -1.6894526848236207, -1.6641480686178605, 
-1.6259632840554623, -1.5797272221734517, -1.5304526042341449, 
-1.4822669020749997, -1.4382956330131802, -1.4035060707178593, 
-1.3713669023251123, -1.3450502765298256, -1.3237256384601355, 
-1.3067129170769922, -1.2934500517352099, -1.2834685894614450, 
-1.2763751114007325, -1.2718369293238887, -1.2695709511047486, 
-1.2693349265318459, -1.2709205012150404, -1.2741476583276263, 
-1.2788602360621875, -1.2849222865616337, -1.2922150988216596, 
-1.3006347498340813, -1.3100900793094747, -1.3205010066475469, 
-1.3317971265033985, -1.3439165328146825, -1.3568048315786447, 
-1.3704143107765658, -1.3847032422068080, -1.3981299691008315, 
-1.4129052576537908, -1.4270089113970650, -1.4397191028500018, 
-1.4504056791814679, -1.4584865818252370, -1.4635205552532708, 
-1.4652317783540327, -1.7313234421408250, -1.7224077392331143, 
-1.6969680615586336, -1.6584679067096657, -1.6116389361314500, 
-1.5614180769369117, -1.5119029466800880, -1.4662531826439609, 
-1.4297067367963008, -1.3954729415091374, -1.3669552133730019, 
-1.3433618018045970, -1.3240441071703386, -1.3084654965842641, 
-1.2961779695716644, -1.2868044653051507, -1.2800252835002999, 
-1.2755675452667901, -1.2731969284870357, -1.2727111247554883, 
-1.2739346134381790, -1.2767144536300286, -1.2809168702548683, 
-1.2864244653144024, -1.2931339254650718, -1.3009541268779243, 
-1.3098045606239479, -1.3196140186663421, -1.3303194933794660, 
-1.3418652533895781, -1.3542020661965040, -1.3672865440379205, 
-1.3796849665465143, -1.3934277519010361, -1.4066300981594699, 
-1.4185902847166141, -1.4286874491279504, -1.4363456828504941, 
-1.4411258102336859, -1.4427523377231137, -1.7714870728398626, 
-1.7625417292086618, -1.7369772937418770, -1.6981636934845088, 
-1.6507178409844299, -1.5994847366155227, -1.5485213261652608, 
-1.5010187326568816, -1.4625133702278401, -1.4259239368021090, 
-1.3949108670954313, -1.3687262271014926, -1.3467573747878254, 
-1.3284971607732650, -1.3135217253169766, -1.3014737286949511, 
-1.2920495261389870, -1.2849892460819992, -1.2800690326757556, 
-1.2770949208472120, -1.2758979564391044, -1.2763302757723287, 
-1.2782619317018378, -1.2815783058314156, -1.2861779850107260, 
-1.2919710086582441, -1.2988774146675497, -1.3068260276416985, 
-1.3157534453608042, -1.3256031887219406, -1.3363249876204379, 
-1.3478741808946535, -1.3589563176826316, -1.3713720212428107, 
-1.3834112035408617, -1.3943999297995158, -1.4037312421913219, 
-1.4108389104953003, -1.4152878069330641, -1.4168037212117539, 
-1.8173442372088855, -1.8083808192409692, -1.7827232375598652, 
-1.7436395619653813, -1.6956205502758055, -1.6434072409105165, 
-1.5910054741459130, -1.5416304623950559, -1.5011201513712504, 
-1.4620941304477635, -1.4284748794298705, -1.3995599358473640, 
-1.3747743932104890, -1.3536424914488505, -1.3357665276094923, 
-1.3208109808195063, -1.3084904087639249, -1.2985601109677141, 
-1.2908088485254325, -1.2850531109336709, -1.2811325600224133, 
-1.2789063789319917, -1.2782503238427165, -1.2790543264596519, 
-1.2812205319365160, -1.2846616839675526, -1.2892997889228381, 
-1.2950650060590878, -1.3018947223498030, -1.3097327793041940, 
-1.3185288259742511, -1.3282377776862477, -1.3377344216072609, 
-1.3485440314820967, -1.3591689044972834, -1.3689714395427843, 
-1.3773642584593246, -1.3837953385828565, -1.3878363884278506, 
-1.3892159674915960, -1.8670195973128079, -1.8580539805507861, 
-1.8323499006805064, -1.7930707979340630, -1.7445765630130068, 
-1.6914973907730297, -1.6377794715348075, -1.5866518252833970, 
-1.5442350249747181, -1.5028626838504822, -1.4667054543958244, 
-1.4351048132077000, -1.4075226534779897, -1.3835142107985507, 
-1.3627080209737454, -1.3447908696206115, -1.3294963418444490, 
-1.3165960064801534, -1.3058925545186322, -1.2972144052680152, 
-1.2904114277463745, -1.2853515186565512, -1.2819178449434068, 
-1.2800066068636839, -1.2795252123768950, -1.2803907793318581, 
-1.2825289010128254, -1.2858726249525485, -1.2903616058042331, 
-1.2959414014051860, -1.3025628876195225, -1.3101817725931653, 
-1.3178678809552822, -1.3268338105849768, -1.3358268014212811, 
-1.3442543623098393, -1.3515550781479571, -1.3571964129354666, 
-1.3607603987268102, -1.3619803325128153, -1.9138230283668030, 
-1.9048704576036848, -1.8791699460847042, -1.8397915112529752, 
-1.7909771553615228, -1.7372547696032585, -1.6825113682962196, 
-1.6299790108230816, -1.5860056583003708, -1.5426903695664984, 
-1.5044057129939821, -1.4705297520705798, -1.4405552011769716, 
-1.4140634919808559, -1.3907056082966485, -1.3701877132107896, 
-1.3522602250857916, -1.3367094132848019, -1.3233508605850792, 
-1.3120243264524718, -1.3025896742756058, -1.2949236157584763, 
-1.2889170895023290, -1.2844731366125191, -1.2815051694326427, 
-1.2799355539453581, -1.2796944445261869, -1.2807188233538795, 
-1.2829517071039003, -1.2863414914568054, -1.2908414100695764, 
-1.2964090894412017, -1.3023084925449759, -1.3094453376061881, 
-1.3168109547826767, -1.3238609953865847, -1.3300633118896903, 
-1.3349079203806098, -1.3379896145693795, -1.3390479811954583, 
-1.9656707919578276, -1.9567483005516131, -1.9311007763411741, 
-1.8917002725535073, -1.8426641728888571, -1.7884098409337092, 
-1.7327564433610776, -1.6789298668082191, -1.6334891693190119, 
-1.5883115767900062, -1.5479595461635292, -1.5118465920934072, 
-1.4794953576107133, -1.4505128471457709, -1.4245721448924114, 
-1.4013987138253987, -1.3807599848633887, -1.3624573461816620, 
-1.3463199087689401, -1.3321996041863906, -1.3199672940296461, 
-1.3095096567242503, -1.3007266781684677, -1.2935296163341061, 
-1.2878393415338583, -1.2835849772273495, -1.2807027834034943, 
-1.2791352374304150, -1.2788302769877475, -1.2797406771274777, 
-1.2818235392490709, -1.2850398742576561, -1.2888793210767957, 
-1.2938956921425702, -1.2993645194549652, -1.3048013115950339, 
-1.3097120920071419, -1.3136169436949643, -1.3161285415477417, 
-1.3169957179519460, -2.0173964970609317, -2.0085189619844748, 
-1.9829712569031033, -1.9436331678215786, -1.8945038153270981, 
-1.8398935219692027, -1.7835522943725295, -1.7286914654517427, 
-1.6820411687839121, -1.6352955275891834, -1.5931734587505717, 
-1.5551189754310295, -1.5206804156079374, -1.4894867625405346, 
-1.4612301782550754, -1.4356529160498404, -1.4125373728509742, 
-1.3916984289059846, -1.3729774784812330, -1.3562377279858004, 
-1.3413604563629602, -1.3282420150005296, -1.3167914025715466, 
-1.3069282917989771, -1.2985814152280415, -1.2916872391089467, 
-1.2861888707760962, -1.2820351570721793, -1.2791799405379525, 
-1.2775814470732088, -1.2772017841468293, -1.2780065328115024, 
-1.2797200956889823, -1.2825350992483449, -1.2860233832815156, 
-1.2897670676709549, -1.2933165572665801, -1.2962275886056669, 
-1.2981349166156619, -1.2987992074576973, -2.0689808446365792, 
-2.0601617047912524, -2.0347561971929764, -1.9955572114037576, 
-1.9464517330428588, -1.8916467443426284, -1.8348225879639968, 
-1.7791691321112002, -1.7315508480460782, -1.6835156386657220, 
-1.6399080121273695, -1.6001980483467229, -1.5639558962419822, 
-1.5308291180858602, -1.5005259865952560, -1.4728029698283081, 
-1.4474552140775829, -1.4243092077854784, -1.4032170559789217, 
-1.3840519606264745, -1.3667046158800049, -1.3510803061032688, 
-1.3370965502543364, -1.3246811759605521, -1.3137707353867851, 
-1.3043091960293345, -1.2962468551030419, -1.2895394377715352, 
-1.2841473481807542, -1.2800350488608980, -1.2771705491128775, 
-1.2755249868935494, -1.2750645922320625, -1.2756145719537462, 
-1.2770528594424884, -1.2790348112013554, -1.2811615784385832, 
-1.2830303814215154, -1.2843025246592259, -1.2847532974536671, 
-2.1204070372279933, -2.1116584311223483, -2.0864334652791885, 
-2.0474431847886767, -1.9984683155625746, -1.9436163195734899, 
-1.8864980093775623, -1.8302761241557892, -1.7819157931407810, 
-1.7328537125958587, -1.6880315659875988, -1.6469416005267148, 
-1.6091722895435931, -1.5743866454359166, -1.5423062380686960, 
-1.5126992207582413, -1.4853712212649588, -1.4601583143729748, 
-1.4369215296177413, -1.4155425070151797, -1.3959200225705235, 
-1.3779671810225806, -1.3616091266381307, -1.3467811610083307, 
-1.3334271843980645, -1.3214983973986314, -1.3109522145675752, 
-1.3017513528780107, -1.2938630661718591, -1.2872585031461121, 
-1.2819121712231611, -1.2778014923483201, -1.2751392399043828, 
-1.2733803440381553, -1.2727159057497204, -1.2728808697360892, 
-1.2735335518842397, -1.2743186583171844, -1.2749287434139891, 
-1.2751566915712753, -2.1716604705581508, -2.1629933594481692, 
-2.1379836216209722, -2.0992652008272006, -2.0505181693341550, 
-1.9957543331500207, -1.9385156040751639, -1.8819329804424632, 
-1.8330414084955660, -1.7831995164379026, -1.7374200756702078, 
-1.6952141251787691, -1.6561853973564693, -1.6200095479209240, 
-1.5864188573959903, -1.5551907712330415, -1.5261391811790208, 
-1.4991076961668699, -1.4739643787215657, -1.4505975757562255, 
-1.4289125770952134, -1.4088289076216010, -1.3902781101264461, 
-1.3732019125566213, -1.3575506999080507, -1.3432822304952541, 
-1.3303605507818501, -1.3187550737794114, -1.3084397941880055, 
-1.2993926196489523, -1.2915948022020534, -1.2850304576454514, 
-1.2801609432663217, -1.2760717559265595, -1.2732711206052949, 
-1.2715794795724740, -1.2707186601471128, -1.2703869356618769, 
-1.2703129858787283, -1.2703104168336661, -2.2227284658645003, 
-2.2141527440613684, -2.1893896001940210, -2.1510003227080894, 
-2.1025696580486075, -2.0480176058997888, -1.9908181840763450, 
-1.9340669226394820, -1.8848403718858537, -1.8344503622494603, 
-1.7879568597692042, -1.7448868205870249, -1.7048565962938300, 
-1.6675520303266667, -1.6327138052545449, -1.6001264708864837, 
-1.5696100991129116, -1.5410138438349670, -1.5142109030200881, 
-1.4890945265733493, -1.4655748138448090, -1.4435761141259451, 
-1.4230348925507590, -1.4038979589644101, -1.3861209828656735, 
-1.3696672363465143, -1.3545065209880733, -1.3406142452593923, 
-1.3279706270251255, -1.3165600019471999, -1.3063702233212762, 
-1.2973921425524308, -1.2903347418511986, -1.2839189406319997, 
-1.2789705126509172, -1.2754006723972307, -1.2730008826902068, 
-1.2715290278878373, -1.2707548899821295, -1.2705160385376904, 
-2.2736000371180269, -2.2651246309491686, -2.2406364303094586, 
-2.2026282301740920, -2.1545944973578512, -2.1003672146082715, 
-2.0433537924428848, -1.9866113047617884, -1.9372321205343221, 
-1.8865106933235025, -1.8395323448080321, -1.7958375411637950, 
-1.7550530279938608, -1.7168727428015218, -1.6810437702208820, 
-1.6473558307822687, -1.6156332907491264, -1.5857290001025033, 
-1.5575194756568718, -1.5309010868955315, -1.5057869980283223, 
-1.4821046863384117, -1.4597939038458063, -1.4388049829776934, 
-1.4190974114446002, -1.4006386196414320, -1.3834029374907659, 
-1.3673706879971441, -1.3525273927628789, -1.3388630709364557, 
-1.3263716179488312, -1.3150502542398863, -1.3058513168036534, 
-1.2971403101454575, -1.2900570361945771, -1.2846079050411301, 
-1.2806597369773154, -1.2780358897661530, -1.2765522305256201, 
-1.2760735969548027, -2.3242656881897137, -2.3158986449095331, 
-2.2917109934680786, -2.2541309265874050, -2.2065673969807711, 
-2.1527680648712524, -2.0960752200020463, -2.0395051074176265, 
-1.9901423678782704, -1.9392916791893176, -1.8920437929133562, 
-1.8479507054632860, -1.8066477238443845, -1.7678351438272502, 
-1.7312647742859815, -1.6967298573701748, -1.6640574121590133, 
-1.6331023364631174, -1.6037428028676037, -1.5758766198769267, 
-1.5494183208504249, -1.5242968070746938, -1.5004534161833689, 
-1.4778403192726117, -1.4564191734621057, -1.4361599739817112, 
-1.4170400629254480, -1.3990432618387705, -1.3821591031385343, 
-1.3663821415905830, -1.3517113320881766, -1.3381494640588130, 
-1.3268843595219471, -1.3159398472782036, -1.3067618878189069, 
-1.2994554195648798, -1.2939677308600490, -1.2901931598115950, 
-1.2879985277263135, -1.2872792394048411, -2.3747172357881046, 
-2.3664658035717787, -2.3426018104490010, -2.3054924813951194, 
-2.2584657441688396, -2.2051885100545254, -2.1489395690068225, 
-2.0926924734256542, -2.0435026502445295, -1.9927108217666947, 
-1.9453950176425032, -1.9011171680261207, -1.8595196712108322, 
-1.8203077876743738, -1.7832366940636899, -1.7481018037489156, 
-1.7147314202932191, -1.6829810851138074, -1.6527291738952863, 
-1.6238734254943918, -1.5963281749847915, -1.5700221233368272, 
-1.5448965190266746, -1.5209036574565482, -1.4980056262856418, 
-1.4761732411944473, -1.4553851289814328, -1.4356269244227318, 
-1.4168905548436852, -1.3991735924307995, -1.3824786593478235, 
-1.3668128749735897, -1.3535878239335606, -1.3405042138192724, 
-1.3293015641346246, -1.3201853239704981, -1.3131875089273986, 
-1.3082783821574462, -1.3053803266333708, -1.3044225209799525, 
-2.4249476546250137, -2.4168183546379192, -2.3932988545481257, 
-2.3566988033364975, -2.3102693233342335, -2.2576000110029120, 
-2.2019078589783292, -2.1461222816371852, -2.0972499024590801, 
-2.0466915744985701, -1.9994960924030702, -1.9552340613212895, 
-1.9135538279754420, -1.8741645420975261, -1.8368237007545207, 
-1.8013278372050634, -1.7675054579514522, -1.7352116149583947, 
-1.7043236855249808, -1.6747380561714837, -1.6463674911069910, 
-1.6191390240268835, -1.5929922528103146, -1.5678779457508161, 
-1.5437568889546511, -1.5205989199656804, -1.4983821042204999, 
-1.4770920197831257, -1.4567211227826489, -1.4372681716640590, 
-1.4187376931703524, -1.4011394771804508, -1.3860930918734908, 
-1.3709996931783164, -1.3578746867208424, -1.3470243883402713, 
-1.3385686784311883, -1.3325578859424316, -1.3289741225932654, 
-1.3277833478255114, -2.4749509417955622, -2.4669496332091829, 
-2.4437933874820175, -2.4077374404006133, -2.3619600673511800, 
-2.3099768318068428, -2.2549446704673839, -2.1997477559821137, 
-2.1513260612654959, -2.1011629758086117, -2.0542630552522794, 
-2.0102046134555982, -1.9686410918975610, -1.9292847422238677, 
-1.8918946227116209, -1.8562676233125215, -1.8222316588510483, 
-1.7896404421208745, -1.7583694267538221, -1.7283126286203079, 
-1.6993801152717092, -1.6714960086559281, -1.6445968853326738, 
-1.6186304860560106, -1.5935546664234126, -1.5693365347004564, 
-1.5459517335602118, -1.5233838304802771, -1.5016237877449516, 
-1.4806694880067586, -1.4605252956100394, -1.4412016376855612, 
-1.4245060900952466, -1.4075689946267962, -1.3926586081977783, 
-1.3801805575244177, -1.3703443050263615, -1.3632833044443666, 
-1.3590429101956265, -1.3576285388176783, -2.5247219978021387, 
-2.5168539365218332, -2.4940778149774219, -2.4585974031062094, 
-2.4135218366451450, -2.3622957675129967, -2.3080178229141799, 
-2.2535261069440691, -2.2056776953217780, -2.1560592978261468, 
-2.1096176132761912, -2.0659379467393069, -2.0246782309516198, 
-1.9855532866631986, -1.9483232350839110, -1.9127848280588056, 
-1.8787648700013224, -1.8461151671097156, -1.8147086105201762, 
-1.7844361142610785, -1.7552042064429489, -1.7269331256158860, 
-1.6995553116055075, -1.6730142065376152, -1.6472633005664121, 
-1.6222653703105947, -1.5979918677467633, -1.5744224244234226, 
-1.5515444411536874, -1.5293527374323579, -1.5078492381755388, 
-1.4870426783661275, -1.4689044076854749, -1.4503279569786900, 
-1.4338058234258546, -1.4198391900651719, -1.4087270753573935, 
-1.4006877213972166, -1.3958323372609407, -1.3942079852309770, 
-2.5742565220245575, -2.5665264138030137, -2.5441455594919802, 
-2.5092690081527778, -2.4649402227006467, -2.4145359001679934, 
-2.3610980831813433, -2.3074182028563861, -2.2602556604797308, 
-2.2113197109824805, -2.1654868492224759, -2.1223488616379038, 
-2.0815677804059085, -2.0428606818371247, -2.0059884814888034, 
-1.9707475404992323, -1.9369632900989491, -1.9044853325255098, 
-1.8731836410485634, -1.8429455916467932, -1.8136736335315977, 
-1.7852834572338008, -1.7577025549742196, -1.7308690934328486, 
-1.7047310370370869, -1.6792454726666250, -1.6543780957033545, 
-1.6301028236880164, -1.6064015082528913, -1.5832637190775500, 
-1.5606865758464092, -1.5386746059812659, -1.5193344739262000, 
-1.4993622021466775, -1.4814402223248995, -1.4661590443409327, 
-1.4539051330711217, -1.4449814397809666, -1.4395664520926930, 
-1.4377503942254646, -2.6235509207528507, -2.6159629692834545, 
-2.5939909478712386, -2.5597437398971810, -2.5162023730607901, 
-2.4666783800213836, -2.4141589016909788, -2.3613882685925840, 
-2.3150147790171256, -2.2668879648018074, -2.2218029325921091, 
-2.1793576101094829, -2.1392179119852721, -2.1011030404023732, 
-2.0647746330021963, -2.0300286193201003, -1.9966890240585773, 
-1.9646031969577347, -1.9336381071678803, -1.9036774457455199, 
-1.8746193518612853, -1.8463746280790705, -1.8188653449303083, 
-1.7920237596503787, -1.7657914914275312, -1.7401189078742483, 
-1.7149646860467123, -1.6902955171723129, -1.6660859279968068, 
-1.6423181938721647, -1.6189823198251430, -1.5960760662835789, 
-1.5758088686569485, -1.5547238034351651, -1.5356532360107615, 
-1.5192680474402840, -1.5060376078154130, -1.4963473796475291, 
-1.4904430422229333, -1.4884586115264082, -2.6726022261646540, 
-2.6651601766796453, -2.6436091120518079, -2.6100141274551238, 
-2.5672968352712453, -2.5187062290950588, -2.4671761734144817, 
-2.4154036094010536, -2.3699135414838488, -2.3227120849790239, 
-2.2785028369814366, -2.2368896618289247, -2.1975422800260294, 
-2.1601820394454987, -2.1245713899576559, -2.0905059674141437, 
-2.0578085550583110, -2.0263244224253070, -1.9959176938696270, 
-1.9664685006611617, -1.9378707400484243, -1.9100303128807086, 
-1.8828637452957264, -1.8562971240955077, -1.8302652926653269, 
-1.8047112665568514, -1.7795858364347503, -1.7548473318393183, 
-1.7304615227471107, -1.7064016376612747, -1.6826484772974621, 
-1.6591906021908187, -1.6383038494417907, -1.6164280494730110, 
-1.5964999459901241, -1.5792589005033337, -1.5652498739213174, 
-1.5549361271319773, -1.5486285772045529, -1.5465045325165925, 
-2.7214080248541812, -2.7141152036919634, -2.6929959011794513, 
-2.6600736355212535, -2.6182134175479970, -2.5706041646584485, 
-2.5201280212487385, -2.4694343578021138, -2.4249138298337449, 
-2.3787440866491592, -2.3355280651027526, -2.2948754663366659, 
-2.2564598490919652, -2.2200048438677360, -2.1852739321328958, 
-2.1520627391378930, -2.1201931365938398, -2.0895086743386875, 
-2.0598710063297951, -2.0311570743410878, -2.0032568798033457, 
-1.9760717209431249, -1.9495128054549342, -1.9235001726798160, 
-1.8979618764494031, -1.8728333922092051, -1.8480572209479096, 
-1.8235826686035814, -1.7993657834903805, -1.7753694362152379, 
-1.7515635267627241, -1.7279253021185699, -1.7067571916147233, 
-1.6844504018785864, -1.6639952469705541, -1.6461845965597799, 
-1.6316285974900673, -1.6208606776732593, -1.6142527877593971, 
-1.6120236302686066, -2.7699663947109645, -2.7628257452640299, 
-2.7421478037314375, -2.7099165672534760, -2.6689430642296239, 
-2.6223584404389917, -2.5729945995671337, -2.5234532416255111, 
-2.4799806605393431, -2.4349397036026454, -2.3928243825947821, 
-2.3532502137226663, -2.3158947070846150, -2.2804840000735744, 
-2.2467829228838712, -2.2145874853369483, -2.1837191079863176, 
-2.1540201344632068, -2.1253503025641614, -2.0975839459546664, 
-2.0706077630757731, -2.0443190351064997, -2.0186242071006428, 
-1.9934377698206962, -1.9686813970215946, -1.9442833057457585, 
-1.9201778167113546, -1.8963050988194703, -1.8726110865911769, 
-1.8490475621872431, -1.8255723946439364, -1.8021499280911779, 
-1.7810664475579745, -1.7587237622830574, -1.7381101770470315, 
-1.7200539533848609, -1.7052166581366481, -1.6941909436159603, 
-1.6874029399744985, -1.6851091550759028, -2.8182758491094164, 
-2.8112899645180405, -2.7910618784208334, -2.7595379777829878, 
-2.7194777443179046, -2.6739567036510876, -2.6257579159617679, 
-2.5774353714173488, -2.5350819464235252, -2.4912581330820420, 
-2.4503415614602515, -2.4119535960657483, -2.3757758674562255, 
-2.3415373047376256, -2.3090044726798484, -2.2779742415008712, 
-2.2482681375722717, -2.2197279287158023, -2.1922121327677564, 
-2.1655932288564443, -2.1397554132038494, -2.1145927850349593, 
-2.0900078795640979, -2.0659104880254313, -2.0422167219364624, 
-2.0188482919439279, -1.9957319818164971, -1.9727993061155740, 
-1.9499863461854106, -1.9272337635135615, -1.9044869921971872, 
-1.8816966130488959, -1.8610877390703027, -1.8391361809903648, 
-1.8187685537772893, -1.8008272943284056, -1.7860080650326127, 
-1.7749481306336632, -1.7681178974054699, -1.7658060940558153, 
-2.8663352875092589, -2.8595064404254904, -2.8397356928215416, 
-2.8089335970969427, -2.7698103516207486, -2.7253878661487865, 
-2.6784016693971160, -2.6313580455869303, -2.5901882759872583, 
-2.5476617957041756, -2.5080331337319923, -2.4709295714944055, 
-2.4360370637251676, -2.4030876530629830, -2.3718500673113354, 
-2.3421225645562203, -2.3137273984189388, -2.2865064728096973, 
-2.2603178851572316, -2.2350331442975162, -2.2105349091947835, 
-2.1867151368599029, -2.1634735582016402, -2.1407164229663072, 
-2.1183554719425670, -2.0963071079324216, -2.0744917477504030, 
-2.0528333464452979, -2.0312590924746559, -2.0096992788854995, 
-1.9880873606274447, -1.9663602117040770, -1.9466351998479869, 
-1.9255291536164159, -1.9058440795045986, -1.8884124435823486, 
-1.8739430259912366, -1.8630991292889720, -1.8563821067986579, 
-1.8541050224411399, -2.9141439516858649, -2.9074741213966560, 
-2.8881672687911260, -2.8580997612003283, -2.8199346151936169, 
-2.7766419882021474, -2.7309111031776876, -2.6852005717962495, 
-2.6452727090634980, -2.6041161099829000, -2.5658561557679809, 
-2.5301261324165134, -2.4966165391054576, -2.4650628705707653, 
-2.4352364658012520, -2.4069375238380037, -2.3799896818606756, 
-2.3542357397605489, -2.3295342397498047, -2.3057566932486377, 
-2.2827853048740185, -2.2605110838629212, -2.2388322624487413, 
-2.2176529623170942, -2.1968820667911837, -2.1764322695351046, 
-2.1562192814881525, -2.1361611872639634, -2.1161779509011751, 
-2.0961910789414708, -2.0761234564155915, -2.0558993782757522, 
-2.0374811821832814, -2.0176966613145559, -1.9991581020211509, 
-1.9826612372086743, -1.9689033721720841, -1.9585511173032706, 
-1.9521196949913120, -1.9499360307473701, -2.9617013869115523, 
-2.9551922840784846, -2.9363550338872515, -2.9070333506012989, 
-2.8698450189337152, -2.8277101735680072, -2.7832728712953023, 
-2.7389441032178650, -2.7003105876820954, -2.6605892808780673, 
-2.6237709834056298, -2.5894950791807383, -2.5574568336973926, 
-2.5273955320835277, -2.4990855727617580, -2.4723296517112812, 
-2.4469534544506271, -2.4228014540773106, -2.3997335329432419, 
-2.3776222257854895, -2.3563504372943695, -2.3358095258422757, 
-2.3158976729368361, -2.2965184785109818, -2.2775797378480136, 
-2.2589923684453725, -2.2406694656534376, -2.2225254753783430, 
-2.2044754811819027, -2.1864346122397018, -2.1683175881454968, 
-2.1500384265791581, -2.1333573311994960, -2.1153851120043750, 
-2.0984782342273838, -2.0833667845931507, -2.0707085873537849, 
-2.0611466233575633, -2.0551889246170947, -2.0531629735542918, 
-3.0090074074965596, -3.0026604967419495, -2.9842977780758670, 
-2.9557317352832211, -2.9195367293187351, -2.8785844746759444, 
-2.8354749168674172, -2.7925714884010011, -2.7552793610875614, 
-2.7170521017627065, -2.6817410580630576, -2.6489918001799513, 
-2.6185045713471577, -2.5900227711921340, -2.5633242896173885, 
-2.5382148591613052, -2.5145228640791508, -2.4920952180822464, 
-2.4707940369863124, -2.4504939094062648, -2.4310796214061128, 
-2.4124442279486513, -2.3944873903724870, -2.3771139184142336, 
-2.3602324698300454, -2.3437543720949643, -2.3275925401812163, 
-2.3116604729848493, -2.2958713193718920, -2.2801370137381283, 
-2.2643674910261500, -2.2484700028193805, -2.2339566097041152, 
-2.2182943314550250, -2.2035180449278813, -2.1902617426133735, 
-2.1791127357896114, -2.1706593613472909, -2.1653773220274926, 
-2.1635783539152951, -3.0560620661739275, -3.0498785867186862, 
-3.0319946151175743, -3.0041927254296947, -2.9690055304016467, 
-2.9292578068822475, -2.8875063615084327, -2.8460671335912995, 
-2.8101584239094888, -2.7734777691800243, -2.7397327037567978, 
-2.7085750591847209, -2.6797102479707893, -2.6528860831324503, 
-2.6278843487709747, -2.6045143214476885, -2.5826077010414719, 
-2.5620145762438158, -2.5426001594359344, -2.5242420993613992, 
-2.5068282304708593, -2.4902546529769500, -2.4744240625552032, 
-2.4592442665556624, -2.4446268367725810, -2.4304858588177196, 
-2.4167367461186098, -2.4032950934537021, -2.3900755515724112, 
-2.3769907116386038, -2.3639499968220394, -2.3508585692378143, 
-2.3389363259859870, -2.3260797312752648, -2.3139380277545536, 
-2.3030178810835471, -2.2938026205558351, -2.2867911994729258, 
-2.2823978573426897, -2.2808992276296118, -3.1028656268776276, 
-3.0968466114146751, -3.0794449480949093, -3.0524145272581702, 
-3.0182477652774860, -2.9797238708608762, -2.9393574045926143, 
-2.8994168764470123, -2.8649289665433511, -2.8298417097522934, 
-2.7977149349083739, -2.7682067905006575, -2.7410280228524395, 
-2.7159311236495909, -2.6927021344323179, -2.6711543386404739, 
-2.6511233197579380, -2.6324630236748896, -2.6150425684823162, 
-2.5987436155436523, -2.5834581639848855, -2.5690866642424557, 
-2.5555363697985984, -2.5427198628338679, -2.5305537013196151, 
-2.5189571435019706, -2.5078509118492036, -2.4971559631892823, 
-2.4867922357137568, -2.4766773475508947, -2.4667252266164827, 
-2.4568446585377024, -2.4479221754389209, -2.4383557435371479, 
-2.4293480316605267, -2.4212472161136147, -2.4143975269802693, 
-2.4091706730005256, -2.4058866155407319, -2.4047645756660385, 
-3.1494185405196453, -3.1435648324875718, -3.1266484386094513, 
-3.1003957033954435, -3.0672602833293476, -3.0299770823029228, 
-2.9910192314718143, -2.9526078701842273, -2.9195738368565469, 
-2.8861214186030173, -2.8556592747338532, -2.8278519023749631, 
-2.8024155141684277, -2.7791075060971560, -2.7577184934695587, 
-2.7380661755376532, -2.7199905266784050, -2.7033499650623565, 
-2.6880182505913384, -2.6738819316951834, -2.6608382069047249, 
-2.6487930991438065, -2.6376598629556343, -2.6273575603361818, 
-2.6178097514213161, -2.6089432532629822, -2.6006869242359558, 
-2.5929704339123960, -2.5857229790942906, -2.5788719066803600, 
-2.5723412038668263, -2.5660498168038881, -2.5605132558624897, 
-2.5547005578254227, -2.5493112902836477, -2.5445049604547414, 
-2.5404509032371076, -2.5373534715885775, -2.5354024370508559, 
-2.5347346388687302, -3.1957214234204159, -3.1900336928251689, 
-3.1736049792349785, -3.1481351372958608, -3.1160403926405928, 
-3.0800125081864440, -3.0424839298046797, -3.0056284772661606, 
-2.9740774123902778, -2.9422963086600333, -2.9135395839481628, 
-2.8874780889325300, -2.8638335997562701, -2.8423685987093164, 
-2.8228785392867222, -2.8051858851171909, -2.7891354396545425, 
-2.7745906302702541, -2.7614305072752656, -2.7495472836293349, 
-2.7388442856978052, -2.7292342162292496, -2.7206376520751179, 
-2.7129817138228978, -2.7061988542542204, -2.7002257185203669, 
-2.6950020318833539, -2.6904694712612987, -2.6865704749346033, 
-2.6832469408344184, -2.6804387580976345, -2.6780821094513882, 
-2.6762879583305224, -2.6746621267973567, -2.6733501161731903, 
-2.6722944798405770, -2.6714542916015702, -2.6708253179043080, 
-2.6704290109990398, -2.6702927207916076, -3.2417750380893873, 
-3.2362537960067339, -3.2203146688628141, -3.1956320012620028, 
-3.1645858170302619, -3.1298258089584832, -3.0937444132378991, 
-3.0584681718273647, -3.0284254822817900, -2.9983475702192171, 
-2.9713318994676290, -2.9470556507983248, -2.9252462239409143, 
-2.9056713236967089, -2.8881314513843548, -2.8724541193135584, 
-2.8584893237572153, -2.8461059527169423, -2.8351888979754678, 
-2.8256367038423398, -2.8173596281491622, -2.8102780209642133, 
-2.8043209472928945, -2.7994249943353280, -2.7955332133549673, 
-2.7925941519066440, -2.7905609346719507, -2.7893903507231141, 
-2.7890419017018302, -2.7894767589763414, -2.7906565679989463, 
-2.7925420244204417, -2.7948105749395848, -2.7977653242410816, 
-2.8009532297528081, -2.8040743472614174, -2.8068437391869110, 
-2.8090075892079676, -2.8103798564944920, -2.8108499235555930, 
-3.2875802760892157, -3.2822258879673734, -3.2667777906189079, 
-3.2428857276790430, -3.2128946572300481, -3.1794131860417232, 
-3.1447943517550940, -3.1111174500931660, -3.0826051381833186, 
-3.0542580401646395, -3.0290142827577493, -3.0065573244582775, 
-2.9866202110473430, -2.9689759595536231, -2.9534302729279318, 
-2.9398159305389222, -2.9279884081658873, -2.9178224163874731, 
-2.9092091360446699, -2.9020539903165821, -2.8962748349324050, 
-2.8918004773960457, -2.8885694567694760, -2.8865290300678010, 
-2.8856343212637010, -2.8858475952605289, -2.8871376225029093, 
-2.8894791003271632, -2.8928520945721927, -2.8972414589222049, 
-2.9026361791449302, -2.9090285736522739, -2.9156384073063109, 
-2.9235200513918769, -2.9315845764213888, -2.9392674530193439, 
-2.9460087800079742, -2.9512659053111601, -2.9546065104789587, 
-2.9557531698084816, -3.3331381427491080, -3.3279508406190628, 
-3.3129947920700746, -3.2898959831189747, -3.2609653557751139, 
-3.2287713341408009, -3.1956281080763129, -3.1635677481182998, 
-3.1366046735035464, -3.1100120802606153, -3.0865666774468785, 
-3.0659581203235331, -3.0479250860681177, -3.0322459477243098, 
-3.0187317083345961, -3.0072205670009500, -2.9975736883777118, 
-2.9896718770253528, -2.9834129446746154, -2.9787096175126062, 
-2.9754878717043658, -2.9736856144973007, -2.9732516492375285, 
-2.9741448779209883, -2.9763337061050072, -2.9797956231320373, 
-2.9845169362614516, -2.9904926407265942, -2.9977264088807529, 
-3.0062306800785215, -3.0160268279564568, -3.0271453719807004, 
-3.0383291119960698, -3.0514300018446798, -3.0646933577132809, 
-3.0772719639336166, -3.0883038919414578, -3.0969217091625656, 
-3.1024100437929438, -3.1042966683322111, -3.3784497435207275, 
-3.3734296372111676, -3.3589662674698779, -3.3366626450102936, 
-3.3087966652285026, -3.2778973978787520, -3.2462406795514465, 
-3.2158113662269980, -3.1904134903443944, -3.1655954639529851, 
-3.1439707758087709, -3.1252351693907126, -3.1091329028147383, 
-3.0954477045601823, -3.0839959225952898, -3.0746212645096014, 
-3.0671907175882569, -3.0615913626724662, -3.0577278793524356, 
-3.0555205975372641, -3.0549039909255478, -3.0558255371347127, 
-3.0582448909595783, -3.0621333338827394, -3.0674734762192597, 
-3.0742591993039614, -3.0824958347406111, -3.0922005865092248, 
-3.1034032100624951, -3.1161469706446141, -3.1304899109248638, 
-3.1465064653269872, -3.1624479877071998, -3.1810017235276766, 
-3.1997248780147425, -3.2174737448208885, -3.2330621041512728, 
-3.2452666064119358, -3.2530557513298910, -3.2557367006423048, 
-3.4235162717944507, -3.4186633592387925, -3.4046929418236038, 
-3.3831857806032852, -3.3563876194006705, -3.3267889323432547, 
-3.2966276450450773, -3.2678413995900581, -3.2440220135504600, 
-3.2209952711405960, -3.2012098937063476, -3.1843675783320382, 
-3.1702180797611650, -3.1585504393065493, -3.1491863437781125, 
-3.1419750371198121, -3.1367893906947040, -3.1335228583098140, 
-3.1320871230700500, -3.1324102992894129, -3.1344355927696572, 
-3.1381203523686190, -3.1434354685917354, -3.1503650934618941, 
-3.1589066719709207, -3.1690712904076861, -3.1808843620131859, 
-3.1943866869084658, -3.2096359423434313, -3.2267086825146283, 
-3.2457029563711490, -3.2667416894265160, -3.2875748993836837, 
-3.3117535688341335, -3.3361317043165468, -3.3592596817193532, 
-3.3796101867900448, -3.3955789297906573, -3.4057905265352648, 
-3.4093092587031202, -3.4683389980150681, -3.4636531747296115, 
-3.4501756565771116, -3.4294656279914384, -3.4037375072625959, 
-3.3754438671660729, -3.3467851163589279, -3.3196516744231754, 
-3.2974216113303774, -3.2761997904026332, -3.2582688535844886, 
-3.2433362928021121, -3.2311572436848683, -3.2215259786864321, 
-3.2142694699107333, -3.2092424686298937, -3.2063237239751792, 
-3.2054130789109476, -3.2064292600794073, -3.2093082330532980, 
-3.2140020343992193, -3.2204780221646105, -3.2287185106051379, 
-3.2387207757743557, -3.2504974380381761, -3.2640772474369788, 
-3.2795063198902499, -3.2968498985704433, -3.3161947479556999, 
-3.3376523316245725, -3.3613629837912131, -3.3875013662751634, 
-3.4133105495373397, -3.4432241102505725, -3.4733845707236117, 
-3.5020312067515027, -3.5272846305336754, -3.5471416923682537, 
-3.5598620754104551, -3.5642496899583729, -3.5129192599546433, 
-3.5084003277593614, -3.4954153567565585, -3.4755025789761667, 
-3.4508458492834206, -3.4238604737978808, -3.3967096937817258, 
-3.3712366893352863, -3.3506045219487421, -3.3311984281926645, 
-3.3151338750999040, -3.3021239687112187, -3.2919290811235897, 
-3.2843485984986049, -3.2792146812150174, -3.2763875066496855, 
-3.2757516331120637, -3.2772132347684422, -3.2806980335043123, 
-3.2861498075641316, -3.2935293965889088, -3.3028141536352282, 
-3.3139978205600791, -3.3270908265362324, -3.3421210327315656, 
-3.3591349715835466, -3.3781996591373087, -3.3994050967551743, 
-3.4228676284756658, -3.4487343886586399, -3.4771891706947149, 
-3.5084601860050904, -3.5392818468152831, -3.5749796237985958, 
-3.6109824465374616, -3.6452172466305139, -3.6754474601634484, 
-3.6992608432621439, -3.7145388097558651, -3.7198131664759924, 
-3.5572584540166647, -3.5529061290633455, -3.5404130794055919, 
-3.5212971635855799, -3.4977123759524558, -3.4720373356752963, 
-3.4463984253956306, -3.4225915613963664, -3.4035637860242760, 
-3.3859816245357464, -3.3717924729805282, -3.3607148511870486, 
-3.3525141975931199, -3.3469948625162056, -3.3439940584656429, 
-3.3433772606702741, -3.3450347118588502, -3.3488787925035570, 
-3.3548420916072166, -3.3628760660405592, -3.3729502162142340, 
-3.3850517376728053, -3.3991856357054933, -3.4153753161821494, 
-3.4336636931645832, -3.4541148851730457, -3.4768166106399243, 
-3.5018834435182344, -3.5294611586200366, -3.5597324926153715, 
-3.5929247855652382, -3.6293201730451807, -3.6651461826862164, 
-3.7066203059048046, -3.7484612341938708, -3.7882858217118662, 
-3.8235008625669562, -3.8512825958806558, -3.8691290514426724, 
-3.8752945622578361, -3.6013580274599395, -3.5971719476263622, 
-3.5851699431837312, -3.5668500360787543, -3.5443370082708388, 
-3.5199733210077038, -3.4958487698036280, -3.4737119765312179, 
-3.4562931840043931, -3.4405407747901813, -3.4282333617143528, 
-3.4190946609262323, -3.4128949844511749, -3.4094434688592834, 
-3.4085822080654249, -3.4101818053092563, -3.4141380133035675, 
-3.4203692347298591, -3.4288147269838674, -3.4394334070921522, 
-3.4522031925304457, -3.4671208463374503, -3.4842023241495506, 
-3.5034836496548265, -3.5250223764315494, -3.5488997315079907, 
-3.5752235834724120, -3.6041324414387574, -3.6358007792825604, 
-3.6704461055770317, -3.7083383846039766, -3.7498126919313250, 
-3.7905945024346597, -3.8377849844607899, -3.8854006688950049, 
-3.9307536183662366, -3.9708996701054620, -4.0026086063572590, 
-4.0229981384078570, -4.0300462618517399, -3.6452194714433102, 
-3.6411992031466842, -3.6296871390048184, -3.6121619622852514, 
-3.5907198400207307, -3.5676675579388979, -3.5450585619740758, 
-3.5245941438796273, -3.5087871784186047, -3.4948681570604041, 
-3.4844463666776542, -3.4772504876230546, -3.4730554932434350, 
-3.4716751039182192, -3.4729560942724707, -3.4767739896726368, 
-3.4830298353602127, -3.4916478209209281, -3.5025736124350617, 
-3.5157732957917371, -3.5312328744831043, -3.5489582986804975, 
-3.5689760332853568, -3.5913342041852214, -3.6161043974116129, 
-3.6433842291536438, -3.6733008607313247, -3.7060157089656562, 
-3.7417307098653727, -3.7806966492997454, -3.8232243058827384, 
-3.8696995042439206, -3.9153531378603237, -3.9681542001894701, 
-4.0214291427930613, -4.0721930356531901, -4.1171609273401284, 
-4.1527079559326987, -4.1755821802295534, -4.1834925713885838, 
-3.6888443148034336, -3.6849893592814555, -3.6739659216077905, 
-3.6572338081458704, -3.6368611216400484, -3.6151194118631338, 
-3.5940259819272207, -3.5752347537942546, -3.5610408605432644, 
-3.5489568648718230, -3.5404223413246414, -3.5351706901542657, 
-3.5329813173288076, -3.5336723038235522, -3.5370948788770007, 
-3.5431292535663066, -3.5516815118250453, -3.5626813516361824, 
-3.5760805367550850, -3.5918519695738733, -3.6099893355524619, 
-3.6305073038658158, -3.6534423013323929, -3.6788539106958336, 
-3.7068269835227943, -3.7374746067381923, -3.7709421260940643, 
-3.8074125183195040, -3.8471135295387890, -3.8903271816444134, 
-3.9374025246441366, -3.9887729384898658, -4.0391844470926248, 
-4.0974516552872631, -4.1562263521659490, -4.2122364426788623, 
-4.2618700580727147, -4.3011252170110730, -4.3263975554428882, 
-4.3351397523784705, -3.7322341184875949, -3.7285439175910602, 
-3.7180076019637514, -3.7020665293348554, -3.6827612455484622, 
-3.6623284646969831, -3.6427495260141893, -3.6256309391757187, 
-3.6130499011380333, -3.6028007447419017, -3.5961530900738943, 
-3.5928448031948195, -3.5926594805503860, -3.5954193233896294, 
-3.6009797685069378, -3.6092254511056479, -3.6200672100675888, 
-3.6334399378911462, -3.6493011431898656, -3.6676301429996103, 
-3.6884278418687897, -3.7117170895468821, -3.7375436428407647, 
-3.7659777934138940, -3.7971167658292337, -3.8310880438379971, 
-3.8680538545103276, -3.9082171392007856, -3.9518294828129723, 
-3.9992016827114494, -4.0507179558800273, -4.1068552750988951, 
-4.1618863736969933, -4.2254441377219401, -4.2895238413190109, 
-4.3505776391082600, -4.4046834934799861, -4.4474843047951618, 
-4.4750457181610823, -4.4845811926778811, -3.7753904705722277, 
-3.7718644121091973, -3.7618135404338058, -3.7466611618565731, 
-3.7284207327863212, -3.7092944959280385, -3.6912279805622541, 
-3.6757802398701696, -3.6648105049404660, -3.6563943383045716, 
-3.6516312965415776, -3.6502634499393474, -3.6520783326957194, 
-3.6569020124036555, -3.6645938696402047, -3.6750426821105244, 
-3.6881637361873927, -3.7038967771222135, -3.7222046728619715, 
-3.7430727148287537, -3.7665085185713036, -3.7925425225099154, 
-3.8212291178693039, -3.8526484809770647, -3.8869092245019532, 
-3.9241520421762495, -3.9645545994722586, -4.0083380315795898, 
-4.0557755669241926, -4.1072040267513774, -4.1630393041741316, 
-4.2237974765150428, -4.2832910869940388, -4.3519401191480567, 
-4.4211036684416021, -4.4869707486427775, -4.5453279625315810, 
-4.5914882588397266, -4.6212133998722056, -4.6314977754765438, 
-3.8183149818054187, -3.8149524044733627, -3.8053851406012966, 
-3.7910188135208562, -3.7738402208422168, -3.7560174652789837, 
-3.7394603976810554, -3.7256805698780506, -3.7163193686291431, 
-3.7097328286667577, -3.7068504567865377, -3.7074182606070645, 
-3.7112274514733765, -3.7181076990863589, -3.7279220513160749, 
-3.7405631315328938, -3.7559503482559493, -3.7740279368919412, 
-3.7947637150386542, -3.8181484802745977, -3.8441960186398139, 
-3.8729437276384502, -3.9044538933492636, -3.9388157008452893, 
-3.9761481048900804, -4.0166037493858138, -4.0603742072275653, 
-4.1076969289596814, -4.1588644574868674, -4.2142367169475152, 
-4.2742575662900562, -4.3394774068885882, -4.4032628954017685, 
-4.4767872843431800, -4.5507955289083792, -4.6212269606523178, 
-4.6835969281927019, -4.7329154836651428, -4.7646687573923012, 
-4.7756540029258669, -3.8610092816184305, -3.8578094795575426, 
-3.8487238437102622, -3.8351406562112462, -3.8190204525574813, 
-3.8024974968412324, -3.7874460730433643, -3.7753301871441924, 
-3.7675736419881050, -3.7628119906966306, -3.7618048172480330, 
-3.7643017964133310, -3.7700975507217596, -3.7790250805146135, 
-3.7909508154666445, -3.8057709170443661, -3.8234085780779581, 
-3.8438121472097522, -3.8669539657365415, -3.8928298507907240, 
-3.9214591977221547, -3.9528857103226063, -3.9871788039341522, 
-4.0244357672825695, -4.0647848185085307, -4.1083892551790271, 
-4.1554529854056854, -4.2062278501693378, -4.2610233252695195, 
-4.3202194568402641, -4.3842842901123200, -4.4537976880016306, 
-4.5216956347800430, -4.5998692779407087, -4.6784727320196513, 
-4.7532096484393120, -4.8193448391553142, -4.8716132454594314, 
-4.9052543525291137, -4.9168907916404434, -3.9034750145572343, 
-3.9004372415552151, -3.8918311236491547, -3.8790279188692218, 
-3.8639622660071193, -3.8487348645463859, -3.8351845254710049, 
-3.8247276657188900, -3.8185708920259613, -3.8156281449629308, 
-3.8164893170725760, -3.8209074786985799, -3.8286803945597048, 
-3.8396441197686215, -3.8536681747309340, -3.8706519448093690, 
-3.8905220617457204, -3.9132306021038432, -3.9387539958133817, 
-3.9670925832598933, -3.9982707978234195, -4.0323379865497380, 
-4.0693699184542504, -4.1094710716120764, -4.1527778412059462, 
-4.1994628771163622, -4.2497408501167291, -4.3038760733927699, 
-4.3621925915214774, -4.4250876255371852, -4.4930496858329549, 
-4.5666833298954570, -4.6385097267745232, -4.7211019525517024, 
-4.8040474386727041, -4.8828284294243591, -4.9524799428589859, 
-5.0074893453784419, -5.0428780132974262, -5.0551160379345390, 
-3.9457138370903930, -3.9428373104663619, -3.9347084824253558, 
-3.9226818811254733, -3.9086665852667246, -3.8947299788564438, 
-3.8826754781720147, -3.8738718700975201, -3.8693090698222350, 
-3.8681781150635492, -3.8708995345422221, -3.8772295229144871, 
-3.8869687171859670, -3.8999559495457556, -3.9160635375669810, 
-3.9351937733846856, -3.9572763791233778, -3.9822667708720179, 
-4.0101450293937502, -4.0409155200242077, -4.0746071421230878, 
-4.1112742240891613, -4.1509981169736845, -4.1938895819341111, 
-4.2400921187406606, -4.2897864504345637, -4.3431964719229565, 
-4.4005971012520666, -4.4623246627044439, -4.5287907150441695, 
-4.6005006734922400, -4.6780792590001994, -4.7536490821602220, 
-4.8404293772143028, -4.9274655381443395, -5.0100327024813911, 
-5.0829563804818738, -5.1405028753714683, -5.1775026232329422, 
-5.1902940523096968, -3.9835799636259526, -3.9808480296361450, 
-3.9731472645471912, -3.9618173565878498, -3.9487446199748049, 
-3.9359666145908871, -3.9252549202087574, -3.9179353742158858, 
-3.9148031637923397, -3.9152977799100817, -3.9196875009760821, 
-3.9277311059957700, -3.9392315896738395, -3.9540300489992943, 
-3.9720010706779649, -3.9930492946288920, -4.0171069296438766, 
-4.0441320694709146, -4.0741077117481224, -4.1070414254643213, 
-4.1429656499787724, -4.1819386440507076, -4.2240461403870588, 
-4.2694038036974504, -4.3181606427516641, -4.3705035956822309, 
-4.4266636017688645, -4.4869236058595252, -4.5516291349495637, 
-4.6212023751731248, -4.6961611198935289, -4.7771446562054010, 
-4.8559305524455310, -4.9462902908163970, -5.0368021043804152, 
-5.1225666138759474, -5.1982372437692890, -5.2579025084722550, 
-5.2962424144076925, -5.3094932100241561, -4.0235163358691359, 
-4.0209369098522343, -4.0136877045628792, -4.0030929052658450, 
-3.9910149528406427, -3.9794599803562041, -3.9701658713864689, 
-3.9644129100663887, -3.9627905115746866, -3.9650001297310440, 
-3.9711491071304623, -3.9809987258151374, -3.9943542432685559, 
-4.0110589132767993, -4.0309894698254691, -4.0540527631137619, 
-4.0801833327156674, -4.1093417721075634, -4.1415137911372071, 
-4.1767099251611635, -4.2149658762926148, -4.2563435072883022, 
-4.3009325455605278, -4.3488530973755788, -4.4002591250945562, 
-4.4553431095726266, -4.5143422146328485, -4.5775464046734999, 
-4.6453091616930617, -4.7180617396118905, -4.7963323407164484, 
-4.8807723030159567, -4.9628080461689299, -5.0567664320773149, 
-5.1507524109611227, -5.2396938407685267, -5.3180780965766292, 
-5.3798257587302949, -5.4194777571963257, -5.4331773658949114, 
-4.0601705715247327, -4.0577311044257094, -4.0508963550148778, 
-4.0409764026449153, -4.0298118853819345, -4.0193801049854265, 
-4.0113879023198500, -4.0070735222845704, -4.0068372276610384, 
-4.0106207447775093, -4.0183834362079072, -4.0298886803840546, 
-4.0449436093303683, -4.0633932468380936, -4.0851160756048683, 
-4.1100207347273594, -4.1380436404495473, -4.1691473895762812, 
-4.2033198556673694, -4.2405739292562803, -4.2809478894226993, 
-4.3245064287627866, -4.3713423905648243, -4.4215793195161179, 
-4.4753749801222620, -4.5329260664606776, -4.5944744220145237, 
-4.6603152229945515, -4.7308077745893877, -4.8063898624453341, 
-4.8875970506513653, -4.9750890248214779, -5.0599815015787044, 
-5.1570874100519646, -5.2540948468336124, -5.3457830874548868, 
-5.4265003261285036, -5.4900293817578643, -5.5307996113984927, 
-5.5448809478682275, -4.0921218469951697, -4.0898043520856282, 
-4.0833308324720496, -4.0739991597177587, -4.0636310413666177, 
-4.0541786484311153, -4.0473216661444011, -4.0442615159114821, 
-4.0452333975797563, -4.0503882903473292, -4.0595563919698519, 
-4.0725027521130004, -4.0890359795451063, -4.1090024781091055, 
-4.1322820824585218, -4.1587848053589926, -4.1884484960813086, 
-4.2212372736182795, -4.2571406477370823, -4.2961732811084197, 
-4.3383753812657035, -4.3838137454955159, -4.4325835182929847, 
-4.4848107633942353, -4.5406560051803053, -4.6003189636682018, 
-4.6640448024606007, -4.7321323437784253, -4.8049449009096188, 
-4.8829246715821144, -4.9666120853023479, -5.0566722060336371, 
-5.1439589310892631, -5.2436911170205311, -5.3432070561604279, 
-5.4371640649736639, -5.5197987198084455, -5.5847854905500531, 
-5.6264677816700752, -5.6408599182254617, -4.1182703251639898, 
-4.1160526204564505, -4.1098746609321966, -4.1010243841650817, 
-4.0913080514503610, -4.0826572632055722, -4.0767294114275892, 
-4.0746956591564887, -4.0766560239388481, -4.0829325648519053, 
-4.0932496771728673, -4.1073736707774984, -4.1251142547294410, 
-4.1463188503015145, -4.1708682802113328, -4.1986735541863904, 
-4.2296735556481027, -4.2638334966546338, -4.3011440563648584, 
-4.3416211577777766, -4.3853063725321615, -4.4322679775766272, 
-4.4826027238289905, -4.5364384191592508, -4.5939374806849100, 
-4.6553016806781731, -4.7207784054338449, -4.7906688810866234, 
-4.8653390164492389, -4.9452338059768008, -5.0308966853532722, 
-5.1229959404299832, -5.2121773960107047, -5.3139802734249972, 
-5.4154658573368000, -5.5111967189815241, -5.5953244123068275, 
-5.6614422215288149, -5.7038302915346319, -5.7184625887067524, 
-4.1376420699529497, -4.1354982714370792, -4.1295392213128901, 
-4.1210455285681240, -4.1118120356255465, -4.1037550989913640, 
-4.0985155534174114, -4.0972420748416649, -4.0999344890863609, 
-4.1070414821470758, -4.1182090219546543, -4.1332042900584636, 
-4.1518377483216975, -4.1739575073425215, -4.1994450538472634, 
-4.2282120632127258, -4.2601981057195077, -4.2953691161296561, 
-4.3337165434545151, -4.3752571367244508, -4.4200333572408654, 
-4.4681144415672680, -4.5195981756433339, -4.5746134824782470, 
-4.6333239784083178, -4.6959327220902836, -4.7626884753132668, 
-4.8338939291864786, -4.9099165451325328, -4.9912029528772006, 
-5.0782982966614014, -5.1718726286802577, -5.2624211841544870, 
-5.3657138905297250, -5.4686121098986487, -5.5656110684602753, 
-5.6508028056335569, -5.7177244982256585, -5.7606131890354781, 
-5.7754156519505289, -4.1495445934604005, -4.1474461942343197, 
-4.1416216215399624, -4.1333469939019212, -4.1244101471528047, 
-4.1167180689925669, -4.1119014030159589, -4.1110949788525364, 
-4.1142370471777010, -4.1218540432935695, -4.1335437314688344, 
-4.1490738028586858, -4.1682551557366185, -4.1909362972619348, 
-4.2169990937121673, -4.2463555990016273, -4.2789457723949420, 
-4.3147359564879215, -4.3537180332816066, -4.3959092147861556, 
-4.4413524590363078, -4.4901175360199863, -4.5423028040319657, 
-4.5980377989411991, -4.6574867913013565, -4.7208535353296410, 
-4.7883875285862576, -4.8603922355321236, -4.9372359238483110, 
-5.0193660549401935, -5.1073286187871076, -5.2017945106798260, 
-5.2931679265009057, -5.3973578239007756, -5.5011049313835567, 
-5.5988642662124422, -5.6846927039937807, -5.7520945574785038, 
-5.7952818693029586, -5.8101857634209413, -4.1535627692639228, 
-4.1514796944190042, -4.1457005150020603, -4.1374998356717692, 
-4.1286631284529127, -4.1210942177257257, -4.1164203035293516, 
-4.1157715360435949, -4.1190653766541958, -4.1268544983271189, 
-4.1387203898669878, -4.1544309100231134, -4.1737970999280058, 
-4.1966675960050370, -4.2229243878089360, -4.2524796518491890, 
-4.2852734730780266, -4.3212723256099217, -4.3604682308183387, 
-4.4028785494534040, -4.4485463987934777, -4.4975417194101475, 
-4.5499630520962837, -4.6059401274500127, -4.6656374230172215, 
-4.7292589119579347, -4.7970543219711237, -4.8693273575103007, 
-4.9464465339664203, -5.0288595649517411, -5.1171126924448709, 
-5.2118770567882624, -5.3035263736328675, -5.4080160764135359, 
-5.5120465441628825, -5.6100594175410556, -5.6960999357639297, 
-5.7636615838119729, -5.8069482085127868, -5.8218858237140445
};

/*========================================================================= */
static void setalpha(double alpha, double oneminusalpha, double twominusalpha)
{
/* functions setalpha might be made faster by initializing arrays */
double sinangle, sinhalfangle;
int i;

/* If alpha is not in permissible range then print message but do nothing */
if((alpha>2.) | (alpha <= 0.)){
	return;
}

/* If alpha is 2 or the same as the last time, then do nothing. */
if ((twominusalpha == 0) | ((alpha == previous_alpha) &
	(oneminusalpha == previous_oneminusalpha) &
	(twominusalpha == previous_twominusalpha) ))return;

/* On first call, read array margins as vectors and tabulated arrays */
if (previous_alpha == -999.){

/* Compute reciprocals of denominators for later interpolation */
	calc_recip_denom(nx1,Vx1,xdenom1);
	calc_recip_denom(ny1,Vy1,ydenom1);
	calc_recip_denom(nx2,Vx2,xdenom2);
	calc_recip_denom(ny2,Vy2,ydenom2);
	calc_recip_denom(nx3,Vx3,xdenom3);
	calc_recip_denom(ny3,Vy3,ydenom3);
	calc_recip_denom(nx4,Vx4,xdenom4);
	calc_recip_denom(ny4,Vy4,ydenom4);
	calc_recip_denom(nx5,Vx5,xdenom5);
	calc_recip_denom(ny5,Vy5,ydenom5);
	calc_recip_denom(nx6,Vx6,xdenom6);
	calc_recip_denom(ny6,Vy6,ydenom6);
	calc_recip_denom(nx7,Vx7,xdenom7);
	calc_recip_denom(ny7,Vy7,ydenom7);

/* Also calculate Gaussian distribution for tabulated x's for use with table4 */
	for (i=0; i<nx4; i++){
		f4_alpha2[i]=normaltail(Vx4[i]/sqrt(2.));
		d4_alpha2[i]=1/(sqrt(4*pi))*exp(-Vx4[i]*Vx4[i]*.25);
	}

}			/* end of initialization */
/* ===================================================== */
/* Store standard numbers which vary with alpha */
previous_alpha=alpha;
previous_oneminusalpha=oneminusalpha;
previous_twominusalpha=twominusalpha;
distributiontabulated=FALSE;

alphasmall= alpha < .5;
alphalarge= alpha > 1.7;

/* Interpolate appropriate table(s) and store the result(s) */
/* Case when alpha < .5 */
if(alphasmall){
	absam1=fabs(oneminusalpha);
	a1=alpha/absam1;
	sa2=twominusalpha/(2*alpha);
	nu=pow(absam1,(-1/alpha));
	eta=tan(hpi*alpha);
	Calpha_C=exp(LogGamma(alpha))*sin(pi*alpha)/pi;
	ximid=.2/alpha;
/*	midpoint=pow((absam1/ximid),(1/a1))*alpha */
	logmidpoint=log(absam1/ximid)/a1+log(alpha);
/*	xi=absam1*(z/alpha)**a1) */
	Clogd=log(nu/sqrt(2*pi*alpha));

	interpolate_over_alpha(nx1,ny1,Vy1,alpha,tablef1,tabled1,f1,d1,ydenom1);
	interpolate_over_alpha(nx2,ny2,Vy2,alpha,tablef2,tabled2,f2,d2,ydenom2);
	interpolate_over_alpha(nx3,ny3,Vy3,alpha,tablef3,tabled3,f3,d3,ydenom3);
	}
else{
/* Case when alpha > .5 */
	if(alpha>1) alphastar=1/alpha;
	else alphastar=alpha;
	ximid=.4;
	if(alphastar == 1.){
		midpoint=(-log(hpi*ximid)-1)/hpi;
		nu=1;
		eta=0;
		logscalef=log(hpi);
/* Lower limit where xi=10**30; take density to be zero below here */
		xlowlimit=-(1+log(hpi*1.E30))/hpi;
		}
	else{
		ratio=oneminusalpha/alpha;
		angle=hpi*oneminusalpha;
		sinangle=sin(angle);
		C1=alpha/sinangle;
		C2=oneminusalpha/alpha;
		C3=oneminusalpha/sinangle;
		C4=(2*pow(sin(.25*pi*oneminusalpha),2)-oneminusalpha)/sinangle;
		midpoint=C1*expm1(C2*log(C3/ximid))+C4;
		nu=pow(fabs(oneminusalpha),(-1/alpha));
		if(alphalarge) eta=-tan(hpi*twominusalpha);
		else eta=1/tan(angle);
	/* Lower limit where xi=10**30; take density to be zero below here */
		xlowlimit=C1*expm1(C2*log(C3/1.E30))+C4;

		if(alphalarge){
			sinhalfangle=sin(.25*pi*twominusalpha);
			logscalef=log1p(-2*sinhalfangle*sinhalfangle)/alpha;
		} else logscalef=log(fabs(sinangle))/alpha;
	}
	sa2=twominusalpha/(2*alpha);
	Clogd=log(nu/sqrt(2*pi*alpha));
	if(alpha < 1)sinangle=sin(hpi*alpha); else
		sinangle=sin(hpi*twominusalpha);
	Calpha_M=exp(LogGamma(alpha))*sinangle/pi;

	interpolate_over_alpha(nx1,ny1,Vy1,alphastar,
		tablef1,tabled1,f1,d1,ydenom1);
	if(alphalarge){
		interpolate_over_alpha(nx4,ny4,Vy4,alpha,
			tablef4,tabled4,f4,d4,ydenom4);
		interpolate_over_alpha(nx5,ny5,Vy5,alpha,
			tablef5,tabled5,f5,d5,ydenom5);
		}
	else{
		interpolate_over_alpha(nx6,ny6,Vy6,alpha,
			tablef6,tabled6,f6,d6,ydenom6);
		interpolate_over_alpha(nx7,ny7,Vy7,alpha,
			tablef7,tabled7,f7,d7,ydenom7);
	}
}
}
/*========================================================================= */
void tailsMSS(int n,double x[],double d[],double logd[],double F[],
	double logF[],double cF[],double logcF[],
	double alpha,double oneminusalpha, double twominusalpha,
	double location,double logscale)

/*  Only need to return logd,F and cF. */
/*  For left-skewed, need to swap F and cF. */
{
/* Computes density, distribution function and complement for a maximally skew
	stable distribution skewed to the right */
/* When alpha < 0.5:
	MSS variable is exp(logscale)*(parametrization C standard)-location
	log MSS variable is exp{location-exp(logscale)*(parametrization C standard)} 
   When alpha >=0.5:
	MSS variable is exp(logscale)*(parametrization M=S0 standard)-location
	log MSS variable is exp{location-exp(logscale)*(parametrization M=S0 standard)}
   In both cases the log MSS variable = exp( - MSS variable) and
		MSS variable = -log( log MSS variable).		*/

static const double roothalf=.7071067811865475244;
static const double log_density_mode2=-1.2655121234846454;
double z,y,dy,difference,logz,t,temp,temp2,approx,scale;
int i;

/* If appropriate, set up for new alpha */
setalpha(alpha, oneminusalpha, twominusalpha);

/* Case when alpha is precisely 2 */
scale=exp(logscale);
if (twominusalpha == 0){
	for(i=0; i<n; i++){
		z=(x[i]-location)/scale;
		logd[i]=log_density_mode2-.25*z*z-logscale;
		d[i]=exp(logd[i]);
		if(z<0){
			F[i]=normaltail(-z*roothalf);
			cF[i]=1-F[i];
		}
		else{
			cF[i]=normaltail(z*roothalf);
		F[i]=1-cF[i];
		}
	}
}
/* Case when alpha is less than 0.5 */
else if(alphasmall){
    for(i=0; i<n; i++){
	difference=x[i]-location;
	if(difference <= 0){
		F[i]=0.;
		logF[i]=neglarge;
		cF[i]=1.;
		logcF[i]=0.;
		d[i]=0.;
		logd[i]=neglarge;
	}
	else{
		logz=log(difference)-logscale;

	 /*Case covered by table 1: small x */
		if(logz<logmidpoint){
		/*xi=absam1*(z/alpha)**(-a1) */
			xi=absam1*exp(-a1*(logz-log(alpha)));
			t=.2/(alpha*xi);
			interpolate(t,&ffound,&dfound,nx1,Vx1,f1,d1,xdenom1);
			logd[i]=Clogd+sa2*log(xi)-xi+log(dfound)-logscale;
			if(d[i]<logposlarge) d[i]=exp(logd[i]);
			else d[i]=poslarge;
			logF[i]=-.5*log(2*pi*alpha*xi)-xi+log(ffound);
			F[i]=exp(logF[i]);
			logcF[i]=log1p(-F[i]);
			cF[i]=1.-F[i];
		}

	 /*Case covered by table 2: small alpha, middle range for x */
		else if(logz<0.){
			temp2=expm1(-alpha*logmidpoint); /*independent of z */
			temp=exp(-alpha*logz);		/*= z**(-alpha) */
			t=expm1(-alpha*logz)/temp2;
			interpolate(t,&ffound,&dfound,nx2,Vx2,f2,d2,xdenom2);
			logF[i]=log(ffound*(1+EulersC*alpha*temp))-temp;
			F[i]=exp(logF[i]);
			logcF[i]=log1p(-F[i]);
			cF[i]=1-F[i];
			logd[i]=log(dfound*(1-EulersC*alpha+EulersC*alpha*temp)*alpha*temp)-temp-logz-logscale;
			d[i]=exp(logd[i]);
		}
	 /*Case covered by table 3: small alpha, upper range for x */
		else{
			t=exp(-alpha*logz);	/*= z**(-alpha) */
			interpolate(t,&ffound,&dfound,nx3,Vx3,f3,d3,xdenom3);
			approx=Calpha_C*t;
			cF[i]=ffound*approx;
			logcF[i]=log(cF[i]);
			F[i]=1-cF[i];
			logF[i]=log1p(-cF[i]);
			logd[i]=log(dfound*approx*alpha)-logz-logscale;
			d[i]=exp(logd[i]);
		}
	}
    }
}

/* Case when alpha is between 1.7 and 2 */
else if(alphalarge){
    scale=exp(logscale);
    for(i=0; i<n; i++){
	z=(x[i]-location)/scale;

/* Case when z is below limit where xi can be calculated */
	if(z<xlowlimit){
		F[i]=0.;
		logF[i]=neglarge;
		cF[i]=1.;
		logcF[i]=0.;
		d[i]=0.;
		logd[i]=neglarge;
	}
/* Case covered by table 1: low range for x */
	else if(z<midpoint){
		xi=C3/exp(log1p((z-C4)/C1)/C2);
		t=.2/(alphastar*xi);
		interpolate(t,&ffound,&dfound,nx1,Vx1,f1,d1,xdenom1);
		logd[i]=Clogd+sa2*log(xi)-xi+log(dfound)-logscale+logscalef;
		d[i]=exp(logd[i]);
		logF[i]=-.5*log(2*pi*alpha*xi)-xi+log(ffound);
		F[i]=exp(logF[i]);
		logcF[i]=log1p(-F[i]);
		cF[i]=1.-F[i];
	}

/* Case covered by table 4: large alpha, middle range for x */
	else if(z<20.){
		interpolate(z,&ffound,&dfound,nx4,Vx4,f4,d4,xdenom4);
		cF[i]=normaltail(z*sqrt(.5))+ffound*twominusalpha;
		logcF[i]=log(cF[i]);
		F[i]=1.-cF[i];
		logF[i]=log1p(-cF[i]);
/* 		d[i]=exp(log_density_mode2-.25*z*z-logscale)+dfound*(2-alpha); */
		d[i]=(exp(log_density_mode2-.25*z*z)+dfound*twominusalpha)/scale;
		logd[i]=log(d[i]);
	}

/* Case covered by table 5: large alpha, upper range for x */
	else{
/* Find y such that x=y+eta*(y**(1-alpha)-1) */
		y=z;
/* A good first approximation */
		do{
			dy=(z-y-expm1(oneminusalpha*log(y))*eta)/
				(1+eta*oneminusalpha*pow(y,(-alpha)));
			y=y+dy;
		} while(fabs(dy)>1.e-10*y);
		t=pow((y/19.5),(-alpha));
		interpolate(t,&ffound,&dfound,nx5,Vx5,f5,d5,xdenom5);
		logapprox=log(2*Calpha_M)-alpha*log(y);
		logcF[i]=logapprox+log(ffound);
		cF[i]=exp(logcF[i]);
		F[i]=1.-cF[i];
		logF[i]=log1p(-cF[i]);
		logd[i]=logapprox-logscale+log(alpha*dfound/y);
		d[i]=exp(logd[i]);
	}
    }
}
/*======================================== */
/*Case when alpha is between 0.5 and 1.7 */
else{
    scale=exp(logscale);
    for(i=0; i<n; i++){
	z=(x[i]-location)/scale;

/* Case when z is below limit where xi can be calculated */
	if(z<xlowlimit){
		F[i]=0.;
		logF[i]=neglarge;
		cF[i]=1.;
		logcF[i]=0.;
		d[i]=0.;
		logd[i]=neglarge;
	}
/* Case covered by table 1: low range for x */
	else if(z<midpoint){
		if(alpha == 1) xi=exp(-1-hpi*z)/hpi;
		else xi=C3/exp(log1p((z-C4)/C1)/C2);
		t=.2/(alphastar*xi);
		interpolate(t,&ffound,&dfound,nx1,Vx1,f1,d1,xdenom1);
		logd[i]=Clogd+sa2*log(xi)-xi+log(dfound)-logscale+logscalef;
		d[i]=exp(logd[i]);
		logF[i]=-.5*log(2*pi*alpha*xi)-xi+log(ffound);
		F[i]=exp(logF[i]);
		logcF[i]=log1p(-F[i]);
		cF[i]=1.-F[i];
	}

/* Case covered by table 7: middle range for alpha, middle range for x */
	else if(z<7.3){
		t=(z-midpoint)/(7.3-midpoint);
		interpolate(t,&ffound,&dfound,nx7,Vx7,f7,d7,xdenom7);
		logcF[i]=ffound;
		cF[i]=exp(ffound);
		F[i]=1.-cF[i];
		logF[i]=log1p(-cF[i]);
		logd[i]=dfound-logscale;
		d[i]=exp(logd[i]);
	}

/* Case covered by table 6: middle range for alpha, upper range for x */
	else{
		if(alpha == 1){
			y=z;
			do{
				dy=(z-y-log(y)/hpi)/(1+1/(y*hpi));
				y=y+dy;
			}
			while(fabs(dy)>1.e-10*y);
		}
		else{
		/* Find y such that x=y+eta*(y**oneminusalpha-1) */
		/*=y+expm1(oneminusalpha*log(y))/tan(hpi*oneminusalpha) */
			temp=1/tan(hpi*oneminusalpha);
			y=z;
/* A good first approximation */
			do{
				dy=(z-y-expm1(oneminusalpha*log(y))*eta)/(1+eta*oneminusalpha*pow(y,(-alpha)));
				y=y+dy;
			}
			while(fabs(dy)>1.e-10*y);
		}
		t=pow((y/5.),(-alpha));
		interpolate(t,&ffound,&dfound,nx6,Vx6,f6,d6,xdenom6);
		logapprox=log(2*Calpha_M)-alpha*log(y);
		logcF[i]=logapprox+log(ffound);
		cF[i]=exp(logcF[i]);
		F[i]=1.-cF[i];
		logF[i]=log1p(-cF[i]);
		logd[i]=logapprox-logscale+log(alpha*dfound/y);
		d[i]=exp(logd[i]);
	}
    }
}
}
/*====================================================================== */
void tailslogMSS(int n,double x[],double d[],double F[],double cF[],
	double logd[],double logF[],double logcF[],
	double alpha,double oneminusalpha, double twominusalpha,
	double location,double logscale)
/* Computes density, distribution function and complement for a
	log maximally skew stable distribution */
{
static const double neglarge=-1.7E308;
int i;
double *nlogx;
nlogx = (double *)R_alloc(n, sizeof(double));

for (i=0; i<n; i++)  nlogx[i]=-log(x[i]); 
tailsMSS(n,nlogx,d,logd,cF,logcF,F,logF,alpha,oneminusalpha, twominusalpha,
		location,logscale);
for (i=0; i<n; i++) if(x[i] > 0){
	logd[i] += nlogx[i];
	d[i] = exp(logd[i]);
	/*Perhaps more accurate sometimes?:	d[i]=d[i]/x[i];  */
} else {
	logd[i] = neglarge;
	logF[i] = neglarge;
	d[i] = 0;
	F[i] = 0;
	cF[i] = 1;
	logcF[i] = 0;
}
}
/*====================================================================== */
void RtailsMSS(double *Ralpha,double *Roneminusalpha,
	double *Rtwominusalpha,double *Rlocation,double *Rlogscale,
	double *Rn,double *x,double *d,double *logd,double *F,
	double *logF,double *cF,double *logcF)
/* Computes density, distribution function and complement for a maximally skew
	stable distribution skewed to the right */
{
int n;
double alpha,oneminusalpha, twominusalpha,location,logscale;
n = *Rn;
alpha = *Ralpha;
oneminusalpha = *Roneminusalpha;
twominusalpha = *Rtwominusalpha;
location = *Rlocation;
logscale = *Rlogscale;
tailsMSS(n,x,d,logd,F,logF,cF,logcF,alpha,oneminusalpha, twominusalpha,
	location,logscale);
}
/*====================================================================== */
void RtailslogMSS(double *Ralpha,double *Roneminusalpha,
	double *Rtwominusalpha,double *Rlocation,double *Rlogscale,
	double *Rn,double *x,double *d, double *logd,double *F,double *logF,
	double *cF,double *logcF)
/* Computes density, distribution function and complement for a
	log maximally skew stable distribution */
{
int n;
double alpha,oneminusalpha, twominusalpha,location,logscale;
n = *Rn;
alpha = *Ralpha;
oneminusalpha = *Roneminusalpha;
twominusalpha = *Rtwominusalpha;
location = *Rlocation;
logscale = *Rlogscale;
tailslogMSS(n,x,d,F,cF,logd,logF,logcF,alpha,oneminusalpha, twominusalpha,
		location,logscale);
}
