% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dbHasCompleted.R
\name{dbHasCompleted}
\alias{dbHasCompleted}
\title{Completion status}
\usage{
dbHasCompleted(res, ...)
}
\arguments{
\item{res}{An object inheriting from \linkS4class{DBIResult}.}

\item{...}{Other arguments passed on to methods.}
}
\value{
\code{dbHasCompleted()} returns a logical scalar.
For a query initiated by \code{\link[DBI:dbSendQuery]{DBI::dbSendQuery()}} with non-empty result set,
\code{dbHasCompleted()} returns \code{FALSE} initially
and \code{TRUE} after calling \code{\link[DBI:dbFetch]{DBI::dbFetch()}} without limit.
For a query initiated by \code{\link[DBI:dbSendStatement]{DBI::dbSendStatement()}},
\code{dbHasCompleted()} always returns \code{TRUE}.
}
\description{
This method returns if the operation has completed.
A \code{SELECT} query is completed if all rows have been fetched.
A data manipulation statement is always completed.

\Sexpr[results=rd,stage=render]{DBI:::methods_as_rd("dbHasCompleted")}
}
\section{The data retrieval flow}{


This section gives a complete overview over the flow
for the execution of queries that return tabular data as data frames.

Most of this flow, except repeated calling of \code{\link[=dbBind]{dbBind()}} or \code{\link[=dbBindArrow]{dbBindArrow()}},
is implemented by \code{\link[=dbGetQuery]{dbGetQuery()}}, which should be sufficient
unless you want to access the results in a paged way
or you have a parameterized query that you want to reuse.
This flow requires an active connection established by \code{\link[=dbConnect]{dbConnect()}}.
See also \code{vignette("dbi-advanced")} for a walkthrough.
\enumerate{
\item Use \code{\link[=dbSendQuery]{dbSendQuery()}} to create a result set object of class
\linkS4class{DBIResult}.
\item Optionally, bind query parameters with \code{\link[=dbBind]{dbBind()}} or \code{\link[=dbBindArrow]{dbBindArrow()}}.
This is required only if the query contains placeholders
such as \verb{?} or \verb{$1}, depending on the database backend.
\item Optionally, use \code{\link[=dbColumnInfo]{dbColumnInfo()}} to retrieve the structure of the result set
without retrieving actual data.
\item Use \code{\link[=dbFetch]{dbFetch()}} to get the entire result set, a page of results,
or the remaining rows.
Fetching zero rows is also possible to retrieve the structure of the result set
as a data frame.
This step can be called multiple times.
Only forward paging is supported, you need to cache previous pages
if you need to navigate backwards.
\item Use \code{\link[=dbHasCompleted]{dbHasCompleted()}} to tell when you're done.
This method returns \code{TRUE} if no more rows are available for fetching.
\item Repeat the last four steps as necessary.
\item Use \code{\link[=dbClearResult]{dbClearResult()}} to clean up the result set object.
This step is mandatory even if no rows have been fetched
or if an error has occurred during the processing.
It is good practice to use \code{\link[=on.exit]{on.exit()}} or \code{\link[withr:defer]{withr::defer()}}
to ensure that this step is always executed.
}
}

\section{Failure modes}{


Attempting to query completion status for a result set cleared with
\code{\link[DBI:dbClearResult]{DBI::dbClearResult()}} gives an error.

}

\section{Specification}{


The completion status for a query is only guaranteed to be set to
\code{FALSE} after attempting to fetch past the end of the entire result.
Therefore, for a query with an empty result set,
the initial return value is unspecified,
but the result value is \code{TRUE} after trying to fetch only one row.


Similarly, for a query with a result set of length n,
the return value is unspecified after fetching n rows,
but the result value is \code{TRUE} after trying to fetch only one more
row.

}

\examples{
\dontshow{if (requireNamespace("RSQLite", quietly = TRUE)) withAutoprint(\{ # examplesIf}
con <- dbConnect(RSQLite::SQLite(), ":memory:")

dbWriteTable(con, "mtcars", mtcars)
rs <- dbSendQuery(con, "SELECT * FROM mtcars")

dbHasCompleted(rs)
ret1 <- dbFetch(rs, 10)
dbHasCompleted(rs)
ret2 <- dbFetch(rs)
dbHasCompleted(rs)

dbClearResult(rs)
dbDisconnect(con)
\dontshow{\}) # examplesIf}
}
\seealso{
Other DBIResult generics: 
\code{\link{DBIResult-class}},
\code{\link{dbBind}()},
\code{\link{dbClearResult}()},
\code{\link{dbColumnInfo}()},
\code{\link{dbFetch}()},
\code{\link{dbGetInfo}()},
\code{\link{dbGetRowCount}()},
\code{\link{dbGetRowsAffected}()},
\code{\link{dbGetStatement}()},
\code{\link{dbIsReadOnly}()},
\code{\link{dbIsValid}()},
\code{\link{dbQuoteLiteral}()},
\code{\link{dbQuoteString}()}

Other DBIResultArrow generics: 
\code{\link{DBIResultArrow-class}},
\code{\link{dbBind}()},
\code{\link{dbClearResult}()},
\code{\link{dbFetchArrow}()},
\code{\link{dbFetchArrowChunk}()},
\code{\link{dbIsValid}()}

Other data retrieval generics: 
\code{\link{dbBind}()},
\code{\link{dbClearResult}()},
\code{\link{dbFetch}()},
\code{\link{dbFetchArrow}()},
\code{\link{dbFetchArrowChunk}()},
\code{\link{dbGetQuery}()},
\code{\link{dbGetQueryArrow}()},
\code{\link{dbSendQuery}()},
\code{\link{dbSendQueryArrow}()}
}
\concept{DBIResult generics}
\concept{DBIResultArrow generics}
\concept{data retrieval generics}
