/*
 * file:       TurboProjectReader.java
 * author:     Jon Iles
 * copyright:  (c) Packwood Software 2018
 * date:       09/01/2018
 */

/*
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

package net.sf.mpxj.turboproject;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import net.sf.mpxj.ChildTaskContainer;
import net.sf.mpxj.CustomFieldContainer;
import net.sf.mpxj.Day;
import net.sf.mpxj.Duration;
import net.sf.mpxj.EventManager;
import net.sf.mpxj.FieldContainer;
import net.sf.mpxj.FieldType;
import net.sf.mpxj.MPXJException;
import net.sf.mpxj.ProjectCalendar;
import net.sf.mpxj.ProjectCalendarException;
import net.sf.mpxj.ProjectCalendarWeek;
import net.sf.mpxj.ProjectConfig;
import net.sf.mpxj.ProjectFile;
import net.sf.mpxj.Relation;
import net.sf.mpxj.RelationType;
import net.sf.mpxj.Resource;
import net.sf.mpxj.ResourceAssignment;
import net.sf.mpxj.ResourceField;
import net.sf.mpxj.Task;
import net.sf.mpxj.TaskField;
import net.sf.mpxj.common.StreamHelper;
import net.sf.mpxj.listener.ProjectListener;
import net.sf.mpxj.reader.AbstractProjectReader;

/**
 * This class creates a new ProjectFile instance by reading a TurboProject PEP file.
 */
public final class TurboProjectReader extends AbstractProjectReader
{
   /**
    * {@inheritDoc}
    */
   @Override public void addProjectListener(ProjectListener listener)
   {
      if (m_projectListeners == null)
      {
         m_projectListeners = new LinkedList<ProjectListener>();
      }
      m_projectListeners.add(listener);
   }

   /**
    * {@inheritDoc}
    */
   @Override public ProjectFile read(InputStream stream) throws MPXJException
   {
      try
      {
         m_projectFile = new ProjectFile();
         m_eventManager = m_projectFile.getEventManager();
         m_tables = new HashMap<String, Table>();

         ProjectConfig config = m_projectFile.getProjectConfig();
         config.setAutoResourceID(false);
         config.setAutoCalendarUniqueID(false);
         config.setAutoResourceUniqueID(false);
         config.setAutoTaskID(false);
         config.setAutoTaskUniqueID(false);
         config.setAutoOutlineLevel(true);
         config.setAutoOutlineNumber(true);
         config.setAutoWBS(true);

         m_projectFile.getProjectProperties().setFileApplication("TurboProject");
         m_projectFile.getProjectProperties().setFileType("PEP");

         m_eventManager.addProjectListeners(m_projectListeners);

         applyAliases();

         readFile(stream);
         readCalendars();
         readResources();
         readTasks();
         readRelationships();
         readResourceAssignments();

         //
         // Ensure that the unique ID counters are correct
         //
         config.updateUniqueCounters();

         return m_projectFile;
      }

      catch (IOException ex)
      {
         throw new MPXJException("Failed to parse file", ex);
      }

      finally
      {
         m_projectFile = null;
         m_eventManager = null;
         m_projectListeners = null;
         m_tables = null;
      }
   }

   /**
    * Reads a PEP file from the input stream.
    *
    * @param is input stream representing a PEP file
    */
   private void readFile(InputStream is) throws IOException
   {
      StreamHelper.skip(is, 64);
      int index = 64;

      ArrayList<Integer> offsetList = new ArrayList<Integer>();
      List<String> nameList = new ArrayList<String>();

      while (true)
      {
         byte[] table = new byte[32];
         is.read(table);
         index += 32;

         int offset = PEPUtility.getInt(table, 0);
         offsetList.add(Integer.valueOf(offset));
         if (offset == 0)
         {
            break;
         }

         nameList.add(PEPUtility.getString(table, 5).toUpperCase());
      }

      StreamHelper.skip(is, offsetList.get(0).intValue() - index);

      for (int offsetIndex = 1; offsetIndex < offsetList.size() - 1; offsetIndex++)
      {
         String name = nameList.get(offsetIndex - 1);
         Class<? extends Table> tableClass = TABLE_CLASSES.get(name);
         if (tableClass == null)
         {
            tableClass = Table.class;
         }

         Table table;
         try
         {
            table = tableClass.newInstance();
         }

         catch (Exception ex)
         {
            throw new RuntimeException(ex);
         }

         m_tables.put(name, table);
         table.read(is);
      }
   }

   /**
    * Read calendar data from a PEP file.
    */
   private void readCalendars()
   {
      //
      // Create the calendars
      //
      for (MapRow row : getTable("NCALTAB"))
      {
         ProjectCalendar calendar = m_projectFile.addCalendar();
         calendar.setUniqueID(row.getInteger("UNIQUE_ID"));
         calendar.setName(row.getString("NAME"));
         calendar.setWorkingDay(Day.SUNDAY, row.getBoolean("SUNDAY"));
         calendar.setWorkingDay(Day.MONDAY, row.getBoolean("MONDAY"));
         calendar.setWorkingDay(Day.TUESDAY, row.getBoolean("TUESDAY"));
         calendar.setWorkingDay(Day.WEDNESDAY, row.getBoolean("WEDNESDAY"));
         calendar.setWorkingDay(Day.THURSDAY, row.getBoolean("THURSDAY"));
         calendar.setWorkingDay(Day.FRIDAY, row.getBoolean("FRIDAY"));
         calendar.setWorkingDay(Day.SATURDAY, row.getBoolean("SATURDAY"));

         for (Day day : Day.values())
         {
            if (calendar.isWorkingDay(day))
            {
               // TODO: this is an approximation
               calendar.addDefaultCalendarHours(day);
            }
         }
      }

      //
      // Set up the hierarchy and add exceptions
      //
      Table exceptionsTable = getTable("CALXTAB");
      for (MapRow row : getTable("NCALTAB"))
      {
         ProjectCalendar child = m_projectFile.getCalendarByUniqueID(row.getInteger("UNIQUE_ID"));
         ProjectCalendar parent = m_projectFile.getCalendarByUniqueID(row.getInteger("BASE_CALENDAR_ID"));
         if (child != null && parent != null)
         {
            child.setParent(parent);
         }

         addCalendarExceptions(exceptionsTable, child, row.getInteger("FIRST_CALENDAR_EXCEPTION_ID"));

         m_eventManager.fireCalendarReadEvent(child);
      }
   }

   /**
    * Read exceptions for a calendar.
    *
    * @param table calendar exception data
    * @param calendar calendar
    * @param exceptionID first exception ID
    */
   private void addCalendarExceptions(Table table, ProjectCalendar calendar, Integer exceptionID)
   {
      Integer currentExceptionID = exceptionID;
      while (true)
      {
         MapRow row = table.find(currentExceptionID);
         if (row == null)
         {
            break;
         }

         Date date = row.getDate("DATE");
         ProjectCalendarException exception = calendar.addCalendarException(date, date);
         if (row.getBoolean("WORKING"))
         {
            exception.addRange(ProjectCalendarWeek.DEFAULT_WORKING_MORNING);
            exception.addRange(ProjectCalendarWeek.DEFAULT_WORKING_AFTERNOON);
         }

         currentExceptionID = row.getInteger("NEXT_CALENDAR_EXCEPTION_ID");
      }
   }

   /**
    * Read resource data from a PEP file.
    */
   private void readResources()
   {
      for (MapRow row : getTable("RTAB"))
      {
         Resource resource = m_projectFile.addResource();
         setFields(RESOURCE_FIELDS, row, resource);
         m_eventManager.fireResourceReadEvent(resource);
         // TODO: Correctly handle calendar
      }
   }

   /**
    * Read task data from a PEP file.
    */
   private void readTasks()
   {
      Integer rootID = Integer.valueOf(1);
      readWBS(m_projectFile, rootID);
      readTasks(rootID);
      m_projectFile.getTasks().synchronizeTaskIDToHierarchy();
   }

   /**
    * Recursively read the WBS structure from a PEP file.
    *
    * @param parent parent container for tasks
    * @param id initial WBS ID
    */
   private void readWBS(ChildTaskContainer parent, Integer id)
   {
      Integer currentID = id;
      Table table = getTable("WBSTAB");

      while (currentID.intValue() != 0)
      {
         MapRow row = table.find(currentID);
         Integer taskID = row.getInteger("TASK_ID");
         Task task = readTask(parent, taskID);
         Integer childID = row.getInteger("CHILD_ID");
         if (childID.intValue() != 0)
         {
            readWBS(task, childID);
         }
         currentID = row.getInteger("NEXT_ID");
      }
   }

   /**
    * Read leaf tasks attached to the WBS.
    *
    * @param id initial WBS ID
    */
   private void readTasks(Integer id)
   {
      Integer currentID = id;
      Table table = getTable("WBSTAB");

      while (currentID.intValue() != 0)
      {
         MapRow row = table.find(currentID);
         Task task = m_projectFile.getTaskByUniqueID(row.getInteger("TASK_ID"));
         readLeafTasks(task, row.getInteger("FIRST_CHILD_TASK_ID"));
         Integer childID = row.getInteger("CHILD_ID");
         if (childID.intValue() != 0)
         {
            readTasks(childID);
         }
         currentID = row.getInteger("NEXT_ID");
      }
   }

   /**
    * Read the leaf tasks for an individual WBS node.
    *
    * @param parent parent task
    * @param id first task ID
    */
   private void readLeafTasks(Task parent, Integer id)
   {
      Integer currentID = id;
      Table table = getTable("A1TAB");
      while (currentID.intValue() != 0)
      {
         if (m_projectFile.getTaskByUniqueID(currentID) == null)
         {
            readTask(parent, currentID);
         }
         currentID = table.find(currentID).getInteger("NEXT_TASK_ID");
      }
   }

   /**
    * Read data for an individual task from the tables in a PEP file.
    *
    * @param parent parent task
    * @param id task ID
    * @return task instance
    */
   private Task readTask(ChildTaskContainer parent, Integer id)
   {
      Table a0 = getTable("A0TAB");
      Table a1 = getTable("A1TAB");
      Table a2 = getTable("A2TAB");
      Table a3 = getTable("A3TAB");
      Table a4 = getTable("A4TAB");

      Task task = parent.addTask();
      MapRow a1Row = a1.find(id);
      MapRow a2Row = a2.find(id);

      setFields(A0TAB_FIELDS, a0.find(id), task);
      setFields(A1TAB_FIELDS, a1Row, task);
      setFields(A2TAB_FIELDS, a2Row, task);
      setFields(A3TAB_FIELDS, a3.find(id), task);
      setFields(A5TAB_FIELDS, a4.find(id), task);

      task.setStart(task.getEarlyStart());
      task.setFinish(task.getEarlyFinish());
      if (task.getName() == null)
      {
         task.setName(task.getText(1));
      }

      m_eventManager.fireTaskReadEvent(task);

      return task;
   }

   /**
    * Read relationship data from a PEP file.
    */
   private void readRelationships()
   {
      for (MapRow row : getTable("CONTAB"))
      {
         Task task1 = m_projectFile.getTaskByUniqueID(row.getInteger("TASK_ID_1"));
         Task task2 = m_projectFile.getTaskByUniqueID(row.getInteger("TASK_ID_2"));

         if (task1 != null && task2 != null)
         {
            RelationType type = row.getRelationType("TYPE");
            Duration lag = row.getDuration("LAG");
            Relation relation = task2.addPredecessor(task1, type, lag);
            m_eventManager.fireRelationReadEvent(relation);
         }
      }
   }

   /**
    * Read resource assignment data from a PEP file.
    */
   private void readResourceAssignments()
   {
      for (MapRow row : getTable("USGTAB"))
      {
         Task task = m_projectFile.getTaskByUniqueID(row.getInteger("TASK_ID"));
         Resource resource = m_projectFile.getResourceByUniqueID(row.getInteger("RESOURCE_ID"));
         if (task != null && resource != null)
         {
            ResourceAssignment assignment = task.addResourceAssignment(resource);
            m_eventManager.fireAssignmentReadEvent(assignment);
         }
      }
   }

   /**
    * Retrieve a table by name.
    *
    * @param name table name
    * @return Table instance
    */
   private Table getTable(String name)
   {
      Table table = m_tables.get(name);
      if (table == null)
      {
         table = EMPTY_TABLE;
      }
      return table;
   }

   /**
    * Configure column aliases.
    */
   private void applyAliases()
   {
      CustomFieldContainer fields = m_projectFile.getCustomFields();
      for (Map.Entry<FieldType, String> entry : ALIASES.entrySet())
      {
         fields.getCustomField(entry.getKey()).setAlias(entry.getValue());
      }
   }

   /**
    * Set the value of one or more fields based on the contents of a database row.
    *
    * @param map column to field map
    * @param row database row
    * @param container field container
    */
   private void setFields(Map<String, FieldType> map, MapRow row, FieldContainer container)
   {
      if (row != null)
      {
         for (Map.Entry<String, FieldType> entry : map.entrySet())
         {
            container.set(entry.getValue(), row.getObject(entry.getKey()));
         }
      }
   }

   /**
    * Configure the mapping between a database column and a field.
    *
    * @param container column to field map
    * @param name column name
    * @param type field type
    */
   private static void defineField(Map<String, FieldType> container, String name, FieldType type)
   {
      defineField(container, name, type, null);
   }

   /**
    * Configure the mapping between a database column and a field, including definition of
    * an alias.
    *
    * @param container column to field map
    * @param name column name
    * @param type field type
    * @param alias field alias
    */
   private static void defineField(Map<String, FieldType> container, String name, FieldType type, String alias)
   {
      container.put(name, type);
      if (alias != null)
      {
         ALIASES.put(type, alias);
      }
   }

   private ProjectFile m_projectFile;
   private EventManager m_eventManager;
   private List<ProjectListener> m_projectListeners;
   private HashMap<String, Table> m_tables;

   private static final Table EMPTY_TABLE = new Table();

   private static final Map<String, Class<? extends Table>> TABLE_CLASSES = new HashMap<String, Class<? extends Table>>();
   static
   {
      TABLE_CLASSES.put("RTAB", TableRTAB.class);
      TABLE_CLASSES.put("A0TAB", TableA0TAB.class);
      TABLE_CLASSES.put("A1TAB", TableA1TAB.class);
      TABLE_CLASSES.put("A2TAB", TableA2TAB.class);
      TABLE_CLASSES.put("A3TAB", TableA3TAB.class);
      TABLE_CLASSES.put("A5TAB", TableA5TAB.class);
      TABLE_CLASSES.put("CONTAB", TableCONTAB.class);
      TABLE_CLASSES.put("USGTAB", TableUSGTAB.class);
      TABLE_CLASSES.put("NCALTAB", TableNCALTAB.class);
      TABLE_CLASSES.put("CALXTAB", TableCALXTAB.class);
      TABLE_CLASSES.put("WBSTAB", TableWBSTAB.class);
   }

   private static final Map<FieldType, String> ALIASES = new HashMap<FieldType, String>();
   private static final Map<String, FieldType> RESOURCE_FIELDS = new HashMap<String, FieldType>();
   private static final Map<String, FieldType> A0TAB_FIELDS = new HashMap<String, FieldType>();
   private static final Map<String, FieldType> A1TAB_FIELDS = new HashMap<String, FieldType>();
   private static final Map<String, FieldType> A2TAB_FIELDS = new HashMap<String, FieldType>();
   private static final Map<String, FieldType> A3TAB_FIELDS = new HashMap<String, FieldType>();
   private static final Map<String, FieldType> A5TAB_FIELDS = new HashMap<String, FieldType>();

   static
   {
      defineField(RESOURCE_FIELDS, "ID", ResourceField.ID);
      defineField(RESOURCE_FIELDS, "UNIQUE_ID", ResourceField.UNIQUE_ID);
      defineField(RESOURCE_FIELDS, "NAME", ResourceField.NAME);
      defineField(RESOURCE_FIELDS, "GROUP", ResourceField.GROUP);
      defineField(RESOURCE_FIELDS, "DESCRIPTION", ResourceField.NOTES);
      defineField(RESOURCE_FIELDS, "PARENT_ID", ResourceField.PARENT_ID);

      defineField(RESOURCE_FIELDS, "RATE", ResourceField.NUMBER1, "Rate");
      defineField(RESOURCE_FIELDS, "POOL", ResourceField.NUMBER2, "Pool");
      defineField(RESOURCE_FIELDS, "PER_DAY", ResourceField.NUMBER3, "Per Day");
      defineField(RESOURCE_FIELDS, "PRIORITY", ResourceField.NUMBER4, "Priority");
      defineField(RESOURCE_FIELDS, "PERIOD_DUR", ResourceField.NUMBER5, "Period Dur");
      defineField(RESOURCE_FIELDS, "EXPENSES_ONLY", ResourceField.FLAG1, "Expenses Only");
      defineField(RESOURCE_FIELDS, "MODIFY_ON_INTEGRATE", ResourceField.FLAG2, "Modify On Integrate");
      defineField(RESOURCE_FIELDS, "UNIT", ResourceField.TEXT1, "Unit");

      defineField(A0TAB_FIELDS, "UNIQUE_ID", TaskField.UNIQUE_ID);

      defineField(A1TAB_FIELDS, "ORDER", TaskField.ID);
      defineField(A1TAB_FIELDS, "PLANNED_START", TaskField.BASELINE_START);
      defineField(A1TAB_FIELDS, "PLANNED_FINISH", TaskField.BASELINE_FINISH);

      defineField(A2TAB_FIELDS, "DESCRIPTION", TaskField.TEXT1, "Description");

      defineField(A3TAB_FIELDS, "EARLY_START", TaskField.EARLY_START);
      defineField(A3TAB_FIELDS, "LATE_START", TaskField.LATE_START);
      defineField(A3TAB_FIELDS, "EARLY_FINISH", TaskField.EARLY_FINISH);
      defineField(A3TAB_FIELDS, "LATE_FINISH", TaskField.LATE_FINISH);

      defineField(A5TAB_FIELDS, "ORIGINAL_DURATION", TaskField.DURATION);
      defineField(A5TAB_FIELDS, "REMAINING_DURATION", TaskField.REMAINING_DURATION);
      defineField(A5TAB_FIELDS, "PERCENT_COMPLETE", TaskField.PERCENT_COMPLETE);
      defineField(A5TAB_FIELDS, "TARGET_START", TaskField.DATE1, "Target Start");
      defineField(A5TAB_FIELDS, "TARGET_FINISH", TaskField.DATE2, "Target Finish");
      defineField(A5TAB_FIELDS, "ACTUAL_START", TaskField.ACTUAL_START);
      defineField(A5TAB_FIELDS, "ACTUAL_FINISH", TaskField.ACTUAL_FINISH);
   }
}
