/* Copyright (C) 2000-2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   There are special exceptions to the terms and conditions of the GPL as it
   is applied to this software. View the full text of the exception in file
   EXCEPTIONS in the directory of this software distribution.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MYODBCSetupDataSourceDialog.h"

#include "MySQL-16.xpm"
#include "TopImage.xpm"


MYODBCSetupDataSourceDialog::MYODBCSetupDataSourceDialog( QWidget *pwidgetParent, MYODBCUTIL_DATASOURCE *pDataSource )
    : QDialog( pwidgetParent, "MYODBCSetupDataSourceDialog", true, Qt::WDestructiveClose )
{
    this->hDBC              = NULL;
    this->pDataSource       = pDataSource;

    doInit();
}

MYODBCSetupDataSourceDialog::MYODBCSetupDataSourceDialog( QWidget *pwidgetParent, SQLHDBC hDBC, MYODBCUTIL_DATASOURCE *pDataSource )
    : QDialog( pwidgetParent, "MYODBCSetupDataSourceDialog", true, Qt::WDestructiveClose )
{
    this->hDBC              = hDBC;
    this->pDataSource       = pDataSource;

    doInit();
}

MYODBCSetupDataSourceDialog::~MYODBCSetupDataSourceDialog()
{
    // Are we getting deleted - is our destructor getting executed?
    // printf( "[PAH][%s][%d]\n", __FILE__, __LINE__ );
}

// toggle display of diagnostic info
void MYODBCSetupDataSourceDialog::slotDiagnostics()
{
}

// invoke browser with online help
void MYODBCSetupDataSourceDialog::slotHelp()
{
    QProcess  * pprocess    = new QProcess( this );
    QString     stringURL   = "http://dev.mysql.com/doc/mysql/en/ODBC_Connector.html";

#ifdef Q_WS_WIN
    pprocess->addArgument( "c:\\program files\\internet explorer\\IEXPLORE.EXE" );
    pprocess->addArgument( stringURL );
    if ( !pprocess->start() )
    {
        QMessageBox::warning( this, "MyODBC Help", "Failed to execute a browser.\n\nPlease visit\n" + stringURL, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
        delete pprocess;
    }
#else
#ifdef Q_WS_MACX
    pprocess->addArgument( "open" );
//    pprocess->addArgument( "/Applications/Safari.app" );
    pprocess->addArgument( stringURL );
    if ( !pprocess->start() )
    {
        QMessageBox::warning( this, "MyODBC Help", "Failed to execute a browser.\n\nPlease visit\n" + stringURL, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
        delete pprocess;
    }
#else
    pprocess->addArgument( "netscape" );            // generic
    pprocess->addArgument( stringURL );
    if ( !pprocess->start() )
    {
        delete pprocess;
        pprocess = new QProcess( this );
        pprocess->addArgument( "konqueror" );       // KDE folks
        pprocess->addArgument( stringURL );
        if ( !pprocess->start() )
        {
            delete pprocess;
            pprocess = new QProcess( this );
            pprocess->addArgument( "htmlview" );    // GNOME folks
            pprocess->addArgument( stringURL );
            if ( !pprocess->start() )
            {
                QMessageBox::warning( this, "MyODBC Help", "Failed to execute a browser.\n\nPlease visit\n" + stringURL, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
                delete pprocess;
            }
        }
    }
#endif
#endif

}

// try to connect/disconnect
void MYODBCSetupDataSourceDialog::slotTest()
{
    if ( pDataSource->nMode == MYODBCUTIL_DATASOURCE_MODE_DRIVER_CONNECT )
        doTestUsingDriver();
    else
        doTestUsingDriverManager();
}

// save dsn info
void MYODBCSetupDataSourceDialog::slotOk()
{
    switch ( pDataSource->nMode )
    {
        case MYODBCUTIL_DATASOURCE_MODE_DSN_ADD:
            if ( ptab1->getDataSourceName().isEmpty() )
            {
                ptabwidget->setCurrentPage( 0 );
                ptab1->plineeditDataSourceName->setFocus();
                QMessageBox::warning( this, "MyODBC", tr("Missing Data Source Name"), tr("&Ok"), QString::null, QString::null, 0, 1 );
                return;
            }
            if ( ptab1->getUser().isEmpty() )
            {
                ptabwidget->setCurrentPage( 0 );
                ptab1->plineeditUser->setFocus();
                QMessageBox::warning( this, "MyODBC", tr("Missing User ID"), tr("&Ok"), QString::null, QString::null, 0, 1 );
                return;
            }

            /*!
                ODBC RULE
        
                If the data source name matches an existing data source name and hwndParent is null, 
                ConfigDSN overwrites the existing name. If it matches an existing name and hwndParent 
                is not null, ConfigDSN prompts the user to overwrite the existing name.        
            */
            if ( MYODBCUtilDSNExists( (char *)(ptab1->getDataSourceName().latin1()) ) )
            {
                int n = QMessageBox::warning( this, "MyODBC", tr("Data Source Name (%1) exists. It will be replaced?") .arg( ptab1->getDataSourceName() ), tr("&Ok"), tr("&Cancel"), QString::null, 0, 1 );
                if ( n )
                    return;
            }
            break;

        case MYODBCUTIL_DATASOURCE_MODE_DSN_EDIT:
            if ( ptab1->getDataSourceName().isEmpty() )
            {
                ptabwidget->setCurrentPage( 0 );
                ptab1->plineeditDataSourceName->setFocus();
                QMessageBox::warning( this, "MyODBC", tr("Missing Data Source Name"), tr("&Ok"), QString::null, QString::null, 0, 1 );
                return;
            }
            break;

        case MYODBCUTIL_DATASOURCE_MODE_DSN_VIEW:
            break;

        case MYODBCUTIL_DATASOURCE_MODE_DRIVER_CONNECT:
            /*
                MYODBC RULE

                A UID is really the only thing we MUST have for a driver connect.
            */                
            if ( ptab1->getUser().isEmpty() )
            {
                ptabwidget->setCurrentPage( 0 );
                ptab1->plineeditUser->setFocus();
                QMessageBox::warning( this, "MyODBC", tr("Missing User ID"), tr("&Ok"), QString::null, QString::null, 0, 1 );
                return;
            }
            break;

        default:
            break;
    }

    if ( pDataSource->nMode != MYODBCUTIL_DATASOURCE_MODE_DSN_VIEW )
    {
        // update MYODBCUTIL_DATASOURCE here
        MYODBCUtilClearDataSource( pDataSource );

        if ( !ptab1->getDataSourceName().isEmpty() )
            pDataSource->pszDSN = strdup( ptab1->getDataSourceName().latin1() );
        if ( !ptab1->getDescription().isEmpty() )
            pDataSource->pszDESCRIPTION = strdup( ptab1->getDescription().latin1() );

        if ( !ptab1->getServer().isEmpty() )
            pDataSource->pszSERVER = strdup( ptab1->getServer().latin1() );
        if ( !ptab1->getUser().isEmpty() )
            pDataSource->pszUSER = strdup( ptab1->getUser().latin1() );
        if ( !ptab1->getPassword().isEmpty() )
            pDataSource->pszPASSWORD = strdup( ptab1->getPassword().latin1() );
        if ( !ptab1->getDatabase().isEmpty() )
            pDataSource->pszDATABASE = strdup( ptab1->getDatabase().latin1() );
        if ( !ptab2->getPort().isEmpty() )
            pDataSource->pszPORT = strdup( ptab2->getPort().latin1() );
        if ( !ptab2->getSocket().isEmpty() )
            pDataSource->pszSOCKET = strdup( ptab2->getSocket().latin1() );
        if ( !ptab2->getInitialStatement().isEmpty() )
            pDataSource->pszSTMT = strdup( ptab2->getInitialStatement() );

        unsigned int nFlags = ptab3->getFlags();

        if ( nFlags > 0 )
        {
            pDataSource->pszOPTION = (char*)malloc( 50 );
            sprintf( pDataSource->pszOPTION, "%d", nFlags );
        }
    }

    // exit 
    done( QDialog::Accepted );
}

void MYODBCSetupDataSourceDialog::doInit()
{
    setIcon( QPixmap( MySQL_16_xpm ) );

    playoutTop = new QVBoxLayout( this );
    playoutTop->setMargin( 5 );
    playoutTop->setSpacing( 5 );

    playoutLabels = new QHBoxLayout( playoutTop );
    playoutLabels->setMargin( 0 );
    playoutLabels->setSpacing( 0 );

    // labels (text/image)
    plabelText = new QLabel( "", this );
    plabelText->setPaletteBackgroundColor( Qt::white );
    plabelText->setAlignment( Qt::AlignHCenter | Qt::AlignVCenter );
    playoutLabels->addWidget( plabelText );
    playoutLabels->setStretchFactor( plabelText, 10 );

    plabelImage = new QLabel( this );
    plabelImage->setBackgroundPixmap( TopImage_xpm );
    plabelImage->setMinimumSize( 500, 63 );
    plabelImage->setMaximumSize( 500, 63 );
    QToolTip::add( plabelImage, "Brought to you by the database elves." );
    playoutLabels->addWidget( plabelImage );

    // splitter
    psplitter = new QSplitter( this );
    playoutTop->addWidget( psplitter );

    // tab widget in the middle
    ptabwidget = new QTabWidget( psplitter );

    // text browser (assist) to the right)
    ptextbrowserAssist = new QTextBrowser( psplitter );

    // tabs
    ptab1 = new MYODBCSetupDataSourceTab1( ptabwidget );
    ptabwidget->addTab( ptab1, "Login" ); 
    ptab2 = new MYODBCSetupDataSourceTab2( ptabwidget );
    ptabwidget->addTab( ptab2, "Connect Options" ); 
    ptab3 = new MYODBCSetupDataSourceTab3( ptabwidget );
    ptabwidget->addTab( ptab3, "Advanced" ); 

    // buttons near bottom
    playoutButtons = new QHBoxLayout( playoutTop );
    playoutButtons->addStretch( 10 );
#ifndef __SPARC__
    ppushbuttonTest = new QPushButton( "&Test", this );
    playoutButtons->addWidget( ppushbuttonTest );
#endif
    ppushbuttonGuru = new QPushButton( "&Guru", this );
    ppushbuttonGuru->setToggleButton( TRUE );
    playoutButtons->addWidget( ppushbuttonGuru );
    ppushbuttonHelp = new QPushButton( "&Help", this );
    playoutButtons->addWidget( ppushbuttonHelp );
    playoutButtons->addStretch( 2 );
    ppushbuttonOk = new QPushButton( "&Ok", this );
    playoutButtons->addWidget( ppushbuttonOk );
    ppushbuttonCancel = new QPushButton( "&Cancel", this );
    playoutButtons->addWidget( ppushbuttonCancel );

    // lets hide this until we have completed the feature
    ppushbuttonGuru->hide();

    // diagnostics along bottom
    ptexteditDiagnostics = new QTextEdit( this );
    ptexteditDiagnostics->hide();
    playoutTop->addWidget( ptexteditDiagnostics );

    // set values
    ptab1->setDatabase( pDataSource->pszDATABASE );
    ptab1->setDescription( pDataSource->pszDESCRIPTION );
    ptab1->setDataSourceName( pDataSource->pszDSN );
    ptab1->setPassword( pDataSource->pszPASSWORD );
    ptab2->setPort( pDataSource->pszPORT );
    ptab1->setServer( pDataSource->pszSERVER );
    ptab2->setSocket( pDataSource->pszSOCKET );
    ptab2->setInitialStatement( pDataSource->pszSTMT );
    ptab1->setUser( pDataSource->pszUSER );

    if ( pDataSource->pszOPTION )
    {
        ulong nOptions = atol( pDataSource->pszOPTION );
        ptab3->ptab3a->pcheckboxDontOptimizeColumnWidth->setChecked( nOptions & (1 << 0) ? TRUE : FALSE );
        ptab3->ptab3a->pcheckboxReturnMatchingRows->setChecked( nOptions & (1 << 1) ? TRUE : FALSE );
        ptab3->ptab3d->pcheckboxTraceDriverCalls->setChecked( nOptions & (1 << 2) ? TRUE : FALSE );
        ptab3->ptab3a->pcheckboxAllowBigResults->setChecked( nOptions & (1 << 3) ? TRUE : FALSE );
        ptab3->ptab3b->pcheckboxDontPromptOnConnect->setChecked( nOptions & (1 << 4) ? TRUE : FALSE );
        ptab3->ptab3b->pcheckboxEnableDynamicCursor->setChecked( nOptions & (1 << 5) ? TRUE : FALSE );
        ptab3->ptab3b->pcheckboxIgnorePoundInTable->setChecked( nOptions & (1 << 6) ? TRUE : FALSE );
        ptab3->ptab3b->pcheckboxUseManagerCursors->setChecked( nOptions & (1 << 7) ? TRUE : FALSE );
        ptab3->ptab3b->pcheckboxDontUseSetLocale->setChecked( nOptions & (1 << 8) ? TRUE : FALSE );
        ptab3->ptab3b->pcheckboxPadCharToFullLen->setChecked( nOptions & (1 << 9) ? TRUE : FALSE );
        ptab3->ptab3c->pcheckboxReturnTableNamesSQLDescribeCol->setChecked( nOptions & (1 << 10) ? TRUE : FALSE );
        ptab3->ptab3a->pcheckboxUseCompressedProtocol->setChecked( nOptions & (1 << 11) ? TRUE : FALSE );
        ptab3->ptab3c->pcheckboxIgnoreSpaceAfterFunctionNames->setChecked( nOptions & (1 << 12) ? TRUE : FALSE ); 
        ptab3->ptab3c->pcheckboxForceUseOfNamedPipes->setChecked( nOptions & (1 << 13) ? TRUE : FALSE );          
        ptab3->ptab3a->pcheckboxChangeBIGINTColumnsToInt->setChecked( nOptions & (1 << 14) ? TRUE : FALSE );
        ptab3->ptab3c->pcheckboxNoCatalog->setChecked( nOptions & (1 << 15) ? TRUE : FALSE );
        ptab3->ptab3c->pcheckboxReadOptionsFromMyCnf->setChecked( nOptions & (1 << 16) ? TRUE : FALSE );          
        ptab3->ptab3a->pcheckboxSafe->setChecked( nOptions & (1 << 17) ? TRUE : FALSE );
        ptab3->ptab3c->pcheckboxDisableTransactions->setChecked( nOptions & (1 << 18) ? TRUE : FALSE );           
        ptab3->ptab3d->pcheckboxSaveQueries->setChecked( nOptions & (1 << 19) ? TRUE : FALSE );
        ptab3->ptab3b->pcheckboxDontCacheResults->setChecked( nOptions & (1 << 20) ? TRUE : FALSE );
        ptab3->ptab3c->pcheckboxForceUseOfForwardOnlyCursors->setChecked( nOptions & (1 << 21) ? TRUE : FALSE );  
    }

#ifndef __SPARC__
    connect( ppushbuttonTest, SIGNAL(clicked()), SLOT(slotTest()) );
#endif
    connect( ppushbuttonGuru, SIGNAL(toggled(bool)), SLOT(slotToggleGuru(bool)) );
    connect( ppushbuttonHelp, SIGNAL(clicked()), SLOT(slotHelp()) );
    connect( ppushbuttonOk, SIGNAL(clicked()), SLOT(slotOk()) );
    connect( ppushbuttonCancel, SIGNAL(clicked()), SLOT(reject()) );

#ifndef __SPARC__
    connect( ptab1, SIGNAL(signalRequestDatabaseNames()), SLOT(slotLoadDatabaseNames()) );
#endif

    connect( ptab1->plineeditDataSourceName, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab1->plineeditDescription, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab1->plineeditServer, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab1->plineeditUser, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab1->plineeditPassword, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab1->pcomboboxDatabase, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab2->plineeditPort, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab2->plineeditSocket, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab2->plineeditInitialStatement, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3a->pcheckboxDontOptimizeColumnWidth, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3a->pcheckboxReturnMatchingRows, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3a->pcheckboxAllowBigResults, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3a->pcheckboxUseCompressedProtocol, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3a->pcheckboxChangeBIGINTColumnsToInt, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3a->pcheckboxSafe, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3b->pcheckboxDontPromptOnConnect, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3b->pcheckboxEnableDynamicCursor, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3b->pcheckboxIgnorePoundInTable, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3b->pcheckboxUseManagerCursors, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3b->pcheckboxDontUseSetLocale, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3b->pcheckboxPadCharToFullLen, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3b->pcheckboxDontCacheResults, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3c->pcheckboxReturnTableNamesSQLDescribeCol, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3c->pcheckboxIgnoreSpaceAfterFunctionNames, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3c->pcheckboxForceUseOfNamedPipes, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3c->pcheckboxNoCatalog, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3c->pcheckboxReadOptionsFromMyCnf, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3c->pcheckboxDisableTransactions, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3c->pcheckboxForceUseOfForwardOnlyCursors, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3d->pcheckboxTraceDriverCalls, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );
    connect( ptab3->ptab3d->pcheckboxSaveQueries, SIGNAL(signalAssistText(const QString&)), ptextbrowserAssist, SLOT(setText(const QString&)) );

    doApplyMode();
}

void MYODBCSetupDataSourceDialog::doApplyMode()
{
    switch ( pDataSource->nMode )
    {
        case MYODBCUTIL_DATASOURCE_MODE_DSN_EDIT:
            setCaption( "Connector/ODBC - Configure Data Source Name" );
            break;
        case MYODBCUTIL_DATASOURCE_MODE_DSN_ADD:
            setCaption( "Connector/ODBC - Add Data Source Name" );
            break;
        case MYODBCUTIL_DATASOURCE_MODE_DSN_VIEW:
            setCaption( "Connector/ODBC - View Data Source Name" );
            break;
        case MYODBCUTIL_DATASOURCE_MODE_DRIVER_CONNECT:
            setCaption( "Connector/ODBC - Driver Connect" );
            ppushbuttonTest->hide();
            switch ( pDataSource->nPrompt )
            {
                case MYODBCUTIL_DATASOURCE_PROMPT_COMPLETE:
                    /*
                        ODBC RULE

                        We assume caller has already tried to connect and failed. So we 
                        do same thing as PROMPT.
                    */    
                case MYODBCUTIL_DATASOURCE_PROMPT_PROMPT:
                    /* 
                        MYODBC RULE
                        
                        Hide DSN fields if we are doing a DSN-less connect. 
                    */
                    if ( pDataSource->nConnect != MYODBCUTIL_DATASOURCE_CONNECT_DSN )
                    {
                        ptab1->plabelDataSourceName->hide();
                        ptab1->plineeditDataSourceName->hide();

                        ptab1->plabelDescription->hide();
                        ptab1->plineeditDescription->hide();

                        ptab1->plineeditUser->setFocus();
                    }
                    break;
                case MYODBCUTIL_DATASOURCE_PROMPT_REQUIRED:
                    /*
                        ODBC RULE

                        We assume caller has already tried to connect and failed. So we 
                        prompt - but only with the required fields.
                    */    
                    if ( pDataSource->nConnect == MYODBCUTIL_DATASOURCE_CONNECT_DSN )
                    {
                        /*
                            MYODBC RULE

                            Disable everything except UID and PWD. The rest must be configured
                            in the DSN.
                        */    
                        ptab1->plineeditDataSourceName->setDisabled( true );
                        ptab1->plineeditDescription->setDisabled( true );
                        ptab1->plineeditServer->setDisabled( true );
                        ptab1->pcomboboxDatabase->setDisabled( true );
                        ptab2->setDisabled( true );
                        ptab3->setDisabled( true );
                    }
                    else
                    {
                        /*
                            MYODBC RULE

                            Hide DSN specific stuff. Disable server and database only
                            if provided. UID and PWD is enabled and remaining disabled. 
                        */                            
                        ptab1->plabelDataSourceName->hide();
                        ptab1->plineeditDataSourceName->hide();

                        ptab1->plabelDescription->hide();
                        ptab1->plineeditDescription->hide();

                        if ( pDataSource->pszSERVER )
                            ptab1->plineeditServer->setDisabled( true );
                        if ( pDataSource->pszDATABASE )
                            ptab1->pcomboboxDatabase->setDisabled( true );
                        ptab2->setDisabled( true );
                        ptab3->setDisabled( true );
                    }

                    ptab1->plineeditUser->setFocus();
                    break;
                case MYODBCUTIL_DATASOURCE_PROMPT_NOPROMPT:
                    /* this should not happen */
                    break;
            }
            break;
        default:
            setCaption( "Connector/ODBC - Unknown Mode" );
    }
}

/*!
    \brief  Try connect/disconnect.

            Here we try the connect/disconnect by calling into
            the driver manager. This is a good test and is used when
            we are working on behalf of the ODBC administrator but
            we can not do a test this way when we are being called by
            the driver because we do not want to create another
            ODBC environment within the same app context and we can
            not use an existing DBC (from SQLDriverConnect) because
            it would be the drivers DBC - not the DM's DBC.

    \sa     doTestUsingMySQL            
*/
BOOL MYODBCSetupDataSourceDialog::doTestUsingDriverManager()
{
    SQLHENV     hEnv        = SQL_NULL_HENV;
    SQLHDBC     hDbc        = SQL_NULL_HDBC;
    SQLRETURN   nReturn;
    QString     stringConnectIn;

    /* build connect string in */
#ifdef Q_WS_MACX
    /*!
        \note OSX

              Yet another place where iODBC fails to act like the ODBC
              specification. SQLDriverConnect() should accept the 
              driver *description* and should allow for the curly
              braces but fails on both accounts deciding instead to
              interpret the whole value as a driver file name.

              Its possible to do the xref here, for SQLDriverConnect,
              but due to time constraints the file is hard-coded.
    */ 
    stringConnectIn  = "Driver=/usr/lib/libmyodbc3.bundle";
#else
    stringConnectIn  = "DRIVER=" + QString( pDataSource->pszDRIVER );
#endif
    stringConnectIn += ";UID=" + ptab1->getUser();
    stringConnectIn += ";PWD=" + ptab1->getPassword();
    stringConnectIn += ";SERVER=" + ptab1->getServer();
    if ( !ptab1->getDatabase().isEmpty() )
        stringConnectIn += ";DATABASE=" + ptab1->getDatabase();
    if ( !ptab2->getPort().isEmpty() )
        stringConnectIn += ";PORT=" + ptab2->getPort();
    if ( !ptab2->getSocket().isEmpty() )
        stringConnectIn += ";SOCKET=" + ptab2->getSocket();
//    if ( !ptab1->getOptions().isEmpty() )
//        stringConnectIn += ";OPTION=" + ptab1->getOptions();
    if ( !ptab2->getInitialStatement().isEmpty() )
        stringConnectIn += ";STMT=" + ptab2->getInitialStatement();

    nReturn = SQLAllocHandle( SQL_HANDLE_ENV, NULL, &hEnv );
    if ( nReturn != SQL_SUCCESS )
        slotShowDiagnostics( nReturn, SQL_HANDLE_ENV, NULL );
    if ( !SQL_SUCCEEDED(nReturn) )
        return FALSE;

    nReturn = SQLSetEnvAttr( hEnv, SQL_ATTR_ODBC_VERSION, (SQLPOINTER)SQL_OV_ODBC3, 0 );
    if ( nReturn != SQL_SUCCESS )
        slotShowDiagnostics( nReturn, SQL_HANDLE_ENV, NULL );
    if ( !SQL_SUCCEEDED(nReturn) )
        goto slotTestExit1;

    nReturn = SQLAllocHandle( SQL_HANDLE_DBC, hEnv, &hDbc );
    if ( nReturn != SQL_SUCCESS )
        slotShowDiagnostics( nReturn, SQL_HANDLE_ENV, hEnv );
    if ( !SQL_SUCCEEDED(nReturn) )
        goto slotTestExit1;

    nReturn = SQLDriverConnect( hDbc, NULL, (SQLCHAR*)stringConnectIn.latin1(), SQL_NTS, NULL, 0, NULL, SQL_DRIVER_NOPROMPT );
    if ( nReturn != SQL_SUCCESS )
        slotShowDiagnostics( nReturn, SQL_HANDLE_DBC, hDbc );
    if ( !SQL_SUCCEEDED(nReturn) )
        goto slotTestExit2;

    QMessageBox::information( this, "Connector/ODBC", "Success; connection was made!\n", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );

    nReturn = SQLDisconnect( hDbc );

slotTestExit2:
    nReturn = SQLFreeHandle( SQL_HANDLE_DBC, hDbc );
slotTestExit1:
    nReturn = SQLFreeHandle( SQL_HANDLE_ENV, hEnv );

    return SQL_SUCCEEDED(nReturn);
}

/*!
    \brief  Try connect/disconnect.

            Here we try to use the driver directly as this is the dbc we
            have at hand.

    \note   The creation of this method was driven by problems while 
            coding support for SQLDriverConnect() prompting.
            
    \sa     doTestUsingDriverManager
*/
BOOL MYODBCSetupDataSourceDialog::doTestUsingDriver()
{
    QMessageBox::warning( this, "Connector/ODBC", "Testing not implemented for SQLDriverConnect()", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
    return TRUE;
}

BOOL MYODBCSetupDataSourceDialog::doLoadDatabaseNamesUsingDriverManager()
{
    SQLHENV     hEnv        = SQL_NULL_HENV;
    SQLHDBC     hDbc        = hDBC;
    SQLHSTMT    hStmt;
    SQLRETURN   nReturn;
    QString     stringConnectIn;
    QStringList stringlistDatabases;
    SQLCHAR     szCatalog[MYODBC_DB_NAME_MAX]; 
    SQLINTEGER  nCatalog;

    stringlistDatabases += " ";

    /* build connect string in */
#ifdef Q_WS_MACX
    /*!
        \note OSX

              Yet another place where iODBC fails to act like the ODBC
              specification. SQLDriverConnect() should accept the 
              driver *description* and should allow for the curly
              braces but fails on both accounts deciding instead to
              interpret the whole value as a driver file name.

              Its possible to do the xref here, for SQLDriverConnect,
              but due to time constraints the file is hard-coded.
    */ 
    stringConnectIn  = "Driver=/usr/lib/libmyodbc3.bundle";
#else
    stringConnectIn  = "DRIVER=" + QString( pDataSource->pszDRIVER );
#endif
    stringConnectIn += ";UID=" + ptab1->getUser();
    stringConnectIn += ";PWD=" + ptab1->getPassword();
    stringConnectIn += ";SERVER=" + ptab1->getServer();
    if ( !ptab2->getPort().isEmpty() )
        stringConnectIn += ";PORT=" + ptab2->getPort();
    if ( !ptab2->getSocket().isEmpty() )
        stringConnectIn += ";SOCKET=" + ptab2->getSocket();
//    if ( !plineeditOptions->text().isEmpty() )
//        stringConnectIn += ";OPTION=" + plineeditOptions->text();

    if ( hDBC == SQL_NULL_HDBC )
    {
        nReturn = SQLAllocHandle( SQL_HANDLE_ENV, NULL, &hEnv );
        if ( nReturn != SQL_SUCCESS )
            slotShowDiagnostics( nReturn, SQL_HANDLE_ENV, NULL );
        if ( !SQL_SUCCEEDED(nReturn) )
            return FALSE;
    
        nReturn = SQLSetEnvAttr( hEnv, SQL_ATTR_ODBC_VERSION, (SQLPOINTER)SQL_OV_ODBC3, 0 );
        if ( nReturn != SQL_SUCCESS )
            slotShowDiagnostics( nReturn, SQL_HANDLE_ENV, NULL );
        if ( !SQL_SUCCEEDED(nReturn) )
            goto slotLoadDatabaseNamesExit1;
    
        nReturn = SQLAllocHandle( SQL_HANDLE_DBC, hEnv, &hDbc );
        if ( nReturn != SQL_SUCCESS )
            slotShowDiagnostics( nReturn, SQL_HANDLE_ENV, hEnv );
        if ( !SQL_SUCCEEDED(nReturn) )
            goto slotLoadDatabaseNamesExit1;
    }

    nReturn = SQLDriverConnect( hDbc, NULL, (SQLCHAR*)stringConnectIn.latin1(), SQL_NTS, NULL, 0, NULL, SQL_DRIVER_NOPROMPT );
    if ( nReturn != SQL_SUCCESS )
        slotShowDiagnostics( nReturn, SQL_HANDLE_DBC, hDbc );
    if ( !SQL_SUCCEEDED(nReturn) )
        goto slotLoadDatabaseNamesExit2;

    nReturn = SQLAllocHandle( SQL_HANDLE_STMT, hDbc, &hStmt );
    if ( nReturn != SQL_SUCCESS )
        slotShowDiagnostics( nReturn, SQL_HANDLE_DBC, hDbc );
    if ( !SQL_SUCCEEDED(nReturn) )
        goto slotLoadDatabaseNamesExit2;

    nReturn = SQLTables( hStmt, (SQLCHAR*)SQL_ALL_CATALOGS, SQL_NTS, (SQLCHAR*)"", SQL_NTS, (SQLCHAR*)"", 0, (SQLCHAR*)"", 0 );
    if ( nReturn != SQL_SUCCESS )
        slotShowDiagnostics( nReturn, SQL_HANDLE_STMT, hStmt );
    if ( !SQL_SUCCEEDED(nReturn) )
        goto slotLoadDatabaseNamesExit3;

    nReturn = SQLBindCol( hStmt, 1, SQL_C_CHAR, szCatalog, MYODBC_DB_NAME_MAX, &nCatalog );
    while ( TRUE )
    {
        nReturn = SQLFetch( hStmt );
        if ( nReturn == SQL_NO_DATA )
            break;
        else if ( nReturn != SQL_SUCCESS )
            slotShowDiagnostics( nReturn, SQL_HANDLE_STMT, hStmt );
        if ( SQL_SUCCEEDED(nReturn) )
            stringlistDatabases += (const char*)szCatalog;
        else
            break;
    }

slotLoadDatabaseNamesExit3:
    nReturn = SQLFreeHandle( SQL_HANDLE_STMT, hStmt );
slotLoadDatabaseNamesExit2:
    nReturn = SQLDisconnect( hDbc );
    if ( hDBC == SQL_NULL_HDBC )
        nReturn = SQLFreeHandle( SQL_HANDLE_DBC, hDbc );
slotLoadDatabaseNamesExit1:
    if ( hDBC == SQL_NULL_HDBC )
        nReturn = SQLFreeHandle( SQL_HANDLE_ENV, hEnv );

    ptab1->pcomboboxDatabase->clear();
    ptab1->pcomboboxDatabase->insertStringList( stringlistDatabases );

    return TRUE;
}

BOOL MYODBCSetupDataSourceDialog::doLoadDatabaseNamesUsingDriver()
{
    QMessageBox::warning( this, "Connector/ODBC", "Loading database list not implemented for SQLDriverConnect()", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
    return TRUE;
}

void MYODBCSetupDataSourceDialog::slotShowDiagnostics( SQLRETURN nReturn, SQLSMALLINT nHandleType, SQLHANDLE h )
{
    BOOL bMessageShown = FALSE;

    if ( h )
    {
        SQLSMALLINT nRec = 1;
        SQLCHAR     szSQLState[6];
        SQLINTEGER  nNative;
        SQLCHAR     szMessage[SQL_MAX_MESSAGE_LENGTH];
        SQLSMALLINT nMessage;

        *szSQLState = '\0';
        *szMessage  = '\0';

        while ( SQL_SUCCEEDED( SQLGetDiagRec( nHandleType,
                                              h,
                                              nRec,
                                              szSQLState,
                                              &nNative,
                                              szMessage,
                                              SQL_MAX_MESSAGE_LENGTH,
                                              &nMessage ) ) )
        {
            szSQLState[5]               = '\0';
            szMessage[SQL_MAX_MESSAGE_LENGTH - 1]  = '\0';

            if ( nReturn == SQL_ERROR )
                QMessageBox::critical( this, "MYODBCConfig", (char*)szMessage, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
            else if ( nReturn == SQL_SUCCESS_WITH_INFO )
                QMessageBox::warning( this, "MYODBCConfig", (char*)szMessage, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
            else
                QMessageBox::information( this, "MYODBCConfig", (char*)szMessage, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );

            bMessageShown = TRUE;
            nRec++;

            *szSQLState = '\0';
            *szMessage  = '\0';
        }
    }

    if ( !bMessageShown )
    {
        switch ( nReturn )
        {
            case SQL_ERROR:
                QMessageBox::critical( this, "MYODBCConfig", "Request returned with SQL_ERROR. Sorry; no diagnostic information.", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
                break;
            case SQL_SUCCESS_WITH_INFO:
                QMessageBox::warning( this, "MYODBCConfig", "Request return with SQL_SUCCESS_WITH_INFO. Sorry; no diagnostic information.", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
                break;
            case SQL_INVALID_HANDLE:
                QMessageBox::critical( this, "MYODBCConfig", "Request returned with SQL_INVALID_HANDLE.", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
                break;
            default:
                QMessageBox::information( this, "MYODBCConfig", "Request did not return with SQL_SUCCESS. Sorry; no diagnostic information.", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
        }
    }
}

void MYODBCSetupDataSourceDialog::slotShowInstallerError()
{
    WORD      nRecord = 1;
    DWORD     nError;
    char      szError[SQL_MAX_MESSAGE_LENGTH];
    RETCODE   nReturn;

    nReturn = SQLInstallerError( nRecord, &nError, szError, SQL_MAX_MESSAGE_LENGTH - 1, 0 );
    if ( SQL_SUCCEEDED( nReturn ) )
        QMessageBox::critical( this, "MYODBCConfig", (char*)szError, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
    else
        QMessageBox::critical( this, "MYODBCConfig", "ODBC Installer error (unknown)", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
}

void MYODBCSetupDataSourceDialog::slotToggleGuru( bool b )
{
    if ( b )
        ptexteditDiagnostics->show();
    else
        ptexteditDiagnostics->hide();
}

void MYODBCSetupDataSourceDialog::slotLoadDatabaseNames()
{
    if ( hDBC )
        doLoadDatabaseNamesUsingDriver();
    else
        doLoadDatabaseNamesUsingDriverManager();
}


