/////////////////////////////////////////////////////////////////////////////
// Name:      MainWin.cpp
// Author:    Alex Thuering
// Created:   7.07.2012
// Copyright: (c) Alex Thuering
// Licence:   GPL
/////////////////////////////////////////////////////////////////////////////
#include "MainWin.h"

//(*InternalHeaders(MainWin)
#include <wx/intl.h>
#include <wx/string.h>
//*)
#include <wx/toolbar.h>
#include <wx/progdlg.h>
#include <wx/file.h>
#include <wx/regex.h>
#include <wx/artprov.h>
#include <wx/aboutdlg.h>
#include "ProgramProcess.h"
#include "Languages.h"
#include "Config.h"
#include "Version.h"
#include "utils.h"
#include "../resources/mp4joiner.png.h"
#include "../resources/add.png.h"
#include "../resources/remove.png.h"
#include "../resources/run.png.h"
#include "../resources/preferences.png.h"
#include <map>

/**
 * Processes output of MP4Box tool
 */
class MP4Process: public ProgramProcess {
public:
	MP4Process(wxProgressDialog* parent, int step, vector<MediaFile*> mediaFiles):
			ProgramProcess(parent), files(mediaFiles) {
		this->step = step;
		pattern.Compile(".*\\(([0-9]+)/100\\).*");
	}
	
	virtual void OnTerminate(int pid, int status) {
		ProgramProcess::OnTerminate(pid, status); // read the rest of the output
		if (status == 0)
			Update(-1, _("Successful"));
	}
    
	virtual void ProcessOutput(const wxString& line, bool errorStream) {
    	if (line.StartsWith("Error") || line.StartsWith("Unknown")
    			|| line.StartsWith("No suitable media tracks to cat")) {
    		cerr << line << endl;
			wxLogError(line);
    	} else if (line.StartsWith("IsoMedia import")) {
			Update(line);
		} else if (line.StartsWith("Appending file")) {
			Update(step*100, line);
			step++;
		} else if (line.StartsWith("Saving to")) {
			Update(step*100, line.Index(':') > 0 ? line.BeforeLast(':') : line);
			step++;
		} else if (line.StartsWith("Importing")) {
			cerr << line << endl;
		} else if (pattern.Matches(line)) {
			cerr << line << endl;
			long percent = 0;
			if (pattern.GetMatch(line, 1).ToLong(&percent)) {
				Update((step > 0 ? (step-1)*100 : 0) + percent);
			}
		} else
			cerr << line << endl;
    }

private:
	vector<MediaFile*>& files;
	int step;
	wxRegEx pattern;
};

/**
 * Processes output of avconv tool
 */
class AvConvProcess: public ProgramProcess {
public:
	AvConvProcess(wxProgressDialog* parent, int step, long totalFrames) : ProgramProcess(parent) {
		this->totalFrames = totalFrames;
		this->step = step;
		pattern.Compile(wxT("frame=[[:space:]]*([0-9]+).*"));
	}
	
	virtual void ProcessOutput(const wxString& line, bool errorStream) {
		if (line.Find(wxT("buffer underflow i=1")) >= 0
				|| line.Find(wxT("packet too large, ignoring buffer limits")) >= 0
				|| line.Find(wxT("Last message repeated 1 times")) >= 0)
			return;
		cerr << line << endl;
    	if (line.StartsWith("Error") || line.StartsWith("Unknown")) {
			wxLogError(line);
    	} else if (pattern.Matches(line)) {
			long frame = 0;
			pattern.GetMatch(line, 1).ToLong(&frame);
			int percent = (totalFrames > 0) ? (frame * 100) / totalFrames : 0;
			Update(step*100 + percent);
		}
    }

private:
	long totalFrames;
	int step;
	wxRegEx pattern;
};


//(*IdInit(MainWin)
const long MainWin::ID_MEDIA_LISTBOX = wxNewId();
const long MainWin::ID_TOOLBAR = wxNewId();
const long MainWin::ID_STATUSBAR = wxNewId();
//*)
const long ADD_FILE_ID = wxNewId();
const long REMOVE_FILE_ID = wxNewId();
const long RUN_ID = wxNewId();
const long SETTINGS_ID = wxNewId();
const long ABOUT_ID = wxNewId();

BEGIN_EVENT_TABLE(MainWin,wxFrame)
	//(*EventTable(MainWin)
	//*)
END_EVENT_TABLE()

MainWin::MainWin() {
	//(*Initialize(MainWin)
	Create(0, wxID_ANY, _("MP4 Joiner"), wxDefaultPosition, wxDefaultSize, wxDEFAULT_FRAME_STYLE, _T("wxID_ANY"));
	SetFocus();
	mediaListBox = new MediaListBox(this, ID_MEDIA_LISTBOX, wxPoint(128,176), wxSize(300,400), 0, 0, 0, wxDefaultValidator, _T("ID_MEDIA_LISTBOX"));
	statusBar = new wxStatusBar(this, ID_STATUSBAR, 0, _T("ID_STATUSBAR"));
	int __wxStatusBarWidths_1[1] = { -10 };
	int __wxStatusBarStyles_1[1] = { wxSB_NORMAL };
	statusBar->SetFieldsCount(1,__wxStatusBarWidths_1);
	statusBar->SetStatusStyles(1,__wxStatusBarStyles_1);
	SetStatusBar(statusBar);
	Center();
	
	Connect(ID_MEDIA_LISTBOX,wxEVT_COMMAND_LISTBOX_SELECTED,(wxObjectEventFunction)&MainWin::OnMediaSelect);
	//*)

	toolbar = new wxToolBar(this, ID_TOOLBAR, wxDefaultPosition, wxDefaultSize, wxTB_TEXT|wxTB_FLAT);
	SetToolBar(toolbar);
	toolbar->AddTool(ADD_FILE_ID, _("Add Video"), wxBITMAP_FROM_MEMORY(add), wxNullBitmap, wxITEM_NORMAL,
			_("Add video file"), wxT(""));
	toolbar->AddTool(REMOVE_FILE_ID,  _("Remove"), wxBITMAP_FROM_MEMORY(remove), wxNullBitmap, wxITEM_NORMAL,
			_("Remove selected video"), wxT(""));
	toolbar->AddTool(RUN_ID,  _("Join"), wxBITMAP_FROM_MEMORY(run), wxNullBitmap, wxITEM_NORMAL,
			_("Select output file and start join"), wxT(""));
#if wxCHECK_VERSION(2, 9, 0)
	toolbar->AddStretchableSpace();
#endif
	toolbar->AddTool(SETTINGS_ID, _("Language"), wxBITMAP_FROM_MEMORY(preferences),
			wxNullBitmap, wxITEM_NORMAL, _("Select language"), wxT(""));
	toolbar->AddTool(ABOUT_ID, _("About"), wxArtProvider::GetBitmap(wxART_INFORMATION, wxART_TOOLBAR),
			wxNullBitmap, wxITEM_NORMAL, _("About the program"), wxT(""));
	toolbar->Realize();
	UpdateToolBar();

	Connect(ADD_FILE_ID, wxEVT_COMMAND_TOOL_CLICKED, (wxObjectEventFunction)&MainWin::OnAddFileBt);
	Connect(REMOVE_FILE_ID, wxEVT_COMMAND_TOOL_CLICKED, (wxObjectEventFunction)&MainWin::OnRemoveFileBt);
	Connect(RUN_ID, wxEVT_COMMAND_TOOL_CLICKED, (wxObjectEventFunction)&MainWin::OnRunBt);
	Connect(SETTINGS_ID, wxEVT_COMMAND_TOOL_CLICKED, (wxObjectEventFunction)&MainWin::OnSettingsBt);
	Connect(ABOUT_ID, wxEVT_COMMAND_TOOL_CLICKED, (wxObjectEventFunction)&MainWin::OnAboutBt);
	Connect(wxEVT_CLOSE_WINDOW, (wxObjectEventFunction)&MainWin::OnClose);
	
	// Restore frame size/position
	if (s_config.IsMainWinMaximized()) {
		Maximize(true);
	} else {
		wxRect rect = s_config.GetMainWinLocation();
		if (rect.width > 0 && rect.height > 0)
			SetSize(rect);
		else
			SetSize(wxSize(500,550));
	}
}

MainWin::~MainWin() {
	//(*Destroy(MainWin)
	//*)
	VECTOR_CLEAR(files, MediaFile)
}

void MainWin::UpdateToolBar() {
	toolbar->EnableTool(REMOVE_FILE_ID, mediaListBox->GetSelection() >= 0);
	toolbar->EnableTool(RUN_ID, files.size() > 1);
	long totalDuration = 0;
	long totalSize = 0;
	for (vector<MediaFile*>::iterator it = files.begin(); it != files.end(); it++) {
		if ((*it)->GetDuration() > 0)
			totalDuration += (int) (*it)->GetDuration();
		totalSize += (*it)->GetSize() / 1024;
	}
	wxString statusStr = _("Total duration");
	int mins = totalDuration / 60;
	int secs = totalDuration % 60;
	int hours = mins / 60;
	mins %= 60;
	statusStr += wxString::Format(" %02d:%02d:%02d", hours, mins, secs);
	statusStr += " / ";
	statusStr += _("total size");
	if (totalSize > 1024)
		statusStr += wxString::Format(": %0.1f ", (double) totalSize / 1024) + _("GB");
	else
		statusStr += wxString::Format(": %ld ", totalSize) + _("MB");
	statusBar->SetStatusText(statusStr, 0);
}

void MainWin::OnAddFileBt(wxCommandEvent& event) {
	wxFileDialog fileDlg(this, _("Choose a file"), wxT(""), wxT(""),
			wxString(_("MP4 Files")) + wxT(" (*.mp4)|*.mp4|")
			+ wxString(_("All Files")) + wxT(" (*.*)|*.*"), wxFD_OPEN | wxFD_MULTIPLE);
	fileDlg.SetDirectory(s_config.GetLastAddDir() + wxFILE_SEP_PATH);
	if (fileDlg.ShowModal() != wxID_OK)
		return;
	s_config.SetLastAddDir(fileDlg.GetDirectory());
	wxArrayString paths;
	fileDlg.GetPaths(paths);
	for (unsigned int i = 0; i < paths.GetCount(); i++) {
		MediaFile* mediaFile = new MediaFile;
		if (mediaFile->Init(paths[i]))
			files.push_back(mediaFile);
		else
			delete mediaFile;
	}
	mediaListBox->RefreshAll();
	UpdateToolBar();
}

void MainWin::OnRemoveFileBt(wxCommandEvent& event) {
	delete files[mediaListBox->GetSelection()];
	files.erase(files.begin() + mediaListBox->GetSelection());
	mediaListBox->RefreshAll();
	UpdateToolBar();
}

void RemoveTempFiles(const map<int, wxString>& tempFiles) {
	for (map<int, wxString>::const_iterator it = tempFiles.begin(); it != tempFiles.end(); it++) {
		if (wxFileExists(it->second)) {
			wxRemoveFile(it->second);
		}
	}
}

void MainWin::OnRunBt(wxCommandEvent& event) {
	// choose a file to save
	wxFileDialog fileDlg(this, _("Choose a file to save"), wxT(""), _("Output.mp4"),
		wxString(_("MP4 Files")) + wxT(" (*.mp4)|*.mp4|")
		+ wxString(_("All Files")) + wxT(" (*.*)|*.*"), wxFD_SAVE | wxFD_OVERWRITE_PROMPT);
	if (fileDlg.ShowModal() != wxID_OK)
		return;
	wxString fileName = fileDlg.GetPath();
	if (wxFile::Exists(fileName) && !wxRemoveFile(fileName))
		return;
	
	// check if reencoding is needed
	map<int, wxString> tempFiles;
	MediaFile* mediaFile1 = files[0];
	for (unsigned int idx = 1; idx < files.size(); idx++) {
		MediaFile* mediaFile = files[idx];
		if (mediaFile->HasCompatibleStreams(mediaFile1))
			continue;
		tempFiles[idx] = wxFileName::CreateTempFileName(wxFileName::GetTempDir() + wxFILE_SEP_PATH) + ".mp4";
	}
	
	// show progress dialog
	int stepCount = tempFiles.size() + files.size() + 1;
	wxProgressDialog progDlg(_("MP4 Joiner"), _("Joning the files"), stepCount*100, this,
			wxPD_APP_MODAL | wxPD_CAN_ABORT | wxPD_ELAPSED_TIME | wxPD_ESTIMATED_TIME);
	progDlg.Show();
	int step = 0;
	
	// reencode
	for (map<int, wxString>::iterator it = tempFiles.begin(); it != tempFiles.end(); it++) {
		int fileIdx = it->first;
		wxString tempFileName = it->second;
		
		MediaFile* mediaFile = files[fileIdx];
		progDlg.Update(step * 100, wxString::Format(_("Reencode audio from file %s"), mediaFile->GetFileName().c_str()));
		
		// build avconv/ffmpeg command
		wxString cmd = "ffmpeg";
#if defined(__WXMSW__) || defined(__WXMAC__)
		cmd = '"' + wxGetAppPath() + cmd + '"';
#endif
		cmd += " -i \"" + mediaFile->GetFileName() + '"';
		cmd += " -c:v copy";
		
		unsigned int audioIdx = 0;
		unsigned int streamIdx1 = 0;
		for (unsigned int i = 0; i < mediaFile->GetStreams().size(); i++) {
			if (mediaFile->GetStreams()[i]->GetType() == stAUDIO) {
				// find corresponding audio stream in mediaFile1 
				while (streamIdx1 < mediaFile1->GetStreams().size()
						&& mediaFile1->GetStreams()[streamIdx1]->GetType() != stAUDIO)
					streamIdx1++;
				if (streamIdx1 >= mediaFile1->GetStreams().size())
					break;
				// set audio codec
				wxString codec = mediaFile1->GetStreams()[streamIdx1]->GetCodecName();
				if (codec == "aac")
					codec = "libfaac";
				cmd += wxString::Format(" -c:a:%d ", audioIdx) + codec;
				cmd += wxString::Format(" -ac:a:%d %d", audioIdx, mediaFile1->GetStreams()[i]->GetChannelNumber());
				cmd += wxString::Format(" -ar:%d %d", audioIdx, mediaFile1->GetStreams()[i]->GetSampleRate());
				audioIdx++;
				streamIdx1++;
			}
		}
		cmd += " \"" + tempFileName + '"';
		cerr << cmd << endl;
		
		// execute avconv/ffmpeg
		AvConvProcess proc(&progDlg, step++, lround(mediaFile->GetDuration() * mediaFile->GetVideoStream()->GetFps()));
		if (!proc.Execute(cmd)) {
			RemoveTempFiles(tempFiles);
			return;
		}
		cerr << endl;
	}
	
	// build MP4Box command
	wxString cmd = "MP4Box";
#if defined(__WXMSW__) || defined(__WXMAC__)
	cmd = '"' + wxGetAppPath() + cmd + '"';
#endif
	for (unsigned int idx = 0; idx < files.size(); idx++) {
		wxString fileName = tempFiles.find(idx) != tempFiles.end() ? tempFiles[idx] : files[idx]->GetFileName();
		cmd += " -cat \"" + fileName + '"';
	}
	cmd += " \"" + fileName + '"';
	cerr << cmd << endl;
	
	// execute MP4Box
	MP4Process proc(&progDlg, step, files);
	proc.Execute(cmd);
	RemoveTempFiles(tempFiles);
}

void MainWin::OnSettingsBt(wxCommandEvent& event) {
	int langId = wxLANGUAGE_ENGLISH;
	if (wxLocale::FindLanguageInfo(s_config.GetLanguageCode()))
		langId = wxLocale::FindLanguageInfo(s_config.GetLanguageCode())->Language;
	int lang = ChooseLanguage(langId);
	if (lang == wxLANGUAGE_UNKNOWN)
		return;
	wxString languageCode = wxLocale::GetLanguageInfo(lang)->CanonicalName;
	if (languageCode == s_config.GetLanguageCode())
		return;
	s_config.SetLanguageCode(languageCode);
	wxMessageBox(_("Language change will not take effect until MP4Joiner is restarted"),
			GetTitle(), wxOK|wxICON_INFORMATION, this);
}

wxString FixEmail(const wxString& str) {
	wxString result = str;
	result.Replace(" at ", "@");
	return result;
}

void MainWin::OnAboutBt(wxCommandEvent& event) {
	wxAboutDialogInfo info;
	info.SetName("MP4Joiner");
	info.SetVersion(APP_VERSION);
	info.SetDescription(_("Simple tool to join multiply MP4 files into one."));
	info.SetCopyright(FixEmail(wxString("(C) 2013 Alex Thring <alex at thuering.biz>", wxConvISO8859_1)));
	info.SetWebSite("http://www.mp4joiner.org");
#ifdef __WXMSW__
	info.SetIcon(wxICON(mp4joiner));
#else
	info.SetIcon(wxICON_FROM_MEMORY(mp4joiner));
#endif
	wxAboutBox(info);
	
}

void MainWin::OnMediaSelect(wxCommandEvent& event) {
	UpdateToolBar();
}

void MainWin::OnClose(wxCloseEvent &event) {
	if (!IsIconized())
		s_config.SetMainWinLocation(GetRect(), IsMaximized());
	Destroy();
}
