/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: itkRobustAutomaticThresholdCalculator.h,v $
  Language:  C++
  Date:      $Date: 2009-04-23 03:43:42 $
  Version:   $Revision: 1.2 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkRobustAutomaticThresholdCalculator_h
#define __itkRobustAutomaticThresholdCalculator_h

#include "itkMacro.h"
#include "itkImage.h"

namespace itk
{

/** \class RobustAutomaticThresholdCalculator
 * \brief Compute the robust automatic threshold
 *
 * \ingroup Operators
 */
template < class TInputImage, class TGradientImage >
class ITK_EXPORT RobustAutomaticThresholdCalculator : public Object
{
public:
  /** Standard class typedefs. */
  typedef RobustAutomaticThresholdCalculator Self;
  typedef Object                             Superclass;
  typedef SmartPointer<Self>                 Pointer;
  typedef SmartPointer<const Self>           ConstPointer;

  /** Method for creation through the object factory. */
  itkNewMacro(Self);

  /** Run-time type information (and related methods). */
  itkTypeMacro(RobustAutomaticThresholdCalculator, Object);

  /** Extract the dimension of the image. */
  itkStaticConstMacro(ImageDimension, unsigned int,
                      TInputImage::ImageDimension);

  /** Standard image type within this class. */
  typedef TInputImage    InputImageType;
  typedef TGradientImage GradientImageType;

  /** Standard image type pointer within this class. */
  typedef typename InputImageType::Pointer         InputImagePointer;
  typedef typename InputImageType::ConstPointer    InputImageConstPointer;
  typedef typename GradientImageType::Pointer      GradientImagePointer;
  typedef typename GradientImageType::ConstPointer GradientImageConstPointer;
  typedef typename InputImageType::PixelType       InputPixelType;
  typedef typename GradientImageType::PixelType    GradientPixelType;

  /** Set the input image. */
  virtual void SetInput( const InputImageType * image )
    {
    if ( m_Input != image )
      {
      m_Input = image;
      this->Modified();
      m_Valid = false;
      }
    }

  virtual void SetGradient( const GradientImageType * image )
    {
    if ( m_Gradient != image )
      {
      m_Gradient = image;
      this->Modified();
      m_Valid = false;
      }
    }

  itkSetMacro(Pow, double);
  itkGetConstMacro(Pow, double);

  /** Compute moments of a new or modified image.
   * This method computes the moments of the image given as a
   * parameter and stores them in the object.  The values of these
   * moments and related parameters can then be retrieved by using
   * other methods of this object. */
  void Compute( void );
  
  const InputPixelType & GetOutput() const;

protected:
  RobustAutomaticThresholdCalculator();
  virtual ~RobustAutomaticThresholdCalculator() {};
  void PrintSelf(std::ostream& os, Indent indent) const;

private:
  RobustAutomaticThresholdCalculator(const Self&); //purposely not implemented
  void operator=(const Self&); //purposely not implemented

  bool           m_Valid;                      // Have moments been computed yet?
  double         m_Pow;
  InputPixelType m_Output;

  InputImageConstPointer    m_Input;
  GradientImageConstPointer m_Gradient;

};  // class RobustAutomaticThresholdCalculator

} // end namespace itk


#ifndef ITK_MANUAL_INSTANTIATION
#include "itkRobustAutomaticThresholdCalculator.txx"
#endif

#endif /* __itkRobustAutomaticThresholdCalculator_h */
