/*
    Copyright 2014-2015 Thibaut Paumard

    This file is part of Gyoto.

    Gyoto is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Gyoto is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Gyoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "GyotoObject.h"
#include "GyotoProperty.h"
#include "GyotoValue.h"
#include "GyotoError.h"
#include "GyotoFactoryMessenger.h"
#include "GyotoMetric.h"
#include "GyotoAstrobj.h"
#include "GyotoSpectrum.h"
#include "GyotoSpectrometer.h"
#include "GyotoScreen.h"

#include <iostream>
#include <cstdlib>

using namespace std ;
using namespace Gyoto ;

// We do have a propert list. In contains a single item, wich is a
// link to the NULL pointer, meaning end-of-the list.  This is used to
// terminate the property list of our descendents in a
// forward-compatible manner, i.e., we may well add very generic
// Properties in the future.
GYOTO_PROPERTY_START(Object)
GYOTO_PROPERTY_END(Object, NULL)


Gyoto::Object::Object(std::string const &name):kind_(name) {}
Gyoto::Object::Object():kind_("") {}
Gyoto::Object::Object(Object const &o):kind_(o.kind_) {}
Gyoto::Object::~Object() {}


void Object::set(Property const &p,
		 Value val,
		 std::string const &unit) {
  GYOTO_DEBUG_EXPR(p.type);
  switch (p.type) {
  case Property::empty_t:
    throwError("Attempt to set empty_t Property");
    return;
  case Property::double_t:
    {
      Property::set_double_unit_t setu = p.setter_unit.set_double;
      if (setu) {
	GYOTO_DEBUG << "double Property which supports unit" << endl;
	(this->*setu)(val, unit);
      } else {
	GYOTO_DEBUG << "double Property which does not support unit" << endl;
	if (unit != "") throwError("Can't set this property with unit");
	set(p, val);
      }
    }
    return;
  case Property::vector_double_t:
    {
      Property::set_vector_double_unit_t setu = p.setter_unit.set_vdouble;
      if (setu) {
	GYOTO_DEBUG << "vector<double> Property which supports unit" << endl;
	(this->*setu)(val, unit);
      } else {
	GYOTO_DEBUG << "vector<double> Property which does not support unit" << endl;
	if (unit != "") throwError("Can't set this property with unit");
	set(p, val);
      }
    }
    return;
  default:
    GYOTO_DEBUG<< "Not a double_t, vector_double_t or empty_t Property" << endl;
    if (unit != "")
      throwError("Can't set this property with unit (not a double)");
    set(p, val);
    return;
  }

}

void Object::set(Property const &p, Value val) {
# define ___local_case(type)			\
  case Property::type##_t:			\
    {						\
      GYOTO_DEBUG <<"Setting property of type " #type << endl;	\
      Property::set_##type##_t set = p.setter.set_##type;	\
      GYOTO_DEBUG_EXPR(set);					\
      if (!set) throwError("Can't set this Property");	\
      (this->*set)(val);				\
    }							\
    break
  
  switch (p.type) {
  case Property::empty_t:
    return;
    ___local_case(double);
    ___local_case(bool);
    ___local_case(long);
    ___local_case(unsigned_long);
  case Property::filename_t:
    ___local_case(string);
  case Property::vector_double_t:
    {
      Property::set_vector_double_t set = p.setter.set_vdouble;
      if (!set) throwError("Can't set this Property");
      (this->*set)(val);
    }
    break;
  case Property::vector_unsigned_long_t:
    {
      Property::set_vector_unsigned_long_t set = p.setter.set_vulong;
      if (!set) throwError("Can't set this Property");
      (this->*set)(val);
    }
    break;
    ___local_case(metric);
    ___local_case(astrobj);
    ___local_case(spectrum);
    ___local_case(spectrometer);
    ___local_case(screen);
  default:
    throwError("Unimplemented Property type in Object::set");
  }
# undef ___local_case
}

void Object::set(std::string const &pname, Value val) {
  Property const * p = property(pname);
  if (!p) throwError("No Property by that name");
  set(*p, ((p->type == Property::bool_t && pname == p->name_false)?
	  Value(!val):val));
}

void Object::set(std::string const &pname, Value val, std::string const &unit) {
  Property const * p = property(pname);
  if (!p) throwError("No Property by that name");
  set(*p, ((p->type == Property::bool_t && pname == p->name_false)?
	  Value(!val):val), unit);
}

Value Object::get(Property const &p,
			    std::string const &unit) const {

  if (p.type == Property::double_t) {
    Property::get_double_unit_t getu = p.getter_unit.get_double;
    if (getu) return (this->*getu)(unit);
    if (unit != "") throwError("Can't get this property with unit");
    return get(p);
  }

  if (p.type == Property::vector_double_t) {
    Property::get_vector_double_unit_t getu = p.getter_unit.get_vdouble;
    if (getu) return (this->*getu)(unit);
    if (unit != "") throwError("Can't get this property with unit");
    return get(p);
  }

  if (unit != "")
    throwError("Can't set this property with unit (not a double)");

  return get(p);
}

Value Object::get(Property const &p) const {
# define ___local_case(type) \
  case Property::type##_t:     \
    {			     \
    Property::get_##type##_t get = p.getter.get_##type;	\
    if (!get) throwError("Can't get this Property");	\
    val = Value((this->*get)());			\
    }							\
    break

  Gyoto::Value val;
  switch (p.type) {
  case Property::empty_t:
    throwError("Can't get empty property");
    ___local_case(bool);
    ___local_case(double);
    ___local_case(long);
    ___local_case(unsigned_long);
  case Property::filename_t:
    ___local_case(string);
  case Property::vector_double_t:
    {
      Property::get_vector_double_t get = p.getter.get_vdouble;
      if (!get) throwError("Can't get this Property");
      val = (this->*get)();
    }
    break;
  case Property::vector_unsigned_long_t:
    {
      Property::get_vector_unsigned_long_t get = p.getter.get_vulong;
      if (!get) throwError("Can't get this Property");
      val = (this->*get)();
    }
    break;
    ___local_case(metric);
    ___local_case(astrobj);
    ___local_case(spectrum);
    ___local_case(spectrometer);
    ___local_case(screen);
  default:
    throwError("Unimplemented Property type in Object::get");
  }
  return val;
# undef ___local_case
}

Value Object::get(std::string const &pname) const {
  Property const * p = property(pname);
  if (!p) throwError("No Property by that name");
  Value res = get(*p);
  if (p->type == Property::bool_t && pname == p->name_false)
    return !bool(res);
  return res;
}

Value Object::get(std::string const &pname, std::string const &unit) const{
  Property const * p = property(pname);
  if (!p) throwError("No Property by that name");
  Value res = get(*p, unit);
  if (p->type == Property::bool_t && pname == p->name_false)
    return !bool(res);
  return res;
}

Property const * Object::property(std::string const pname) const {
  Property const * prop = getProperties(); 
  while (prop) {
    if (*prop) {
      GYOTO_DEBUG_EXPR(prop->name);
      if (prop->name == pname ||
	  (prop->type==Property::bool_t && prop->name_false == pname))
	return prop;
      ++prop;
    } else prop=prop->parent;
  }
  return NULL;
}

#ifdef GYOTO_USE_XERCES
void Object::fillProperty(Gyoto::FactoryMessenger *fmp, Property const &p) const {
  FactoryMessenger * childfmp=NULL;
  string name=p.name;
  switch (p.type) {
  case Property::empty_t:
    break;
  case Property::bool_t:
    fmp->setParameter(get(p)?name:p.name_false);
    break;
  case Property::long_t:
    fmp->setParameter(name, long(get(p)));
    break;
  case Property::unsigned_long_t:
    fmp->setParameter(name, (unsigned long)(get(p)));
    break;
  case Property::double_t:
    fmp->setParameter(name, double(get(p)));
    break;
  case Property::string_t:
  case Property::filename_t:
    fmp->setParameter(name, std::string(get(p)));
    break;
  case Property::vector_double_t:
    fmp->setParameter(name, get(p).operator std::vector<double>());
    break;
  case Property::vector_unsigned_long_t:
    fmp->setParameter(name, get(p).operator std::vector<unsigned long>());
    break;
  case Property::metric_t:
    fmp->metric(get(p));
    break;
  case Property::astrobj_t:
    fmp->astrobj(get(p));
    break;
  case Property::screen_t:
    fmp->screen(get(p));
    break;
  case Property::spectrum_t:
    {
      SmartPointer<Spectrum::Generic> sp=get(p);
      if (!sp) return;
      childfmp = fmp -> makeChild ( name );
      sp -> fillElement(childfmp);
      delete childfmp;
    }
    break;
  case Property::spectrometer_t:
    {
      SmartPointer<Spectrometer::Generic> spr=get(p);
      if (!spr) return;
      childfmp = fmp -> makeChild ( name );
      spr -> fillElement(childfmp);
      delete childfmp;
    }
    break;
  default:
    throwError("Property type unimplemented in Object::fillProperty()");
  }
}

void Object::fillElement(Gyoto::FactoryMessenger *fmp) const {
  fmp -> setSelfAttribute("kind", kind_);
  Property const * prop = getProperties(); 
  while (prop) {
    if (*prop) {
      fillProperty(fmp, *prop);
      ++prop;
    } else prop=prop->parent;
  }
}

void Object::setParameters(Gyoto::FactoryMessenger *fmp)  {
  string name="", content="", unit="";
  FactoryMessenger * child = NULL;
  if (fmp)
    while (fmp->getNextParameter(&name, &content, &unit)) {
      GYOTO_DEBUG << "Setting '" << name << "' to '" << content
		  << "' (unit='"<<unit<<"')" << endl;
      Property const * prop = property(name);
      if (!prop) {;
	GYOTO_DEBUG << "'" << name << "' not found, calling setParameter()"
		    << endl;
	// The specific setParameter() implementation may well know
	// this entity
	setParameter(name, content, unit);
      } else {
	GYOTO_DEBUG << "'" << name << "' found "<< endl;
	switch (prop->type) {
	case Property::metric_t:
	  set(*prop, fmp->metric());
	  break;
	case Property::astrobj_t:
	  set(*prop, fmp->astrobj());
	  break;
	case Property::screen_t:
	  set(*prop, fmp->screen());
	  break;
	case Property::spectrum_t:
	  content = fmp -> getAttribute("kind");
	  child = fmp -> getChild();
	  set(*prop, (*Spectrum::getSubcontractor(content))(child) );
	  delete child;
	  break;
	case Property::spectrometer_t:
	  content = fmp -> getAttribute("kind");
	  child = fmp -> getChild();
	  set(*prop, (*Spectrometer::getSubcontractor(content))(child) );
	  delete child;
	  break;
	case Property::filename_t:
	  content = fmp->fullPath(content);
	  // no 'break;' here, we need to proceed
	default:
	  setParameter(*prop, name, content, unit);
	}
      }
    }
  GYOTO_DEBUG << "Done processing parameters" << endl;
}

#endif

void Object::setParameter(Property const &p, string const &name,
			  string const & content, string const & unit) {
  Value val;
  switch (p.type) {
  case Property::bool_t:
    val = (name==p.name);
    break;
  case Property::long_t:
    val = strtol(content.c_str(), NULL, 0);
    break;
  case Property::unsigned_long_t:
    val = strtoul(content.c_str(), NULL, 0);
    break;
  case Property::double_t:
    val = atof(content.c_str());
    set(p, val, unit);
    return;
  case Property::filename_t:
  case Property::string_t:
    val = content;
    break;
  case Property::vector_double_t:
    val = FactoryMessenger::parseArray(content);
    set(p, val, unit);
    return;
  case Property::vector_unsigned_long_t:
    val = FactoryMessenger::parseArrayULong(content);
    break;
  case Property::metric_t:
    throwError("Metric can't be set using setParameter()");
  default:
    throwError("Property type unimplemented in Object::setParameter()");
  }
  set(p, val);
}

int Object::setParameter(string name, string content, string unit) {
  Property const * prop = property(name);
  if (!prop) return 1;
  setParameter(*prop, name, content, unit);
  return 0;
}
