RCS_ID("$Id: FFInputEvent.m 521 2005-10-15 16:51:57Z ravemax $")

#import "FFInputEvent.h"
#import "FFMainController.h"

@implementation FFInputEvent

#define NUM_FUNCTION_KEYS	(NSModeSwitchFunctionKey - NSUpArrowFunctionKey + 1)
static const NSString* FunctionKeyNames[NUM_FUNCTION_KEYS]; // Forward declaration. Real is at the end.
static NSString*	OtherKnownKeyNames[256] = { NULL };

// Modifier glyphs
static NSString*	CommandGlyph = nil;
static NSString*	OptionGlpyh, *ControlGlyph, *ShiftGlyph;

// Storage keys 
static NSString*	TypeKey		= @"type";
static NSString*	ActionKey	= @"action";
static NSString*	ModifierKey	= @"modifier";
static NSString*	KeyKey		= @"key";
static NSString*	ButtonKey	= @"button";
static NSString*	CommandKey	= @"command";
static NSString*	ScriptKey	= @"script";

// Script
static FFMenuAction	ScriptAction = MENU_SCRIPTS;

#pragma mark -
#pragma mark Creation & cleanup

+ (void)initialize {
	if (CommandGlyph == nil) { // No double initialization
		// Modifier glyphs
		CommandGlyph	= [[NSString alloc] initWithFormat:@"%C", 0x2318];
		OptionGlpyh		= [[NSString alloc] initWithFormat:@"%C", 0x2325];
		ControlGlyph	= [[NSString alloc] initWithFormat:@"%C", 0x2303];
		ShiftGlyph		= [[NSString alloc] initWithFormat:@"%C", 0x21E7];		

		// Other known keys besides the function keys
		OtherKnownKeyNames[0x03] = @"UpTab";
		OtherKnownKeyNames[0x09] = @"Tab";
		OtherKnownKeyNames[0x0D] = @"Enter";
		OtherKnownKeyNames[0x1B] = @"Esc";
		OtherKnownKeyNames[0x20] = @"Space";
		OtherKnownKeyNames[0x7F] = @"Delete (->)";
		OtherKnownKeyNames[0x08] = @"Backspace (<-)";
	}
}

- (id)init {
	self = [super init];
	if (self != nil) {
		m_type		= EVENT_NONE;
		m_action	= 0;
		m_modifier	= 0;
		m_key		= 0;
		m_button	= -1;
		m_command	= nil;
		m_script	= nil;
	}
	return self;
}

- (id)initWithType:(FFInputEventType)type action:(FFMenuAction)action 
			modifier:(unsigned)mod key:(unichar)key button:(int)btn 
		  andCommand:(NSString*)cmd {

	self = [super init];
	if (self != nil) {
		m_type		= type;
		m_action	= action;
		m_modifier	= mod;
		m_key		= key;
		m_button	= btn;
		m_command	= cmd;
	}
	return self;
}

+ (FFInputEvent*)event {
	return [[[self alloc] init] autorelease];
}

+ (FFInputEvent*)eventWithKey:(unichar)key modifier:(unsigned)modifier andAction:(FFMenuAction)action {
	return [[[self alloc] initWithType:EVENT_KEY action:action 
							  modifier:modifier key:key button:-1
							andCommand:nil] autorelease];
}

+ (FFInputEvent*)eventWithButton:(int)button modifier:(unsigned)modifier andAction:(FFMenuAction)action {
	return [[[self alloc] initWithType:EVENT_BUTTON action:action
							  modifier:modifier key:0 button:button
							andCommand:nil] autorelease];
}

+ (FFInputEvent*)eventWithSpeechCommand:(NSString*)cmd andAction:(FFMenuAction)action {
	return [[[self alloc] initWithType:EVENT_SPEECH action:action
							  modifier:0 key:0 button:-1
							andCommand:cmd] autorelease];
}

- (void)dealloc {
	if (m_command != nil)
		[m_command release];
	
	[super dealloc];
}

#pragma mark -
#pragma mark Getters

- (FFInputEventType)type	{ return m_type; }
- (FFMenuAction)action		{ return m_action; }
- (unsigned)modifier		{ return m_modifier; }
- (unichar)key				{ return m_key; }
- (int)button				{ return m_button; }
- (NSString*)command		{ return m_command; }
- (NSString*)script			{ return m_script; }

#pragma mark -
#pragma mark Setters

- (void)setType:(FFInputEventType)type	{ m_type = type; }
- (void)setAction:(FFMenuAction)action	{ m_action = action; }
- (void)setModifier:(unsigned)modifier	{ m_modifier = modifier; }
- (void)setKey:(unichar)key				{ m_key = key; }
- (void)setButton:(int)button			{ m_button = button; }

- (void)setCommand:(NSString*)command {
	if (m_command != nil)
		[m_command release];
	m_command = [command retain];
}

- (void)setScript:(NSString*)script {
	if (m_script != nil)
		[m_script release];
	m_script = [script retain];
}

#pragma mark -
#pragma mark Storage

enum {
	MOD_CMD	= 0,
	MOD_OPT,
	MOD_SHIFT,
	MOD_CTRL,
	MOD_NUM,

	NUM_MOD
};

#define NO_MOD_CHARACTER '-'

- (NSString*)_modifierToString:(unsigned)mod {
	char mca[NUM_MOD];

	memset(mca, (int)NO_MOD_CHARACTER, (size_t)NUM_MOD);
	if (mod & NSCommandKeyMask)		mca[MOD_CMD]	= 'C';
	if (mod & NSAlternateKeyMask)	mca[MOD_OPT]	= 'O';
	if (mod & NSShiftKeyMask)		mca[MOD_SHIFT]	= 'S';
	if (mod & NSControlKeyMask)		mca[MOD_CTRL]	= 'T';
	if (mod & NSNumericPadKeyMask)	mca[MOD_NUM]	= 'N';

	return [NSString stringWithCString:(const char*)mca length:NUM_MOD];
}

- (unsigned)_modifierFromString:(NSString*)str {
	unsigned mod = 0;

	if ([str characterAtIndex:MOD_CMD] != NO_MOD_CHARACTER)
		mod |= NSCommandKeyMask;
	if ([str characterAtIndex:MOD_OPT] != NO_MOD_CHARACTER)
		mod |= NSAlternateKeyMask;
	if ([str characterAtIndex:MOD_SHIFT] != NO_MOD_CHARACTER)
		mod |= NSShiftKeyMask;
	if ([str characterAtIndex:MOD_CTRL] != NO_MOD_CHARACTER)
		mod |= NSControlKeyMask;
	if ([str characterAtIndex:MOD_NUM] != NO_MOD_CHARACTER)
		mod |= NSNumericPadKeyMask;

	return mod;
}

- (NSDictionary*)exportAsDictionary {
	if (![self isCompleteDoIgnoreAction:FALSE])
		return nil;
	
	NSMutableDictionary* dict = [NSMutableDictionary dictionaryWithCapacity:3];
	[dict setObject:[NSNumber numberWithInt:(int)m_type] forKey:TypeKey];
	[dict setObject:[NSNumber numberWithInt:(int)m_action] forKey:ActionKey];
	if ([self isAssignedToScript])
		[dict setObject:m_script forKey:ScriptKey];
	
	if (m_type == EVENT_SPEECH)
		[dict setObject:m_command forKey:CommandKey];
	else {
		[dict setObject:[self _modifierToString:m_modifier] forKey:ModifierKey];
		if (m_type == EVENT_KEY)
			[dict setObject:[NSString stringWithFormat:@"0x%04X", m_key] forKey:KeyKey];			
		else
			[dict setObject:[NSNumber numberWithInt:m_button] forKey:ButtonKey];
	}
	return dict;
}

- (NSString*)description {
	return [[self exportAsDictionary] description];
}

- (id)initWithDictionary:(NSDictionary*)dict {
	self = [self init];
	if (self != nil) {
		m_type		= (FFInputEventType)[[dict objectForKey:TypeKey] intValue];
		m_action	= (FFMenuAction)[[dict objectForKey:ActionKey] intValue];
		if ([self isAssignedToScript])
			m_script = [[dict objectForKey:ScriptKey] retain];
		
		if (m_type == EVENT_SPEECH) {
			m_command	= [[dict objectForKey:CommandKey] retain];
			m_modifier	= 0;
			m_key		= 0;
			m_button	= -1;
		} else {
			m_modifier	= [self _modifierFromString:[dict objectForKey:ModifierKey]];
			m_command	= nil;
			if (m_type == EVENT_KEY) {
				unsigned ku;
				[[NSScanner scannerWithString:[dict objectForKey:KeyKey]] scanHexInt:&ku];
				m_key = (unichar)ku;
				
				// Uppercase & lowercase fix
				if (m_modifier & NSShiftKeyMask) {
					if ((m_key >= 'a') && (m_key <= 'z'))
						m_key += 'A' - 'a';
				} else if ((m_key >= 'A') && (m_key <= 'Z'))
					m_key -= 'A' - 'a';
					
				m_button	= -1;
			} else {
				m_button	= [[dict objectForKey:ButtonKey] intValue];
				m_key		= 0;
			}
		}
	}
	return self;
}

+ (FFInputEvent*)eventFromDictionary:(NSDictionary*)dict {
	return [[[self alloc] initWithDictionary:dict] autorelease];
}

#pragma mark -
#pragma mark Misc

- (BOOL)isCompleteDoIgnoreAction:(BOOL)ignoreAction {
	if ((m_type == EVENT_NONE) || (!ignoreAction && (m_action == 0)))
		return FALSE;
	if (m_type == EVENT_KEY)
		return (m_key > 0);
	if (m_type == EVENT_BUTTON)
		return (m_button != -1);
	return ((m_command != nil) && ([m_command length] > 0));
}

- (BOOL)isEqualToInputEvent:(FFInputEvent*)otherEvent {
	if (m_type != [otherEvent type])
		return FALSE;
	if (m_type == EVENT_NONE)
		return TRUE;
	if (m_type == EVENT_SPEECH)
		return [m_command isEqualToString:[otherEvent command]];
	if (m_modifier != [otherEvent modifier])
		return FALSE;
	if (m_type == EVENT_KEY)
		return (m_key == [otherEvent key]);
	return (m_button == [otherEvent button]);
}

- (BOOL)isAssignedToScript {
	return (m_action == ScriptAction);
}

- (NSString*)_modifiersAsString {
	NSString* modStr = @"";
	
	if (m_modifier & NSCommandKeyMask)
		modStr = [modStr stringByAppendingString:CommandGlyph];
	if (m_modifier & NSAlternateKeyMask)
		modStr = [modStr stringByAppendingString:OptionGlpyh];
	if (m_modifier & NSShiftKeyMask)
		modStr = [modStr stringByAppendingString:ShiftGlyph];
	if (m_modifier & NSControlKeyMask)
		modStr = [modStr stringByAppendingString:ControlGlyph];
	if (m_modifier & NSNumericPadKeyMask)
		modStr = [modStr stringByAppendingString:FFTR(@"<numeric>")];
	
	if ([modStr length] > 0)
		return [modStr stringByAppendingString:@" + "];
	return modStr;
}

- (NSString*)asString {
	if (m_type == EVENT_KEY) {
		NSString*	keyStr = nil;
		
		// A function key
		if ((m_key >= NSUpArrowFunctionKey) && (m_key <= NSModeSwitchFunctionKey))
			keyStr = (NSString*)FunctionKeyNames[(m_key - NSUpArrowFunctionKey)];
		
		// Maybe some other known "special" key
		else if (m_key < 256)
			keyStr = OtherKnownKeyNames[m_key];
		
		// Ok something different
		//if (iswctype((wint_t)m_key, wctype("alnum")))		
		if (keyStr == nil)
			keyStr = [NSString stringWithFormat:@"%C", m_key];

		return [NSString stringWithFormat:@"%@%@", [self _modifiersAsString], keyStr];
	} else if (m_type == EVENT_BUTTON)
		return [NSString stringWithFormat:FFTRC(@"%@Button %d", @"0=Modifier, 1=Button no."),
				[self _modifiersAsString], m_button];
	else
		return [NSString stringWithFormat:FFTRC(@"Speech \"%@\"", @"0=Command"), m_command];
}

+ (NSArray*)specialKeyNames { // Dirty
	NSMutableArray* skn = [NSMutableArray arrayWithCapacity:NUM_FUNCTION_KEYS];
	
	int	i;
	for (i = 0; i < 256; i++)
		if (OtherKnownKeyNames[i] != NULL)
			[skn addObject:OtherKnownKeyNames[i]];
	
	[skn addObjectsFromArray:[NSArray arrayWithObjects:(id*)FunctionKeyNames count:(unsigned)NUM_FUNCTION_KEYS]];
	return skn;
}

+ (NSArray*)specialKeyKeys { // Dirty
	NSMutableArray* skk = [NSMutableArray arrayWithCapacity:NUM_FUNCTION_KEYS];
	int	i;
	
	for (i = 0; i < 256; i++)
		if (OtherKnownKeyNames[i] != NULL)
			[skk addObject:[NSNumber numberWithUnsignedShort:(unsigned short)i]];
	
	for (i = NSUpArrowFunctionKey; i <= NSModeSwitchFunctionKey; i++)
		[skk addObject:[NSNumber numberWithUnsignedShort:(unsigned short)i]];
	
	return skk;
}

#pragma mark -
#pragma mark The function keys

static const NSString* FunctionKeyNames[NUM_FUNCTION_KEYS] = {
	@"UpArrow",
	@"DownArrow",
	@"LeftArrow",
	@"RightArrow",
	@"F1",
	@"F2",
	@"F3",
	@"F4",
	@"F5",
	@"F6",
	@"F7",
	@"F8",
	@"F9",
	@"F10",
	@"F11",
	@"F12",
	@"F13",
	@"F14",
	@"F15",
	@"F16",
	@"F17",
	@"F18",
	@"F19",
	@"F20",
	@"F21",
	@"F22",
	@"F23",
	@"F24",
	@"F25",
	@"F26",
	@"F27",
	@"F28",
	@"F29",
	@"F30",
	@"F31",
	@"F32",
	@"F33",
	@"F34",
	@"F35",
	@"Insert",
	@"FwdDelete",
	@"Home",
	@"Begin",
	@"End",
	@"PageUp",
	@"PageDown",
	@"PrintScreen",
	@"ScrollLock",
	@"Pause",
	@"SysReq",
	@"Break",
	@"Reset",
	@"Stop",
	@"Menu",
	@"User",
	@"System",
	@"Print",
	@"ClearLine",
	@"ClearDisplay",
	@"InsertLine",
	@"DeleteLine",
	@"InsertChar",
	@"DeleteChar",
	@"Prev",
	@"Next",
	@"Select",
	@"Execute",
	@"Undo",
	@"Redo",
	@"Find",
	@"Help",
	@"ModeSwitch"
};

@end
